/***************************************************************************
                             th-title-selector.c
                             -------------------
    begin                : Mon Nov 08 2004
    copyright            : (C) 2004 by Tim-Philipp Mller
    email                : t.i.m@orange.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#include "th-device-pool.h"
#include "th-disc-drive-pool.h"
#include "th-job.h"
#include "th-title-preview.h"
#include "th-title-selector.h"
#include "th-utils.h"

#include <gtk/gtk.h>
#include <gst/gst.h>
#include <glib/gi18n.h>
#include <string.h>

typedef enum
{
	MSG_DETECTING_DRIVES,
	MSG_NO_DRIVES_DETECTED,
	MSG_INSERT_DVD,
	MSG_RETRIEVING_DETAILS,
	MSG_RETRIEVAL_FAILED,
	MSG_NONE
} TitleListMsg;

enum
{
	COL_TITLE_JOB = 0,
	COL_SELECTED
};

struct _ThTitleSelectorPrivate
{
	ThDiscDrivePool  *disc_pool;
	ThDevicePool     *dev_pool;

	ThDiscDrive      *active_drive;
	
	GtkListStore     *drive_list;
	GtkComboBox      *drive_combobox;
	gulong            drive_list_checkid;
	gulong            drive_detect_id;

	gboolean          detecting_drives; /* TRUE until we get a drive or nothing found from pool */

	GtkCellRenderer  *titles_togglecell;
	GtkListStore     *titles_list;
	GtkTreeView      *titles_view;
	GtkWidget        *titles_vbox;
	gulong            titles_check_id;

	GtkWidget        *ok_button;
	GtkWidget        *cancel_button;

	GtkWidget        *preview;

	TitleListMsg      cur_msg;
};

static void             title_selector_class_init       (ThTitleSelectorClass *klass);

static void             title_selector_instance_init    (ThTitleSelector *ts);

static void             title_selector_finalize         (GObject *object);

static void             title_selector_drive_added_cb   (ThTitleSelector *ts, 
                                                         ThDiscDrive     *drive, 
                                                         ThDiscDrivePool *pool);

static void             title_selector_disc_changed_cb  (ThTitleSelector *ts, 
                                                         ThDiscDrive     *drive, 
                                                         ThDiscDrivePool *pool);

static void             title_selector_drive_removed_cb (ThTitleSelector *ts,
                                                         ThDiscDrive     *drive, 
                                                         ThDiscDrivePool *pool);

static void             title_selector_no_drives_found   (ThTitleSelector *ts);

static void             title_selector_setup_drive_list  (ThTitleSelector *ts);

static void             title_selector_setup_titles_list (ThTitleSelector *ts);

static void             title_selector_on_drive_changed  (ThTitleSelector *ts,
                                                          GtkComboBox     *combobox);

static void             title_selector_update_ok_button  (ThTitleSelector *ts);

/* variables */

static GtkDialogClass  *ts_parent_class; /* NULL */

/***************************************************************************
 *
 *   title_selector_class_init
 *
 ***************************************************************************/

static void
title_selector_class_init (ThTitleSelectorClass *klass)
{
	GObjectClass  *object_class; 

	object_class = G_OBJECT_CLASS (klass);
	
	ts_parent_class = g_type_class_peek_parent (klass);

	object_class->finalize = title_selector_finalize;
}

/***************************************************************************
 *
 *   title_selector_detect_drives_delayed
 *
 *   idle timeout. detect drives only after window is set up and visible
 *
 ***************************************************************************/

static gboolean
title_selector_detect_drives_delayed (ThTitleSelector *ts)
{
	th_device_pool_add_tracker (ts->priv->dev_pool, TH_DEVICE_TRACKER (ts->priv->disc_pool));

	g_object_unref (ts->priv->disc_pool); /* device pool adds own reference */

	ts->priv->drive_detect_id = 0;
	
	return FALSE; /* only call once */
}

/***************************************************************************
 *
 *   title_selector_instance_init
 *
 ***************************************************************************/

static void
title_selector_instance_init (ThTitleSelector *ts)
{
	ts->priv = g_new0 (ThTitleSelectorPrivate, 1);

	ts->priv->dev_pool = th_device_pool_new ();
	
	ts->priv->disc_pool = th_disc_drive_pool_new ();

	g_signal_connect_swapped (ts->priv->disc_pool, 
	                          "drive-added", 
	                          G_CALLBACK (title_selector_drive_added_cb), 
	                          ts);

	g_signal_connect_swapped (ts->priv->disc_pool, 
	                          "disc-changed", 
	                          G_CALLBACK (title_selector_disc_changed_cb), 
	                          ts);

	g_signal_connect_swapped (ts->priv->disc_pool, 
	                          "drive-removed", 
	                          G_CALLBACK (title_selector_drive_removed_cb), 
	                          ts);

	g_signal_connect_swapped (ts->priv->disc_pool,
	                          "no-drive-found",
	                          G_CALLBACK (title_selector_no_drives_found),
	                          ts);

	/* we add the disc pool to the device tracker later
	 *  in _new(), because we want the list store and the
	 *  combo box to be set up when the first drive-added
	 *  signals start coming in */
	ts->priv->drive_detect_id = g_idle_add ((GSourceFunc) title_selector_detect_drives_delayed, ts);

	ts->priv->cancel_button = th_fake_dialog_add_button (TH_FAKE_DIALOG (ts), 
	                                                     GTK_STOCK_CANCEL, 
	                                                     GTK_RESPONSE_REJECT);

	/* makes cancel button stick to the left; let's see
	 *  how long it takes until some HIGian complains ;) */
	gtk_button_box_set_child_secondary (GTK_BUTTON_BOX (TH_FAKE_DIALOG(ts)->action_area),
	                                    ts->priv->cancel_button, 
	                                    TRUE);

	ts->priv->ok_button = th_fake_dialog_add_button (TH_FAKE_DIALOG (ts), 
	                                                 GTK_STOCK_OK, 
	                                                 GTK_RESPONSE_ACCEPT);

	gtk_widget_set_sensitive (ts->priv->ok_button, FALSE);
	
	ts->priv->cur_msg = MSG_NONE;

	/* set to FALSE as soon as we find a drive
	 *  or we know that there are no drives */
	ts->priv->detecting_drives = TRUE;

	gtk_widget_set_size_request (GTK_WIDGET (ts), -1, gdk_screen_height () / 2);
}

/***************************************************************************
 *
 *   title_selector_finalize
 *
 ***************************************************************************/

static void
title_selector_finalize (GObject *object)
{
	ThTitleSelector *ts;

	ts = (ThTitleSelector*) object;

	th_log ("ThTitleSelector: finalize\n");
	
	g_object_unref (ts->priv->dev_pool);

	g_object_unref (ts->priv->drive_list);
	g_object_unref (ts->priv->titles_list);

	if (ts->priv->drive_list_checkid > 0)
		g_source_remove (ts->priv->drive_list_checkid);
	
	if (ts->priv->titles_check_id > 0)
		g_source_remove (ts->priv->titles_check_id);
	
	if (ts->priv->drive_detect_id > 0)
		g_source_remove (ts->priv->drive_detect_id);

	memset (ts->priv, 0xab, sizeof (ThTitleSelectorPrivate));
	g_free (ts->priv);
	
	/* chain up */
	G_OBJECT_CLASS (ts_parent_class)->finalize (object);
}


/***************************************************************************
 *
 *   th_title_selector_get_type
 *
 ***************************************************************************/

GType
th_title_selector_get_type (void)
{
	static GType type; /* 0 */

	if (type == 0)
	{
		static GTypeInfo info =
		{
			sizeof (ThTitleSelectorClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) title_selector_class_init,
			NULL, NULL,
			sizeof (ThTitleSelector),
			0,
			(GInstanceInitFunc) title_selector_instance_init
		};

		type = g_type_register_static (TH_TYPE_FAKE_DIALOG, "ThTitleSelector", &info, 0);
	}

	return type;
}


/***************************************************************************
 *
 *   title_selector_response_cb
 *
 ***************************************************************************/

static void
title_selector_response_cb (ThTitleSelector *ts, gint r, GtkWidget *w)
{
	g_return_if_fail (TH_IS_TITLE_SELECTOR (ts));
	
	if (ts->priv->preview) {
		gtk_widget_destroy (ts->priv->preview);
		ts->priv->preview = NULL;
	}
}

/***************************************************************************
 *
 *   th_title_selector_new
 *
 ***************************************************************************/

GtkWidget *
th_title_selector_new (void)
{
	ThTitleSelector *ts;
	GtkWidget       *glade_window = NULL;
	GtkWidget       *toplevel_vbox;
	
	ts = (ThTitleSelector *) g_object_new (TH_TYPE_TITLE_SELECTOR, NULL);
	
	if (!th_utils_ui_load_interface ("th-title-selector.glade", 
	                                 FALSE,
	                                 "th-title-selector",  &glade_window,
	                                 "th-toplevel-vbox",   &toplevel_vbox,
	                                 "th-drive-combobox",  &ts->priv->drive_combobox,
	                                 "th-titles-vbox",     &ts->priv->titles_vbox,
	                                 "th-titles-treeview", &ts->priv->titles_view,
	                                 NULL))
	{
		g_warning ("th_utils_ui_load_interface (\"th-ui-title-selector.glade\") failed.\n");
		if (glade_window)
			gtk_widget_destroy (glade_window);
		gtk_widget_destroy (GTK_WIDGET (ts));
		return NULL;
	}
	
	g_object_ref (toplevel_vbox);
	gtk_container_remove (GTK_CONTAINER (glade_window), toplevel_vbox);
	gtk_container_add (GTK_CONTAINER (ts), toplevel_vbox);
	g_object_unref (toplevel_vbox);

	title_selector_setup_drive_list (ts);
	title_selector_setup_titles_list (ts);

	gtk_widget_show_all (ts->priv->titles_vbox);

	g_signal_connect_swapped (ts, "response", 
	                          G_CALLBACK (title_selector_response_cb),
	                          ts);

	return GTK_WIDGET (ts);
}

/***************************************************************************
 *
 *   title_selector_fill_titles_list
 *
 ***************************************************************************/

static void
title_selector_fill_titles_list (ThTitleSelector *ts)
{
	const GList *l;
	gboolean     is_main_feature;

	th_log ("Filling view... (active_drive = %p)\n", ts->priv->active_drive);
	
	gtk_list_store_clear (ts->priv->titles_list);
	
	g_object_set (ts->priv->titles_togglecell, "visible", TRUE, NULL);
	
	for (l = th_disc_drive_get_titles (ts->priv->active_drive);  l;  l = l->next)
	{
		GtkTreeIter  iter;
		ThJob       *job;
		
		job = TH_JOB (l->data);
	
		gtk_list_store_append (ts->priv->titles_list, &iter);
	
		/* pre-select main feature(s) */
		g_object_get (job, "is-main-title", &is_main_feature, NULL);
		
		gtk_list_store_set (ts->priv->titles_list, &iter, 
		                    COL_TITLE_JOB, job,
		                    COL_SELECTED, is_main_feature,
		                    -1);
	
		ts->priv->cur_msg = MSG_NONE;
	}

	title_selector_update_ok_button (ts);
}

/***************************************************************************
 *
 *   title_selector_set_titles_list_msg
 *
 ***************************************************************************/

static void
title_selector_set_titles_list_msg (ThTitleSelector *ts, TitleListMsg new_msg)
{
	GtkTreeIter iter;

	/* If we already have the same message, just return
	 *  (rather than having the tree view redraw the same
	 *  message multiple times per second) */
	if (gtk_tree_model_iter_n_children (GTK_TREE_MODEL (ts->priv->titles_list), NULL) == 1)
	{
		ThJob *job;
	
		gtk_tree_model_iter_nth_child (GTK_TREE_MODEL (ts->priv->titles_list), &iter, NULL, 0);

		gtk_tree_model_get (GTK_TREE_MODEL (ts->priv->titles_list), &iter,
		                    COL_TITLE_JOB, &job, 
		                    -1);

		if (job)
			g_object_unref (job);
		
		if (job == NULL  &&  ts->priv->cur_msg == new_msg)
			return;
	}

	ts->priv->cur_msg = new_msg;
	
	gtk_list_store_clear (ts->priv->titles_list);
	
	gtk_list_store_append (ts->priv->titles_list, &iter);

	gtk_list_store_set (ts->priv->titles_list, &iter, COL_TITLE_JOB, NULL, -1);
	
	g_object_set (ts->priv->titles_togglecell, "visible", FALSE, NULL);

	gtk_widget_set_sensitive (ts->priv->ok_button, TRUE);
}

/***************************************************************************
 *
 *   title_selector_ask_for_drive_device
 *
 *   Pops up a dialog and asks the user for the drive device 
 *
 ***************************************************************************/

static void
title_selector_ask_for_drive_device (ThTitleSelector *ts)
{
	ThDiscDrive *drive;
	const gchar *dev;
	GtkWidget   *dlg, *entry, *hbox, *label;
	GError      *err = NULL;
	gchar       *msg;
	
	msg = g_markup_printf_escaped ("<b>%s</b>\n\n%s\n", 
	                               _("Enter DVD drive device"),
	                               _("Could not find any DVD drives. Please\n"
	                                 "enter the device of your DVD drive\n"
	                                 "(e.g. /dev/hdc or /dev/sr1)"));
	
	dlg = gtk_message_dialog_new_with_markup (NULL, GTK_DIALOG_MODAL,
	                                          GTK_MESSAGE_QUESTION,
	                                          GTK_BUTTONS_OK_CANCEL,
	                                          msg);
	
	hbox = gtk_hbox_new (FALSE, 0);
	label = gtk_label_new (_("Device:"));
	entry = gtk_entry_new ();
	gtk_container_set_border_width (GTK_CONTAINER (hbox), 20);
	gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX (hbox), entry, FALSE, FALSE, 12);
	gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dlg)->vbox), hbox, FALSE, FALSE, 2);
	gtk_widget_show_all (hbox);

askfordevice:

	/* run dialog until we get a valid-looking device
	 *  string, or until the user hits Cancel or Delete */
	do
	{
		if (gtk_dialog_run (GTK_DIALOG (dlg)) != GTK_RESPONSE_OK)
			goto done;
	
		dev = gtk_entry_get_text (GTK_ENTRY (entry));
	}
	while (dev == NULL || !g_path_is_absolute (dev));

	if (!th_utils_device_might_be_dvd_drive (dev, &err))
	{
		GtkWidget *err_dlg;
		
		g_free (msg);
	
		msg = g_markup_printf_escaped ("<b>%s</b>\n\n%s\n", 
		                               _("Device invalid or not accessible"),
		                               err->message);
		
		err_dlg = gtk_message_dialog_new_with_markup (GTK_WINDOW (dlg), 
		                                              GTK_DIALOG_MODAL,
		                                              GTK_MESSAGE_ERROR,
		                                              GTK_BUTTONS_CLOSE,
		                                              msg);

		(void) gtk_dialog_run (GTK_DIALOG (err_dlg));
		
		gtk_widget_destroy (err_dlg);
		g_error_free (err);
		err = NULL;
		
		goto askfordevice;
	}
	
	drive = th_disc_drive_new (dev, dev, _("Possible DVD Drive"), " ");

	if (th_disc_drive_pool_add_drive (ts->priv->disc_pool, drive)) {
	  g_object_set (drive, 
	                "poll-media-required", TRUE, 
	                "poll-media", TRUE, 
	                NULL);
	}

	g_object_unref (drive); /* list store added ref */

	title_selector_set_titles_list_msg (ts, MSG_DETECTING_DRIVES);

done:

	gtk_widget_destroy (dlg);
	g_free (msg);
}

/***************************************************************************
 *
 *   title_selector_titles_check_cb
 *
 *   Checks whether we need to fill the titles list, or 
 *     display a progess message etc. Assumes we are called
 *     frequently enough in order to not miss an empty drive
 *     when you DVDs are changed (e.g. at least 1/sec)
 *
 ***************************************************************************/

static gboolean
title_selector_titles_check_cb (ThTitleSelector *ts)
{
	gint num;

	/* If no drive selected, clear titles list if it has rows */
	if (ts->priv->active_drive == NULL)
	{
		gtk_widget_set_sensitive (ts->priv->ok_button, FALSE);
		
		if (ts->priv->detecting_drives)
		{
			title_selector_set_titles_list_msg (ts, MSG_DETECTING_DRIVES);
			return TRUE; /* call again */
		}
		
		if (ts->priv->cur_msg == MSG_NO_DRIVES_DETECTED)
			return TRUE; /* call again */
		
		title_selector_set_titles_list_msg (ts, MSG_NO_DRIVES_DETECTED);

		title_selector_ask_for_drive_device (ts);
		
		return TRUE; /* call again */
	}

	g_object_get (ts->priv->active_drive, "num-titles", &num, NULL);
	
	switch (num)
	{
		case -2:
			title_selector_set_titles_list_msg (ts, MSG_INSERT_DVD);
			gtk_widget_set_sensitive (ts->priv->ok_button, FALSE);
			break;

		case -1:
			title_selector_set_titles_list_msg (ts, MSG_RETRIEVING_DETAILS);
			gtk_widget_set_sensitive (ts->priv->ok_button, FALSE);
			break;

		case 0:
			title_selector_set_titles_list_msg (ts, MSG_RETRIEVAL_FAILED);
			gtk_widget_set_sensitive (ts->priv->ok_button, FALSE);
			break;
		
		default:
			if (gtk_tree_model_iter_n_children (GTK_TREE_MODEL (ts->priv->titles_list), NULL) != num
			 || ts->priv->cur_msg != MSG_NONE) /* second condition in case disc has only one title */
				title_selector_fill_titles_list (ts);
			break;
	}

	return TRUE; /* call again */
}

/***************************************************************************
 *
 *   title_selector_preview_title
 *
 ***************************************************************************/

static void
title_selector_preview_title (ThTitleSelector *ts, ThJob *job)
{
	GError *err = NULL;
	gchar  *device, *title_tag;
	guint   title_num;

	g_return_if_fail (TH_IS_JOB (job));

	g_object_get (job, 
	              "title-num", &title_num, 
	              "title-tag", &title_tag, 
	              NULL);

	g_object_get (ts->priv->active_drive, "device", &device, NULL);


	if (ts->priv->preview) {
		gtk_widget_destroy (ts->priv->preview);
		ts->priv->preview = NULL;
	}

	ts->priv->preview =
	    th_title_preview_new (device, title_tag, title_num + 1, &err);

	g_free (title_tag);
	g_free (device);

	if (err != NULL || ts->priv->preview == NULL) {
		GtkWidget *dlg, *appwin;

		appwin = gtk_widget_get_toplevel (GTK_WIDGET (ts));
		
		dlg = gtk_message_dialog_new (GTK_WINDOW (appwin),
		                              GTK_DIALOG_MODAL,
		                              GTK_MESSAGE_ERROR,
		                              GTK_BUTTONS_OK,
		                              _("Can't preview DVD title %d"),
		                              title_num + 1);

		gtk_message_dialog_format_secondary_text (GTK_MESSAGE_DIALOG (dlg),
		    "%s", (err) ? err->message : _("unknown error"));

		(void) gtk_dialog_run (GTK_DIALOG (dlg));
		gtk_widget_destroy (dlg);
		if (err)                             
			g_error_free (err);
		return;
	}

	/* make sure ts->priv->preview is automatically set to NULL if it
	 * gets destroyed other than by us (e.g. via close button) */
	{
		GtkWidget **widget_addr = &ts->priv->preview;

		g_object_add_weak_pointer (G_OBJECT (ts->priv->preview),
		                           (gpointer *) widget_addr);
	}

	gtk_widget_show (ts->priv->preview);
}

/***************************************************************************
 *
 *   title_selector_title_preview
 *
 ***************************************************************************/

static void
title_selector_title_preview (ThTitleSelector *ts, GtkWidget *menuitem)
{
	GtkTreeSelection *sel;
	GtkTreeModel     *model;
	GtkTreePath      *path;
	GtkTreeIter       iter;
	GList            *list;
	
	sel = gtk_tree_view_get_selection (ts->priv->titles_view);
	list = gtk_tree_selection_get_selected_rows (sel, NULL);
	g_return_if_fail (list != NULL);
	
	path = (GtkTreePath*) list->data;
	model = GTK_TREE_MODEL (ts->priv->titles_list);
	
	if (gtk_tree_model_get_iter (model, &iter, path))
	{
		ThJob *job;
	
		gtk_tree_model_get (model, &iter, COL_TITLE_JOB, &job, -1);

		if (TH_IS_JOB (job))
		{
			title_selector_preview_title (ts, job);
			g_object_unref (job);
		}
	}
	
	gtk_tree_path_free (path);
	g_list_free (list);
}

/***************************************************************************
 *
 *   title_selector_popup_context_menu
 *
 ***************************************************************************/

static void
title_selector_popup_context_menu (ThTitleSelector *ts, GdkEventButton *event)
{
	GtkTreeSelection  *sel;
	GtkTreeIter        iter;
	GtkWidget         *menu;
	GtkWidget         *menuitem;

	menu = gtk_menu_new ();
	
	menuitem = gtk_menu_item_new_with_label (_("Preview"));
	
	g_signal_connect_swapped (menuitem, "activate", 
	                          G_CALLBACK (title_selector_title_preview),
	                          ts);

	sel = gtk_tree_view_get_selection (ts->priv->titles_view);
	
	if (!gtk_tree_selection_get_selected (sel, NULL, &iter)
	  || ts->priv->preview != NULL)
	{
		gtk_widget_set_sensitive (menuitem, FALSE);
	}
	else
	{
		ThJob *job;

		gtk_tree_model_get (GTK_TREE_MODEL (ts->priv->titles_list), &iter,
		                    COL_TITLE_JOB, &job, 
		                    -1);

		/* It's a status/error mesage, not a title */
		if (job == NULL)
		{
			gtk_widget_destroy (menu);
			gtk_widget_destroy (menuitem);
			return;
		}
		
		gtk_widget_set_sensitive (menuitem, TRUE);
		
		g_object_unref (job);
	}

	gtk_menu_shell_append (GTK_MENU_SHELL (menu), menuitem);
	
	gtk_widget_show_all (menu);
	
	gtk_menu_popup (GTK_MENU (menu), NULL, NULL, NULL, NULL,
	                (event) ? event->button : 0,
	                gdk_event_get_time ((GdkEvent*)event));
}

/***************************************************************************
 *
 *   title_selector_title_popup_menu
 *
 *   Callback for Shift-F10, the keyboard accelerator for pop-up menus
 *
 ***************************************************************************/

static gboolean
title_selector_title_popup_menu (ThTitleSelector *ts, GtkTreeView *view)
{
	title_selector_popup_context_menu (ts, NULL);
	return TRUE; /* we handled this */
}

/***************************************************************************
 *
 *   title_selector_title_button_press
 *
 ***************************************************************************/

static gboolean
title_selector_title_button_press (ThTitleSelector *ts, GdkEventButton *event, GtkTreeView *view)
{
	GtkTreeSelection *sel;
	GtkTreePath      *path;

	if (event->type != GDK_BUTTON_PRESS || event->button != 3)
		return FALSE; /* not handled */

	sel = gtk_tree_view_get_selection (ts->priv->titles_view);
	if (gtk_tree_view_get_path_at_pos (ts->priv->titles_view, event->x, event->y, &path, NULL, NULL, NULL))
	{
		gtk_tree_selection_select_path (sel, path);
		gtk_tree_path_free (path);
	}
	else
	{
		gtk_tree_selection_unselect_all (sel);
	}
	
	title_selector_popup_context_menu (ts, event);

	return TRUE; /* we handled this */
}

/***************************************************************************
 *
 *   title_selector_update_ok_button
 *
 ***************************************************************************/

static void
title_selector_update_ok_button (ThTitleSelector *ts)
{
	GtkTreeModel *model;
	GtkTreeIter   iter;
	gboolean      active, ok;
	guint         num_selected;
	
	/* count number of selected titles */
	num_selected = 0;
	model = GTK_TREE_MODEL (ts->priv->titles_list);
	
	ok = gtk_tree_model_get_iter_first (model, &iter);
	while (ok)
	{
		gtk_tree_model_get (model, &iter, COL_SELECTED, &active, -1);
		
		if (active)
			++num_selected;

		ok = gtk_tree_model_iter_next (model, &iter);
	}
	
	if (num_selected > 0)
		gtk_widget_set_sensitive (ts->priv->ok_button, TRUE);
	else
		gtk_widget_set_sensitive (ts->priv->ok_button, FALSE);
}

/***************************************************************************
 *
 *   title_selector_title_toggled
 *
 ***************************************************************************/

static void
title_selector_title_toggled (ThTitleSelector       *ts,
                              const gchar           *pathstr,
                              GtkCellRendererToggle *cell_not_used)
{
	GtkTreeModel *model;
	GtkTreePath  *path;
	GtkTreeIter   iter;
	gboolean      active;
	
	path = gtk_tree_path_new_from_string (pathstr);
	model = GTK_TREE_MODEL (ts->priv->titles_list);
	
	if (gtk_tree_model_get_iter (model, &iter, path))
	{
		gtk_tree_model_get (model, &iter, COL_SELECTED, &active, -1);
		
		gtk_list_store_set (ts->priv->titles_list, &iter, COL_SELECTED, !active, -1);
	}

	gtk_tree_path_free (path);

	title_selector_update_ok_button (ts);
}

/***************************************************************************
 *
 *   ts_title_cell_edited
 *
 ***************************************************************************/

static void
ts_title_cell_edited (ThTitleSelector *ts, 
                      const gchar     *pathstr, 
                      const gchar     *newstr, 
                      gpointer         cell)
{
	GtkTreeModel *model;
	GtkTreeIter   iter;
	ThJob        *job;
	
	model = GTK_TREE_MODEL (ts->priv->titles_list);
	if (!gtk_tree_model_get_iter_from_string (model, &iter, pathstr))
		g_return_if_reached ();
		
	gtk_tree_model_get (model, &iter, COL_TITLE_JOB, &job, -1);

	if (job == NULL || newstr == NULL || *newstr == 0x00)
		return;

	g_object_set (job, "title-tag", newstr, NULL);
	th_job_save_config (job);

	g_object_unref (job);
}

/***************************************************************************
 *
 *   ts_title_cell_data_func
 *
 ***************************************************************************/

static void
ts_title_cell_data_func (GtkTreeViewColumn  *col,
                         GtkCellRenderer    *cell,
                         GtkTreeModel       *model,
                         GtkTreeIter        *iter,
                         gpointer            ts)
{
	const gchar *s;
	ThJob       *job;
	gchar       *markup;

	gtk_tree_model_get (model, iter, COL_TITLE_JOB, &job, -1);
	
	if (job)
	{
		gchar *title_tag;
	
		g_object_get (job, "title-tag", &title_tag, NULL);
		g_object_set (cell, "text", title_tag, "editable", TRUE, NULL);
	
		g_free (title_tag);
		g_object_unref (job);
		return;
	}
	
	/* job == NULL => show the current error/warning message */
	switch (TH_TITLE_SELECTOR (ts)->priv->cur_msg)
	{
		case MSG_DETECTING_DRIVES:
			s = _("Detecting DVD drives...");
			break;
		case MSG_NO_DRIVES_DETECTED:
			s = _("No DVD drives detected.\n\n"
			      "Possibly you do not have the Hardware Abstraction\n"
			      "Layer (HAL) package installed, or the hald daemon\n"
			      "is not running. HAL may require a fairly recent\n"
			      "kernel on some operating systems (e.g. 2.6.x or\n"
			      "newer on Linux systems)\n"
			      "On Linux 2.4 kernels, only IDE/SCSI drives will be\n"
			      "detected (ie. USB drives will not be found).\n"
			      "For other operating systems, DVD drive detection\n"
			      "is not implemented yet, sorry.");
			break;
		case MSG_INSERT_DVD:
			s = _("Please insert a DVD into the drive");
			break;
		case MSG_RETRIEVING_DETAILS:
			s = _("Retrieving DVD title details ...");
			break;
		case MSG_RETRIEVAL_FAILED:
			s = _("Failed to retrieve DVD title details for some reason.\n"
			      "Maybe you do not have libdvdcss2 installed?");
			break;
		default:
			s = NULL;
			g_object_set (cell, "markup", "<b>[FIXME]</b>", NULL);
			g_return_if_reached ();
	}

	markup = g_markup_printf_escaped ("<i>%s</i>", s);
	g_object_set (cell, "markup", markup, "editable", FALSE, NULL);
	g_free (markup);
}

/***************************************************************************
 *
 *   ts_title_len_unit_cell_data_func
 *
 ***************************************************************************/

static void
ts_title_len_unit_cell_data_func (GtkTreeViewColumn  *col,
                                  GtkCellRenderer    *cell,
                                  GtkTreeModel       *model,
                                  GtkTreeIter        *iter,
                                  gpointer            ts)
{
	const gchar *txt = "";
	ThJob      *job;

	gtk_tree_model_get (model, iter, COL_TITLE_JOB, &job, -1);
	
	if (job)
	{
		guint len_secs;
	
		g_object_get (job, "title-length", &len_secs, NULL);

		if (len_secs >= 3600)
			txt = _("hours");
		else if (len_secs >= 60)
			txt = _("mins");
		else
			txt = _("secs");

		g_object_unref (job);
	}

	g_object_set (cell, "text", txt, "scale", (gdouble) PANGO_SCALE_SMALL, NULL);
}

/***************************************************************************
 *
 *   ts_title_len_cell_data_func
 *
 ***************************************************************************/

static void
ts_title_len_cell_data_func (GtkTreeViewColumn  *col,
                             GtkCellRenderer    *cell,
                             GtkTreeModel       *model,
                             GtkTreeIter        *iter,
                             gpointer            ts)
{
	ThJob *job;
	gchar  txt[64];

	gtk_tree_model_get (model, iter, COL_TITLE_JOB, &job, -1);

	txt[0] = 0x00;

	if (job)
	{
		guint len_secs, hours, mins, secs;
	
		g_object_get (job, "title-length", &len_secs, NULL);

		hours = len_secs / 3600;
		mins = (len_secs % 3600) / 60;
		secs = len_secs % 60;

		if (hours == 0)
		{
			if (mins == 0)
				g_snprintf (txt, sizeof(txt), "%u", secs);
			else
				g_snprintf (txt, sizeof(txt), "%u:%02u", mins, secs);
		}
		else
		{
			g_snprintf (txt, sizeof(txt), "%u:%02u", hours, mins);
		}
		
		g_object_unref (job);
	}
	
	g_object_set (cell, "text", txt, "scale", (gdouble) PANGO_SCALE_SMALL, NULL);
}

/***************************************************************************
 *
 *   title_selector_setup_titles_list
 *
 ***************************************************************************/

static void
title_selector_setup_titles_list (ThTitleSelector *ts)
{
	GtkTreeViewColumn *col;
	GtkTreeSelection  *sel;
	GtkCellRenderer   *cell;

	ts->priv->titles_list = gtk_list_store_new (2, TH_TYPE_JOB, G_TYPE_BOOLEAN);
	
	col = gtk_tree_view_column_new ();
	gtk_tree_view_column_set_sizing (col, GTK_TREE_VIEW_COLUMN_AUTOSIZE);
	gtk_tree_view_column_set_expand (col, FALSE);

	/* toggle cell renderer */
	cell = gtk_cell_renderer_toggle_new ();
	ts->priv->titles_togglecell = cell;

	g_object_set (cell, "activatable", TRUE, NULL);
	
	gtk_tree_view_column_pack_start (col, cell, FALSE);
	
	gtk_tree_view_column_set_attributes (col, cell, 
	                                     "active", COL_SELECTED,
	                                     NULL);

	g_signal_connect_swapped (cell, "toggled", 
	                          G_CALLBACK (title_selector_title_toggled), 
	                          ts);

	gtk_tree_view_append_column (ts->priv->titles_view, col);

	/* title text cell renderer (set editable on a per-row basis) */
	col = gtk_tree_view_column_new ();
	gtk_tree_view_column_set_sizing (col, GTK_TREE_VIEW_COLUMN_AUTOSIZE);
	gtk_tree_view_column_set_expand (col, TRUE);

	cell = gtk_cell_renderer_text_new ();
	
	g_signal_connect_swapped (cell, "edited",
	                          G_CALLBACK (ts_title_cell_edited),
	                          ts);
	
	gtk_tree_view_column_pack_start (col, cell, TRUE);

	gtk_tree_view_column_set_cell_data_func (col, cell, ts_title_cell_data_func, ts, NULL);

	gtk_tree_view_append_column (ts->priv->titles_view, col);
	
	/* length text cell renderer */
	col = gtk_tree_view_column_new ();
	gtk_tree_view_column_set_sizing (col, GTK_TREE_VIEW_COLUMN_AUTOSIZE);
	gtk_tree_view_column_set_expand (col, FALSE);
	
	cell = gtk_cell_renderer_text_new ();
	g_object_set (cell, "xalign", (gdouble) 1.0, NULL);
	gtk_tree_view_column_pack_start (col, cell, TRUE);
	gtk_tree_view_column_set_cell_data_func (col, cell, ts_title_len_cell_data_func, ts, NULL);

	cell = gtk_cell_renderer_text_new ();
	g_object_set (cell, "xalign", (gdouble) 0.0, NULL);
	gtk_tree_view_column_pack_start (col, cell, FALSE);
	gtk_tree_view_column_set_cell_data_func (col, cell, ts_title_len_unit_cell_data_func, ts, NULL);

	gtk_tree_view_append_column (ts->priv->titles_view, col);

	gtk_tree_view_set_model (ts->priv->titles_view, GTK_TREE_MODEL (ts->priv->titles_list));

	ts->priv->titles_check_id = g_timeout_add (333, (GSourceFunc) title_selector_titles_check_cb, ts);

	sel = gtk_tree_view_get_selection (ts->priv->titles_view);
	gtk_tree_selection_set_mode (sel, GTK_SELECTION_SINGLE);
	
	g_signal_connect_swapped (ts->priv->titles_view, "popup-menu", 
	                          G_CALLBACK (title_selector_title_popup_menu), 
	                          ts);

	g_signal_connect_swapped (ts->priv->titles_view, "button-press-event", 
	                          G_CALLBACK (title_selector_title_button_press), 
	                          ts);
}

/***************************************************************************
 *
 *   title_selector_drive_list_check
 *
 *   Called regularly to check if we have drives in the list. If not,
 *     we add a row with a NULL pointer as drive, so we can render a 
 *     'no DVD drives found' row
 *
 ***************************************************************************/

static gboolean
title_selector_drive_list_check (ThTitleSelector *ts)
{
	GtkTreeIter iter;

	if (gtk_tree_model_iter_n_children (GTK_TREE_MODEL (ts->priv->drive_list), NULL) == 0)
	{
		gtk_list_store_append (ts->priv->drive_list, &iter);
		gtk_list_store_set (ts->priv->drive_list, &iter, 0, NULL, -1);
		gtk_combo_box_set_active_iter (ts->priv->drive_combobox, &iter);
	
		gtk_list_store_clear (ts->priv->titles_list);
	}

	return TRUE; /* call again */
}

/***************************************************************************
 *
 *   title_selector_drive_data_func
 *
 ***************************************************************************/

static void
title_selector_drive_data_func (GtkCellLayout   *cl, 
                                GtkCellRenderer *cell,
                                GtkTreeModel    *model,
                                GtkTreeIter     *iter,
                                gpointer         foo)
{
	ThDiscDrive *drive;
	gchar       *title, *driveid, *markup;
	gchar       *disc_title, *vendor, *product;
	
	gtk_tree_model_get (model, iter, 0, &drive, -1);
	
	if (drive == NULL)
	{
		gchar *s = g_strdup_printf ("<i>%s</i>", _("No DVD drives detected."));
		g_object_set (cell, "markup", s, NULL);
		g_free (s);
		return;
	}
	
	g_object_get (drive, 
	              "disc-title", &disc_title, 
	              "product", &product, 
	              "vendor", &vendor, 
	              NULL);

	if (disc_title == NULL)
		title = g_strdup_printf ("<i>%s</i>", _("(no DVD in drive)"));
	else
		title = g_strdup_printf ("<b>%s</b>", disc_title);
	
	driveid = g_strdup_printf ("<span size='small'>%s %s</span>", vendor, product);

	markup = g_strdup_printf ("%s\n%s", title, driveid);
	
	g_object_set (cell, "markup", markup, NULL);
	
	g_free (markup);
	g_free (driveid);
	g_free (vendor);
	g_free (product);
	g_free (title);
	g_free (disc_title);

	g_object_unref (drive);
}

/***************************************************************************
 *
 *   title_selector_on_drive_changed
 *
 *   Called when the drive combo box selection changes 
 *
 ***************************************************************************/

static void
title_selector_on_drive_changed (ThTitleSelector *ts, GtkComboBox *combobox)
{
	ThDiscDrive **drive_addr;
	GtkTreeIter  iter;
	gint         row;
	
	g_return_if_fail (TH_IS_TITLE_SELECTOR (ts));
	g_return_if_fail (GTK_IS_COMBO_BOX (combobox));

	drive_addr = &ts->priv->active_drive;

	g_object_get (combobox, "active", &row, NULL);
	
	if (ts->priv->active_drive)
	{
		g_object_remove_weak_pointer (G_OBJECT (ts->priv->active_drive),
		                              (gpointer *) drive_addr);
		ts->priv->active_drive = NULL;
	}
	
	if (row < 0)
		return; /* nothing selected */
	
	if (!gtk_tree_model_iter_nth_child (GTK_TREE_MODEL (ts->priv->drive_list), &iter, NULL, row))
		g_return_if_reached ();

	gtk_tree_model_get (GTK_TREE_MODEL (ts->priv->drive_list), &iter, 
	                    0, &ts->priv->active_drive,
	                    -1);

	/* Might be NULL if the 'no drives detected' row shows */
	if (ts->priv->active_drive == NULL)
		return;

	/* ref added by gtk_tree_model_get() */
	g_object_unref (ts->priv->active_drive);

	g_object_add_weak_pointer (G_OBJECT (ts->priv->active_drive),
	                           (gpointer *) drive_addr);

	th_log ("active drive now: %p\n", ts->priv->active_drive);
	ts->priv->cur_msg = MSG_RETRIEVING_DETAILS;
}

/***************************************************************************
 *
 *   title_selector_setup_drive_list
 *
 ***************************************************************************/

static void
title_selector_setup_drive_list (ThTitleSelector *ts)
{
	GtkCellRenderer *cell;

	ts->priv->drive_list = gtk_list_store_new (1, TH_TYPE_DISC_DRIVE);
	
	cell = gtk_cell_renderer_text_new ();
	
	gtk_cell_layout_pack_start (GTK_CELL_LAYOUT (ts->priv->drive_combobox), cell, TRUE);
	
	gtk_cell_layout_set_cell_data_func (GTK_CELL_LAYOUT (ts->priv->drive_combobox),
	                                    cell, title_selector_drive_data_func, NULL, NULL);

	gtk_combo_box_set_model (ts->priv->drive_combobox, GTK_TREE_MODEL (ts->priv->drive_list));

	ts->priv->drive_list_checkid = g_timeout_add (1000, (GSourceFunc) title_selector_drive_list_check, ts);

	g_signal_connect_swapped (ts->priv->drive_combobox,
	                          "changed",
	                          G_CALLBACK (title_selector_on_drive_changed),
	                          ts);
}



/***************************************************************************
 *
 *   title_selector_drive_added_cb
 *
 ***************************************************************************/

static void
title_selector_drive_added_cb (ThTitleSelector *ts, 
                               ThDiscDrive     *drive, 
                               ThDiscDrivePool *pool)
{
	GtkTreeIter iter;
	
	/* g_print ("DVD DRIVE ADDED: %s\n", drive->udi); */

	/* first drive? clear 'detecting' message */
	if (ts->priv->detecting_drives)
		gtk_list_store_clear (ts->priv->drive_list);
	
	gtk_list_store_append (ts->priv->drive_list, &iter);
	
	gtk_list_store_set (ts->priv->drive_list, &iter, 0, drive, -1);
	
	gtk_combo_box_set_active_iter (ts->priv->drive_combobox, &iter);
	
	drive->iter = iter;

	ts->priv->detecting_drives = FALSE;
	ts->priv->cur_msg = MSG_NONE;
}

/***************************************************************************
 *
 *   title_selector_no_drives_found
 *
 ***************************************************************************/

static void
title_selector_no_drives_found (ThTitleSelector *ts)
{
	ts->priv->detecting_drives = FALSE;
	th_log ("========== no drives found =============\n");
}

/***************************************************************************
 *
 *   title_selector_disc_changed_cb
 *
 ***************************************************************************/

static void
title_selector_disc_changed_cb (ThTitleSelector *ts, 
                                ThDiscDrive     *drive, 
                                ThDiscDrivePool *pool)
{
	GtkTreePath *path;
	gchar       *disc_label;
	
	g_object_get (drive, "disc-title", &disc_label, NULL);

	path = gtk_tree_model_get_path (GTK_TREE_MODEL (ts->priv->drive_list), &drive->iter);
	gtk_tree_model_row_changed (GTK_TREE_MODEL (ts->priv->drive_list), path, &drive->iter);
	gtk_tree_path_free (path);

	if (disc_label)
	{
		gtk_combo_box_set_active_iter (ts->priv->drive_combobox, &drive->iter);
		th_log ("DVD DISC INSERTED: %s\n", disc_label);
	}
	else
	{
		th_log ("DVD DISC REMOVED.\n");
	}
	
	g_free (disc_label);
}

/***************************************************************************
 *
 *   title_selector_drive_removed_cb
 *
 ***************************************************************************/

static void
title_selector_drive_removed_cb (ThTitleSelector *ts,
                                 ThDiscDrive     *drive, 
                                 ThDiscDrivePool *pool)
{
	gtk_list_store_remove (ts->priv->drive_list, &drive->iter);
}

/***************************************************************************
 *
 *   title_selector_add_job_if_selected
 *
 ***************************************************************************/

static GList *
title_selector_add_job_if_selected (ThTitleSelector *ts, GtkTreeIter *iter, GList *job_list)
{
	gboolean   title_selected;
	GList     *ret;
	gchar     *disc_title, *fn;
	ThJob     *job;

	gtk_tree_model_get (GTK_TREE_MODEL (ts->priv->titles_list), iter, 
	                    COL_SELECTED, &title_selected, 
	                    -1);

	if (title_selected == FALSE)
		return job_list;

	gtk_tree_model_get (GTK_TREE_MODEL (ts->priv->titles_list), iter, 
	                    COL_TITLE_JOB, &job,
	                    -1);

	g_return_val_if_fail (job != NULL, job_list);

	g_object_get (ts->priv->active_drive, "disc-title", &disc_title, NULL);

	g_object_get (job, "output-fn", &fn, NULL);

	if (fn == NULL)
	{
		th_job_set_filename_from_disc_title (job, disc_title);
	}
	
	ret = g_list_append (job_list, job);
	g_object_ref (job); /* for list */

	g_free (disc_title);
	g_free (fn);
	g_object_unref (job); /* ref added by gtk_tree_model_get() */

	return ret;
}

/***************************************************************************
 *
 *   title_selector_get_selected_jobs
 *
 ***************************************************************************/

static GList *
title_selector_get_selected_jobs (ThTitleSelector *ts)
{
	GtkTreeModel *model;
	GtkTreeIter   iter;
	gboolean      rowexists;
	GList        *jobs = NULL;
		
	model = GTK_TREE_MODEL (ts->priv->titles_list);
		
	rowexists = gtk_tree_model_get_iter_first (model, &iter);
	while (rowexists)
	{
		jobs = title_selector_add_job_if_selected (ts, &iter, jobs);

		rowexists = gtk_tree_model_iter_next (model, &iter);
	}
	
	return jobs;
}

/***************************************************************************
 *
 *   title_selector_set_drive_polling_active
 *
 *   tells drives that they don't need to poll for media changes
 *    for the time being, or make them start polling again.
 *
 ***************************************************************************/

static void
title_selector_set_drive_polling_active (ThTitleSelector *ts, gboolean do_poll)
{
	GtkTreeModel *model;
	GtkTreeIter   iter;
	gboolean      rowexists, polling_required;
		
	model = GTK_TREE_MODEL (ts->priv->drive_list);
		
	rowexists = gtk_tree_model_get_iter_first (model, &iter);
	while (rowexists)
	{
		ThDiscDrive *drive;

		gtk_tree_model_get (model, &iter, 0, &drive, -1);

		g_assert (drive == NULL || TH_IS_DISC_DRIVE (drive));

		if (drive)
		{
			g_object_unref (drive); /* ref added by gtk_tree_model_get() */

			g_object_get (drive, "poll-media-required", &polling_required, NULL);

			if (polling_required)
				g_object_set (drive, "poll-media", do_poll, NULL);
		}

		rowexists = gtk_tree_model_iter_next (model, &iter);
	}
}

/***************************************************************************
 *
 *   th_title_selector_run
 *
 ***************************************************************************/

GList *
th_title_selector_run (ThTitleSelector *ts)
{
	GList *jobs = NULL;
	gint   ret;

	g_return_val_if_fail (TH_IS_TITLE_SELECTOR (ts), NULL);
	
	title_selector_set_drive_polling_active (ts, TRUE);
	
	ret = th_fake_dialog_run (TH_FAKE_DIALOG (ts));

	if (ret == TH_RESPONSE_DESTROYED)
		return NULL;

	if (ret == GTK_RESPONSE_ACCEPT)
		jobs = title_selector_get_selected_jobs (ts);

	title_selector_set_drive_polling_active (ts, FALSE);
	
	return jobs;
}

/***************************************************************************
 *
 *   th_title_selector_add_vob_folder
 *
 ***************************************************************************/

gboolean
th_title_selector_add_vob_folder (ThTitleSelector *ts, const gchar *dir)
{
  ThDiscDrive *drive;

  g_return_val_if_fail (TH_IS_TITLE_SELECTOR (ts), FALSE);
  g_return_val_if_fail (dir != NULL, FALSE);
  g_return_val_if_fail (g_path_is_absolute (dir), FALSE);

  drive = th_disc_drive_pool_get_drive_from_udi (ts->priv->disc_pool, dir);

  if (drive == NULL) {
    /* FIXME: find nice string and make translatable */
    drive = th_disc_drive_new (dir, dir, "Directory with .VOB Files", "");
    g_return_val_if_fail (drive != NULL, FALSE);

    g_object_set (drive, "disc-title", dir, NULL);

    th_disc_drive_pool_add_drive (ts->priv->disc_pool, drive);

    /* the title selector list store holds a reference now */
    g_object_unref (drive);
  }

  g_signal_emit_by_name (ts->priv->disc_pool, "disc-changed", drive);

  return TRUE;
}

/***************************************************************************
 *
 *   th_title_selector_add_iso_file
 *
 ***************************************************************************/

gboolean
th_title_selector_add_iso_file (ThTitleSelector *ts, const gchar *iso_fn)
{
  ThDiscDrive *drive;

  g_return_val_if_fail (TH_IS_TITLE_SELECTOR (ts), FALSE);
  g_return_val_if_fail (iso_fn != NULL, FALSE);
  g_return_val_if_fail (g_path_is_absolute (iso_fn), FALSE);

  drive = th_disc_drive_pool_get_drive_from_udi (ts->priv->disc_pool, iso_fn);

  if (drive == NULL) {
    gchar *base_name;

    base_name = g_path_get_basename (iso_fn);

    /* FIXME: find nice string and make translatable */
    drive = th_disc_drive_new (iso_fn, iso_fn, "ISO image", "");

    g_return_val_if_fail (drive != NULL, FALSE);

    g_object_set (drive, "disc-title", base_name, NULL);

    th_disc_drive_pool_add_drive (ts->priv->disc_pool, drive);

    /* the title selector list store holds a reference now */
    g_object_unref (drive);
    g_free (base_name);
  }

  g_signal_emit_by_name (ts->priv->disc_pool, "disc-changed", drive);

  return TRUE;
}

