/* -*- Mode: C; tab-width: 5; indent-tabs-mode: t; c-basic-offset: 5 -*- */

#include <string.h>
#include <unistd.h>
#include "nm-gsm-modem.h"
#include "nm-device-private.h"
#include "nm-device-interface.h"
#include "nm-setting-connection.h"
#include "nm-setting-gsm.h"
#include "nm-modem-types.h"
#include "nm-utils.h"

#include "nm-gsm-device-glue.h"

G_DEFINE_TYPE (NMGsmModem, nm_gsm_modem, NM_TYPE_MODEM_DEVICE)

#define NM_GSM_MODEM_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), NM_TYPE_GSM_MODEM, NMGsmModemPrivate))

enum {
	MODEM_STATE_BEGIN,
	MODEM_STATE_ENABLE,
	MODEM_STATE_SET_PIN,
	MODEM_STATE_SET_PUK,
	MODEM_STATE_SET_APN,
	MODEM_STATE_SET_BAND,
	MODEM_STATE_SET_NETWORK_MODE,
	MODEM_STATE_CONNECTED_CHECKPOINT,
	MODEM_STATE_REGISTER,
	MODEM_STATE_FAILED,
};

typedef struct {
	int modem_state;
	int reason;
} NMGsmModemPrivate;

NMDevice *
nm_gsm_modem_new (const char *path,
			   const char *data_device,
			   const char *driver)
{
	g_return_val_if_fail (path != NULL, NULL);
	g_return_val_if_fail (data_device != NULL, NULL);
	g_return_val_if_fail (driver != NULL, NULL);

	return (NMDevice *) g_object_new (NM_TYPE_GSM_MODEM,
							    NM_DEVICE_INTERFACE_UDI, path,
							    NM_DEVICE_INTERFACE_IFACE, data_device,
							    NM_DEVICE_INTERFACE_DRIVER, driver,
							    NM_DEVICE_INTERFACE_MANAGED, TRUE,
							    NM_MODEM_DEVICE_PATH, path,
							    NULL);
}

static NMSetting *
get_setting (NMGsmModem *modem, GType setting_type)
{
	NMActRequest *req;
	NMSetting *setting = NULL;

	req = nm_device_get_act_request (NM_DEVICE (modem));
	if (req) {
		NMConnection *connection;

		connection = nm_act_request_get_connection (req);
		if (connection)
			setting = nm_connection_get_setting (connection, setting_type);
	}

	return setting;
}

#define get_proxy(dev,iface) (nm_modem_device_get_proxy(NM_MODEM_DEVICE (dev), iface))

int state_machine_singleton = 0;

static void
state_machine (DBusGProxy *proxy, DBusGProxyCall *call_id, gpointer user_data)
{
	NMGsmModem *modem = NM_GSM_MODEM (user_data);
	NMGsmModemPrivate *priv = NM_GSM_MODEM_GET_PRIVATE (modem);
	NMSettingGsm *setting;
	const char *secret = NULL;
	const char *secret_name = NULL;
	gboolean retry_secret = FALSE;
	GError *error = NULL;
	static int bTriedPUK = 0;   /* Testing: assuming only one modem is connection phase and stuck on PUK */
	static int numServiceUnavail = 0; /* Testing: If service unavailable retry once more just to give it time */
	int connected = 0;

	setting = NM_SETTING_GSM (get_setting (modem, NM_TYPE_SETTING_GSM));

	if (call_id)
		dbus_g_proxy_end_call (proxy, call_id, &error, G_TYPE_INVALID);

	if (!setting)
		return;
		

	if (error) {

		if (dbus_g_error_has_name (error, MM_MODEM_ERROR_SIM_PIN)) {
			secret = setting->pin;
			secret_name = NM_SETTING_GSM_PIN;
			priv->modem_state = MODEM_STATE_SET_PIN;
		} else if (dbus_g_error_has_name (error, MM_MODEM_ERROR_SIM_PUK)) {
			g_debug("MM_MODEM_ERROR_SIM_PUK");
			if (setting->pin == NULL) {
				g_debug("pin == NULL");
				nm_device_state_changed (NM_DEVICE (modem), NM_DEVICE_STATE_NEED_AUTH, NM_DEVICE_STATE_REASON_NONE);
				nm_act_request_request_connection_secrets (nm_device_get_act_request (NM_DEVICE (modem)),
					NM_SETTING_GSM_SETTING_NAME,
					retry_secret,
					SECRETS_CALLER_GSM,
					NM_SETTING_GSM_PIN,
					NULL);
			}
			if (bTriedPUK > 1) {
				g_debug("fail with PUK");
				priv->modem_state = MODEM_STATE_FAILED;
				priv->reason = NM_DEVICE_STATE_REASON_GSM_PUK_CHECK_FAILED;
				bTriedPUK = 0;
			} else {
				bTriedPUK++;
			secret = setting->puk;
			secret_name = NM_SETTING_GSM_PUK;
				priv->modem_state = MODEM_STATE_SET_PUK;
			}
		} else if (dbus_g_error_has_name (error, MM_MODEM_ERROR_SIM_WRONG)) {
			g_free (setting->pin);
			setting->pin = NULL;
			secret_name = NM_SETTING_GSM_PIN;
			retry_secret = TRUE;
			priv->modem_state = MODEM_STATE_SET_PIN;
		} else if (dbus_g_error_has_name (error, MM_MODEM_ERROR_WRONG_PASSWORD)) {
			priv->modem_state = MODEM_STATE_FAILED;
			priv->reason = NM_DEVICE_STATE_REASON_GSM_PIN_CHECK_FAILED;
			g_debug("pin check failed");
		} else if (dbus_g_error_has_name (error, MM_MODEM_ERROR_INVALID_PARAMETER)) {
			if (priv->modem_state == MODEM_STATE_ENABLE) {
				priv->modem_state = MODEM_STATE_FAILED;
				priv->reason = NM_DEVICE_STATE_REASON_GSM_PIN_CHECK_FAILED;
			}
			g_debug ("Invalid parameter");
		} else if (dbus_g_error_has_name (error, MM_MODEM_ERROR_SERVICE_UNAVAILABLE)) {
			if (priv->modem_state == MODEM_STATE_REGISTER) {
				if (numServiceUnavail++ > 1) {
				priv->modem_state = MODEM_STATE_FAILED;
				priv->reason = NM_DEVICE_STATE_REASON_SERVICE_UNAVAILABLE;
				} else
					priv->modem_state = MODEM_STATE_BEGIN;
			}
			
			g_debug ("Service Unavailable");
		}


		/* FIXME: Hacks to ignore failures of setting band and network mode for now
		   since only Huawei module supports it. Remove when ModemManager rules.
		*/
		else if (dbus_g_error_has_name (error, MM_MODEM_ERROR_OPERATION_NOT_SUPPORTED) &&
			    (priv->modem_state == MODEM_STATE_SET_BAND ||
				priv->modem_state == MODEM_STATE_SET_NETWORK_MODE)) {

			nm_warning ("Modem does not support setting %s, ignoring",
					  priv->modem_state == MODEM_STATE_SET_BAND ? "band" : "network mode");
		} else {
			g_debug("modem_state = %d", priv->modem_state);
			if (priv->modem_state == MODEM_STATE_ENABLE) {
				nm_warning("Slow starter this modem...");
				priv->modem_state = MODEM_STATE_BEGIN;
			} else {
				if (dbus_g_error_has_name (error, MM_SERIAL_RESPONSE_TIMEOUT)) {
					g_debug("Work-around for bad COPS: serial timeout ignored");
					
				} else {
			priv->modem_state = MODEM_STATE_FAILED;
			nm_warning ("GSM modem connection failed: %s", error->message);
		}
		}
		}

		g_error_free (error);
	}

 again:

	g_debug("modem_state = %d", priv->modem_state);
	
	switch (priv->modem_state) {
	case MODEM_STATE_BEGIN:
		state_machine_singleton = 1;
		priv->modem_state = MODEM_STATE_ENABLE;
		dbus_g_proxy_begin_call (get_proxy (modem, MM_DBUS_INTERFACE_MODEM),
							"Enable", state_machine,
							modem, NULL,
							G_TYPE_BOOLEAN, TRUE,
							G_TYPE_INVALID);
		break;

	case MODEM_STATE_SET_PIN:
		if (secret) {
			priv->modem_state = MODEM_STATE_ENABLE;
			dbus_g_proxy_begin_call (get_proxy (modem, MM_DBUS_INTERFACE_MODEM_GSM_CARD),
								"SendPin", state_machine,
								modem, NULL,
								G_TYPE_STRING, secret,
								G_TYPE_INVALID);
		} else {
			nm_device_state_changed (NM_DEVICE (modem), NM_DEVICE_STATE_NEED_AUTH, NM_DEVICE_STATE_REASON_NONE);
			nm_act_request_request_connection_secrets (nm_device_get_act_request (NM_DEVICE (modem)),
											   NM_SETTING_GSM_SETTING_NAME,
											   retry_secret,
											   SECRETS_CALLER_GSM,
											   secret_name,
											   NULL);

		}
		break;

	case MODEM_STATE_SET_PUK:
		g_debug("set_puk");
		if (setting->puk) {
			g_debug("got puk");
			priv->modem_state = MODEM_STATE_ENABLE;
			dbus_g_proxy_begin_call (get_proxy (modem, MM_DBUS_INTERFACE_MODEM_GSM_CARD),
								"SendPuk", state_machine,
								modem, NULL,
								G_TYPE_STRING, setting->puk,
								G_TYPE_STRING, setting->pin,
								G_TYPE_INVALID);
		} else {
			g_debug("secret_name = %s", secret_name);
			nm_device_state_changed (NM_DEVICE (modem), NM_DEVICE_STATE_NEED_AUTH, NM_DEVICE_STATE_REASON_NONE);
			nm_act_request_request_connection_secrets (nm_device_get_act_request (NM_DEVICE (modem)),
				NM_SETTING_GSM_SETTING_NAME,
				retry_secret,
				SECRETS_CALLER_GSM,
				secret_name,
				NULL);

		}
		break;

	case MODEM_STATE_ENABLE:
		priv->modem_state = MODEM_STATE_SET_APN;
		if (setting->apn)
			dbus_g_proxy_begin_call (get_proxy (modem, MM_DBUS_INTERFACE_MODEM_GSM_NETWORK),
								"SetApn", state_machine,
								modem, NULL,
								G_TYPE_STRING, setting->apn,
								G_TYPE_INVALID);
		else
			goto again;

		break;
	case MODEM_STATE_SET_APN:
		priv->modem_state = MODEM_STATE_SET_BAND;
		if (setting->band)
			dbus_g_proxy_begin_call (get_proxy (modem, MM_DBUS_INTERFACE_MODEM_GSM_NETWORK),
								"SetBand", state_machine,
								modem, NULL,
								G_TYPE_UINT, (guint32) setting->band,
								G_TYPE_INVALID);
		else
			goto again;

		break;

	case MODEM_STATE_SET_BAND:
		priv->modem_state = MODEM_STATE_SET_NETWORK_MODE;
		if (setting->network_type)
			dbus_g_proxy_begin_call (get_proxy (modem, MM_DBUS_INTERFACE_MODEM_GSM_NETWORK),
								"SetNetworkMode", state_machine,
								modem, NULL,
								G_TYPE_UINT, (guint32) setting->network_type,
								G_TYPE_INVALID);
		else
			goto again;

		break;

	case MODEM_STATE_SET_NETWORK_MODE:
		priv->modem_state = MODEM_STATE_CONNECTED_CHECKPOINT;
		dbus_g_proxy_begin_call_with_timeout (get_proxy (modem, MM_DBUS_INTERFACE_MODEM_GSM_NETWORK),
									   "Register", state_machine,
									   modem, NULL, 120000,
									   G_TYPE_STRING, setting->network_id ? setting->network_id : "",
									   G_TYPE_INVALID);
		break;

	case MODEM_STATE_CONNECTED_CHECKPOINT:
		if (numServiceUnavail++ > 4) {
			nm_device_state_changed (NM_DEVICE (modem), NM_DEVICE_STATE_FAILED, priv->reason);
			break;
		}
		if (!dbus_g_proxy_call (get_proxy (modem, MM_DBUS_INTERFACE_MODEM_GSM_NETWORK),
						    "GetConnectedStatus", &error,
						    G_TYPE_INVALID,
						    G_TYPE_UINT, &connected,
						    G_TYPE_INVALID)) {
			g_warning ("Error getting connected status: %s", error->message);
			g_error_free(error);
			priv->modem_state = MODEM_STATE_REGISTER;
		}
		g_debug("connected = %d", connected);
		if (connected)
			priv->modem_state = MODEM_STATE_REGISTER;
		else {
			sleep(5);
			state_machine (NULL, NULL, modem);
			break;
		}

	case MODEM_STATE_REGISTER:
		nm_modem_device_connect (NM_MODEM_DEVICE (modem), setting->number);
		bTriedPUK = 0;
		numServiceUnavail = 0;
		state_machine_singleton = 0;
		break;
	case MODEM_STATE_FAILED:
	default:
		state_machine_singleton = 0;
		nm_device_state_changed (NM_DEVICE (modem), NM_DEVICE_STATE_FAILED, priv->reason);
		break;
	}
}

static NMActStageReturn
real_act_stage1_prepare (NMDevice *device, NMDeviceStateReason *reason)
{
	NMGsmModemPrivate *priv = NM_GSM_MODEM_GET_PRIVATE (device);

	if (!state_machine_singleton) {
		priv->modem_state = MODEM_STATE_BEGIN;
		priv->reason = NM_DEVICE_STATE_REASON_NONE;
		state_machine (NULL, NULL, device);

	} else
		g_debug ("prevented by state_machine_singleton");
	return NM_ACT_STAGE_RETURN_POSTPONE;
}

static NMConnection *
real_get_best_auto_connection (NMDevice *dev,
                               GSList *connections,
                               char **specific_object)
{
	GSList *iter;

	for (iter = connections; iter; iter = g_slist_next (iter)) {
		NMConnection *connection = NM_CONNECTION (iter->data);
		NMSettingConnection *s_con;

		s_con = (NMSettingConnection *) nm_connection_get_setting (connection, NM_TYPE_SETTING_CONNECTION);
		g_assert (s_con);

		if (!s_con->autoconnect)
			continue;

		if (strcmp (s_con->type, NM_SETTING_GSM_SETTING_NAME))
			continue;

		return connection;
	}
	return NULL;
}

static void
real_connection_secrets_updated (NMDevice *dev,
                                 NMConnection *connection,
                                 GSList *updated_settings,
                                 RequestSecretsCaller caller)
{
	NMActRequest *req;
	gboolean found = FALSE;
	GSList *iter;

	g_debug("real_connection_secrets_updated");
	
	if (caller == SECRETS_CALLER_PPP) {
		NMPPPManager *ppp_manager;
		NMSettingGsm *s_gsm = NULL;

		ppp_manager = nm_modem_device_get_ppp_manager (NM_MODEM_DEVICE (dev));
		g_return_if_fail (ppp_manager != NULL);

		s_gsm = (NMSettingGsm *) nm_connection_get_setting (connection, NM_TYPE_SETTING_GSM);
		if (!s_gsm) {
			/* Shouldn't ever happen */
			nm_ppp_manager_update_secrets (ppp_manager,
			                               nm_device_get_iface (dev),
			                               NULL,
			                               NULL,
			                               "missing GSM setting; no secrets could be found.");
		} else {
			nm_ppp_manager_update_secrets (ppp_manager,
			                               nm_device_get_iface (dev),
			                               s_gsm->username ? s_gsm->username : "",
			                               s_gsm->password ? s_gsm->password : "",
			                               NULL);
		}
		return;
	}

	g_return_if_fail (caller == SECRETS_CALLER_GSM);

	g_debug("real_connection_secret %d", nm_device_get_state (dev));
	
	g_return_if_fail (nm_device_get_state (dev) == NM_DEVICE_STATE_NEED_AUTH);

	for (iter = updated_settings; iter; iter = g_slist_next (iter)) {
		const char *setting_name = (const char *) iter->data;

		if (!strcmp (setting_name, NM_SETTING_GSM_SETTING_NAME))
			found = TRUE;
		else
			nm_warning ("Ignoring updated secrets for setting '%s'.", setting_name);
	}

	if (!found)
		return;

	req = nm_device_get_act_request (dev);
	g_assert (req);

	g_return_if_fail (nm_act_request_get_connection (req) == connection);

	nm_device_activate_schedule_stage1_device_prepare (dev);
}

static const char *
real_get_ppp_name (NMModemDevice *device, NMConnection *connection)
{
	NMSettingGsm *s_gsm;

	s_gsm = (NMSettingGsm *) nm_connection_get_setting (connection, NM_TYPE_SETTING_GSM);
	g_assert (s_gsm);

	return s_gsm->username;
}

/*****************************************************************************/

static void
nm_gsm_modem_init (NMGsmModem *self)
{
	nm_device_set_device_type (NM_DEVICE (self), NM_DEVICE_TYPE_GSM);
}

static void
nm_gsm_modem_class_init (NMGsmModemClass *klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);
	NMDeviceClass *device_class = NM_DEVICE_CLASS (klass);
	NMModemDeviceClass *modem_class = NM_MODEM_DEVICE_CLASS (klass);

	g_type_class_add_private (object_class, sizeof (NMGsmModemPrivate));

	/* Virtual methods */
	device_class->get_best_auto_connection = real_get_best_auto_connection;
	device_class->connection_secrets_updated = real_connection_secrets_updated;
	device_class->act_stage1_prepare = real_act_stage1_prepare;
	modem_class->get_ppp_name = real_get_ppp_name;

	dbus_g_object_type_install_info (G_TYPE_FROM_CLASS (klass),
							   &dbus_glib_nm_gsm_device_object_info);
}
