/* -*- Mode: C; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*- */
/*
  Additions to NetworkManager, network-manager-applet and modemmanager
  for supporting Ericsson modules like F3507g.

  Copyright (C) 2008 Ericsson MBM

  Author: Per Hallsmark <per.hallsmark@ericsson.com>
          Bjorn Runaker <bjorn.runaker@ericsson.com>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the

  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <unistd.h>
#include <string.h>
#include <stdlib.h>
#include <arpa/inet.h>
#include <dbus/dbus-glib.h>
#include "mm-modem-mbm.h"
#include "mm-serial.h"
#include "mm-errors.h"
#include "mm-callback-info.h"


static void impl_mbm_authenticate (MMModemMbm *self,
                                   const char *username,
                                   const char *password,
                                   const char *pin,
                                   int bEnable,
                                   DBusGMethodInvocation *context);

#include "mm-modem-gsm-mbm-glue.h"

static void
mbm_enable (MMModem *modem,
            gboolean bEnable,
            MMModemFn callback,
            gpointer user_data);

static gpointer mm_modem_mbm_parent_class = NULL;

#define MM_MODEM_MBM_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), MM_TYPE_MODEM_MBM, MMModemMbmPrivate))

typedef struct {
    char    *network_device;
    GRegex  *msg_waiting_regex;
    GRegex  *ciev_regex;
    GRegex    *e2cfun_regex;
    gpointer std_parser;
    guint32  signal_quality;
    char    *username;
    char    *password;
    char    *pin;
    int      bEnable;
    int      bModemEnabled;
    int      bModemEnableRunning;
    int      register_completed;
  int       enap_completed;
    int       bEnap_done;
    int      bCOPS_register;
    char    *network_id;
  int       bDisabled;
} MMModemMbmPrivate;

enum {
    PROP_0,
    PROP_NETWORK_DEVICE,

    LAST_PROP
};

static void
init_do_wait4creg_loop (MMSerial *serial, gpointer user_data);

static void
init_do_wait4creg (MMSerial *serial,
                      GString *response,
                      GError *error,
                      gpointer user_data);


MMModem *
mm_modem_mbm_new (const char *serial_device,
                  const char *network_device,
                  const char *driver)
{
    g_return_val_if_fail (serial_device != NULL, NULL);
    g_return_val_if_fail (network_device != NULL, NULL);
    g_return_val_if_fail (driver != NULL, NULL);

    return MM_MODEM (g_object_new (MM_TYPE_MODEM_MBM,
                                   MM_SERIAL_DEVICE, serial_device,
                                   MM_SERIAL_SEND_DELAY, (guint64) 10000,
                                   MM_MODEM_DRIVER, driver,
                                   MM_MODEM_MBM_NETWORK_DEVICE, network_device,
                                   NULL));
}

static void
eiapsw_done (MMSerial *serial,
             GString *response,
             GError *error,
             gpointer user_data)
{
    MMCallbackInfo *info = (MMCallbackInfo *) user_data;

    if (error)
        info->error = g_error_copy (error);

    mm_callback_info_schedule (info);
}

static void
eiac_done (MMSerial *serial,
           GString *response,
           GError *error,
           gpointer user_data)
{
    MMCallbackInfo *info = (MMCallbackInfo *) user_data;
    char *command;

    if (error)
        info->error = g_error_copy (error);

    command = g_strdup_printf ("AT*EIAPSW=1,1,\"%s\"",
                               (char *) mm_callback_info_get_data (info, "apn"));


    mm_serial_queue_command (serial, command, 10, eiapsw_done, info);
    g_free (command);
}

static void
eiad_done (MMSerial *serial,
           GString *response,
           GError *error,
           gpointer user_data)
{
    MMCallbackInfo *info = (MMCallbackInfo *) user_data;

    if (error)
        info->error = g_error_copy (error);

    mm_serial_queue_command (serial, "AT*EIAC=1", 10, eiac_done, info);
}

static void
set_apn (MMModemGsmNetwork *modem,
         const char *apn,
         MMModemFn callback,
         gpointer user_data)
{
    MMCallbackInfo *info;

    info = mm_callback_info_new (MM_MODEM (modem), callback, user_data);
    mm_callback_info_set_data (info, "apn", g_strdup (apn), g_free);

    mm_serial_queue_command (MM_SERIAL (modem), "AT*EIAD=0", 10, eiad_done, info);
}

static void
set_user_pass (MMModemGsmNetwork *modem,
               const char *username,
               const char *password,
               const char *pin,
               int bEnable,
               MMModemFn callback,
               gpointer user_data)
{
    MMModemMbmPrivate *priv = MM_MODEM_MBM_GET_PRIVATE (modem);

    MMCallbackInfo *info = (MMCallbackInfo *) user_data;

    g_debug("set_user_pass %s %s", username?username:"", password?password:"");

    if (priv->username)
        g_free(priv->username);
    priv->username = g_strdup(username?username:"");
    if (priv->password)
        g_free(priv->password);
    priv->password = g_strdup(password?password:"");
    if (priv->pin)
        g_free(priv->pin);
    priv->pin = g_strdup(pin?pin:"");
}

static void
do_cmer_done (MMSerial *serial,
              GString *response,
              GError *error,
              gpointer user_data)
{
    MMCallbackInfo *info = (MMCallbackInfo *) user_data;

    if (error)
        info->error = g_error_copy (error);

    mm_serial_queue_command (serial, "AT*E2CFUN=1", 3, NULL, NULL);
    mm_callback_info_schedule (info);
}

/*
 * CMER   should be done after we are registred.
 */
static void
do_register_done (MMSerial *serial,
                  GString *response,
                  GError *error,
                  gpointer user_data)
{
    MMCallbackInfo *info = (MMCallbackInfo *) user_data;
    MMModemMbmPrivate *priv = MM_MODEM_MBM_GET_PRIVATE (MM_MODEM(serial));

    if (error) {
        info->error = g_error_copy (error);
    } else {
        priv->bEnap_done = 1;
        priv->bCOPS_register = 0;
    }

    mm_serial_queue_command (serial, "AT+CMER=3,0,0,1", 20, do_cmer_done, info);
}

static void
init_do_creg (MMSerial *serial,
                 GString *response,
                 GError *error,
                 gpointer user_data)
{
    MMCallbackInfo *info = (MMCallbackInfo *) user_data;
    MMModemMbmPrivate *priv = MM_MODEM_MBM_GET_PRIVATE (MM_MODEM(serial));

	priv->register_completed = 0;
    mm_serial_queue_command (serial, "+CREG=1", 25, init_do_wait4creg, user_data);
}

static void
init_do_wait4enap_loop (MMSerial *serial, gpointer user_data);


static void
check_enap_done (MMSerial *serial,
                 GString *response,
                 GError *error,
                 gpointer user_data)
{
    MMCallbackInfo *info = (MMCallbackInfo *) user_data;
    MMModemMbmPrivate *priv = MM_MODEM_MBM_GET_PRIVATE (info->modem);
    const char *reply = response->str;

    priv->enap_completed = 0;
    
    if (error)
        info->error = g_error_copy (error);

    if (g_str_has_prefix (reply, "*ENAP: ")) {
        int bEnabled;
        if (sscanf (reply + 7, "%d", &bEnabled)) {
            if (bEnabled == 1)
                priv->enap_completed = 1;
            if (bEnabled == 0) {
                g_debug("enap failed bringing modem down");
                g_set_error (error, MM_SERIAL_ERROR, MM_SERIAL_SEND_FAILED,
                             "%s", "ENAP failed");
                mm_callback_info_schedule (info);
                return;
            }
        }
    }

    mm_serial_flash (serial, 1000, init_do_wait4enap_loop, user_data);
}



static void
init_do_wait4enap_loop (MMSerial *serial, gpointer user_data)
{
    MMCallbackInfo *info = (MMCallbackInfo *) user_data;
    MMModemMbmPrivate *priv = MM_MODEM_MBM_GET_PRIVATE(info->modem);

    GError *error;

    if (!priv)
        return;

    if (!priv->bModemEnabled || priv->bDisabled) {
        mm_callback_info_schedule (info); /* TODO: is this the best way to free structures */
        return;
    }

    if (!priv->enap_completed) {
        g_debug("still waiting for enap to complete\n");
        mm_serial_queue_command (serial, "*enap?", 3, check_enap_done, info);

    } else {
        g_debug("registration completed\n");
        do_register_done (serial, "", NULL, info);
//        mm_serial_queue_command(serial, "AT*ENAP=1,1", 20, do_register_done, info);
    }  
}


/*
 * ENAP=1,1 should be done after a successful registration
 */
static void
init_do_enap (MMSerial *serial,
                   GString *response,
                   GError *error,
                   gpointer user_data)
{
    MMCallbackInfo *info = (MMCallbackInfo *) user_data;

	g_debug("init_do_enap");

    if (error) {
		g_debug("error %s", error->message);
/*        info->error = g_error_copy (error);
        mm_callback_info_schedule (info);
        return;*/
    }

    sleep(4);
    mm_serial_queue_command (serial, "*ENAP=1,1", 25, check_enap_done, user_data);
}

static void
do_register_authenticate (MMSerial *serial,
              GString *response,
              GError *error,
              gpointer user_data)
{
    char *command;
    MMCallbackInfo *info = (MMCallbackInfo *) user_data;
    MMModemMbmPrivate *priv = MM_MODEM_MBM_GET_PRIVATE (MM_MODEM(serial));

    if (error)
        info->error = g_error_copy (error);


    command = g_strdup_printf ("AT*EIAAUW=1,1,\"%s\",\"%s\"",
//                               mm_generic_gsm_get_cid (MM_GENERIC_GSM (info->modem)),
                               priv->username ? priv->username : "",
                               priv->password ? priv->password : "");

    g_debug("do_register_authenticate %s", command);

//  mm_serial_queue_command (serial, command, 3, do_register_step_enap,
//  info);

    mm_serial_queue_command (serial, command, 3, init_do_creg, info);
    g_free (command);
}


static void
check_creg (MMSerial *serial,
              GString *response,
              GError *error,
              gpointer user_data)
{
    MMCallbackInfo *info = (MMCallbackInfo *) user_data;
    MMModemMbmPrivate *priv = MM_MODEM_MBM_GET_PRIVATE (info->modem);
    const char *reply = response->str;

    if (error)
        info->error = g_error_copy (error);

    if (g_str_has_prefix (reply, "+CREG: ")) {
        int n,bEnabled;
        if (sscanf (reply + 7, "%d,%d", &n,&bEnabled)) {
            if (bEnabled && !(bEnabled == 2 || bEnabled == 3 || bEnabled == 4))
                priv->register_completed = 1;
        }
    }
    
    mm_serial_flash (serial, 1000, init_do_wait4creg_loop, user_data);
}

static void
init_do_wait4creg_loop (MMSerial *serial, gpointer user_data)
{
    MMCallbackInfo *info = (MMCallbackInfo *) user_data;
    MMModemMbmPrivate *priv = MM_MODEM_MBM_GET_PRIVATE(info->modem);

    GError *error;
    
    if (!priv)
        return;
    
    if (!priv->bModemEnabled || priv->bDisabled) {
        mm_callback_info_schedule (info); /* TODO: is this the best way to free structures */
        return;
    }
    
    if (!priv->register_completed) {
        g_debug("still waiting for registration to complete\n");
        mm_serial_queue_command (serial, "+creg?", 3, check_creg, info);
        
//        mm_serial_flash (serial, 1000, init_do_wait4creg_loop, user_data);
    } else {
        g_debug("registration completed\n");
        if (!priv->bCOPS_register) {
            char *command;

            if (priv->network_id) {
            g_debug("reg_network-id=%s", priv->network_id);

                command = g_strdup_printf ("+COPS=4,2,\"%s\"", priv->network_id);
            } else
            {
                command = g_strdup_printf ("+COPS=0");
            }

            priv->register_completed = 0;
            priv->bCOPS_register = 1;
            mm_serial_queue_command (serial, command, 180, init_do_creg, info);
        } else
        {
            mm_serial_queue_command(serial, "AT*ENAP=1,1", 20, check_enap_done, info);
        }
    }
}

static void
init_do_wait4creg (MMSerial *serial,
                          GString *response,
                          GError *error,
                          gpointer user_data)
{
    MMCallbackInfo *info = (MMCallbackInfo *) user_data;
    MMModemMbmPrivate *priv = MM_MODEM_MBM_GET_PRIVATE (info->modem);
    
    if (!priv)
        return;

    if (error) {
        info->error = g_error_copy (error);
        mm_callback_info_schedule (info);
        return;
    }

    mm_serial_flash (serial, 1000, init_do_wait4creg_loop, user_data);
}

static void
do_register (MMModemGsmNetwork *modem,
             const char *network_id,
             MMModemFn callback,
             gpointer user_data)
{
    char *command;
    MMCallbackInfo *info;

    info = mm_callback_info_new (MM_MODEM (modem), callback, user_data);
    MMModemMbmPrivate *priv = MM_MODEM_MBM_GET_PRIVATE (info->modem);
    
    
    g_debug("reg-network-id = %s", network_id);

    if (priv->network_id)
        g_free(priv->network_id);


    if (network_id && *network_id) {
        priv->network_id = g_strdup (network_id);
    } else
    {
        priv->network_id = NULL;
    }
    if (priv->username && *(priv->username)) {
        mm_serial_queue_command (MM_SERIAL (modem), "+CFUN=1", 5, do_register_authenticate, info);
        return;
    }

    mm_serial_queue_command (MM_SERIAL(modem), "+CFUN=1", 5, init_do_creg, info);    
}

static void
mbm_enabled (MMModem *modem,
             GError *error,
             gpointer user_data)
{
    MMCallbackInfo *info = (MMCallbackInfo *) user_data;

    if (error)
        info->error = g_error_copy (error);
    
    g_debug("mbm_enabled");
    
    mm_callback_info_schedule (info);
}

static void
mbm_disabled (MMModem *modem,
              GError *error,
              gpointer user_data)
{
    MMCallbackInfo *info = (MMCallbackInfo *) user_data;

    g_debug("mbm_disabled");

    if (error) {
        info->error = g_error_copy (error);
        mm_callback_info_schedule (info);
    } else {
        mbm_enable (modem, TRUE, mbm_enabled, info);
    }
}

static void
auth_done (MMSerial *serial,
           GString *response,
           GError *error,
           gpointer user_data)
{
    MMCallbackInfo *info = (MMCallbackInfo *) user_data;

    g_debug("auth_done");

    if (error) {
        info->error = g_error_copy (error);
        mm_callback_info_schedule (info);
    } else {
        /* success, kill any existing connections first */
        mbm_enable (MM_MODEM (serial), FALSE, mbm_disabled, info);
    }
}

void
mbm_authenticate (MMModemGsmNetwork *modem,
                  const char *username,
                  const char *password,
                  MMModemFn callback,
                  gpointer user_data)
{
    MMCallbackInfo *info;
    char *command;


    g_return_if_fail (callback != NULL);

    info = mm_callback_info_new (MM_MODEM (modem), callback, user_data);
    command = g_strdup_printf ("AT*EIAAUW=%d,1,\"%s\",\"%s\"",
                         mm_generic_gsm_get_cid (MM_GENERIC_GSM (modem)),
                           password ? password : "",
                           username ? username : "");

    g_debug("mbm_authenticate %s", command);

    mm_serial_queue_command (MM_SERIAL (modem), command, 3, auth_done, info);
    g_free (command);
}

/*****************************************************************************/
/*    Modem class override functions                                         */
/*****************************************************************************/

static void
enable_done (MMSerial *serial,
             GString *response,
             GError *error,
             gpointer user_data)
{
    MMCallbackInfo *info = (MMCallbackInfo *) user_data;
    MMModemMbmPrivate *priv = MM_MODEM_MBM_GET_PRIVATE (MM_MODEM(serial));

    g_debug("enable_done");

    sleep(1);
    
    priv->bModemEnableRunning = 0;
/*    if (error) {
        info->error = g_error_copy (error);
    }*/
        priv->bModemEnabled = 1;        

    mm_callback_info_schedule (info);
}

/* Seems always needed
 * NOTE: this makes it always on when enabled */
static void
init_do_creg0 (MMSerial *serial,
                  GString *response,
                  GError *error,
                  gpointer user_data)
{
    MMCallbackInfo *info = (MMCallbackInfo *) user_data;
    MMModemMbmPrivate *priv = MM_MODEM_MBM_GET_PRIVATE (MM_MODEM(serial));

    if (error) {
        priv->bModemEnableRunning = 0;
        info->error = g_error_copy (error);
        mm_callback_info_schedule (info);
    } else {
        sleep(4);
        if (priv->bModemEnableRunning < 6) {
            priv->bModemEnableRunning = 6;
            mm_serial_queue_command (serial, "+CREG=0", 25, enable_done, user_data);
        } else
            mm_serial_queue_command (serial, "", 1, enable_done, user_data);
    }
}

static void
init_do_cops0 (MMSerial *serial,
               GString *response,
               GError *error,
               gpointer user_data)
{
    MMCallbackInfo *info = (MMCallbackInfo *) user_data;
    MMModemMbmPrivate *priv = MM_MODEM_MBM_GET_PRIVATE (MM_MODEM(serial));

    if (error) {
        priv->bModemEnableRunning = 0;
        info->error = g_error_copy (error);
        mm_callback_info_schedule (info);
    } else {
        sleep(4);
        if (priv->bModemEnableRunning < 5) {
            priv->bModemEnableRunning = 5;
            mm_serial_queue_command (serial, "+COPS=0", 25, init_do_creg0, user_data);
        } else
            mm_serial_queue_command (serial, "", 1, init_do_creg0, user_data);

    }

}

/* Seems always needed
 * NOTE: this makes it always on when enabled */
static void
init_do_cfun1 (MMSerial *serial,
                 GString *response,
                 GError *error,
                 gpointer user_data)
{
    MMCallbackInfo *info = (MMCallbackInfo *) user_data;
    MMModemMbmPrivate *priv = MM_MODEM_MBM_GET_PRIVATE (MM_MODEM (serial));

    if (error) {
        priv->bModemEnableRunning = 0;
        info->error = g_error_copy (error);
        mm_callback_info_schedule (info);
    } else {
        sleep(4);
        if (priv->bModemEnableRunning < 4) {
            priv->bModemEnableRunning = 4;
            mm_serial_queue_command (serial, "+CFUN=1", 25, init_do_cops0, user_data);
        } else
            mm_serial_queue_command (serial, "", 1, init_do_cops0, user_data);

    }
}

static void
init_do_cscs (MMSerial *serial,
             GString *response,
             GError *error,
             gpointer user_data)
{
    MMCallbackInfo *info = (MMCallbackInfo *) user_data;
    MMModemMbmPrivate *priv = MM_MODEM_MBM_GET_PRIVATE (MM_MODEM (serial));

    if (error) {
        priv->bModemEnableRunning = 0;
        info->error = g_error_copy (error);
        mm_callback_info_schedule (info);
        
    } else {
        sleep(4);
        if (priv->bModemEnableRunning < 3) {
            priv->bModemEnableRunning = 3;
            mm_serial_queue_command (serial, "+CSCS=\"IRA\"", 1, init_do_cfun1, user_data);
        } else
            mm_serial_queue_command (serial, "", 1, init_do_cfun1, user_data);
    }
}

static void
enable_flash_done (MMSerial *serial, gpointer user_data)
{
    MMModemMbmPrivate *priv = MM_MODEM_MBM_GET_PRIVATE (MM_MODEM (serial));
    g_debug("enable_flash_done %d", priv->bModemEnableRunning);
    if (priv->bModemEnableRunning < 2) {
        priv->bModemEnableRunning = 2;
	mm_serial_queue_command (serial, "&F E0 +CMEE=1", 3, init_do_cscs, user_data);
    } else
        mm_serial_queue_command (serial, "", 1, init_do_cscs, user_data);
}

static void
disable_done (MMSerial *serial,
              GString *response,
              GError *error,
              gpointer user_data)
{
    MMModemMbmPrivate *priv = MM_MODEM_MBM_GET_PRIVATE (MM_MODEM (serial));
    priv->bModemEnabled = 0;
    
    mm_serial_close (serial);
    
    mm_callback_info_schedule ((MMCallbackInfo *) user_data);
}

static void
do_cfun4 (MMSerial *serial,
          GString *response,
          GError *error,
          gpointer user_data)
{
    MMModemMbmPrivate *priv = MM_MODEM_MBM_GET_PRIVATE (MM_MODEM (serial));
	if (!priv->bModemEnableRunning) {
        if (priv->bEnap_done) {
            sleep (5);
            priv->bEnap_done = 0;
        }
    mm_serial_queue_command (serial, "+CFUN=4", 5, disable_done, user_data);
	} else
	{
		g_debug("Not doing a +CFUN=4 when in middle of startup sequence");
        mm_callback_info_schedule ((MMCallbackInfo *) user_data);
    }
}

static void
disable_flash_done (MMSerial *serial, gpointer user_data)
{
    MMModemMbmPrivate *priv = MM_MODEM_MBM_GET_PRIVATE (MM_MODEM (serial));
    if (!priv)
        return;
    if (!priv->bModemEnableRunning) {
        if (!priv->bEnap_done) {
            mm_serial_queue_command (serial, "+CFUN=4", 5, disable_done, user_data);
        } else
        {
            mm_serial_queue_command (serial, "*ENAP=0", 5, do_cfun4, user_data);
        }
	} else
	{
		g_debug("Not doing a *ENAP=0,1 when in middle of startup sequence");
	}
}


static void
mbm_enable (MMModem *modem,
            gboolean bEnable,
            MMModemFn callback,
            gpointer user_data)
{
    MMCallbackInfo *info;
    static int restart_count = 0;
    MMModemMbmPrivate *priv = MM_MODEM_MBM_GET_PRIVATE (modem);

    g_debug("mbm_enable %d %d", bEnable, restart_count);
    
    /* First, reset the previously used CID */
    mm_generic_gsm_set_cid (MM_GENERIC_GSM (modem), 0);

    info = mm_callback_info_new (modem, callback, user_data);

    if (!bEnable) {
        restart_count = 0;
        priv->bDisabled = 1;
        if (mm_serial_is_connected (MM_SERIAL (modem)))
            mm_serial_flash (MM_SERIAL (modem), 1000, disable_flash_done, info);
        else
            disable_flash_done (MM_SERIAL (modem), info);
    } else {
        priv->bDisabled = 0;
        if (priv->bModemEnabled) {
            g_debug("Modem already enabled");
            restart_count = 0;
            mm_callback_info_schedule (info);
        } else
        {
            g_debug("bModemEnableRunning = %d", priv->bModemEnableRunning);
            if (restart_count++ > 4) {
                g_debug("This port is not working!");
                mm_serial_close (MM_SERIAL (modem));
                mm_callback_info_schedule ((MMCallbackInfo *) user_data);
                system ("/etc/init.d/NetworkManager restart");
                sleep(15);
                exit(0);
            }
            if (!priv->bModemEnableRunning) {
                priv->bModemEnableRunning = 1;
            if (mm_serial_open (MM_SERIAL (modem), &info->error))
                mm_serial_flash (MM_SERIAL (modem), 100, enable_flash_done, info);
            } else
                mm_serial_flash (MM_SERIAL (modem), 100, enable_flash_done, info);

            if (info->error)
                mm_callback_info_schedule (info);
        }
    }

}

static void
mbm_connect_done (MMSerial *serial,
                  gpointer user_data)
{
    MMCallbackInfo *info = (MMCallbackInfo *) user_data;

    mm_callback_info_schedule (info);
}

static void
mbm_connect (MMModem *modem,
             const char *number,
             MMModemFn callback,
             gpointer user_data)
{
    MMCallbackInfo *info;

    g_debug("%s", __FUNCTION__);

    info = mm_callback_info_new (modem, callback, user_data);
    mm_callback_info_schedule (info);
#if 0    
    mm_serial_flash (MM_SERIAL (modem), 1000, mbm_connect_done, info);
#endif
}

static void
mbm_disconnect_done (MMSerial *serial, gpointer user_data)
{
    g_debug("%s", __FUNCTION__);

    mm_callback_info_schedule ((MMCallbackInfo *) user_data);
}

static void
mbm_disconnect (MMModem *modem,
                MMModemFn callback,
                gpointer user_data)
{
    MMCallbackInfo *info;
    MMModemMbmPrivate *priv = MM_MODEM_MBM_GET_PRIVATE (modem);
    g_debug("%s", __FUNCTION__);

    info = mm_callback_info_new (modem, callback, user_data);
    if (priv)
        priv->bCOPS_register = 0;
    
    mm_callback_info_schedule (info);
#if 0    
    mm_serial_flash (MM_SERIAL (modem), 1000, mbm_disconnect_done, info);
#endif
}


static gboolean
parse_erinfo (const char *reply, int *mode, int *gsm_rinfo, int *umts_rinfo)
{
    if (reply == NULL || strncmp (reply, "*ERINFO:", 8))
        return FALSE;

    if (sscanf (reply + 8, "%d,%d,%d", mode, gsm_rinfo, umts_rinfo))
        return TRUE;

    return FALSE;
}

static void
get_network_mode_done (MMSerial *serial,
                       GString *response,
                       GError *error,
                       gpointer user_data)
{
    MMCallbackInfo *info = (MMCallbackInfo *) user_data;

    g_debug("get_network_mode_done");
    if (error)
        info->error = g_error_copy (error);
    else {
        int mode;
        int gsm_rinfo;
        int umts_rinfo;
        guint32 result = 0;

        if (parse_erinfo (response->str, &mode, &gsm_rinfo, &umts_rinfo)) {
            if (umts_rinfo == 2)
                result = MM_MODEM_GSM_NETWORK_MODE_HSDPA;
            else if (umts_rinfo && !gsm_rinfo)
                result = MM_MODEM_GSM_NETWORK_MODE_3G;
            else if (umts_rinfo && gsm_rinfo)
                result = MM_MODEM_GSM_NETWORK_MODE_PREFER_3G;
            else if (gsm_rinfo == 1)
                result = MM_MODEM_GSM_NETWORK_MODE_GPRS;
            else if (gsm_rinfo == 2)
                result = MM_MODEM_GSM_NETWORK_MODE_EDGE;
        }

        if (result == 0)
            info->error = g_error_new (MM_MODEM_ERROR, MM_MODEM_ERROR_GENERAL,
                                       "%s", "Could not parse network mode results");
        else
            mm_callback_info_set_result (info, GUINT_TO_POINTER (result), NULL);
    }

    mm_callback_info_schedule (info);
}

static void
get_network_mode (MMModemGsmNetwork *modem,
                  MMModemUIntFn callback,
                  gpointer user_data)
{
    MMCallbackInfo *info;
    MMModemMbmPrivate *priv = MM_MODEM_MBM_GET_PRIVATE (modem);
    info = mm_callback_info_uint_new (MM_MODEM (modem), callback, user_data);
    g_debug("get_network_mode");
    if (priv->bModemEnabled) {
    mm_serial_queue_command (MM_SERIAL (modem), "AT*ERINFO?", 3, get_network_mode_done, info);
    } else
    {
        info->error = g_error_new (MM_MODEM_ERROR, MM_MODEM_ERROR_GENERAL,
                                   "%s", "Device not connected");
        mm_callback_info_schedule (info);
    }
}

/* GetSignalQuality */

static void
get_signal_quality_done (MMSerial *serial,
                         GString *response,
                         GError *error,
                         gpointer user_data)
{
    MMCallbackInfo *info = (MMCallbackInfo *) user_data;
    char *reply = response->str;

    if (error)
        info->error = g_error_copy (error);
    else if (!strncmp (reply, "+CIND: ", 7)) {
        /* Got valid reply */
        int battch;
        int signal;

        reply += 7;

        if (sscanf (reply, "%d,%d", &battch, &signal)) {
           /* Normalize the quality */
            signal = signal * 100 / 5;

            MM_MODEM_MBM_GET_PRIVATE (serial)->signal_quality = signal;
            mm_callback_info_set_result (info, GUINT_TO_POINTER (signal), NULL);
        } else {
            info->error = g_error_new_literal (MM_MODEM_ERROR, MM_MODEM_ERROR_GENERAL,
                                               "Could not parse signal quality results");
        }
    }

    mm_callback_info_schedule (info);
}

static void
get_signal_quality (MMModemGsmNetwork *modem,
                    MMModemUIntFn callback,
                    gpointer user_data)
{
    MMCallbackInfo *info;
	MMModemMbmPrivate *priv = MM_MODEM_MBM_GET_PRIVATE (modem);
	info = mm_callback_info_uint_new (MM_MODEM (modem), callback, user_data);
    g_debug("mbm get_signal_quality");

	if (priv->bModemEnabled) {
    mm_serial_queue_command (MM_SERIAL (modem), "+CIND?", 3, get_signal_quality_done, info);
    } else
    {
        info->error = g_error_new (MM_MODEM_ERROR, MM_MODEM_ERROR_GENERAL,
                                   "%s", "Device not connected");
        mm_callback_info_schedule (info);
    }
}

#if 0
get_con_state_done (MMSerial *serial, gpointer user_data)
{
    MMCallbackInfo *info = (MMCallbackInfo *) user_data;
    MMModemMbmPrivate *priv = MM_MODEM_MBM_GET_PRIVATE (MM_MODEM(serial));
    g_debug("%s", __FUNCTION__);
    mm_callback_info_set_result (info, GUINT_TO_POINTER (priv->bEnap_done), NULL);
    mm_callback_info_schedule ((MMCallbackInfo *) user_data);
}

static void
get_con_state (MMModemGsmNetwork *modem,
               MMModemUIntFn callback,
               gpointer user_data)
{
    MMCallbackInfo *info;
    MMModemMbmPrivate *priv = MM_MODEM_MBM_GET_PRIVATE (modem);
    info = mm_callback_info_uint_new (modem, callback, user_data);

    
    g_debug("mbm get_con_state priv=%d", priv);
    if (priv) {
        mm_serial_flash (MM_SERIAL (modem), 1000, get_con_state_done, info);
    } else
    {
        info->error = g_error_new (MM_MODEM_ERROR, MM_MODEM_ERROR_GENERAL,
                                   "%s", "Device not connected");
        mm_callback_info_schedule (info);
    }
}
#endif

static void
   get_con_state_done (MMSerial *serial,
                          GString *response,
                          GError *error,
                          gpointer user_data)
{
    MMCallbackInfo *info = (MMCallbackInfo *) user_data;
    const char *reply = response->str;

    g_debug("get_con_state_done");
    if (error)
        info->error = g_error_copy (error);
    else {
        if (g_str_has_prefix (reply, "*ENAP: ")) {
            int bEnabled;
            if (sscanf (reply + 7, "%d", &bEnabled)) {
                mm_callback_info_set_result (info, GUINT_TO_POINTER (bEnabled), NULL);
            } else
            {
                info->error = g_error_new (MM_MODEM_ERROR, MM_MODEM_ERROR_GENERAL,
                                           "%s", "Could not parse network mode results");
            }
        } else
        {
            info->error = g_error_new (MM_MODEM_ERROR, MM_MODEM_ERROR_GENERAL,
                                       "%s", "Could not parse network mode results");
        }
    }

    mm_callback_info_schedule (info);
}

static void
   get_con_state (MMModemGsmNetwork *modem,
                     MMModemUIntFn callback,
                     gpointer user_data)
{
    MMCallbackInfo *info;
    MMModemMbmPrivate *priv = MM_MODEM_MBM_GET_PRIVATE (modem);
    info = mm_callback_info_uint_new (MM_MODEM (modem), callback, user_data);
    g_debug("get_con_state");
    if (priv->bModemEnabled) {
        mm_serial_queue_command (MM_SERIAL (modem), "AT*ENAP?", 3, get_con_state_done, info);
    } else
    {
        info->error = g_error_new (MM_MODEM_ERROR, MM_MODEM_ERROR_GENERAL,
                                   "%s", "Device not connected");
        mm_callback_info_schedule (info);
    }
}

power_state_done (MMModem *modem,
                  GError *error,
                  gpointer user_data)
{
    MMCallbackInfo *info = (MMCallbackInfo *) user_data;

    if (error)
        info->error = g_error_copy (error);

    g_debug("power_state_done");

    mm_callback_info_schedule (info);
}

power_state (MMModemGsmNetwork *modem,
             int new_state,
             MMModemFn callback,
             gpointer user_data)
{
    MMCallbackInfo *info;
    MMModemMbmPrivate *priv = MM_MODEM_MBM_GET_PRIVATE (modem);
    info = mm_callback_info_new (MM_MODEM (modem), callback, user_data);
    g_debug("mbm power_state");

    if (new_state == 0) {
        mbm_enable (modem, FALSE, power_state_done, info);
    } else
    {
        mbm_enable (modem, TRUE, power_state_done, info);
    }
}

/*****************************************************************************/

static void
impl_mbm_auth_done (MMModem *modem,
                    GError *error,
                    gpointer user_data)
{
    DBusGMethodInvocation *context = (DBusGMethodInvocation *) user_data;

    if (error)
        dbus_g_method_return_error (context, error);
    else
        dbus_g_method_return (context);
}


static void
impl_mbm_authenticate (MMModemMbm *self,
                       const char *username,
                       const char *password,
                       const char *pin,
                       int bEnable,
                       DBusGMethodInvocation *context)
{
    MMModemMbmPrivate *priv = MM_MODEM_MBM_GET_PRIVATE (self);

    if (priv) {
        priv->username = strdup(username);
        priv->password = strdup(password);
        priv->pin = strdup(pin);
        priv->bEnable = bEnable;
    }
    dbus_g_method_return (context);
}

/*****************************************************************************/

static void
ciev_trig (const char *str, gpointer data)
{
    int event, value;
    guint32 quality;

    if (!str) {
        return;
    }

    event = str[0] - '0';
    value = str[2] - '0';

    switch (event) {
    case 2: /* signal quality, value 0-5 */
        quality = value * 100 / 5;
        g_debug("quality = %d", quality);
        mm_modem_gsm_network_signal_quality (MM_MODEM(data), quality);
        break;
    case 9: /* roaming, value 0 or 1 */
        g_debug("%s: roaming %s\n", __FUNCTION__, value?"active":"inactive");
        break;
    default:
        g_debug("%s: got unknown event %d with value %d\n", __FUNCTION__, event, value);
    }
}

static void
e2cfun_trig (const char *str, gpointer data)
{
    int mode, fun, w_disable;
    guint32 quality;
    MMModemMbmPrivate *priv = MM_MODEM_MBM_GET_PRIVATE(data);

    if (!str) {
        return;
    }

    mode = str[0] - '0';
    fun = str[2] - '0';
    w_disable = str[4] - '0';

    switch (fun) {
    case 0: /* power down */
        priv->register_completed = 0;
        break;
    case 1: /* full functionality */
        break;
    case 4: /* flight mode */
        priv->register_completed = 0;
        break;
    }

    switch (w_disable ) {
    case 0: /* radio active */
        g_debug("radio activated, we should trigger connect here!\n");
        /* TODO: GUI to show radio active and trigger a connect */
        break;
    case 1: /* radio inactive */
        g_debug("wwan disabled");
        mm_modem_gsm_network_signal_quality (MM_MODEM(data), 0);
        /* TODO: GUI to show radio inactive */
        break;
    case 2: /* w_disable not supported */
        break;
    }
}

static gboolean
mbm_parse_response (gpointer data, GString *response, GError **error)
{
    MMModemMbmPrivate *priv = MM_MODEM_MBM_GET_PRIVATE (data);

    mm_util_strip_string (response, priv->ciev_regex, ciev_trig, data);
    mm_util_strip_string (response, priv->msg_waiting_regex, NULL, data);
    mm_util_strip_string (response, priv->e2cfun_regex, e2cfun_trig, data);

    return mm_serial_parser_v1_parse (priv->std_parser, response, error);
}

static void
mm_modem_mbm_init (MMModemMbm *self)
{
    MMModemMbmPrivate *priv = MM_MODEM_MBM_GET_PRIVATE (self);

    priv->msg_waiting_regex = g_regex_new ("\\r\\n[\\*]EMWI\\r\\n", G_REGEX_RAW | G_REGEX_OPTIMIZE, 0, NULL);
    priv->ciev_regex = g_regex_new ("\\r\\n\\+CIEV: (.,.)\\r\\n", G_REGEX_RAW | G_REGEX_OPTIMIZE, 0, NULL);
    priv->e2cfun_regex = g_regex_new ("\\r\\n[\\*]E2CFUN: (.,.,.)\\r\\n", G_REGEX_RAW | G_REGEX_OPTIMIZE, 0, NULL);

    priv->std_parser = (gpointer)mm_serial_parser_v1_new ();

    mm_serial_set_response_parser (MM_SERIAL (self), mbm_parse_response, self, NULL);
    
    priv->username = NULL;
    priv->password = NULL;
    priv->pin = NULL;
    priv->bEnable = 0;
    priv->bModemEnabled = 0;
    priv->bModemEnableRunning = 0;
    priv->register_completed = 0;
    priv->enap_completed = 0;
    priv->bCOPS_register = 0;
    priv->network_id = NULL;
}

static void
modem_init (MMModem *modem_class)
{
    modem_class->enable = mbm_enable;
    modem_class->connect = mbm_connect;
    modem_class->disconnect = mbm_disconnect;
}

static void
modem_gsm_network_init (MMModemGsmNetwork *class)
{
    class->do_register = do_register;
    class->set_apn = set_apn;
    class->set_user_pass = set_user_pass;
    class->get_network_mode = get_network_mode;
    class->get_signal_quality = get_signal_quality;
    class->power_state = power_state;
    class->get_con_state = get_con_state;
}

static GObject*
constructor (GType type,
             guint n_construct_params,
             GObjectConstructParam *construct_params)
{
    GObject *object;
    MMModemMbmPrivate *priv;

    object = G_OBJECT_CLASS (mm_modem_mbm_parent_class)->constructor (type,
                                                                      n_construct_params,
                                                                      construct_params);
    if (!object)
        return NULL;

    priv = MM_MODEM_MBM_GET_PRIVATE (object);

    if (!priv->network_device) {
        g_warning ("No network device provided");
        g_object_unref (object);
        return NULL;
    }

    priv->username = NULL;
    priv->password = NULL;
    priv->pin = NULL;
    priv->bEnable = 0;
    
    priv->register_completed = 0;
    priv->enap_completed = 0;
    priv->bCOPS_register = 0;
    priv->network_id = NULL;
    priv->bEnap_done = 0;
    
    return object;
}

static void
set_property (GObject *object, guint prop_id,
              const GValue *value, GParamSpec *pspec)
{
    MMModemMbmPrivate *priv = MM_MODEM_MBM_GET_PRIVATE (object);

    switch (prop_id) {
    case PROP_NETWORK_DEVICE:
        /* Construct only */
        priv->network_device = g_value_dup_string (value);
        break;
    default:
        G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
        break;
    }
}

static void
get_property (GObject *object, guint prop_id,
              GValue *value, GParamSpec *pspec)
{
    MMModemMbmPrivate *priv = MM_MODEM_MBM_GET_PRIVATE (object);

    switch (prop_id) {
    case PROP_NETWORK_DEVICE:
        g_value_set_string (value, priv->network_device);
        break;
    default:
        G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
        break;
    }
}
static void
finalize (GObject *object)
{
    MMModemMbmPrivate *priv = MM_MODEM_MBM_GET_PRIVATE (object);

    g_free (priv->network_device);
    mm_serial_parser_v1_destroy (priv->std_parser);
    g_regex_unref (priv->ciev_regex);
//    g_regex_unref (priv->creg_regex);
    g_regex_unref (priv->msg_waiting_regex);
    g_regex_unref (priv->e2cfun_regex);

    if (priv->username) {
        g_free(priv->username);
        priv->username = NULL;
    }
    if (priv->password) {
        g_free(priv->password);
        priv->password = NULL;
    }
    if (priv->pin) {
        g_free(priv->pin);
        priv->pin = NULL;
    }
    
    priv->register_completed = 0;
    priv->enap_completed = 0;
    priv->bCOPS_register = 0;
    priv->network_id = NULL;
    priv->bEnap_done = 0;
    
    G_OBJECT_CLASS (mm_modem_mbm_parent_class)->finalize (object);
}

static void
mm_modem_mbm_class_init (MMModemMbmClass *klass)
{
    GObjectClass *object_class = G_OBJECT_CLASS (klass);

    mm_modem_mbm_parent_class = g_type_class_peek_parent (klass);
    g_type_class_add_private (object_class, sizeof (MMModemMbmPrivate));

    /* Virtual methods */
    object_class->constructor = constructor;
    object_class->set_property = set_property;
    object_class->get_property = get_property;
    object_class->finalize = finalize;
    /* Properties */
    g_object_class_install_property
        (object_class, PROP_NETWORK_DEVICE,
         g_param_spec_string (MM_MODEM_MBM_NETWORK_DEVICE,
                              "NetworkDevice",
                              "Network device",
                              NULL,
                              G_PARAM_READWRITE | G_PARAM_CONSTRUCT_ONLY));
}

GType
mm_modem_mbm_get_type (void)
{
    static GType modem_mbm_type = 0;

    if (G_UNLIKELY (modem_mbm_type == 0)) {
        static const GTypeInfo modem_mbm_type_info = {
            sizeof (MMModemMbmClass),
            (GBaseInitFunc) NULL,
            (GBaseFinalizeFunc) NULL,
            (GClassInitFunc) mm_modem_mbm_class_init,
            (GClassFinalizeFunc) NULL,
            NULL,   /* class_data */
            sizeof (MMModemMbm),
            0,      /* n_preallocs */
            (GInstanceInitFunc) mm_modem_mbm_init,
        };

        static const GInterfaceInfo modem_iface_info = {
            (GInterfaceInitFunc) modem_init
        };

        static const GInterfaceInfo modem_gsm_network_info = {
            (GInterfaceInitFunc) modem_gsm_network_init
        };

        modem_mbm_type = g_type_register_static (MM_TYPE_GENERIC_GSM, "MMModemMbm", &modem_mbm_type_info, 0);

        g_type_add_interface_static (modem_mbm_type, MM_TYPE_MODEM, &modem_iface_info);
        g_type_add_interface_static (modem_mbm_type, MM_TYPE_MODEM_GSM_NETWORK, &modem_gsm_network_info);
    }

    return modem_mbm_type;
}
