/*
 * Copyright (C) 2007 Intel
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Authored by Neil Jagdish Patel <njp@o-hand.com>
 *
 */

#include <glib.h>
#include <glib/gstdio.h>
#include <gtk/gtk.h>
#include <gio/gio.h>

#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#include <dbus/dbus-glib.h>
#include <dbus/dbus-glib-bindings.h>

#include "launcher-config.h"
#include "launcher-defines.h"

G_DEFINE_TYPE (LauncherConfig, launcher_config, G_TYPE_OBJECT)

#define LAUNCHER_CONFIG_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE(obj, \
        LAUNCHER_TYPE_CONFIG, LauncherConfigPrivate))

struct _LauncherConfigPrivate
{
  GdkScreen   *screen;  
  GtkSettings *settings;

  GFile        *watch_dir;
  GFileMonitor *dir_monitor;

  DBusGConnection *connection;
  DBusGProxy      *proxy;
};

enum
{
  CONFIG_CHANGED,
  RESUME_EVENT,

  LAST_SIGNAL
};
static guint _config_signals[LAST_SIGNAL] = { 0 };


static gint
get_rows_for_height (gint height)
{
  gint rows = 3;

  if (height < 600)
    rows = 3;
  else if (height < 1025)
    rows = 4;
  else
    rows = 5;

  return rows;
}

static gint
get_cols_for_width (gint width)
{
  gint cols = 3;

  if (width < 801)
    cols = 3;
  else if (width < 1025)
    cols = 4;
  else
    cols = 5;

  return cols;
}

static void
calculate_sizes (LauncherConfig *cfg)
{
  gint          width;
  gint          height;

  width = CSW();
  height = CSH();
  
  cfg->win_width = width - 2* PADDING;
  cfg->win_height = CSH () - PANEL_HEIGHT - (2*PADDING);
  cfg->bar_width = cfg->win_width/5;
  
  cfg->iconview_rows = get_rows_for_height (height);
  cfg->iconview_cols = get_cols_for_width (width);
  cfg->iconview_width = cfg->bar_width * 3;;
  cfg->iconview_height = cfg->win_height-(PANEL_HEIGHT*1.3) - PADDING;
  cfg->iconview_padding = 0;
  
  cfg->icon_width = (cfg->iconview_width - ((cfg->iconview_cols+1)*cfg->iconview_padding)) / cfg->iconview_cols;
  
  cfg->icon_height = (cfg->iconview_height - ((cfg->iconview_rows+1)*cfg->iconview_padding))/cfg->iconview_rows;

  cfg->shortcut_height = (cfg->win_height/N_CATS);
}


static void
set_dpi (LauncherConfig *config)
{
  ClutterBackend *backend = clutter_get_default_backend ();
  gint dpi;

  g_object_get (config->priv->settings, "gtk-xft-dpi", &dpi, NULL);
  dpi /= 1024;

  if (dpi > 0)
  {
    clutter_backend_set_resolution (backend, dpi);
    g_debug ("Font dpi: %d", dpi);
  }
}

static void
set_font (LauncherConfig *config)
{
  gchar *temp;

  g_object_get (config->priv->settings, "gtk-font-name", &temp, NULL);

  memcpy (config->font_name, temp, strlen (temp)+1);

  g_debug ("Font changed: %s\n", config->font_name);

  g_free (temp);
}

/* Callbacks */
static void
on_font_changed (GtkSettings    *settings,
                 GParamSpec     *spec,
                 LauncherConfig *config)
{
  set_font (config);

  g_signal_emit (config, _config_signals[CONFIG_CHANGED], 0);
}

static void
on_font_dpi_changed (GtkSettings    *settings,
                     GParamSpec     *spec,
                     LauncherConfig *config)
{
  set_dpi (config);

  g_signal_emit (config, _config_signals[CONFIG_CHANGED], 0);
}

/*
 * Session switching signals
 */
static void
on_session_active_changed (DBusGProxy     *proxy, 
                           gboolean        active, 
                           LauncherConfig *cfg)
{
  g_debug ("ActiveChanged: %s", active ? "Active" : "Inactive");
  
  if (active)
  {
    g_signal_emit (cfg, _config_signals[RESUME_EVENT], 0);
  }
}

/* GObject functions */
static void
launcher_config_dispose (GObject *object)
{
  LauncherConfig *config = LAUNCHER_CONFIG (object);
  LauncherConfigPrivate *priv;
  
  g_return_if_fail (LAUNCHER_IS_CONFIG (config));
  priv = LAUNCHER_CONFIG (config)->priv;

  g_object_unref (priv->dir_monitor);
  g_object_unref (priv->watch_dir);

  G_OBJECT_CLASS (launcher_config_parent_class)->dispose (object);
}

static void
launcher_config_class_init (LauncherConfigClass *klass)
{
  GObjectClass *obj_class = G_OBJECT_CLASS (klass);

  obj_class->dispose = launcher_config_dispose;

	_config_signals[CONFIG_CHANGED] =
		g_signal_new ("config-changed",
			      G_OBJECT_CLASS_TYPE (obj_class),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET (LauncherConfigClass, config_changed),
			      NULL, NULL,
			      g_cclosure_marshal_VOID__VOID, 
			      G_TYPE_NONE, 0);

	_config_signals[RESUME_EVENT] =
		g_signal_new ("resume-event",
        G_OBJECT_CLASS_TYPE (obj_class),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET (LauncherConfigClass, resume_event),
			      NULL, NULL,
			      g_cclosure_marshal_VOID__VOID, 
			      G_TYPE_NONE, 0);


 
  g_type_class_add_private (obj_class, sizeof (LauncherConfigPrivate)); 
}

static void
launcher_config_init (LauncherConfig *config)
{
  LauncherConfigPrivate *priv;
  DBusGConnection       *connection;
  DBusGProxy            *proxy;
  GError                *error = NULL;
        
  priv = config->priv = LAUNCHER_CONFIG_GET_PRIVATE (config);

  priv->settings = gtk_settings_get_default ();
  set_dpi (config);
  g_signal_connect (priv->settings, "notify::gtk-xft-dpi",
                    G_CALLBACK (on_font_dpi_changed), config);

  set_font (config);
  g_signal_connect (priv->settings, "notify::gtk-font-name",
                    G_CALLBACK (on_font_changed), config);
   
  /* Now load in all the calculations */
  calculate_sizes (config);

  /* Session switch watching */
  priv->connection = connection = dbus_g_bus_get(DBUS_BUS_SYSTEM, &error);

  if (error)
  {
    g_warning ("%s", error->message);
    g_error_free(error);
    return;
  }
  /* Get the current session object */
  proxy = dbus_g_proxy_new_for_name(connection,
                                    "org.freedesktop.ConsoleKit",
                                    "/org/freedesktop/ConsoleKit/Manager",
                                    "org.freedesktop.ConsoleKit.Manager");

  if (!proxy)
  {
    g_warning ("Unable to get the ConsoleKit manager.");
    dbus_g_connection_unref (connection);
    return;
  }
  if (!dbus_g_proxy_call (proxy, "GetCurrentSession", &error, 
                          G_TYPE_INVALID, 
                          DBUS_TYPE_G_PROXY, &priv->proxy,
                          G_TYPE_INVALID))
  {
    g_warning ("Unable to GetCurrentSession from ConsoleKit.Manager");
    g_object_unref (proxy);
    dbus_g_connection_unref (connection);
    return;
  }
  
  dbus_g_proxy_set_interface (priv->proxy, 
                              "org.freedesktop.ConsoleKit.Session");
  dbus_g_proxy_add_signal (priv->proxy, "ActiveChanged",
                           G_TYPE_BOOLEAN, G_TYPE_INVALID);
  dbus_g_proxy_connect_signal (priv->proxy, "ActiveChanged", 
                               G_CALLBACK (on_session_active_changed), config, 
                               NULL);

  g_object_unref (proxy);
}

LauncherConfig*
launcher_config_get_default (void)
{
  static LauncherConfig *config = NULL;
  
  if (config == NULL)
    config = g_object_new (LAUNCHER_TYPE_CONFIG, 
                            NULL);

  return config;
}
