/*
 * Copyright (C) 2008 Canonical Ltd
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as 
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by Neil Jagdish Patel <neil.patel@canonical.com>
 *
 */

#include "task-list.h"

#include <libwnck/libwnck.h>

G_DEFINE_TYPE (TaskList, task_list, GTK_TYPE_HBOX);

#define TASK_LIST_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj),\
  TASK_TYPE_LIST, \
  TaskListPrivate))

struct _TaskListPrivate
{
  WnckScreen *screen;
  
  GHashTable *win_table;
};

/* D&D stuff */
static const GtkTargetEntry drop_types[] =
{
  { "STRING", 0, 0 },
  { "text/plain", 0, 0},
  { "text/uri-list", 0, 0}
};
static const gint n_drop_types = G_N_ELEMENTS(drop_types);

/* Callback */
static gboolean
on_leave_notify_event (GtkWidget        *button,
                       GdkEventCrossing *event,
                       TaskList         *list)
{
  g_return_val_if_fail (TASK_IS_LIST (list), FALSE);

  gtk_widget_queue_draw (GTK_WIDGET (list));

  return FALSE;
}

static gboolean
on_button_clicked (GtkWidget      *button, 
                   GdkEventButton *event,
                   WnckWindow     *window)
{
  g_return_val_if_fail (WNCK_IS_WINDOW (window), FALSE);

  if (event->button == 1)
  {
    WnckWorkspace *space;
    space = wnck_window_get_workspace (window);
    wnck_workspace_activate (space, event->time);
    wnck_window_activate (window, event->time);
    return TRUE;
  }
  else if (event->button == 3)
  {
    GtkWidget *menu = wnck_action_menu_new (window);
    gtk_menu_popup (GTK_MENU (menu), NULL, NULL, NULL, NULL, 
                    event->button, event->time);

    return TRUE;
  }

  return FALSE;
}

static gboolean
on_query_tooltip (GtkWidget *button,
                  gint x, gint y, 
                  gboolean keyboard_mode, 
                  GtkTooltip *tooltip,
                  WnckWindow *window)
{
  g_return_val_if_fail (WNCK_IS_WINDOW (window), FALSE);

  gtk_tooltip_set_text (tooltip, wnck_window_get_name(window));
  gtk_tooltip_set_icon (tooltip, wnck_window_get_icon (window));

  return TRUE;
}

static gboolean
activate_window (GtkWidget *button)
{
  gint active;

  active = GPOINTER_TO_INT (g_object_get_data (G_OBJECT (button), "drag-true"));

  if (active)
  { 
    WnckWindow *window;

    window = (WnckWindow*)g_object_get_data (G_OBJECT (button), "WnckWindow");
    if (WNCK_IS_WINDOW (window))
      wnck_window_activate (window, time (NULL));
  }
  
  g_object_set_data (G_OBJECT (button), "drag-true", GINT_TO_POINTER (0));
  
  return FALSE;
}

static void
on_drag_leave (GtkWidget      *button,
               GdkDragContext *context,
               guint           time)
{
  g_object_set_data (G_OBJECT (button), "drag-true", GINT_TO_POINTER (0));
}

static gboolean
on_drag_motion (GtkWidget      *button,
                GdkDragContext *context,
                gint            x, 
                gint            y,
                guint           t)
{
  gint active;

  active = (gint)g_object_get_data (G_OBJECT (button), "drag-true");
  
  if (!active)
  {
    g_object_set_data (G_OBJECT (button), "drag-true", GINT_TO_POINTER (1));

    g_timeout_add (1000, (GSourceFunc)activate_window, button);
  }

  return FALSE;
}

static void
on_window_opened (WnckScreen *screen, 
                  WnckWindow *window,
                  TaskList   *list)
{
  TaskListPrivate *priv;
  WnckWindowType type;
  GtkWidget *button;

  g_return_if_fail (TASK_IS_LIST (list));
  priv = list->priv;

  type = wnck_window_get_window_type (window);

  if (wnck_window_is_skip_tasklist (window)
      || type == WNCK_WINDOW_DOCK
      || type == WNCK_WINDOW_SPLASHSCREEN
      || type == WNCK_WINDOW_MENU)
    return;

  button = g_object_new (GTK_TYPE_TOGGLE_BUTTON,
                         "has-tooltip", TRUE,
                         "image", gtk_image_new_from_pixbuf (
                                    wnck_window_get_mini_icon (window)),
                         "name", "tasklist-button", 
                         NULL);
  /* D&D */
	gtk_widget_add_events (GTK_WIDGET (button),GDK_ALL_EVENTS_MASK);
	gtk_drag_dest_set (GTK_WIDGET (button),
                           GTK_DEST_DEFAULT_MOTION | GTK_DEST_DEFAULT_DROP,
                           drop_types, n_drop_types,
                           GDK_ACTION_COPY);
	gtk_drag_dest_add_uri_targets (GTK_WIDGET (button));
	gtk_drag_dest_add_text_targets (GTK_WIDGET (button));
  g_signal_connect (button, "drag-motion", 
                    G_CALLBACK (on_drag_motion), NULL);
  g_signal_connect (button, "drag-leave", 
                    G_CALLBACK (on_drag_leave), NULL);
 
  gtk_widget_show (button);
  
  gtk_box_pack_start (GTK_BOX (list), button, FALSE, FALSE, 0);

  g_signal_connect (button, "button-release-event",
                    G_CALLBACK (on_button_clicked), window);
  g_signal_connect (button, "leave-notify-event",
                    G_CALLBACK (on_leave_notify_event), list);
  g_signal_connect (button, "query-tooltip", 
                    G_CALLBACK (on_query_tooltip), window);

  g_hash_table_insert (priv->win_table, window, button);
  g_object_set_data (G_OBJECT (button), "WnckWindow", window);
  
  wnck_window_activate (window, GDK_CURRENT_TIME);
}

static void
on_window_closed (WnckScreen *screen, 
                  WnckWindow *window,
                  TaskList   *list)
{
  TaskListPrivate *priv;
  GtkWidget *button;

  g_return_if_fail (TASK_IS_LIST (list));
  priv = list->priv;

  button = g_hash_table_lookup (priv->win_table, window);

  if (GTK_IS_WIDGET (button))
    gtk_widget_destroy (button);

  g_hash_table_remove (priv->win_table, window);
}

static void
on_active_window_changed (WnckScreen *screen, 
                          WnckWindow *old_window,
                          TaskList   *list)
{
  TaskListPrivate *priv;
  WnckWindow *act_window;
  GtkWidget *old_button;
  GtkWidget *act_button;

  g_return_if_fail (TASK_IS_LIST (list));
  priv = list->priv;

  old_button = g_hash_table_lookup (priv->win_table, old_window);
  if (GTK_IS_WIDGET (old_button))
  {
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (old_button), FALSE);
  }

  act_window = wnck_screen_get_active_window (priv->screen);
  act_button = g_hash_table_lookup (priv->win_table, act_window);
  if (GTK_IS_WIDGET (act_button))
  {
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (act_button), TRUE);
  }
}


/* GObject stuff */
static void
task_list_finalize (GObject *object)
{
  TaskListPrivate *priv;

  priv = TASK_LIST_GET_PRIVATE (object);

  G_OBJECT_CLASS (task_list_parent_class)->finalize (object);
}

static void
task_list_class_init (TaskListClass *klass)
{
  GObjectClass        *obj_class = G_OBJECT_CLASS (klass);

  obj_class->finalize = task_list_finalize;

  g_type_class_add_private (obj_class, sizeof (TaskListPrivate));
}

static void
task_list_init (TaskList *list)
{
  TaskListPrivate *priv;
 	
  priv = list->priv = TASK_LIST_GET_PRIVATE (list);

  priv->screen = wnck_screen_get_default ();

  priv->win_table = g_hash_table_new (NULL, NULL);

  gtk_container_set_border_width (GTK_CONTAINER (list), 0);

  g_signal_connect (priv->screen, "window-opened",
                    G_CALLBACK (on_window_opened), list);
  g_signal_connect (priv->screen, "window-closed",
                    G_CALLBACK (on_window_closed), list);
  g_signal_connect (priv->screen, "active-window-changed",
                    G_CALLBACK (on_active_window_changed), list);
}

GtkWidget *
task_list_new (void)

{
  GtkWidget *list = NULL;

  list = g_object_new (TASK_TYPE_LIST, 
                       "homogeneous", FALSE, 
                       "spacing", 2, 
                       NULL);

  return list;
}
