/** -*- C++ -*-
    @file adept/filterlist.h
    @author Peter Rockai <me@mornfall.net>
*/

#include <qtooltip.h>
#include <kdebug.h>

#include <apt-front/predicate/matchers.h>
#include <apt-front/predicate/combinators.h>
#include <apt-front/predicate/factory.h>
#include <apt-front/cache/entity/entity.h>
#include <apt-front/cache/component/packagetags.h>
#include <apt-front/cache/component/tags.h>

#include <adept/listerpredicate.h>
#include <adept/extendablelist.h>

#ifndef EPT_FILTERLIST_H
#define EPT_FILTERLIST_H

class QPopupMenu;

namespace adept {
using namespace aptFront;
using namespace aptFront::cache;
class Lister;

struct InterfacingPredicate
{
public:
    virtual std::string summary() const = 0;
    virtual void reset() {}
signals:
    void changed();
};


class PredicateInterface: public ItemExtender
{
    Q_OBJECT
public:
    typedef predicate::Predicate< entity::Entity > Predicate;
    PredicateInterface( QWidget *w = 0, const char *n = 0 );
    virtual Predicate predicate() = 0;
signals:
    void predicateAdd( Predicate );
    void predicateDrop( Predicate );
public slots:
    virtual void setPredicate( const Predicate &p ) {
        m_pred = p;
        predicateChanged();
    }
    virtual void widgetsChanged();
    virtual void predicateChanged() = 0;
protected:
    predicate::Predicate< entity::Entity > m_pred;
};

class FilterItem;

class FilterList : public ExtendableList
{
    Q_OBJECT
public:
    friend class FilterItem;
    typedef predicate::Predicate< entity::Entity > Predicate;
    // typedef std::map< KListViewItem *, Predicate > Map;
    FilterList( QWidget *parent = 0, const char *name = 0 );
    typedef predicate::And< entity::Entity > And;
    void appendPredicate( Predicate p );
    void plugLister( Lister *l );
    void emitPredicateChanged();
signals:
    void predicateChanged( ListerPredicate p );
public slots:
    void setPredicate( Predicate p );
    void setHiddenPredicate( Predicate p );
protected slots:
    void editorPredicateDrop( Predicate );
    void editorPredicateAdd( Predicate );
    void contextMenu( QListViewItem *, const QPoint &, int );
    // QPopupMenu *tagMenu();
    void contextActivated( int );
    // void tagMenuActivated( int );
    void updateHeight();
protected:
    void drawContents( QPainter *, int, int, int, int );
    Predicate m_pred;
    Predicate m_hidden;
    FilterItem *m_context;
    bool m_changed;
    Lister *m_lister;
    int m_itemsSeen;
    std::vector< entity::Tag > m_tagMenuMap;
};

class FilterItem : public ExtendableItem
{
public:
    ItemExtender *createExtender();
    FilterItem( FilterList *l ) 
        : ExtendableItem( l ), m_delayedDone( false )
        {
        }
    void setup() {
        ExtendableItem::setup();
        if (m_delayedDone)
            return;
        m_delayedDone = true;
        showExtender();
    }

    QString text( int ) const;

    virtual bool less( const ExtendableItem *other ) const {
        const FilterItem *o = dynamic_cast< const FilterItem * >( other );
        return downcast< InterfacingPredicate >( m_pred ).summary() <
            downcast< InterfacingPredicate >( o->m_pred ).summary();
    }

    void setPredicate( predicate::Predicate< entity::Entity > p ) {
        m_pred = p;
    }

    FilterList::Predicate predicate() const {
        return m_pred;
    }

    FilterList *filterList() { return dynamic_cast< FilterList * >( list() ); }

    void reset() {
        filterList()->editorPredicateDrop( m_pred );
        downcast< InterfacingPredicate >( m_pred ).reset();
        if ( extender() ) {
            downcast< PredicateInterface >( extender() ).setPredicate( m_pred );
            extender()->setFocus();
        }
        filterList()->editorPredicateAdd( m_pred );
    }

protected:
    predicate::Predicate< entity::Entity > m_pred;
    bool m_delayedDone:1;
};

}

#endif
