/*
 * debtags - Implement package tags support for Debian
 *
 * Copyright (C) 2003--2006  Enrico Zini <enrico@debian.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "SmartSearcher.h"
#include "Printer.h"
#include <ept/apt/apt.h>
#include <ept/apt/packagerecord.h>
#include <tagcoll/utils/set.h>
#include <wibble/regexp.h>

using namespace std;
using namespace ept::apt;
using namespace tagcoll;

static string toLower(const std::string& s)
{
	string res;
	for (string::const_iterator i = s.begin(); i != s.end(); ++i)
		res += tolower(*i);
	return res;
}

void SmartSearcher::splitPattern(const std::string& pattern)
{
	patterns.clear();

	wibble::Tokenizer tok(pattern, "[^[:blank:]]+", REG_EXTENDED);
	for (wibble::Tokenizer::const_iterator i = tok.begin();
			i != tok.end(); ++i)
		patterns.push_back(toLower(*i));
}

bool SmartSearcher::patternMatch(const Package& pkg)
{
	using namespace ept;

	std::string name = toLower(pkg);
	std::string desc;
	PackageRecord record(apt.rawRecord(pkg));
	desc = toLower(record.description());

	for (std::vector<std::string>::const_iterator i = patterns.begin();
			i != patterns.end(); ++i)
		if (name.find(*i) == string::npos
				&& desc.find(*i) == string::npos)
			return false;
	return true;
}

bool SmartSearcher::tagMatch(const Package& pkg)
{
	using namespace wibble::operators;
	std::set<Tag> tags = debtags.getTagsOfItem(pkg);

	if (!wanted.empty() && !utils::set_contains(tags, wanted))
		return false;

	if (!unwanted.empty() && !(tags & unwanted).empty())
		return false;

	return true;
}

void SmartSearcher::showSet(const std::set<Tag>& tags, const std::string& type)
{
	for (std::set<Tag>::const_iterator i = tags.begin();
			i != tags.end(); ++i)
	{
		tagsInMenu.push_back(*i);
		cout << tagsInMenu.size() << ") " << i->fullname() << " (" << type << ")" << endl;
	}
}

void SmartSearcher::showInteresting(int max)
{
	for (std::vector<Tag>::const_reverse_iterator i = interesting.rbegin();
			i != interesting.rend() && max > 0; ++i)
	{
		using namespace wibble::operators;
		if (utils::set_contains(wanted, *i)
			|| utils::set_contains(unwanted, *i)
			|| utils::set_contains(ignored, *i)
			|| coll.getCardinality(*i) == 0)
			continue;
		tagsInMenu.push_back(*i);
		cout << tagsInMenu.size() << ") " << i->fullname()
			 << " (" << coll.getCardinality(*i) << "/" << coll.itemCount() << ")" << endl;
		--max;
	}
}

void SmartSearcher::showDiscriminant(int max)
{
	// Compute the most interesting tags by discriminance
	vector<Tag> discr = coll.tagsInDiscriminanceOrder();

	for (std::vector<Tag>::const_reverse_iterator i = discr.rbegin();
			i != discr.rend() && max > 0; ++i)
	{
		using namespace wibble::operators;
		if (utils::set_contains(wanted, *i)
			|| utils::set_contains(unwanted, *i)
			|| utils::set_contains(ignored, *i))
			continue;
		tagsInMenu.push_back(*i);
		cout << tagsInMenu.size() << ") " << i->fullname()
			 << " (" << coll.getCardinality(*i) << "/" << coll.itemCount() << ")" << endl;
		--max;
	}
}

void SmartSearcher::showTags()
{
	tagsInMenu.clear();

	showSet(wanted, "wanted");
	showSet(unwanted, "unwanted");
	showSet(ignored, "ignored");
	cout << endl;
	showInteresting();
	cout << endl;
	showDiscriminant();
}

void SmartSearcher::refilter()
{
	// Regenerate coll
	coll = coll::Fast<Package, Tag>();
	fullColl.output(filter(inserter(coll)));
}

void SmartSearcher::computeInteresting(const std::string& pattern)
{
	splitPattern(pattern);

	coll::Fast<Package, Tag> filtered;
	for (coll::Fast<Package, Tag>::const_iterator i = fullColl.begin();
			i != fullColl.end(); ++i)
		if (patternMatch(i->first))
			filtered.insert(wibble::singleton(i->first), i->second);

	interesting = filtered.tagsInRelevanceOrder(fullColl);
#if 0
	// Compute the set of tags that better represent the keyword search
	TagMetrics<Tag, int> metrics1 = TagMetrics<Tag, int>::computeFromTags(fullColl);
	TagMetrics<Tag, int> metrics2 = TagMetrics<Tag, int>::computeFromTags(filtered);
#if 0
	// Use the jump algorithm
	TagMetrics<Tag, double> jumps = metrics2.jumpsFrom(metrics1);
	interesting = jumps.tagsSortedByMetrics();
#else
	// Use the reduction percentage algorithm (a bit faster, and more
	// optimizable)
	interesting = metrics2.tagsSortedByRelevance(metrics1);
#endif
#endif
}

SmartSearcher::SmartSearcher(const std::string& pattern)
	: apt(env().apt()), debtags(env().debtags())
{
	// Perform the initial filtering using the keyword search
	Apt& apt = env().apt();
	for (Apt::iterator i = apt.begin(); i != apt.end(); ++i)
		fullColl.insert(wibble::singleton(*i), debtags.getTagsOfItem(*i));

	computeInteresting(pattern);

	coll = fullColl;
}

void SmartSearcher::interact()
{
	bool done = false;
	while (!done)
	{
		cout << "Tag selection:" << endl;
		showTags();
		cout << coll.itemCount() << " packages selected so far." << endl;
		string ans;
		bool changed = false;

		// TODO: allow to add tags based on a keyword search on coll
		cout << "Your choice (+#, -#, =#, K word, View, Done, Quit, ?): ";
		if (!getline(cin, ans))
		{
			cout << endl;
			done = true;
		}

		while (ans != "")
		{
			// If we're setting a new keyword search, process now and skip
			// processing as a list
			if (ans[0] == '?')
			{
				cout << "+ number  select the tag with the given number as a tag you want" << endl
					 << "- number  select the tag with the given number as a tag you do not want" << endl
					 << "= number  select the tag with the given number as a tag you don't care about" << endl
					 << "K word    recompute the set of interesting tags from a full-text search using the given word" << endl
					 << "V         view the packages selected so far" << endl
					 << "D         print the packages selected so far and exit" << endl
					 << "Q         quit debtags smart search" << endl
					 << "?         print this help information" << endl;
			}
			else if (ans[0] == 'k' || ans[0] == 'K')
			{
				// Strip initial command and empty spaces
				ans = ans.substr(1);
				while (!ans.empty() && isspace(ans[0]))
					ans = ans.substr(1);
				if (ans == "")
					cout << "The 'k' command needs a keyword to use for finding new interesting tags." << endl;
				else
					computeInteresting(ans);
				ans.clear();
			} else {
				// Split the answer by spaces
				string rest;
				size_t pos = ans.find(" ");
				if (pos != string::npos)
				{
					// Skip spaces
					for ( ; pos < ans.size() && isspace(ans[pos]); ++pos)
						;
					rest = ans.substr(pos);
					ans = ans.substr(0, pos);
				}
				if (ans[0] == '+' || ans[0] == '-' || ans[0] == '=') {
					int idx = strtoul(ans.substr(1).c_str(), NULL, 10);
					if (idx <= 0 || (unsigned)idx > tagsInMenu.size())
						cout << "Tag " << idx << " was not on the menu." << endl;
					else
					{
						Tag tag = tagsInMenu[idx - 1];
						//cout << "Understood " << ans << " as " << ans[0] << tag.fullname() << endl;

						switch (ans[0])
						{
							case '+':
								wanted.insert(tag);
								unwanted.erase(tag);
								ignored.erase(tag);
								break;
							case '-':
								wanted.erase(tag);
								unwanted.insert(tag);
								ignored.erase(tag);
								break;
							case '=':
								wanted.erase(tag);
								unwanted.erase(tag);
								ignored.insert(tag);
								break;
						}
						changed = true;
					}
				} else if (ans == "V" || ans == "v") {
					coll.output(PackagePrinter(PackagePrinter::SHORT));
				} else if (ans == "D" || ans == "d") {
					coll.output(PackagePrinter(PackagePrinter::SHORT));
					done = true;
				} else if (ans == "Q" || ans == "q") {
					done = true;
				} else {
					cout << "Ignoring command \"" << ans << "\"" << endl;
				}
				ans = rest;
			}
		}
		if (changed)
			refilter();
	}
}

void SmartSearcher::outputRelevantTags()
{
	for (std::vector<Tag>::const_iterator i = interesting.begin();
			i != interesting.end(); ++i)
	{
		using namespace wibble::operators;
		if (utils::set_contains(wanted, *i)
			|| utils::set_contains(unwanted, *i)
			|| utils::set_contains(ignored, *i)
			|| coll.getCardinality(*i) == 0)
			continue;
		cout << i->fullname() << " - " << i->shortDescription() << endl;
	}
}

void SmartSearcher::outputDiscriminantTags()
{
	// Compute the most interesting tags by discriminance
	coll::Fast<Package, Tag> filtered;
	for (coll::Fast<Package, Tag>::const_iterator i = fullColl.begin();
			i != fullColl.end(); ++i)
		if (patternMatch(i->first))
			filtered.insert(wibble::singleton(i->first), i->second);

	vector<Tag> discr = filtered.tagsInDiscriminanceOrder();

	for (std::vector<Tag>::const_iterator i = discr.begin();
			i != discr.end(); ++i)
	{
		using namespace wibble::operators;
		if (utils::set_contains(wanted, *i)
			|| utils::set_contains(unwanted, *i)
			|| utils::set_contains(ignored, *i))
			continue;
		cout << i->fullname() << " - " << i->shortDescription()
			 << " (" << filtered.getDiscriminance(*i) * 200 /filtered.itemCount() << "%)" << endl;
	}
}

#include <tagcoll/coll/fast.tcc>

// vim:set ts=4 sw=4:
