/*
 * debtags - Implement package tags support for Debian
 *
 * Copyright (C) 2003--2006  Enrico Zini <enrico@debian.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef DEBTAGS_SMARTSEARCHER_H
#define DEBTAGS_SMARTSEARCHER_H

#include "Environment.h"
#include <tagcoll/coll/fast.h>

#include <vector>

#if 0
template<typename Tag, typename Number = int>
class TagMetrics : public std::map<Tag, Number>
{
	class MetricsOrder
	{
		const TagMetrics& metric;
	public:
		MetricsOrder(const TagMetrics& metric) : metric(metric) {}
		bool operator()(const Tag& t1, const Tag& t2)
		{
			// Returns true if t1 precedes t2, and false otherwise
			return metric.get(t1) < metric.get(t2);
		}
	};
	class DiscriminanceOrder
	{
		const TagMetrics& metric;
		int itemCount;
	public:
		DiscriminanceOrder(const TagMetrics& metric, int itemCount)
			: metric(metric), itemCount(itemCount) {}
		bool operator()(const Tag& t1, const Tag& t2)
		{
			// Returns true if t1 precedes t2, and false otherwise
			return metric.discriminance(t1, itemCount) < metric.discriminance(t2, itemCount);
		}
	};
	class RelevanceOrder
	{
		const TagMetrics& first;
		const TagMetrics& second;
	public:
		RelevanceOrder(const TagMetrics& first, const TagMetrics& second)
			: first(first), second(second) {}
		bool operator()(const Tag& t1, const Tag& t2)
		{
			// Replace 10000* with (double) to be formally precise but
			// use floating point aritmetics
			return 10000 * second.get(t1) / first.get(t1) < 10000 * second.get(t2) / first.get(t2);
		}
	};

	#if 0
	TagMetrics<Tag, double> jumpsFrom(const TagMetrics<Tag, Number>& other) const
	{
		TagMetrics<Tag, double> res;
		for (typename TagMetrics<Tag, Number>::const_iterator i = this->begin(); i != this->end(); ++i)
		{
			//cout << i->first.fullname() << " was " << other.get(i->first) << " is " << i->second << " kept " << i->second * 100 / other.get(i->first) << "%" << endl;
			res.add(i->first, (double)i->second / other.get(i->first));
		}
		return res;
	}
	#endif

public:
	void add(const Tag& tag, const Number& val)
	{
		typename TagMetrics::iterator i = this->find(tag);
		if (i == this->end())
			insert(make_pair(tag, val));
		else
			i->second += val;
	}

	Number get(const Tag& tag) const
	{
		typename TagMetrics::const_iterator i = this->find(tag);
		if (i == this->end())
			return 0;
		else
			return i->second;
	}

	// Get the minimum number of packages that would be eliminated by choosing
	// this tag either as 'wanted' or as 'unwanted'
	Number discriminance(const Tag& tag, Number itemCount) const
	{
		return get(tag) < itemCount - get(tag) ? get(tag) : itemCount - get(tag);
	}

	TagMetrics<Tag, Number> rankMetrics() const
	{
		vector<Tag> sorted = tagsSortedByMetrics();
		TagMetrics<Tag, Number> res;
		for (size_t i = 0; i < sorted.size(); ++i)
			res.add(sorted[i], sorted.size() - i);
		return res;
	}

	TagMetrics<Tag, double> jumpsFrom(const TagMetrics<Tag, Number>& other) const
	{
		// Compute rank metrics
		TagMetrics<Tag, Number> rank1 = other.rankMetrics();
		TagMetrics<Tag, Number> rank2 = rankMetrics();

		TagMetrics<Tag, double> res;
		typename TagMetrics::const_iterator i1 = rank1.begin();
		typename TagMetrics::const_iterator i2 = rank2.begin();
		while (i1 != rank1.end() || i2 != rank2.end())
		{
			if (i1->first == i2->first)
			{
				res.add(i1->first, (i1->second - i2->second) / i2->second);
				++i1;
				++i2;
			} else if (i1 == rank1.end() || (i2 != rank2.end() && i2->first < i1->first)) {
				++i2;
			} else {
				++i1;
			}
		}
		return res;
	}


	vector<Tag> tagsSortedByMetrics() const
	{
		vector<Tag> res;
		for (typename TagMetrics::const_iterator i = this->begin(); i != this->end(); ++i)
			res.push_back(i->first);
		std::sort(res.begin(), res.end(), MetricsOrder(*this));
		return res;
	}

	vector<Tag> tagsSortedByDiscriminance(int itemCount) const
	{
		vector<Tag> res;
		for (typename TagMetrics::const_iterator i = this->begin(); i != this->end(); ++i)
			res.push_back(i->first);
		std::sort(res.begin(), res.end(), DiscriminanceOrder(*this, itemCount));
		return res;
	}

	vector<Tag> tagsSortedByRelevance(const TagMetrics<Tag, Number>& other)
	{
		vector<Tag> res;
		for (typename TagMetrics::const_iterator i = this->begin(); i != this->end(); ++i)
			res.push_back(i->first);
		std::sort(res.begin(), res.end(), RelevanceOrder(other, *this));
		return res;
	}

	template<typename COLL>
	static TagMetrics<Tag, Number> computeFromTags(const COLL& coll)
	{
		TagMetrics<Tag, Number> res;
		for (typename COLL::const_tag_iterator i = coll.tagBegin();
				i != coll.tagEnd(); ++i)
			res.add(i->first, i->second.size());
		return res;
	}

	void dump(const std::string& prefix, ostream& out)
	{
		TagMetrics<Tag, Number> rm = rankMetrics();
		vector<Tag> tags = tagsSortedByMetrics();
		int rank = 0;
		for (typename vector<Tag>::const_iterator i = tags.begin(); i != tags.end(); ++i, ++rank)
			out << prefix << tags.size() - rank << ":" << rm.get(*i) << ") " << i->fullname() << ": " << this->get(*i) << std::endl;
	}
};

template<typename Metrics>
class TagMetricsInserter : public wibble::mixin::OutputIterator< TagMetricsInserter<Metrics> >
{
	Metrics& m;

public:
	TagMetricsInserter(Metrics& m) : m(m) {}

	template<typename Items, typename Tags>
	TagMetricsInserter<Metrics>& operator=(const std::pair<Items, Tags>& data)
	{
		int size = data.first.size();
		for (typename Tags::const_iterator i = data.second.begin();
				i != data.second.end(); ++i)
			m.add(*i, size);
		return *this;
	}
};

template<typename Tag, typename Number>
TagMetricsInserter< TagMetrics<Tag, Number> > tagMetricsInserter(TagMetrics<Tag, Number>& out)
{
	return TagMetricsInserter< TagMetrics<Tag, Number> >(out);
}
#endif


class SmartSearcher
{
protected:
	typedef std::string Package;
	typedef ept::debtags::Tag Tag;

	ept::apt::Apt& apt;
	ept::debtags::Debtags& debtags;

	// TODO:
	//  1) funzione di ranking dei tag che calcola, sul subset indotto da
	//     {wanted, unwanted}, la rilevanza dei tag dopo il filtraggio con la
	//     keyword
	//	2) iterare usando la nuova funzione di ranking
	tagcoll::coll::Fast<Package, Tag> fullColl;
	tagcoll::coll::Fast<Package, Tag> coll;

	std::vector<std::string> patterns;
	std::set<Tag> wanted;
	std::set<Tag> unwanted;
	std::set<Tag> ignored;
	std::vector<Tag> interesting;

	std::vector<Tag> tagsInMenu;

	void splitPattern(const std::string& pattern);
	bool patternMatch(const Package& pkg);
	bool tagMatch(const Package& pkg);

	template<typename OUT>
	class Filter : public wibble::mixin::OutputIterator< Filter<OUT> >
	{
		SmartSearcher& s;
		OUT out;

	public:
		Filter(SmartSearcher& s, const OUT& out) : s(s), out(out) {}

		template<typename ITEMS, typename TAGS>
		Filter<OUT>& operator=(const std::pair<ITEMS, TAGS>& data)
		{
			for (typename ITEMS::const_iterator i = data.first.begin();
					i != data.first.end(); ++i)
			{
				if (!s.tagMatch(*i))
					continue;
#if 0
				bool matches = s.patternMatch(*i);
				for (typename TAGS::const_iterator t = data.second.begin();
						t != data.second.end(); ++t)
				{
					r.incFull(*t);
					++r.totalFull;
					if (matches)
					{
						r.incFilt(*t);
						++r.totalFilt;
					}
				}
#endif
				*out = data;
				++out;
			}
			return *this;
		}
	};
	template<typename OUT>
	Filter<OUT> filter(const OUT& out)
	{
		return Filter<OUT>(*this, out);
	}

	#if 0
	void autoSelect(const std::vector<Tag>& tags, size_t maxAuto = 5, size_t maxUser = 7)
	{
		interesting.clear();
		if (tags.empty())
			return;

		size_t autoCount = (tags.size() - maxUser) / 2;
		if (autoCount > maxAuto) autoCount = maxAuto;
		size_t userCount = tags.size() - (2 * autoCount);
		if (userCount > maxUser) userCount = maxUser;

		// Use the bottom autoCount tags as unwanted
		for (size_t i = 0; i < autoCount; ++i)
			unwanted.insert(tags[i]);

		// Use the top autoCount tags as wanted
		for (size_t i = tags.size() - 1; i >= tags.size() - autoCount; --i)
			wanted.insert(tags[i]);

		// Get the next userCount packages as interesting
		for (size_t i = tags.size() - autoCount - 1; i >= tags.size() - autoCount - userCount; --i)
			interesting.push_back(tags[i]);
	}
	#endif

	void showSet(const std::set<Tag>& tags, const std::string& type);
	void showInteresting(int max = 7);
	void showDiscriminant(int max = 7);
	void showTags();
	void refilter();
	void computeInteresting(const std::string& pattern);

public:
	SmartSearcher(const std::string& pattern);

	void interact();
	void outputRelevantTags();
	void outputDiscriminantTags();


#if 0
	component::PackageTags& debtags;
	Printer<Package, Tag>* printer;
	map<Tag, size_t> tagCount;
	OpSet<Tag> m_topTags;

	virtual void consumeItemUntagged(const Package& pkg)
	{
		printer->consume(pkg);
		CardinalityStore<Package, Tag>::consumeItemUntagged(pkg);
	}
	virtual void consumeItem(const Package& pkg, const Tagcoll::OpSet<Tag>& tags)
	{
		printer->consume(pkg, tags);
		CardinalityStore<Package, Tag>::consumeItem(pkg, tags);
		for (OpSet<Tag>::const_iterator i = tags.begin();
				i != tags.end(); i++)
			++tagCount[*i];
	}
	virtual void consumeItemsUntagged(const Tagcoll::OpSet<Package>& pkgs)
	{
		printer->consume(pkgs);
		CardinalityStore<Package, Tag>::consumeItemsUntagged(pkgs);
	}
	virtual void consumeItems(const Tagcoll::OpSet<Package>& pkgs, const Tagcoll::OpSet<Tag>& tags)
	{
		printer->consume(pkgs, tags);
		CardinalityStore<Package, Tag>::consumeItems(pkgs, tags);
		for (OpSet<Tag>::const_iterator i = tags.begin();
				i != tags.end(); i++)
			tagCount[*i] += pkgs.size();
	}

	vector< pair<Tag, size_t> > getTopTags(size_t count)
	{
		vector< pair<Tag, size_t> > res;
		for (map<Tag, size_t>::const_iterator i = tagCount.begin();
				i != tagCount.end(); i++)
		{
			pair<Tag, size_t> hand = *i;
			for (size_t j = 0; j < count; j++)
			{
				if (j >= res.size())
				{
					res.push_back(hand);
					break;
				}
				else if (hand.second > res[j].second)
				{
					pair<Tag, size_t> tmp = res[j];
					res[j] = hand;
					hand = tmp;
				}
			}
		}
		return res;
	}

public:
	SmartSearcher(component::PackageTags& debtags, Printer<Package, Tag>* printer) :
		debtags(debtags), printer(printer) {}

	void clear()
	{
		CardinalityStore<Package, Tag>::operator=(CardinalityStore<Package, Tag>());
		tagCount.clear();
		m_topTags.clear();
	}

	OpSet<Tag> topTags() { return m_topTags; }

	int outputRelated()
	{
		const size_t relevantTags = 5;
		vector< pair<Tag, size_t> > topTagList = getTopTags(relevantTags);
		Scores<Tag> scores;
		vector< OpSet<Tag> > tagsets;
		float distance = relevantTags;
		int count = 0;

		m_topTags.clear();
		for (vector< pair<Tag, size_t> >::const_iterator i = topTagList.begin();
				i != topTagList.end(); i++)
			m_topTags += i->first;

		// Get the tagsets that score less in weighted distance from topTags
		for (tagsets_t::const_iterator i = this->tagsets.begin();
				i != this->tagsets.end(); i++)
		{
			float d = scores.distance(m_topTags, i->first);
			if (d <= distance)
			{
				if (d < distance)
				{
					tagsets.clear();
					distance = d;
				}
				tagsets.push_back(i->first);
			}
		}

		// print the items they have
		for (vector< OpSet<Tag> >::const_iterator i = tagsets.begin();
				i != tagsets.end(); i++)
		{
			OpSet<Package> pkgs = debtags.tagdb().getItems(*i);
			printer->consume(pkgs, *i);
			count += pkgs.size();
		}
		return count;
	}

	virtual void flush()
	{
		printer->flush();
	}
#endif
};

#endif
// vim:set ts=4 sw=4:
