/*
 * debtags - Implement package tags support for Debian
 *
 * Copyright (C) 2003--2006  Enrico Zini <enrico@debian.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#define APPNAME PACKAGE_NAME
#else
#warning No config.h found: using fallback values
#define APPNAME __FILE__
#define PACKAGE_VERSION "unknown"
#endif

#include <tagcoll/input/stdio.h>
#include <tagcoll/stream/filters.h>
#include <tagcoll/expression.h>
#include <tagcoll/TextFormat.h>
#include <tagcoll/SmartHierarchy.h>
#include <tagcoll/coll/simple.h>
#include <tagcoll/utils/set.h>

#include "Environment.h"
#include "DebtagsOptions.h"
#include "Printer.h"
#include "nullstream.h"
#include "SmartSearcher.h"

#include <ept/apt/packagerecord.h>
#include <ept/debtags/expression.h>
#include <ept/debtags/maint/vocabularymerger.h>
#include <ept/debtags/maint/path.h>
#include <ept/debtags/maint/vocabularyindexer.h>
#include <ept/debtags/maint/debtagsindexer.h>

#include <wibble/sys/fs.h>

#include <cstring>
#include <errno.h>
#include <termios.h>
#include <sys/ioctl.h>
#include <sys/types.h>	// umask
#include <sys/stat.h>	// umask

namespace std {

template<typename TAG, typename _Traits>
basic_ostream<char, _Traits>& operator<<(basic_ostream<char, _Traits>& out, const std::set<TAG>& tags)
{
	for (typename std::set<TAG>::const_iterator i = tags.begin();
			i != tags.end(); i++)
		if (i == tags.begin())
			out << i->fullname();
		else
			out << ", " << i->fullname();
	return out;
}

template<typename TAG, typename _Traits>
basic_ostream<char, _Traits>& operator<<(basic_ostream<char, _Traits>& out, const wibble::Singleton<TAG>& tags)
{
	out << *tags.begin();
	return out;
}

template<typename TAG, typename _Traits>
basic_ostream<char, _Traits>& operator<<(basic_ostream<char, _Traits>& out, const wibble::Empty<TAG>&)
{
	return out;
}

}

using namespace std;
using namespace tagcoll;
using namespace ept;
using namespace ept::debtags;
using namespace ept::apt;


template<typename OUT>
class ExpressionFilter : public wibble::mixin::OutputIterator< ExpressionFilter<OUT> >
{
	tagcoll::Expression expr;
	bool invert;
	OUT out;
public:
	ExpressionFilter(const tagcoll::Expression& expr, bool invert, const OUT& out)
		: expr(expr), invert(invert), out(out) {}

	template<typename ITEMS, typename TAGS>
	ExpressionFilter<OUT>& operator=(const std::pair<ITEMS, TAGS>& data)
	{
		std::set<std::string> tags;
		for (typename TAGS::const_iterator i = data.second.begin();
				i != data.second.end(); ++i)
			tags.insert(i->fullname());
		bool matched = expr(tags);
		if (invert)
			matched = !matched;
		if (matched)
		{
			*out = data;
			++out;
		}
		return *this;
	}
};
template<typename OUT>
ExpressionFilter<OUT> expressionFilter(const tagcoll::Expression& expr, bool invert, const OUT& out)
{
	return ExpressionFilter<OUT>(expr, invert, out);
}

template<typename OUT>
void readCollection(const string& file, const OUT& out)
{
	if (file == "-")
	{
		input::Stdio input(stdin, "<stdin>");
		textformat::parse(input, out);
	}
	else
	{
		input::Stdio input(file);
		textformat::parse(input, out);
	}
}

class SubstringTagMatcher
{
protected:
	vector<string> patterns;

public:
	void add(const std::string& pattern) { patterns.push_back(pattern); }
	
	bool operator()(const Facet& item)
	{
		for (vector<string>::const_iterator i = patterns.begin();
				i != patterns.end(); i++)
		{
			if (strcasestr(item.name().c_str(), i->c_str()))
				return true;
			if (strcasestr(item.shortDescription().c_str(), i->c_str()))
				return true;
			if (strcasestr(item.longDescription().c_str(), i->c_str()))
				return true;
		}
		return false;
	}
	bool operator()(const Tag& item)
	{
		//cerr << "Testing: " << item.fullname() << endl;
		for (vector<string>::const_iterator i = patterns.begin();
				i != patterns.end(); i++)
		{
			if (strcasestr(item.fullname().c_str(), i->c_str()))
				return true;
			if (strcasestr(item.shortDescription().c_str(), i->c_str()))
				return true;
			if (strcasestr(item.longDescription().c_str(), i->c_str()))
				return true;
		}
		//cerr << "  not passed." << endl;
		return false;
	}
};

template<class OUT>
class FacetOnly : public wibble::mixin::OutputIterator< FacetOnly<OUT> >
{
	OUT out;
public:
	FacetOnly(const OUT& out) : out(out) {}

	template<typename ITEMS, typename TAGS>
	FacetOnly<OUT>& operator=(const std::pair<ITEMS, TAGS>& data)
	{
		std::set<std::string> items;
		std::set<std::string> tags;
		for (typename ITEMS::const_iterator i = data.first.begin();
				i != data.first.end(); ++i)
			items.insert(i->name());
		for (typename TAGS::const_iterator i = data.second.begin();
				i != data.second.end(); ++i)
			tags.insert(i->facet().name());
		*out = make_pair(items, tags);
		++out;
		return *this;
	}
};
template<typename OUT>
FacetOnly<OUT> facetOnly(const OUT& out)
{
	return FacetOnly<OUT>(out);
}

// Needs Package to see if it's installed
template<typename OUT>
class TODOFilter : public wibble::mixin::OutputIterator< TODOFilter<OUT> >
{
	Expression nyt;
	OUT out;

public:
	TODOFilter(const OUT& out) : nyt("special::not-yet-tagged"), out(out) {}

	template<typename ITEMS, typename TAGS>
	TODOFilter<OUT>& operator=(const std::pair<ITEMS, TAGS>& data)
	{
		for (typename ITEMS::const_iterator i = data.first.begin();
				i != data.first.end(); ++i)
		{
			PackageState s = env().apt().state(*i);
			if (!(s.isInstalled() || s.isUpgradable() || s.isBroken()))
				continue;
			std::set<string> tags;
			for (typename TAGS::const_iterator j = data.second.begin();
					j != data.second.end(); ++j)
			{
				tags.insert(j->fullname());
			}
			if (nyt(tags))
			{
				*out = make_pair(wibble::singleton(*i), data.second);
				++out;
			}
		}
		return *this;
	}
};

template<typename OUT>
TODOFilter<OUT> todoFilter(const OUT& out)
{
	return TODOFilter<OUT>(out);
}

struct Stats
{
	int seen;
	int onlynyt;
	int nyt;
	int notags;
	int tagged;
	int complete;

	Stats() : seen(0), onlynyt(0), nyt(0), notags(0), tagged(0), complete(0) {}
};

// Might need package in the future to count stats about installed packages
// TODO: count stats about installed packages
class StatsCollector : public wibble::mixin::OutputIterator< StatsCollector >
{
protected:
	Stats& stats;
	Expression has_nyt;
	Expression is_ct;

public:
	StatsCollector(Stats& stats) : stats(stats),
			has_nyt("special::not-yet-tagged*"),
			is_ct("special::completely-tagged") {}

	template<typename ITEMS, typename TAGS>
	StatsCollector& operator=(const std::pair<ITEMS, TAGS>& data)
	{
		++stats.seen;
		if (data.second.empty())
			++stats.notags;
		else {
			std::set<string> tags;
			for (typename TAGS::const_iterator i = data.second.begin();
					i != data.second.end(); ++i)
				tags.insert(i->fullname());

			if (has_nyt(tags)) {
				++stats.nyt;
				++stats.onlynyt;
				for (set<string>::const_iterator i = tags.begin(); i != tags.end(); ++i)
					if (i->substr(0, 23) != "special::not-yet-tagged")
					{
						--stats.onlynyt;
						break;
					}
			} else {
				++stats.tagged;
				if (is_ct(tags))
					++stats.complete;
			}
		}
		return *this;
	}
};

struct VocabularyCheck
{
	int missing_count;
	map<string, int> missing;
	unsigned int mtag_width;
	int max_missing;

	VocabularyCheck() : missing_count(0), mtag_width(0), max_missing(0) {}

	void report(ostream& out)
	{
		if (missing_count > 0)
		{
			out << missing.size() << " tags were found in packages but not in the vocabulary" << endl;
			out << "This happened " << missing_count << " times" << endl;
			out << "The tags found in the collection but not in the vocabulary are:" << endl;

			int mtag_maxdigits = 0;
			for (int i = max_missing; i > 0; i = i / 10)
				mtag_maxdigits++;

			for (map<string, int>::const_iterator i = missing.begin();
					i != missing.end(); i++)
			{
				char buf[1000];
				snprintf(buf, 1000, "\t%s %*s (in %*d package%s)",
					i->first.c_str(), 
					(int)(mtag_width - i->first.size()),
					"", 
					mtag_maxdigits,
					i->second,
					i->second > 1 ? "s" : "");
				out << buf << endl;
			}
		}
	}
};

// Check a collection agaisnt a tag vocabulary
class VocabularyChecker : public wibble::mixin::OutputIterator< VocabularyChecker >
{
protected:
	const Vocabulary& voc;
	VocabularyCheck& res;

public:
	VocabularyChecker(const Vocabulary& voc, VocabularyCheck& res) : voc(voc), res(res) {}

	template<typename ITEMS, typename TAGS>
	VocabularyChecker& operator=(const std::pair<ITEMS, TAGS>& data)
	{
		for (typename TAGS::const_iterator i = data.second.begin();
				i != data.second.end(); ++i)
			if (!voc.hasTag(*i))
			{
				const string& tag = *i;
				res.missing_count++;
				res.missing[tag]++;
				if (tag.size() > res.mtag_width)
					res.mtag_width = tag.size();
				if (res.missing[tag] > res.max_missing)
					res.max_missing = res.missing[tag];
			}
		return *this;
	}
};

template<typename VOC, typename OUT>
class VocabularyFilter : public wibble::mixin::OutputIterator< VocabularyFilter<VOC, OUT> >
{
protected:
	const VOC& voc;
	OUT out;

public:
	VocabularyFilter(const VOC& voc, const OUT& out) : voc(voc), out(out) {}

	template<typename ITEMS, typename TAGS>
	VocabularyFilter& operator=(const std::pair<ITEMS, TAGS>& data)
	{
		std::set< typename TAGS::value_type > tags;
		for (typename TAGS::const_iterator i = data.second.begin();
				i != data.second.end(); ++i)
			if (voc.hasTag(*i))
				tags.insert(*i);
		*out = make_pair(data.first, tags);
		++out;
		return *this;
	}
};
template<typename VOC, typename OUT>
VocabularyFilter<VOC, OUT> vocabularyFilter(const VOC& voc, const OUT& out)
{
	return VocabularyFilter<VOC, OUT>(voc, out);
}

static void printShortVocabularyItem(const Facet& facet)
{
	cout << facet.name() << " (facet) - " << facet.shortDescription() << endl;
}

static void printShortVocabularyItem(const Tag& tag)
{
	cout << tag.fullname() << " - " << tag.shortDescription() << endl;
}

static void printVocabularyItem(const Facet& tag)
{
	string ld = tag.longDescription();
	cout << "Facet: " << tag.name() << endl;
	cout << "Description: " << tag.shortDescription() << endl;
	cout << " " << ld << endl;
	if (ld[ld.size() - 1] != '\n')
		cout << endl;
}
static void printVocabularyItem(const Tag& tag)
{
	string ld = tag.longDescription();
	cout << "Tag: " << tag.fullname() << endl;
	cout << "Description: " << tag.shortDescription() << endl;
	cout << " " << ld << endl;
	if (ld[ld.size() - 1] != '\n')
		cout << endl;
}

template<typename OUT>
class PackageToMaint : public wibble::mixin::OutputIterator< PackageToMaint<OUT> >
{
	OUT out;
public:
	PackageToMaint(const OUT& out) : out(out) {}

	template<typename ITEMS, typename TAGS>
	PackageToMaint& operator=(const std::pair<ITEMS, TAGS>& data)
	{
		std::set<string> items;
		std::set<string> tags;
		for (typename ITEMS::const_iterator i = data.first.begin();
				i != data.first.end(); ++i)
		{
			Version v = env().apt().anyVersion(*i);
			if (v.isValid())
			{
				PackageRecord rec(env().apt().rawRecord(v));
				items.insert(rec.maintainer());
			}
		}
		for (typename TAGS::const_iterator i = data.second.begin();
				i != data.second.end(); ++i)
			if (i->valid())
				tags.insert(i->fullname());
		if (!items.empty())
		{
			*out = make_pair(items, tags);
			++out;
		}
		return *this;
	}
};
template<typename OUT>
PackageToMaint<OUT> packageToMaint(const OUT& out)
{
	return PackageToMaint<OUT>(out);
}


struct PackageScore
{
	// tag -> (installed, total)
	map<Tag, pair<int, int> > tagScore;
	set<string> uninstalled;

	void output(Debtags& debtags, ostream& out)
	{
		map<int, string> sorter;

		for (set<string>::const_iterator i = uninstalled.begin();
				i != uninstalled.end(); ++i)
		{
			// Compute package score
			set<Tag> tags = debtags.getTagsOfItem(*i);
			int score = 0;
			int ntags = 0;
			for (set<Tag>::const_iterator t = tags.begin();
					t != tags.end(); ++t)
			{
				map<Tag, pair<int, int> >::const_iterator j = tagScore.find(*t);
				if (j != tagScore.end())
				{
					score += 100 * j->second.first / j->second.second;
					++ntags;
				}
			}
			if (ntags)
				sorter.insert(make_pair(score / ntags, *i));
			else
				// Packages with no tags get a 0 score
				sorter.insert(make_pair(0, *i));
		}

		for (map<int, string>::const_iterator i = sorter.begin();
				i != sorter.end(); ++i)
		{
			Version ver = env().apt().candidateVersion(i->second);
			if (!ver.isValid())
				ver = env().apt().installedVersion(i->second);
			PackageRecord rec(env().apt().rawRecord(ver));
			out << i->first << " " << i->second << " - " <<
				rec.shortDescription("(short description not available)") << endl;
		}
	}
};

// Needs package for TagsScorer
class PackageScorer : public wibble::mixin::OutputIterator< PackageScorer >
{
protected:
	tagcoll::Expression avoidPkg;
	tagcoll::Expression avoidTag;
	PackageScore& res;

public:
	PackageScorer(PackageScore& res)
		: avoidPkg("role::sw:shlib || role::content:data"), avoidTag("special::*"), res(res) {}

	template<typename ITEMS, typename TAGS>
	PackageScorer& operator=(const std::pair<ITEMS, TAGS>& data)
	{
		// Skip items that need skipping
		if (avoidPkg(data.second))
			return *this;

		for (typename ITEMS::const_iterator i = data.first.begin();
				i != data.first.end(); ++i)
		{
			PackageState s = env().apt().state(*i);
			bool installed = (s.isInstalled() || s.isUpgradable() || s.isBroken());

			if (!installed)
				// Take note of uninstalled packages
				res.uninstalled.insert(*i);

			// Score tags of installed packages
			for (typename TAGS::const_iterator t = data.second.begin();
					t != data.second.end(); ++t)
			{
				std::set<Tag> it;
				it.insert(*t);
				if (!avoidTag(it))
				{
					//cerr << "Count " << t->fullname() << endl;
					map<Tag, pair<int, int> >::iterator j = res.tagScore.find(*t);
					if (j == res.tagScore.end())
						res.tagScore.insert(make_pair(*t, make_pair(installed ? 1 : 0, 1)));
					else
					{
						if (installed)
							++j->second.first;
						++j->second.second;
					}
				}
			}
		}
		return *this;
	}
};

struct PackageToString {
	string operator()(const string& pkg) { return pkg; }
} serPackage;
struct TagToString {
	string operator()(const Tag& tag) { return tag.fullname(); }
} serTag;

int main(int argc, const char* argv[])
{
	wibble::commandline::DebtagsOptions opts;

	try {
		// Install the handler for unexpected exceptions
		wibble::exception::InstallUnexpected installUnexpected;

		if (opts.parse(argc, argv))
			return 0;

		if (opts.out_verbose->boolValue())
			::Environment::get().verbose(true);

		if (opts.out_debug->boolValue())
			::Environment::get().debug(true);

		if (opts.foundCommand() == opts.selfcheck)
		{
			env().init();
			Vocabulary& voc = env().voc();
			Apt& apt = env().apt();

			// ensure that all facets are readable
			std::set<Facet> facets = voc.facets();
			for (std::set<Facet>::const_iterator i = facets.begin(); i != facets.end(); i++)
			{
				cout << "Checking facet " << i->name(string("<invalid name>")) << "..." << endl;
				i->name(string("foo"));
				i->shortDescription(string("foo"));
				i->longDescription(string("foo"));
				i->tags();
			}

			// ensure that all tags are readable
			std::set<Tag> tags = voc.tags();
			for (std::set<Tag>::const_iterator i = tags.begin(); i != tags.end(); i++)
			{
				cout << "Checking tag " << i->fullname(string("<invalid name>")) << "..." << endl;
				i->name(string("foo"));
				i->fullname(string("foo"));
				i->shortDescription(string("foo"));
				i->longDescription(string("foo"));
			}

			// ensure that all packages that declare they are readable, are readable
			for (Apt::iterator i = apt.begin(); i != apt.end(); ++i)
			{
				cout << "Checking package " << *i << "..." << endl;
				//if (i->isValid())
				//{
				//	i->shortDescription(string("foo"));
				//	i->longDescription(string("foo"));
				//	//i->tags(std::set<Tag>());
				//}
			}
			
			for (std::set<Facet>::const_iterator i = facets.begin();
					i != facets.end(); i++)
			{
				if (!voc.hasFacet(i->name()))
					cerr << "Vocabulary is not sure about having facet " << i->name() << endl;

				std::set<Tag> tags = i->tags();
				for (std::set<Tag>::const_iterator j = tags.begin();
						j != tags.end(); j++)
					if (!voc.hasTag(j->fullname()))
						cerr << "Vocabulary is not sure about having tag " << j->fullname() << endl;
			}
			return 0;
		}
		// Output the full package tag database
		else if (opts.foundCommand() == opts.cat)
		{
			env().init();
			Debtags& debtags = env().debtags();
			int count = 0;

			auto_ptr<CollPrinter> printer;

			if (opts.out_names->boolValue())
				printer.reset(new CollPrinter(CollPrinter::NAME, count));
			else if (opts.out_facets->boolValue())
				printer.reset(new CollPrinter(CollPrinter::FACETS, count));
			else if (opts.out_quiet->boolValue())
				printer.reset(new CollPrinter(CollPrinter::QUIET, count));
			else 
				printer.reset(new CollPrinter(CollPrinter::TAGS, count));

			if (opts.hasNext())
			{
				debtags.output(expressionFilter(opts.next(), opts.match_invert->boolValue(), *printer));
			} else
				debtags.output(*printer);

			return count > 0 ? 0 : 1;
		}
		// Output the full package database
		else if (opts.foundCommand() == opts.dumpavail)
		{
			env().init();
			int count = 0;

			auto_ptr<PackagePrinter> printer;
			if (opts.out_names->boolValue())
				printer.reset(new PackagePrinter(PackagePrinter::NAME, count));
			else if (opts.out_short->boolValue())
				printer.reset(new PackagePrinter(PackagePrinter::SHORT, count));
			else if (opts.out_quiet->boolValue())
				printer.reset(new PackagePrinter(PackagePrinter::QUIET, count));
			else 
				printer.reset(new PackagePrinter(PackagePrinter::FULL, count));

			if (opts.hasNext())
			{
				env().debtags().output(expressionFilter(opts.next(), opts.match_invert->boolValue(), *printer));
			} else {
				// If there is no expression filter, dump from the Apt database
				Apt& apt = env().apt();
				PackageRecord record;
				for (Apt::record_iterator i = apt.recordBegin();
						i != apt.recordEnd(); ++i)
				{
					record.scan(*i);
					*printer = record;
				}
			}

			return count > 0 ? 0 : 1;
		}
		// search [-v] <tag expression>\n"
		// Output the names and description of the packages that match\n"
		// the given tag expression\n"
		else if (opts.foundCommand() == opts.search)
		{
			env().init();
			// TODO: complain if no expression found
			int count = 0;

			auto_ptr<PackagePrinter> printer;
			if (opts.out_names->boolValue())
				printer.reset(new PackagePrinter(PackagePrinter::NAME, count));
			else if (opts.out_full->boolValue())
				printer.reset(new PackagePrinter(PackagePrinter::FULL, count));
			else if (opts.out_quiet->boolValue())
				printer.reset(new PackagePrinter(PackagePrinter::QUIET, count));
			else 
				printer.reset(new PackagePrinter(PackagePrinter::SHORT, count));

			if (opts.hasNext())
			{
				env().debtags().output(expressionFilter(opts.next(), opts.match_invert->boolValue(), *printer));
			} else
				env().debtags().output(*printer);

			return count > 0 ? 0 : 1;
		}
		// grep [-v] [-q] <tag expression>
		// Output the lines of the full package tag database that match the
		// given tag expression
		else if (opts.foundCommand() == opts.grep)
		{
			env().init();
			// TODO: complain if no expression found
			Debtags& debtags = env().debtags();
			int count = 0;

			auto_ptr<CollPrinter> printer;

			if (opts.out_names->boolValue())
				printer.reset(new CollPrinter(CollPrinter::NAME, count));
			else if (opts.out_facets->boolValue())
				printer.reset(new CollPrinter(CollPrinter::FACETS, count));
			else if (opts.out_quiet->boolValue())
				printer.reset(new CollPrinter(CollPrinter::QUIET, count));
			else 
				printer.reset(new CollPrinter(CollPrinter::TAGS, count));

			if (opts.hasNext())
			{
				debtags.output(expressionFilter(opts.next(), opts.match_invert->boolValue(), *printer));
			} else
				debtags.output(*printer);

			return count > 0 ? 0 : 1;
		}
		// tagcat
		// Output the entire tag vocabulary
		else if (opts.foundCommand() == opts.tagcat)
		{
			env().init();
			Vocabulary& voc = env().voc();

			std::set<Facet> facets = voc.facets();
			for (std::set<Facet>::const_iterator i = facets.begin();
					i != facets.end(); i++)
			{
				printVocabularyItem(*i);

				std::set<Tag> tags = i->tags();
				for (std::set<Tag>::const_iterator j = tags.begin();
						j != tags.end(); j++)
					printVocabularyItem(*j);
			}
			return 0;
		}
		// tagshow <tag>
		// Show the vocabulary informations about a tag
		else if (opts.foundCommand() == opts.tagshow)
		{
			env().init();
			string tag = opts.next();

			Tag t = env().voc().tagByName(tag);
			if (!t.valid())
			{
				verbose("Tag `%s' was not found in tag vocabulary\n", tag.c_str());
				return 1;
			}
			else
			{
				printVocabularyItem(t);
				return 0;
			}
		}
		// tagsearch <pattern [pattern [pattern [...]]]>
		// Show a summary of all tags matching the given patterns
		else if (opts.foundCommand() == opts.tagsearch)
		{
			env().init();
			SubstringTagMatcher match;

			// Get the patterns to be matched
			bool empty;
			while (opts.hasNext())
			{
				string pattern = opts.next();
				match.add(pattern);
				empty = false;
			}

			if (empty)
			{
				error("No patterns given in commandline\n");
				return 1;
			}

			int matched = 0;

			std::set<Facet> facets = env().voc().facets();
			for (std::set<Facet>::const_iterator i = facets.begin();
					i != facets.end(); i++)
			{
				if (match(*i))
				{
					matched++;
					printShortVocabularyItem(*i);
				}

				std::set<Tag> tags = i->tags();
				for (std::set<Tag>::const_iterator j = tags.begin();
						j != tags.end(); j++)
					if (match(*j))
					{
						matched++;
						printShortVocabularyItem(*j);
					}
			}

			return matched > 0 ? 0 : 1;
		}
		// show <pkg>
		// Call apt-cache show <pkg>, but add tag informations to the output.\n"
		else if (opts.foundCommand() == opts.show)
		{
			env().init();
			while (opts.hasNext())
			{
				string name = opts.next();

				if (env().apt().isValid(name))
				{
					PackagePrinter printer(PackagePrinter::FULL);
					printer = name;
					return 0;
				} else {
					verbose("Package %s not found", name.c_str());
					return 1;
				}
			}
		}
		// related <pkg1[,pkg2[,pkg2,[...]]]>
		// Show packages related to the specified ones
		else if (opts.foundCommand() == opts.related)
		{
			env().init();
			using namespace wibble::operators;

			int maxdist = 0;
			if (opts.misc_distance->boolValue())
				maxdist = opts.misc_distance->intValue();
			string pkg = opts.next();

			// Split the items on commas
			string splititem;
			std::set<string> splititems;
			for (string::const_iterator c = pkg.begin(); c != pkg.end(); c++)
				if (*c == ',')
				{
					if (env().apt().isValid(splititem))
					{
						splititems.insert(splititem);
					} else {
						error("Item \"%s\" does not exist in the collection\n", splititem.c_str());
						return 1;
					}
					splititem = string();
				} else
					splititem += *c;
			if (env().apt().isValid(splititem))
			{
				splititems.insert(splititem);
			} else {
				error("Item \"%s\" does not exist in the collection\n", splititem.c_str());
				return 1;
			}

			// Get the tagset as the intersection of the tagsets of all input items
			std::set<string>::const_iterator i = splititems.begin();
			std::set<Tag> ts = env().debtags().getTagsOfItem(*i);
			for (++i; i != splititems.end(); i++)
				ts &= env().debtags().getTagsOfItem(*i);

			if (ts.empty())
			{
				if (splititems.size() > 1)
					fprintf(stderr, "The packages %s are unrelated: cannot find a barycenter to start computing relationships from.\n", pkg.c_str());
				else
					fprintf(stderr, "The package %s has no tags attached.\n", pkg.c_str());
				return 1;
			}

			std::set<string> related(env().debtags().getRelatedItems(ts, maxdist) - splititems);

			int count = 0;

			auto_ptr<PackagePrinter> printer;
			if (opts.out_names->boolValue())
				printer.reset(new PackagePrinter(PackagePrinter::NAME, count));
			else if (opts.out_full->boolValue())
				printer.reset(new PackagePrinter(PackagePrinter::FULL, count));
			else if (opts.out_quiet->boolValue())
				printer.reset(new PackagePrinter(PackagePrinter::QUIET, count));
			else 
				printer.reset(new PackagePrinter(PackagePrinter::SHORT, count));

			for (std::set<string>::const_iterator i = related.begin();
					i != related.end(); ++i)
			{
				**printer = *i;
				++*printer;
			}

			if (count > 50 && maxdist == 0 && isatty(1))
			{
				string tags;
				for (std::set<Tag>::const_iterator i = ts.begin();
						i != ts.end(); i++)
					if (i == ts.begin())
						tags += i->fullname();
					else
						tags += "%2C" + i->fullname();
				feedback("\nIt seems that this set of packages lacks tag information that could help to better distinguish package similarities.\nYou can help providing better tagging: just point your browser to http://debian.vitavonni.de/packagebrowser/?tags=%s\n", tags.c_str());
			}
			return 0;
		}
		// maintainers
		// Create a tagged collection of maintainers and the tags of the
		// packages they maintain
		else if (opts.foundCommand() == opts.maintainers)
		{
			env().init();
			// Gather maintainer informations
			coll::Simple<string, string> maints;
			env().debtags().output(packageToMaint(inserter(maints)));

			// Print them out
			maints.output(textformat::StdioWriter(stdout));
		}
		// tag
		//   tag [add <package> <tags...>\n"
		//   tag [rm  <package> <tags...>\n"
		//   tag [ls  <package>\n"
		//                View and edit the tags for a package\n");
		else if (opts.foundCommand() == opts.tag)
		{
			std::string cmd = opts.next();

			if (cmd == "add" || cmd == "rm")
			{
				env().init(true);

				string pkg = opts.next();
				if (!env().apt().isValid(pkg))
				{
					error("Package %s not found\n", pkg.c_str());
					return 1;
				}

				std::set<Tag> tagset;
				while (opts.hasNext())
				{
					string tag = opts.next();
					Tag t = env().voc().tagByName(tag);
					if (t)
						tagset.insert(t);
					else
						error("Tag '%s' not found: ignored\n", tag.c_str());
				}

				if (!tagset.empty())
				{
					PatchList<string, Tag> change;
					if (cmd == "add")
						change.addPatch(Patch<string, Tag>(pkg, tagset, std::set<Tag>()));
					else
						change.addPatch(Patch<string, Tag>(pkg, std::set<Tag>(), tagset));
					env().debtags().applyChange(change);
					env().debtags().savePatch();
				} else
					verbose("No tags to add\n");
			}
			else if (cmd == "ls")
			{
				env().init();
				string pkg = opts.next();
				if (env().apt().isValid(pkg))
				{
					std::set<Tag> ts = env().debtags().getTagsOfItem(pkg);
					for (std::set<Tag>::const_iterator i = ts.begin();
							i != ts.end(); i++)
						cout << i->fullname() << endl;
					return 0;
				} else {
					verbose("Package %s not found", pkg.c_str());
					return 1;
				}
			}
			else
				throw wibble::exception::Consistency("parsing the 'tag' subcommand", "command " + cmd + " is not valid working with tags");
		}
		// submit
		// Mail the local updates to the tag database to the central tag
		// repository
		else if (opts.foundCommand() == opts.submit)
		{
			env().init();
			if (opts.hasNext())
			{
				input::Stdio in(opts.next());
				PatchList<string, string> patch;
				textformat::parsePatch(in, inserter(patch));
				env().debtags().sendPatch(patch);
			}
			else
				env().debtags().sendPatch();
		}
		// todo
		// Print a list of the installed packages that are not yet tagged
		else if (opts.foundCommand() == opts.todo)
		{
			env().init();
			int count = 0;

			auto_ptr<PackagePrinter> printer;
			if (opts.out_names->boolValue())
				printer.reset(new PackagePrinter(PackagePrinter::NAME, count));
			else if (opts.out_full->boolValue())
				printer.reset(new PackagePrinter(PackagePrinter::FULL, count));
			else if (opts.out_quiet->boolValue())
				printer.reset(new PackagePrinter(PackagePrinter::QUIET, count));
			else 
				printer.reset(new PackagePrinter(PackagePrinter::SHORT, count));

			env().debtags().output(todoFilter(*printer));

			return count > 0 ? 0 : 1;
		}
		// stats
		// Print statistics about Debtags
		else if (opts.foundCommand() == opts.stats)
		{
			env().init();
			int pkgCount = env().apt().size();
			printf("Total count of packages: %d\n", pkgCount);

			Stats stats;
			env().debtags().outputPatched(StatsCollector(stats));

			printf("Total count of packages (according to APT): %d\n", pkgCount);
			printf("Total count of packages (according to Debtags): %d\n", stats.seen);

			printf("Number of facets: %zi\n", env().voc().facets().size());
			printf("Number of tags: %zi\n", env().voc().tags().size());

#if 0
			// Copied from Debtags class: compute the toplevel facets
			// TODO: use Debtags instead of Environment throughout all Debtags
			CardinalityStore<entity::Package, Facet> coll;
			TagToFacet<entity::Package> tagStripper(coll);
			debtags.outputPatched(tagStripper);
			Facet f;
			SmartHierarchyNode<entity::Package, Facet> node(f, coll, 0);
			printf("Number of automatically computed toplevel facets: %d\n", node.size());
#endif

//			printf("Number of packages with special::completely-tagged tags: %d (%.1f%%)\n",
//					stats.complete, (float)stats.complete*100/stats.seen);
			printf("Number of packages with tags, but no special::not-yet-tagged tags: %d (%.1f%%)\n",
					stats.tagged, (float)stats.tagged*100/stats.seen);
			printf("Number of packages with special::not-yet-tagged tags: %d (%.1f%%)\n",
					stats.nyt, (float)stats.nyt*100/stats.seen);
			printf("Number of packages with only special::not-yet-tagged tags: %d (%.1f%%)\n",
					stats.onlynyt, (float)stats.onlynyt*100/stats.seen);
			printf("Number of packages with no tags: %d (%.1f%%)\n",
					stats.notags, (float)stats.notags*100/stats.seen);

		}
		// check <file>
		// Check that all the tags in the given tagged collection are
		// present in the tag vocabulary.  Checks the main database if no
		// file is specified
		else if (opts.foundCommand() == opts.check)
		{
			env().init();
			if (!opts.hasNext())
				throw wibble::exception::BadOption("you should specify the file with the collection to check");

			string file = opts.next();

			VocabularyCheck results;
			readCollection(file, VocabularyChecker(env().voc(), results));

			if (results.missing_count > 0)
			{
				results.report(cout);
				return 1;
			}
			else
				return 0;
		}
		// score
		// Score uninstalled packages according to how often their tags
		// appear in the packages that are installed already
		else if (opts.foundCommand() == opts.score)
		{
			env().init();
			// Compute package scores
			PackageScore score;
			env().debtags().outputPatched(PackageScorer(score));

			// Print the results
			score.output(env().debtags(), cout);
		}
		// mkpatch [filename]
		// Create a tag patch between the current tag database and the tag
		// collection [filename]
		else if (opts.foundCommand() == opts.diff)
		{
			env().init();
			string file = opts.next();

			coll::Simple<string, Tag> coll;
			env().debtags().outputSystem(file, inserter(coll));

			PatchList<string, Tag> newpatches;
			newpatches.addPatch(env().debtags(), coll);

			textformat::outputPatch(serPackage, serTag, newpatches, stdout);
		}
		// ssearch <word [word1 [word2 ...]]>
		// Perform a keyword search integrated with related packages
		else if (opts.foundCommand() == opts.smartsearch)
		{
			env().init();
			if (!opts.hasNext())
				throw wibble::exception::BadOption("you should specify one pattern to search");

			string keywords;
			while (opts.hasNext())
				if (keywords.empty())
					keywords = opts.next();
				else
					keywords += " " + opts.next();

			SmartSearcher searcher(keywords);

			if (opts.smse_reltags->boolValue())
				searcher.outputRelevantTags();
			else if (opts.smse_disctags->boolValue())
				searcher.outputDiscriminantTags();
			else {
				searcher.interact();
			}
		}
		// update
		// Updates the package tag database (requires root)
		else if (opts.foundCommand() == opts.update)
		{
			using namespace wibble::sys;

			verbose("System source directory: %s\n", Path::debtagsSourceDir().c_str());
			verbose("User source directory: %s\n", Path::debtagsUserSourceDir().c_str());

			if (!opts.misc_reindex->boolValue())
			{
				// Run the fetcher to acquire new data
				string fetcher = SCRIPTDIR "/fetch";
				if (!fs::access(fetcher, X_OK))
					warning("Fetch script %s does not exist or is not executable: skipping acquiring of new data\n", fetcher.c_str());
				else {
					if (opts.out_verbose->boolValue())
						fetcher += " --verbose";
					if (opts.misc_local->boolValue())
						fetcher += " --local";
					if (system(fetcher.c_str()) != 0)
						throw wibble::exception::Consistency("acquiring new data", "fetcher command " + fetcher + " failed");
				}
			}


			// Access the indexes to trigger a rebuild
			//typedef ept::t::cache::debtags::IndexManager<> IndexManager;
			std::string a, b;

			//auto_ptr<Ept> ept = debtagsInit();
			env().init();

			VocabularyIndexer::obtainWorkingVocabulary(a, b);
			verbose("Vocabulary: %s\n", a.c_str());
			verbose("Vocabulary index: %s\n", b.c_str());

			DebtagsIndexer::obtainWorkingDebtags(env().voc(), a, b);
			verbose("Tag database: %s\n", a.c_str());
			verbose("Tag database index: %s\n", b.c_str());

			//mode_t prev_umask = umask(022);
			//umask(prev_umask);
		}
		else if (opts.foundCommand() == opts.vocfilter)
		{
			env().init();
			if (!opts.hasNext())
				throw wibble::exception::BadOption("you should specify the file with the collection to check");

			string file = opts.next();

			if (opts.misc_vocfile->boolValue())
			{
				VocabularyMerger vm;
				input::Stdio input(opts.misc_vocfile->stringValue());
				vm.read(input);
				readCollection(file, vocabularyFilter(vm, textformat::OstreamWriter(cout)));
			}
			else
				readCollection(file, vocabularyFilter(env().voc(), textformat::OstreamWriter(cout)));
		}
#if 0
		// install [-v] [-q] <tag expression>
		// apt-get install the packages that match the given tag expression
		else if (opts.foundCommand() == opts.install)
		{
			component::PackageTags& debtags = debtagsInit();
			wantTagDatabase();
			Installer installer;

			Searcher searcher(debtags, &installer);
			string expression = opts.next();

			searcher.output(expression, opts.matchGroup.invert->boolValue());

			return 1;
		}
#endif
		else
			throw wibble::exception::BadOption(string("unhandled command ") +
						(opts.foundCommand() ? opts.foundCommand()->name() : "(null)"));

		return 0;
	} catch (wibble::exception::BadOption& e) {
		cerr << e.desc() << endl;
		opts.outputHelp(cerr);
		return 1;
	} catch (std::exception& e) {
		cerr << e.what() << endl;
		return 1;
	}
}

#include <ept/debtags/debtags.tcc>
#include <tagcoll/coll/simple.tcc>
#include <tagcoll/coll/fast.tcc>
#include <tagcoll/TextFormat.tcc>

// vim:set ts=4 sw=4:
