/* dbus_service.c
 *
 * D-BUS Service Utilities
 * Provides utilities for construction of D-BUS "Services"  
 *
 * Copyright (C) 2006  Red Hat, Inc. All rights reserved.
 *
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions of
 * the GNU General Public License v.2, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY expressed or implied, including the implied warranties of
 * MERCHANTABILITY or FITNESS FOR A * PARTICULAR PURPOSE.  See the GNU General
 * Public License for more details.  You should have received a copy of the
 * GNU General Public License along with this program; if not, write to the
 * Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.  Any Red Hat trademarks that are incorporated in the
 * source code or documentation are not subject to the GNU General Public
 * License and may only be used or replicated with the express permission of
 * Red Hat, Inc.
 *
 * Red Hat Author(s): Jason Vas Dias
 *                    David Cantrell
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <dbus/dbus.h>
#include <search.h>
#include <sys/time.h>

#include "dbus_service.h"

typedef struct dbcs_s {
    DBusConnection *connection;
    DBusDispatchStatus dispatchStatus;
    uint32_t status;
    dbus_svc_ErrorHandler eh;
    dbus_svc_ErrorHandler dh;
    /* { glibc b-trees: */
    void *roots;
    void *timeouts;
    void *watches;
    void *filters;
    /* } */
    int n;
    fd_set r_fds;
    fd_set s_r_fds;
    fd_set w_fds;
    fd_set s_w_fds;
    fd_set e_fds;
    fd_set s_e_fds;
    DBusMessage *currentMessage;
    int rejectMessage;
} DBusConnectionState;

typedef struct root_s {
    char *path;
    char *if_prefix;
    DBusConnectionState *cs;
    dbus_svc_MessageHandler mh;
    void *object;
    void *tree;
} Root;

typedef struct mhn_s {
    char *path;
    dbus_svc_MessageHandler mh;
    void *object;
} MessageHandlerNode;

typedef struct mfn_s {
    DBusConnectionState *cs;
    dbus_svc_MessageHandler mf;
    int n_matches;
    char **matches;
} MessageFilterNode;

typedef struct dbto_s {
    DBusTimeout *to;
    DBusConnectionState *cs;
    struct timeval tv;
} DBusConnectionTimeout;

#define SHUTDOWN 255

static void no_free(void *p) {}

static int ptr_key_comparator(const void *p1, const void *p2) {
    return ((p1 == p2) ? 0 : ((p1 > p2) ? 1 : -1));
}

static int root_comparator(const void *p1, const void *p2) {
    return strcmp ((const char *) (((const Root *) p1)->path),
                   (const char *) (((const Root *) p2)->path));
}

static int mh_comparator(const void *p1, const void *p2) {
    return strcmp ((const char *) (((const MessageHandlerNode *) p1)->path),
                   (const char *) (((const MessageHandlerNode *) p2)->path));
}

static int mf_comparator(const void *p1, const void *p2) {
    return ptr_key_comparator ((const void *) (((const MessageFilterNode *) p1)->mf), (const void *) (((const MessageFilterNode *) p2)->mf));
}

static void unregister_function(DBusConnection * connection, void *user_data) {}

static DBusHandlerResult message_handler(DBusConnection * connection,
                                          DBusMessage * message,
                                          void *user_data) {
    Root *root = user_data;
    DBusConnectionState *cs = root->cs;
    uint32_t type = dbus_message_get_type (message),
             serial = dbus_message_get_serial (message);
    uint8_t reply = (dbus_message_get_no_reply (message) == 0);
    char sub_path[1024],
         *path = (char *) dbus_message_get_path (message),
         *dest = (char *) dbus_message_get_destination (message),
         *member = (char *) dbus_message_get_member (message),
         *interface = (char *) dbus_message_get_interface (message),
         *sender = (char *) dbus_message_get_sender (message),
         *signature = (char *) dbus_message_get_signature (message),
         *if_suffix = 0L, *obj_suffix = 0L;
    MessageHandlerNode mhn, *mhp, **mhpp;
    dbus_svc_HandlerResult r;
    int len;

    if (cs->dh != 0L)
        (*(cs->dh)) ("message_handler: dest:%s path:%s member:%s interface:%s", dest, path, member, interface);

    if (path == 0L) {
        if (cs->eh != 0L)
            (*(cs->eh)) ("message_handler: message with NULL path");
        return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
    }

    if ((root->if_prefix != 0L) && (interface != 0L)) {
        len = strlen (root->if_prefix);
        if ((strncmp (interface, root->if_prefix, len) == 0) &&
            (interface[len] == '.'))
            if_suffix = interface + len + 1;
    }

    r = DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
    if (strcmp (path, root->path) == 0) {
        r = ((*(root->mh)) (cs, type, reply, serial, dest, path, member, interface, if_suffix, sender, signature, message, 0L, 0L, 0L, root->object));
        if (r != HANDLED_NOW)
            return r;
    }

    if (root->tree == 0L) {
        if (cs->eh != 0L)
            (*(cs->eh)) ("message_handler: not a prefix");
        return r;
    }

    len = strlen (root->path);
    if ((strlen (path) < len) || (strncmp (root->path, path, len) != 0)) {
        if (cs->eh != 0L)
            (*(cs->eh)) ("message_handler: can't happen?!?: path %s not prefixed by %s", path, root->path);
        return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
    }

    obj_suffix = (path + len + 1);

    if (root->tree == root) {	/* no sub-path handlers defined yet */
        r = ((*(root->mh)) (cs, type, reply, serial, dest, path, member, interface, if_suffix, sender, signature, message, root->path, obj_suffix, 0L, root->object));
        if (r != HANDLED_NOW)
            return r;
    }

    if (root->tree != root) {
        if ((if_suffix != 0L) && (member != 0L)) {
            snprintf(sub_path, 1024, "%s.%s.%s", obj_suffix, if_suffix, member);
        } else if ((if_suffix == 0L) && (member != 0L)) {
            snprintf(sub_path, 1024, "%s.%s", obj_suffix, member);
        } else if (member != 0L) {
            snprintf(sub_path, 1024, "%s.%s", obj_suffix, member);
        } else {
            if (cs->eh != 0L)
                (*(cs->eh)) ("message_handler: null member for %s", path);
            return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
        }

        if (cs->dh != 0L)
            (*(cs->dh)) ("message_handler: looking up sub-path %s", sub_path);

        mhn.path = (char *) sub_path;
        mhpp = tfind (&mhn, &(root->tree), mh_comparator);

        if ((mhpp == 0L) || ((mhp = *mhpp) == 0L)) {
            r = ((*(root->mh)) (cs, type, reply, serial, dest, path, member, interface, if_suffix, sender, signature, message, root->path, obj_suffix, 0L, root->object));
            if (r != HANDLED_NOW)
                return r;

            mhpp = tfind (&mhn, &(root->tree), mh_comparator);
            if ((mhpp == 0L) || ((mhp = *mhpp) == 0L)) {
                if (cs->eh != 0L)
                    (*(cs->eh)) ("message_handler: message handler not found under %s for sub-path %s", path, sub_path);

                return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
            }
        }

        r = ((*(mhp->mh)) (cs, type, reply, serial, dest, path, member, interface, if_suffix, sender, signature, message, root->path, obj_suffix, root->object, mhp->object));
        if (r == HANDLED_NOW) {
            mhn.path = (char *) sub_path;
            mhpp = tfind (&mhn, &(root->tree), mh_comparator);
            if ((mhpp == 0L) || ((mhp = *mhpp) == 0L)) {
                if (cs->eh != 0L)
                    (*(cs->eh)) ("message_handler: message handler returned HANDLED_NOW for unhandled path %s", sub_path);
                return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
            }
            r = ((*(mhp->mh)) (cs, type, reply, serial, dest, path, member, interface, if_suffix, sender, signature, message, root->path, obj_suffix, root->object, mhp->object));
        } else {
            return r;
        }
    }

    /* obj.if.member lookup failed */
    if (cs->dh != 0L)
        (*(cs->dh)) ("message_handler: message handler not found for sub-path %s", obj_suffix);
    return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
}

uint8_t dbus_svc_add_path_handler(DBusConnectionState * cs, char *rootPath,
                                  char *ifPrefix, dbus_svc_MessageHandler mh,
                                  void *object, uint8_t isPrefix) {
    DBusObjectPathVTable vtable = {unregister_function, message_handler, NULL,};
    Root *root;
    char *path, *ifp = 0L;

    if ((rootPath == 0L) || (*rootPath == '\0') || (*rootPath != '/')) {
        if (cs->eh != 0L)
            (*(cs->eh)) ("dbus_svc_add_message_handler: invalid path %s", rootPath);
        return (0);
    }

    root = (Root *) malloc (sizeof (Root));
    path = (char *) malloc (strlen (rootPath) + 1);

    if ((path == 0L) || (root == 0L)) {
        if (cs->eh != 0L)
            (*(cs->eh)) ("dbus_svc_add_path_handler: out of memory");
        return (0);
    }

    if (ifPrefix != 0L) {
        if ((ifp = (char *) malloc (strlen (ifPrefix) + 1)) == 0L) {
            if (cs->eh != 0L)
                (*(cs->eh)) ("dbus_svc_add_path_handler: out of memory");
            return (0);
        }

        strcpy(ifp, ifPrefix);
    }

    strcpy(path, rootPath);
    root->cs = cs;
    root->path = path;
    root->if_prefix = ifp;
    root->mh = mh;
    root->tree = isPrefix ? root : 0L;
    root->object = object;

    if (((!isPrefix) &&
	     !dbus_connection_register_object_path(cs->connection, rootPath,
                                              &vtable, root)) ||
        (isPrefix && !dbus_connection_register_fallback(cs->connection,
                                                        rootPath, &vtable,
                                                        root))) {
        if (cs->eh != 0L)
            (*(cs->eh)) ("dbus_svc_add_message_handler failed for: %s: %s", rootPath, isPrefix ? "dbus_connection_register_fallback failed" : "dbus_connection_register_object_path failed");
        return (0);
    }

    return (tsearch (root, &(cs->roots), root_comparator) != 0L);
}

void free_mh(void *mp) {
    MessageHandlerNode *mh = mp;

    free(mh->path);
    free(mh);
}

uint8_t dbus_svc_remove_path_handler(DBusConnectionState * cs, char *rootPath,
                                     void **objectP) {
    Root r;
    const Root *root, *const *rpp;

    if ((rootPath == 0L) || (*rootPath == '\0') || (*rootPath != '/')) {
        if (cs->eh != 0L)
            (*(cs->eh)) ("dbus_svc_remove_path: bad root path %s", rootPath);
        return 0;
    }

    r.path = rootPath;

    rpp = tfind (&r, &(cs->roots), root_comparator);

    if ((rpp == 0L) || ((root = *rpp) == 0L)) {
        if (cs->eh != 0L)
            (*(cs->eh)) ("dbus_svc_remove_path_handler: cannot remove nonexistent root node %s", rootPath);
        return 0;
    }

    tdelete (root, &(cs->roots), root_comparator);

    if (!dbus_connection_unregister_object_path (cs->connection, rootPath)) {
        if (cs->eh != 0L)
            (*(cs->eh)) ("dbus_svc_remove_path_handler:" "dbus_connection_unregister_object_path failed");
        return (0);
    }

    if ((root->tree != root) && (root->tree != 0L))
        tdestroy (root->tree, free_mh);

    if ((objectP != 0L) && (root->object != 0L))
        *objectP = root->object;

    free(root->path);
    free((Root *) root);

    return 1;
}

static uint8_t add_handler(DBusConnectionState * cs, Root * root, char *path,
                           dbus_svc_MessageHandler mh, void *object) {
    MessageHandlerNode *mhn;
    mhn = (MessageHandlerNode *) malloc (sizeof (MessageHandlerNode));

    if (mhn == 0L) {
        if (cs->eh != 0L)
            (*(cs->eh)) ("dbus_svc_add_message_handler: out of memory");
        return (0);
    }

    mhn->mh = mh;
    mhn->path = (char *) malloc (strlen (path) + 1);
    strcpy(mhn->path, path);
    mhn->object = object;

    if (root->tree == root)
        root->tree = 0L;

    return (tsearch ((void *) mhn, &(root->tree), mh_comparator) != 0);
}

static Root *find_root(DBusConnectionState * cs, char *rootPath) {
    Root r;
    const Root *root, *const *rpp;

    if ((rootPath == 0L) || (*rootPath == '\0'))
        return 0L;

    r.path = rootPath;
    rpp = tfind (&r, &(cs->roots), root_comparator);

    if ((rpp == 0L) || ((root = *rpp) == 0L))
        return 0L;

    return (Root *) root;
}

uint8_t dbus_svc_add_handler(DBusConnectionState * cs, char *rootPath,
                             char *objectPath, dbus_svc_MessageHandler mh,
                             void *object) {
    Root *root = find_root (cs, rootPath);
    int len;

    if ((root == 0L) || (objectPath == 0L) || (*objectPath == '\0')) {
        if (cs->eh != 0L)
            (*(cs->eh)) ("dbus_svc_add_handler: bad path %s/%s", rootPath, objectPath);
        return 0;
    }

    len = strlen (root->path);

    if (strncmp (root->path, objectPath, len) == 0) {
        if (*(objectPath + len) == '\0') {
            if (cs->eh != 0L)
                (*(cs->eh)) ("dbus_svc_add_handler: cannot add object path %s - identical to prefix path", objectPath);
            return (0);
        } else if ((*(objectPath + len) == '/')
                   && (*(objectPath + len + 1) != '\0')
                   && (*(objectPath + len + 1) != '/')) {
            (*cs->eh) ("dbus_svc_add_handler: invalid object path %s", (objectPath + len));
            return 0;
        } else if (*(objectPath + len) != '/') {
            (*cs->eh) ("dbus_svc_add_handler: invalid object path %s", (objectPath + len));
            return 0;
        }

        objectPath += len + 1;
    } else if (*objectPath == '/') {
        if ((*(objectPath + 1) == '\0') || (*(objectPath + 1) == '/')) {
            (*cs->eh) ("dbus_svc_add_handler: invalid object path %s", objectPath);
            return 0;
        }

        objectPath += 1;
    } else if (*objectPath == '\0') {
        (*cs->eh) ("dbus_svc_add_handler: empty object path");
        return 0;
    }

    return (add_handler (cs, root, objectPath, mh, object) != 0L);
}

uint8_t dbus_svc_remove_handler(DBusConnectionState * cs, char *rootPath,
                                char *objectPath, void **objectP) {
    MessageHandlerNode mh, *mhp, **mhpp;
    Root *root = find_root (cs, rootPath);

    if ((root == 0L) || (objectPath == 0L) || (*objectPath == '\0')) {
        if (cs->eh != 0L)
            (*(cs->eh)) ("dbus_svc_remove_handler: unhandled path  %s/%s", rootPath, objectPath);
        return (0);
    }

    mh.path = objectPath;

    if (root->tree == root)
        return 0;

    mhpp = tfind (&mh, &(root->tree), mh_comparator);

    if ((mhpp != 0L) && ((mhp = *mhpp) != 0L)) {
        mhpp = tdelete (&mh, &(root->tree), mh_comparator);
        free(mhp->path);

        if (objectP != 0L)
            *objectP = mhp->object;

        free(mhp);
    }

    return (mhpp != 0L);
}

static void filter_current_message(const void *p, const VISIT which,
                                   const int level) {
    MessageFilterNode *mfp;
    MessageFilterNode const *const *mfpp = p;

    if ((mfpp == 0L) || ((mfp = (MessageFilterNode *) * mfpp) == 0L) ||
        ((which != postorder) && (which != leaf)))
        return;

    if (mfp->cs->rejectMessage)
        return;

    DBusMessage *message = mfp->cs->currentMessage;
    uint32_t type = dbus_message_get_type (message);
    uint32_t serial = dbus_message_get_serial (message);
    uint8_t reply = dbus_message_get_no_reply (message) == 0;
    char *path = (char *) dbus_message_get_path (message);
    char *dest = (char *) dbus_message_get_destination (message);
    char *member = (char *) dbus_message_get_member (message);
    char *interface = (char *) dbus_message_get_interface (message);
    char *sender = (char *) dbus_message_get_sender (message);
    char *signature = (char *) dbus_message_get_signature (message);
    char *if_suffix = 0L;

    mfp->cs->rejectMessage = ((*(mfp->mf))
                             (mfp->cs, type, reply, serial, dest, path,
                              member, interface, if_suffix, sender,
                              signature, message, 0L, 0L, 0L, 0L) == HANDLED);
}

static DBusHandlerResult message_filter(DBusConnection * connection,
                                        DBusMessage * message, void *p) {
    DBusConnectionState *cs = p;

    cs->currentMessage = message;
    cs->rejectMessage = 0;

    twalk (cs->filters, filter_current_message);

    cs->currentMessage = 0L;

    if (cs->rejectMessage) {
        cs->rejectMessage = 0;
        return DBUS_HANDLER_RESULT_HANDLED;
    }

    return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
}

uint8_t dbus_svc_add_message_filter(DBusConnectionState * cs,
                                    dbus_svc_MessageHandler mf,
                                    int n_matches,...) {
    MessageFilterNode *mfn;
    char **mp, *m;
    DBusError error;
    va_list va;

    va_start(va, n_matches);

    if (!dbus_connection_add_filter(cs->connection, message_filter, cs, NULL)) {
        if (cs->eh != 0L)
            (*(cs->eh)) ("dbus_svc_add_message_filter: dbus_connection_add_filter failed");
        va_end (va);
        return (0);
    }

    mfn = (MessageFilterNode *) malloc (sizeof (MessageFilterNode));
    if (mfn == 0L) {
        if (cs->eh != 0L)
            (*(cs->eh)) ("dbus_svc_add_message_filter: out of memory");
        va_end (va);
        return (0);
    }

    mfn->mf = mf;
    mfn->cs = cs;
    mfn->n_matches = n_matches;
    mfn->matches = 0L;
    if (n_matches) {
        memset(&error, '\0', sizeof (DBusError));
        dbus_error_init (&error);

        mp = mfn->matches = (char **) calloc(n_matches + 1, sizeof (char *));
        if (mp == 0L) {
            if (cs->eh != 0L)
                (*(cs->eh)) ("dbus_svc_add_message_filter: out of memory");
            va_end (va);
            return (0);
        }

        while (n_matches--) {
            m = va_arg(va, char *);

            if ((*mp = (char *) malloc(strlen (m) + 1)) == 0L) {
                if (cs->eh != 0L)
                    (*(cs->eh)) ("dbus_svc_add_message_filter: out of memory");
                va_end(va);
                return (0);
            }

            strcpy(*mp, m);
            dbus_bus_add_match(cs->connection, *mp, &error);

            if (dbus_error_is_set (&error)) {
                if (cs->eh != 0L)
                    (*(cs->eh)) ("dbus_svc_add_message_filter: dbus_bus_add_match failed for %s: %s %s", *mp, error.name, error.message);
                va_end(va);
                return (0);
            }

            mp++;
        }
    }

    va_end(va);

    return (tsearch (mfn, &(cs->filters), mf_comparator) != 0L);
}

uint8_t dbus_svc_remove_message_filter(DBusConnectionState * cs,
                                       dbus_svc_MessageHandler mf) {
    MessageFilterNode **mfpp = tfind(mf, &(cs->filters), ptr_key_comparator);
    MessageFilterNode *mfp;
    DBusError error;
    int i;

    if ((mfpp != 0L) && ((mfp = *mfpp) != 0L)) {
        mfpp = tdelete(mfp->mf, &(cs->filters), mf_comparator);

        if (mfp->n_matches) {
            memset(&error, '\0', sizeof (DBusError));
            dbus_error_init(&error);

            for (i = 0; i < mfp->n_matches; i++) {
                dbus_bus_remove_match(cs->connection, mfp->matches[i], &error);

                /* XXX fixme: what if other handlers still
                 * want to match this match exp.? */
                if (dbus_error_is_set (&error)) {
                    if (cs->eh != 0L)
                        (*(cs->eh)) ("dbus_svc_remove_message_filter: remove match failed: %s %s", error.name, error.message);
                }
            }

            free(mfp->matches);
        }

        free(mfp);
    }

    if ((mfpp != 0L) && (cs->filters == 0L))
        dbus_connection_remove_filter(cs->connection, message_filter, cs);

    return (mfpp != 0);
}

uint8_t dbus_svc_get_args_va(DBusConnectionState * cs, DBusMessage * msg,
                             dbus_svc_DataType firstType, va_list va) {
    DBusError error;

    memset(&error, '\0', sizeof (DBusError));
    dbus_error_init(&error);

    if ((!dbus_message_get_args_valist (msg, &error, firstType, va)) ||
        dbus_error_is_set (&error)) {
        if (dbus_error_is_set (&error)) {
            if (cs->eh != 0L)
                (*(cs->eh)) ("dbus_svc_get_args failed: %s %s", error.name, error.message);
            dbus_error_free(&error);
        } else if (cs->eh != 0L) {
            (*(cs->eh)) ("dbus_svc_get_args failed: dbus_message_get_args_valist failed");
        }

        return (0);
    }

    return (1);
}

uint8_t dbus_svc_get_args(DBusConnectionState * cs, DBusMessage * msg,
                          dbus_svc_DataType firstType, ...) {
    va_list va;
    uint8_t r;

    va_start(va, firstType);
    r = dbus_svc_get_args_va(cs, msg, firstType, va);
    va_end (va);

    return r;
}

uint8_t dbus_svc_send_va(DBusConnectionState * cs, dbus_svc_MessageType type,
                         int32_t reply_serial, uint32_t * new_serial,
                         char *destination, char *path, char *interface,
                         char *member, dbus_svc_DataType firstType,
                         va_list va) {
    DBusMessageIter iter;
    char *e;
    DBusMessage *msg = dbus_svc_new_message(cs, type, reply_serial,
                                            destination, path, interface,
                                            member);

    if (msg == 0L) {
        if (cs->eh != 0L)
            (*(cs->eh)) ("dbus_svc_send: dbus_svc_new_message failed");
        return 0;
    }

    if (type != DBUS_MESSAGE_TYPE_ERROR) {
        if (!dbus_message_append_args_valist(msg, firstType, va)) {
            if (cs->eh != 0L)
                (*(cs->eh)) ("dbus_svc_send: dbus_message_append_args_valist failed");
            return 0;
        }
    } else {
        if (firstType == DBUS_TYPE_STRING) {
            e = 0L;
            e = va_arg(va, char *);

            if ((e == 0L) || !dbus_message_set_error_name (msg, e)) {
                if (cs->eh != 0L)
                    (*(cs->eh)) ("dbus_svc_send: dbus_message_set_error_name failed");
                return 0;
            }

            firstType = va_arg(va, int);

            if (firstType == DBUS_TYPE_STRING) {
                e = 0L;
                e = va_arg(va, char *);

                if (e == 0L) {
                    if (cs->eh != 0L)
                        (*(cs->eh)) ("dbus_svc_send: NULL error message");
                    return 0;
                }

                dbus_message_iter_init_append(msg, &iter);

                if (!dbus_message_iter_append_basic(&iter, DBUS_TYPE_STRING, &e)) {
                    if (cs->eh != 0L)
                        (*(cs->eh)) ("dbus_svc_send: dbus_message_iter_append_basic failed");
                    return 0;
                }
            }
        } else {
            if (cs->eh != 0L)
                (*(cs->eh)) ("dbus_svc_send: unhandled type for error name: %c", firstType);
            return 0;
        }
    }

    if (!dbus_connection_send(cs->connection, msg, new_serial)) {
        if (cs->eh != 0L)
            (*(cs->eh)) ("dbus_svc_send: dbus_message_send failed");
        return 0;
    }

    if (cs->dh != 0L)
        (*(cs->dh)) ("Sending message");

    dbus_connection_flush(cs->connection);
    return 1;
}

uint8_t dbus_svc_send(DBusConnectionState * cs, dbus_svc_MessageType type,
                      int32_t reply_serial, uint32_t * new_serial,
                      char *destination, char *path, char *interface,
                      char *member, dbus_svc_DataType firstType,...) {
    uint8_t r;
    va_list va;

    va_start(va, firstType);
    r = dbus_svc_send_va(cs, type, reply_serial, new_serial, destination,
                         path, interface, member, firstType, va);
    va_end(va);

    return (r);
}

dbus_svc_MessageHandle dbus_svc_new_message(DBUS_SVC cs,
                                            dbus_svc_MessageType type,
                                            int32_t reply_serial,
                                            char *destination, char *path,
                                            char *interface, char *member) {
    DBusMessage *msg = dbus_message_new(type);

    if (msg == 0L) {
        if (cs->eh != 0L)
            (*(cs->eh)) ("dbus_svc_new_message: dbus_message_set_reply_serial failed");
        return 0;
    }

    if (reply_serial != -1) {
        if (!dbus_message_set_reply_serial(msg, reply_serial)) {
            if (cs->eh != 0L)
                (*(cs->eh)) ("dbus_svc_new_message: dbus_message_set_reply_serial failed");
            return 0;
        }
    }

    if ((destination != 0L) && !dbus_message_set_destination(msg, destination)) {
        if (cs->eh != 0L)
            (*(cs->eh)) ("dbus_svc_new_message: dbus_message_set_destination failed");
        return 0;
    }

    if (!dbus_message_set_path(msg, path)) {
        if (cs->eh != 0L)
            (*(cs->eh)) ("dbus_svc_new_message: dbus_message_set_path failed");
        return 0;
    }

    if (!dbus_message_set_interface(msg, interface)) {
        if (cs->eh != 0L)
            (*(cs->eh)) ("dbus_svc_new_message: dbus_message_set_interface failed - %s", interface);
        return 0;
    }

    if (!dbus_message_set_member(msg, member)) {
        if (cs->eh != 0L)
            (*(cs->eh)) ("dbus_svc_new_message: dbus_message_set_member failed");
        return 0;
    }

    return msg;
}

extern uint8_t dbus_svc_send_message(DBusConnectionState * cs,
                                     dbus_svc_MessageHandle msg,
                                     uint32_t * new_serial) {
    if (!dbus_connection_send(cs->connection, msg, new_serial)) {
        if (cs->eh != 0L)
            (*(cs->eh)) ("dbus_svc_send: dbus_message_send failed");
        return 0;
    }

    if (cs->dh != 0L)
        (*(cs->dh)) ("Sending message");

    dbus_connection_flush(cs->connection);
    return 1;
}

extern uint8_t dbus_svc_message_append_args(DBUS_SVC cs,
                                            dbus_svc_MessageHandle msg,
                                            dbus_svc_DataType firstType, ...) {
    va_list va;

    va_start(va, firstType);

    if (!dbus_message_append_args_valist(msg, firstType, va)) {
        if (cs->eh != 0L)
            (*(cs->eh)) ("dbus_svc_send: dbus_message_append_args failed");
        return 0;
    }

    va_end(va);

    return (1);
}

dbus_svc_MessageHandle dbus_svc_call(DBusConnectionState * cs,
                                     char *destination, char *path,
                                     char *member, char *interface,
                                     dbus_svc_DataType firstType,...) {
    DBusMessage *message = 0L, *reply = 0L;
    va_list va;
    DBusError error;
    int reply_timeout = 20000;

    va_start(va, firstType);

    memset(&error, '\0', sizeof (DBusError));
    dbus_error_init(&error);

    message = dbus_message_new_method_call(destination, path,
                                           interface, member);
    if (message == 0L) {
        if (cs->eh != 0L)
            (*(cs->eh)) ("dbus_svc_call: dbus_message_new_method_call failed");

        va_end(va);
        return (0L);
    }

    if (!dbus_message_append_args_valist(message, firstType, va)) {
        if (cs->eh != 0L)
            (*(cs->eh)) ("dbus_svc_call: dbus_message_append_args_valist failed");

        va_end(va);
        return (0L);
    }

    reply = dbus_connection_send_with_reply_and_block(cs->connection, message,
                                                      reply_timeout, &error);
    if (reply == 0L) {
        if (cs->eh != 0L)
            (*(cs->eh)) ("dbus_svc_call: dbus_message_send_with_reply_and_block failed: %s %s", error.name, error.message);

        va_end (va);
        return (0L);
    }

    return reply;
}

static DBusConnectionState * dbcs_new(DBusConnection * connection) {
    DBusConnectionState *dbcs;

    dbcs = (DBusConnectionState *) malloc(sizeof (DBusConnectionState));
    if (dbcs) {
        memset(dbcs, '\0', sizeof (DBusConnectionState));
        dbcs->connection = connection;
    }

    return (dbcs);
}

static DBusConnectionTimeout * timeout_new(DBusTimeout * timeout) {
    DBusConnectionTimeout *to;

    to = (DBusConnectionTimeout *) malloc(sizeof (DBusConnectionTimeout));
    if (to != 0L) {
        to->to = timeout;
        dbus_timeout_set_data(timeout, to, 0L);
    
        if (dbus_timeout_get_enabled(timeout)) {
            gettimeofday (&(to->tv), 0L);
        } else {
            to->tv.tv_sec = 0;
            to->tv.tv_usec = 0;
        }
    }

    return (to);
}

static dbus_bool_t add_timeout(DBusTimeout * timeout, void *csp) {
    DBusConnectionState *cs = csp;
    DBusConnectionTimeout *to = timeout_new(timeout);

    if (cs->dh != 0L)
        (*(cs->dh)) ("add_timeout: %d", dbus_timeout_get_interval (timeout));

    to->cs = cs;

    if (to) {
        if (tsearch((void *) to, &(cs->timeouts), ptr_key_comparator) != 0L)
            return TRUE;
    }

    if (cs->eh != 0L)
        (*(cs->eh)) ("add_timeout: out of memory");

    return FALSE;
}

static void remove_timeout(DBusTimeout * timeout, void *csp) {
    DBusConnectionState *cs = csp;
    DBusConnectionTimeout *to = dbus_timeout_get_data(timeout);

    if ((to != 0L) && (to->to == timeout)) {
        if (cs->dh != 0L)
            (*(cs->dh)) ("remove_timeout: %d", dbus_timeout_get_interval(to->to));

        if (tdelete((const void *) to, &(cs->timeouts), ptr_key_comparator) != 0L) {
            free(to);
        } else if (cs->eh != 0L) {
            (*(cs->eh)) ("remove_timeout: can't happen?!?: timeout data %p not found", to);
        }
    } else if (cs->eh != 0L) {
        (*(cs->eh)) ("remove_timeout: can't happen?!?: timeout %p did not record data %p %p", timeout, to, ((to != 0L) ? to->to : 0L));
    }
}

static void toggle_timeout(DBusTimeout * timeout, void *csp) {
    DBusConnectionState *cs = csp;
    DBusConnectionTimeout **top = tfind((const void *) dbus_timeout_get_data(timeout), &(cs->timeouts), ptr_key_comparator), *to = 0L;

    if ((top != 0L) && ((to = *top) != 0L) && (to->to == timeout)) {
        if (cs->dh != 0L)
            (*(cs->dh)) ("toggle_timeout: %d", dbus_timeout_get_interval(to->to));

        if (dbus_timeout_get_enabled(timeout)) {
            gettimeofday(&(to->tv), 0L);
        } else {
            to->tv.tv_sec = 0;
            to->tv.tv_usec = 0;
        }
    } else if (cs->eh != 0L) {
        (*(cs->eh)) ("toggle_timeout: can't happen?!?: timeout %p %s %p %p", timeout, ((to == 0L) ? "did not record data" : "not found"), to, ((to != 0L) ? to->to : 0L));
    }
}

static void process_timeout(const void *p, const VISIT which, const int level) {
    DBusConnectionState *cs;
    const void *const *tpp = p;
    DBusConnectionTimeout *to;
    struct timeval tv;
    float now, then, interval;

    gettimeofday(&tv, 0L);

    if ((tpp != 0L) && (*tpp != 0L) && ((which == postorder) || (which == leaf))) {
        to = (DBusConnectionTimeout *) * tpp;
        cs = to->cs;

        if (!dbus_timeout_get_enabled(to->to))
            return;

        cs = dbus_timeout_get_data(to->to);
        then = ((float) to->tv.tv_sec) + (((float) to->tv.tv_usec) / 1000000.0);

        if (then != 0.0) {
            interval = ((float) dbus_timeout_get_interval(to->to)) / 1000.0;
            now = ((float) tv.tv_sec) + (((float) tv.tv_usec) / 1000000.0);

            if ((now - then) >= interval) {
                if (cs->dh != 0L)
                    (*(cs->dh)) ("handle_timeout: %d - %f %f %f", dbus_timeout_get_interval(to->to), then, now, interval);

                dbus_timeout_handle(to->to);
                to->tv = tv;
            }
        } else {
            to->tv = tv;
        }
    }
}

static void process_timeouts(DBusConnectionState * cs) {
    twalk(cs->timeouts, process_timeout);
}

static float next_timeout;

static void find_timeout (const void *p, const VISIT which, const int level) {
    DBusConnectionState *cs;
    const void *const *tpp = p;
    DBusConnectionTimeout *to;
    struct timeval tv;
    float now, then, interval, timeout;

    gettimeofday(&tv, 0L);

    if ((tpp != 0L) && (*tpp != 0L) && ((which == postorder) || (which == leaf))) {
        to = (DBusConnectionTimeout *) * tpp;
        cs = to->cs;

        if (!dbus_timeout_get_enabled(to->to))
            return;

        cs = dbus_timeout_get_data(to->to);
        then = ((float) to->tv.tv_sec) + (((float) to->tv.tv_usec) / 1000000.0);

        if (then != 0.0) {
            interval = ((float) dbus_timeout_get_interval(to->to)) / 1000.0;
            now = ((float) tv.tv_sec) + (((float) tv.tv_usec) / 1000000.0);

	    timeout = then + interval - now;
	    if (timeout < 0)
		timeout = 0;
	    if (next_timeout < 0 || timeout < next_timeout)
		next_timeout = timeout;
        }
    }
}

static struct timeval *find_next_timeout (DBusConnectionState * cs, struct timeval *tv) {
    next_timeout = 1;
    twalk(cs->timeouts, find_timeout);
    printf ("next timeout %f\n", next_timeout);
    if (next_timeout == -1)
	return NULL;
    tv->tv_sec = next_timeout;
    tv->tv_usec = (next_timeout - tv->tv_sec) * 1000000;
    return tv;
}

static void set_watch_fds(DBusWatch * watch, DBusConnectionState * cs) {
    uint8_t flags = dbus_watch_get_flags(watch);
    int fd = dbus_watch_get_unix_fd(watch);

    if (cs->n <= fd)
        cs->n = fd + 1;

    if (dbus_watch_get_enabled(watch)) {
        if (flags & DBUS_WATCH_READABLE)
            FD_SET(fd, &(cs->r_fds));
        else
            FD_CLR(fd, &(cs->r_fds));

        if (flags & DBUS_WATCH_WRITABLE)
            FD_SET(fd, &(cs->w_fds));
        else
            FD_CLR(fd, &(cs->w_fds));

        if (flags & DBUS_WATCH_ERROR)
            FD_SET(fd, &(cs->e_fds));
        else
            FD_CLR(fd, &(cs->e_fds));
    } else {
        FD_CLR(fd, &(cs->r_fds));
        FD_CLR(fd, &(cs->w_fds));
        FD_CLR(fd, &(cs->e_fds));
    }
}

static dbus_bool_t add_watch(DBusWatch * watch, void *csp) {
    DBusConnectionState *cs = csp;

    dbus_watch_set_data(watch, cs, no_free);

    if (cs->dh != 0L)
        (*(cs->dh)) ("add_watch: %d", dbus_watch_get_unix_fd (watch));

    if (tsearch((const void *) watch, &(cs->watches), ptr_key_comparator) == 0L) {
        if (cs->eh != 0L)
            (*(cs->eh)) ("add_watch: out of memory");

        return FALSE;
    }

    set_watch_fds(watch, cs);

    return TRUE;
}

static void remove_watch(DBusWatch * watch, void *csp)
{
    DBusConnectionState *cs = csp;
    int fd = dbus_watch_get_unix_fd(watch);

    if (tdelete((const void *) watch, &(cs->watches), ptr_key_comparator) == 0L)
        if (cs->eh != 0L)
            (*(cs->eh)) ("remove_watch: can't happen?!?: watch not found");

    if (cs->dh != 0L)
        (*(cs->dh)) ("remove_watch: %d", dbus_watch_get_unix_fd(watch));

    FD_CLR(fd, &(cs->r_fds));
    FD_CLR(fd, &(cs->w_fds));
    FD_CLR(fd, &(cs->e_fds));
}

static void toggle_watch(DBusWatch * watch, void *csp) {
    DBusConnectionState *cs = csp;

    if (cs->dh != 0L)
        (*(cs->dh)) ("toggle_watch: %d", dbus_watch_get_unix_fd(watch));

    set_watch_fds(watch, cs);
}

static void process_watch(const void *p, const VISIT which, const int level) {
    const void *const *wpp = p;
    DBusWatch *w;
    int fd;
    uint8_t flags;
    DBusConnectionState *cs;

    if ((wpp != 0L) && (*wpp != 0L) && ((which == postorder) || (which == leaf))) {
        w = (DBusWatch *) * wpp;
        cs = dbus_watch_get_data (w);

        if (cs == 0)
            return;

        if (!dbus_watch_get_enabled(w))
            return;

        fd = dbus_watch_get_unix_fd(w);
        flags = dbus_watch_get_flags(w);

        if (cs->dh != 0L)
            (*(cs->dh)) ("handle_watch: %d", dbus_watch_get_unix_fd(w));

        if ((flags & DBUS_WATCH_READABLE) && (FD_ISSET(fd, &(cs->s_r_fds))))
            dbus_watch_handle(w, DBUS_WATCH_READABLE);

        if ((flags & DBUS_WATCH_WRITABLE) && (FD_ISSET(fd, &(cs->s_w_fds))))
            dbus_watch_handle(w, DBUS_WATCH_READABLE);

        if ((flags & DBUS_WATCH_ERROR) && (FD_ISSET(fd, &(cs->s_e_fds))))
            dbus_watch_handle(w, DBUS_WATCH_ERROR);
    }
}

static void process_watches(DBusConnectionState * cs) {
    twalk(cs->watches, process_watch);
}

static void dispatch_status (DBusConnection * connection, DBusDispatchStatus new_status, void *csp) {
    DBusConnectionState *cs = csp;

    cs->dispatchStatus = new_status;
}

static DBusConnectionState * connection_setup(DBusConnection * connection, dbus_svc_ErrorHandler eh, dbus_svc_ErrorHandler dh) {
    DBusConnectionState *cs = dbcs_new(connection);

    if (cs == 0L) {
        (*(eh)) ("connection_setup: out of memory");
        goto fail;
    }

    cs->eh = eh;
    cs->dh = dh;

    if (!dbus_connection_set_watch_functions(cs->connection, add_watch, remove_watch, toggle_watch, cs, no_free)) {
        if (cs->eh != 0L)
            (*(cs->eh)) ("connection_setup: dbus_connection_set_watch_functions failed");

        goto fail;
    }

    if (!dbus_connection_set_timeout_functions(connection, add_timeout, remove_timeout, toggle_timeout, cs, no_free)) {
        if (cs->eh != 0L)
            (*(cs->eh)) ("connection_setup: dbus_connection_set_timeout_functions failed");

        goto fail;
    }

    dbus_connection_set_dispatch_status_function(connection, dispatch_status, cs, no_free);

    if (dbus_connection_get_dispatch_status(connection) != DBUS_DISPATCH_COMPLETE)
        dbus_connection_ref (connection);

    return cs;

fail:
    if (cs != 0L)
        free(cs);

    dbus_connection_set_dispatch_status_function(connection, NULL, NULL, NULL);
    dbus_connection_set_watch_functions(connection, NULL, NULL, NULL, NULL, NULL);
    dbus_connection_set_timeout_functions(connection, NULL, NULL, NULL, NULL, NULL);

    return 0L;
}

void dbus_svc_main_loop(DBusConnectionState * cs, void (*idle_handler) (DBusConnectionState *)) {
    struct timeval timeout = { 0, 200000 };
    int n_fds;

    while (cs->status != SHUTDOWN) {
        cs->s_r_fds = cs->r_fds;
        cs->s_w_fds = cs->w_fds;
        cs->s_e_fds = cs->e_fds;

	if ((n_fds = select(cs->n, &(cs->s_r_fds), &(cs->s_w_fds), 
			    &(cs->s_e_fds), find_next_timeout (cs, &timeout))) < 0) {
            if (errno != EINTR) {
                if (cs->eh != 0L)
                    (*(cs->eh)) ("select failed: %d : %s", errno, strerror(errno));

                return;
            }
        }

        if (n_fds > 0)
            process_watches(cs);

        process_timeouts(cs);

        if (cs->dispatchStatus != DBUS_DISPATCH_COMPLETE)
            dbus_connection_dispatch(cs->connection);

        if (idle_handler != 0L)
            (*idle_handler) (cs);
    }
}

void dbus_svc_quit(DBusConnectionState * cs) {
    cs->status = SHUTDOWN;
}

DBusConnectionState *dbus_svc_init(dbus_svc_DBUS_TYPE bus, char *name, dbus_svc_ErrorHandler eh, dbus_svc_ErrorHandler dh) {
    DBusConnection *connection;
    DBusError error;
    DBusConnectionState *cs;

    memset(&error, '\0', sizeof (DBusError));

    dbus_error_init(&error);

    if ((connection = dbus_bus_get(bus, &error)) == 0L) {
        (*eh) ("dbus_svc_init failed: %s %s", error.name, error.message);
        return (0L);
    }

    dbus_connection_set_exit_on_disconnect(connection, FALSE);

    if ((cs = connection_setup(connection, eh, dh)) == 0L) {
        (*eh) ("dbus_svc_init failed: connection_setup failed");
        return (0L);
    }

    if (name == 0L)
        return (cs);

    switch (dbus_bus_request_name(connection, name, 0, &error)) {
        case DBUS_REQUEST_NAME_REPLY_PRIMARY_OWNER:
            break;
        case DBUS_REQUEST_NAME_REPLY_EXISTS:
        case DBUS_REQUEST_NAME_REPLY_IN_QUEUE:
        case DBUS_REQUEST_NAME_REPLY_ALREADY_OWNER:
            (*eh) ("dbus_svc_init: dbus_bus_request_name failed:  Name already registered");
            goto give_up;
        default:
            (*eh) ("dbus_svc_init: dbus_bus_request_name failed: %s %s", error.name, error.message);
            goto give_up;
    }

    return (cs);

give_up:
    dbus_connection_close(connection);
    dbus_shutdown();

    return (0L);
}

static void free_root(void *rp) {
    Root *root = rp;

    if (root->tree != root)
        tdestroy(root->tree, free_mh);

    free(root->path);
    free(root);
}

void dbus_svc_shutdown(DBusConnectionState * cs) {
    if (!dbus_connection_set_watch_functions(cs->connection, NULL, NULL, NULL, NULL, NULL))
        if (cs->eh != 0L)
            (*(cs->eh)) ("connection_shutdown: dbus_connection_set_watch_functions: No Memory." "Setting watch functions to NULL failed.");

    if (!dbus_connection_set_timeout_functions(cs->connection, NULL, NULL, NULL, NULL, NULL))
        if (cs->eh != 0L)
            (*(cs->eh)) ("connection_shutdown: dbus_connection_set_timeout_functions: No Memory." "Setting timeout functions to NULL failed.");

    dbus_connection_set_dispatch_status_function(cs->connection, NULL, NULL, NULL);

    tdestroy(cs->timeouts, free);
    cs->timeouts = 0L;
    tdestroy(cs->watches, no_free);
    cs->watches = 0L;
    tdestroy(cs->roots, free_root);
    cs->roots = 0L;

    dbus_connection_close(cs->connection);
    dbus_shutdown();
    free(cs);
}
