/***************************************************************************
 *
 * knetworkmanager-connecioninfodialog.cpp - A NetworkManager frontend for KDE
 *
 * Copyright (C) 2006 Novell, Inc.
 *
 * Author: Helmut Schaa <hschaa@suse.de>, <helmut.schaa@gmx.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 **************************************************************************/

#include <qlabel.h>
#include <qgrid.h>
#include <qprogressbar.h>
#include <qgroupbox.h>
#include <qtabwidget.h>
#include <qtabbar.h>

#include <klocale.h>
#include <kdebug.h>
#include <kactivelabel.h>
#include <kiconloader.h>

#include "knetworkmanager-connectioninfodialog.h"
#include "knetworkmanager-devicestore.h"

ConnectionInfoWidgetImpl::ConnectionInfoWidgetImpl(QWidget* parent, const char* name, WFlags f)
	: ConnectionInfoWidget(parent, name, f)
{

}

QSize ConnectionInfoWidgetImpl::sizeHint() const
{
	return minimumSizeHint();
}

QSize ConnectionInfoWidgetImpl::minimumSizeHint() const
{
	// TODO: dirty hack to set the minimumWidth accroding to the width of the tabs
	QSize size = ConnectionInfoWidget::minimumSizeHint();

	QTabBar helperBar;
	int minWidth = 0;
	for (int i = 0; i < tabWidget->count(); ++i)
	{
		QTab* helper = new QTab(tabWidget->tabIconSet(tabWidget->page(i)), tabWidget->tabLabel(tabWidget->page(i)));
		helperBar.addTab(helper);
		minWidth += helper->rect().width() ;
	}
	if (minWidth > size.width())
		size.setWidth(minWidth);

	return size;
}

ConnectionInfoDialog::ConnectionInfoDialog(KNetworkManager* ctx, QWidget* parent, const char* name, WFlags fl)
	: KDialogBase(KDialogBase::Plain, fl, parent, name, 
                false /*modal*/, i18n("Connection Status"), KDialogBase::Ok)
{
	_infoWidget = NULL;
	_ctx = ctx;
	QVBoxLayout* layout = new QVBoxLayout(plainPage());
	
	// read proc data every 5 seconds
	QTimer* timer = new QTimer(this, "InterfaceStatisticsTimer");
	connect(timer, SIGNAL(timeout()), _ctx->getDeviceStore(), SLOT(getInterfaceStat()));
	timer->start(5000);

	// create info widget and load nice pixmaps
	_infoWidget = new ConnectionInfoWidgetImpl(plainPage());
	_infoWidget->tabWidget->setTabIconSet(_infoWidget->DevicePage, SmallIconSet("wired", QIconSet::Automatic));
	_infoWidget->tabWidget->setTabIconSet(_infoWidget->AddressesPage, SmallIconSet("configure", QIconSet::Automatic));
	_infoWidget->tabWidget->setTabIconSet(_infoWidget->StatisticsPage, SmallIconSet("connect_creating", QIconSet::Automatic));
	_infoWidget->tabWidget->setTabIconSet(_infoWidget->NetworkPage, SmallIconSet("nm_signal_100", QIconSet::Automatic));
	layout->addWidget(_infoWidget);

	// we want to get noticed if the connection changes
	connect(_ctx->getDeviceStore(), SIGNAL(deviceStoreChanged(DeviceStore*)), this, SLOT(deviceStoreChanged(DeviceStore*)));
	deviceStoreChanged(_ctx->getDeviceStore());

	resize(minimumSizeHint());
}

ConnectionInfoDialog::~ConnectionInfoDialog()
{

}

void
ConnectionInfoDialog::deviceStoreChanged(DeviceStore* store)
{
	// did the active device change?
	if (store)
	{
		Device* dev = store->getActiveDevice();
		if (dev != (Device*)_curDev)
		{
			// Device changed
			_curDev = dev;
		}

		// check if network changed
		if (_curDev && _curDev->isWireless())
		{
			Network* net = store->getActiveNetwork(_curDev);
			if ((Network*)_curNet != net)
			{
				// Network changed!!!
				_curNet = net;
			}
		}
	}
	updateConnectionInfo();
}

void
ConnectionInfoDialog::updateConnectionInfo()
{
	if ((Device*)_curDev != NULL)
	{
		_infoWidget->tabWidget->setTabEnabled(_infoWidget->StatisticsPage, true);
		_infoWidget->tabWidget->setTabEnabled(_infoWidget->AddressesPage, true);

		// good device is here, create a connection_infoWidget widget
		_infoWidget->lblConnectionName->setText(_curDev->getProduct());
		_infoWidget->lblVendor->setText(_curDev->getVendor());
		_infoWidget->lblDevice->setText(_curDev->getInterface());
		if (_curDev->getSpeed())
			_infoWidget->lblBandwidth->setText(i18n("%1 Mb/s").arg(_curDev->getSpeed()));
		else
			_infoWidget->lblBandwidth->setText(i18n("Unknown"));
		_infoWidget->lblIP->setText(_curDev->getIPv4Address());
		_infoWidget->lblHardwareAddress->setText(_curDev->getHardwareAddress());
		_infoWidget->lblSubnetmask->setText(_curDev->getSubnetmask());
		_infoWidget->lblBroadcast->setText(_curDev->getBroadcast());
		_infoWidget->lblActive->setText(_curDev->isActive () ? i18n ("yes") : i18n ("no"));

		// RX
		_infoWidget->lblRxBytes->setText(QString("%1").arg(_curDev->getRxBytes()));
		_infoWidget->lblRxMBytes->setText(QString("%1").arg((double)_curDev->getRxBytes() / (1024 * 1024), 0, 'f', 1));
		_infoWidget->lblRxPackets->setText(QString("%1").arg(_curDev->getRxPackets()));
		_infoWidget->lblRxDropped->setText(QString("%1").arg(_curDev->getRxDropped()));
		_infoWidget->lblRxErrors->setText(QString("%1").arg(_curDev->getRxErrors()));
		_infoWidget->lblRxBytesPerSec->setText(QString("%1").arg((double)_curDev->getRxBytesPerSec() / 1024, 0, 'f', 1 ));

		// TX
		_infoWidget->lblTxBytes->setText(QString("%1").arg(_curDev->getTxBytes()));
		_infoWidget->lblTxMBytes->setText(QString("%1").arg((double)_curDev->getTxBytes() / (1024 * 1024), 0, 'f', 1));
		_infoWidget->lblTxPackets->setText(QString("%1").arg(_curDev->getTxPackets()));
		_infoWidget->lblTxDropped->setText(QString("%1").arg(_curDev->getTxDropped()));
		_infoWidget->lblTxErrors->setText(QString("%1").arg(_curDev->getTxErrors()));
		_infoWidget->lblTxBytesPerSec->setText(QString("%1").arg((double)_curDev->getTxBytesPerSec() / 1024, 0, 'f', 1 ));

		if (_curDev->isWired())
		{
			// wired device
			_infoWidget->tabWidget->setTabIconSet(_infoWidget->DevicePage, SmallIconSet("wired", QIconSet::Automatic));
			_infoWidget->tabWidget->setTabEnabled(_infoWidget->NetworkPage, false);

			_infoWidget->lblCarrierDetect->setText(_curDev->hasCarrierDetect () ? i18n ("supported") : i18n ("not supported"));
			_infoWidget->lblCarrierDetect->show();
			_infoWidget->lblCarrierDetectLabel->show();
		}
		else if (_curDev->isWireless())
		{
			// wireless device
			_infoWidget->tabWidget->setTabIconSet(_infoWidget->DevicePage, SmallIconSet("wireless", QIconSet::Automatic));
			_infoWidget->lblCarrierDetect->hide();
			_infoWidget->lblCarrierDetectLabel->hide();
			_infoWidget->tabWidget->setTabEnabled(_infoWidget->NetworkPage, (Network*)_curNet != NULL);
			if (_curNet)
			{
				int signalStrength = _curNet->getStrength();
				if (!signalStrength)
					signalStrength = _curDev->getStrength();

				_infoWidget->lblNetwork->setText(_curNet->getEssid());
				_infoWidget->lblHidden->setText(QString ("%1").arg (_curNet->isHidden () ? i18n ("yes") : i18n ("no")));
				_infoWidget->lblAccesspoint->setText(_curNet->getActiveHardwareAddress());
				_infoWidget->progressSignalstrength->setProgress(signalStrength);
				double freq = _curNet->getFrequency();
				if (freq > 0)
					_infoWidget->lblFrequency->setText(i18n("%1 GHz").arg(freq / 1000000000, 0, 'f', 3));
				else
					_infoWidget->lblFrequency->setText(i18n("Unknown"));
				_infoWidget->lblEncrypted->setText((_curNet->isEncrypted() ? i18n("yes") : i18n("no")));
			}
		}
	}
	else
	{
		_infoWidget->tabWidget->setTabIconSet(_infoWidget->DevicePage, SmallIconSet("nm_no_connection", QIconSet::Automatic));
		_infoWidget->lblConnectionName->setText(i18n("No active device"));
		_infoWidget->lblVendor->setText("-");
		_infoWidget->lblDevice->setText("-");
		_infoWidget->lblBandwidth->setText("-");
		_infoWidget->lblActive->setText("-");
		_infoWidget->lblCarrierDetect->setText("-");

		_infoWidget->tabWidget->setTabEnabled(_infoWidget->NetworkPage, false);
		_infoWidget->tabWidget->setTabEnabled(_infoWidget->StatisticsPage, false);
		_infoWidget->tabWidget->setTabEnabled(_infoWidget->AddressesPage, false);
	}
}

#include "knetworkmanager-connectioninfodialog.moc"

