// -*- C++ -*-
#include <memory>
#include <vector>
#include <iterator>

#ifndef EPT_LIST_H
#define EPT_LIST_H

namespace ept {
namespace core {
namespace list {

template< typename List >
struct ListIterator
{
    typedef std::forward_iterator_tag iterator_category;
    typedef typename List::Type value_type;
    typedef ptrdiff_t difference_type;
    typedef value_type &pointer;
    typedef value_type &reference;

    List l;

    ListIterator &operator++() {
        l = l.tail();
        return *this;
    }

    ListIterator operator++(int) {
        ListIterator i = *this;
        operator++();
        return i;
    }

    typename List::Type operator*() {
        return l.head();
    }

    bool operator==( const ListIterator &o ) const {
        return l.empty() && o.l.empty();
    }

    bool operator!=( const ListIterator &o ) const {
        return !(l.empty() && o.l.empty());
    }

    ListIterator( List _l = List() ) : l( _l )
    {}

};

template< typename List >
struct Sorted
{
    typedef typename List::Type Type;
    List m_list;
    mutable std::auto_ptr< std::vector< Type > > m_sorted;
    int m_pos;

    void sort() const {
        if ( m_sorted.get() )
            return;
        m_sorted.reset( new std::vector< Type >() );
        std::copy( ListIterator< List >( m_list ), ListIterator< List >(),
                   std::back_inserter( *m_sorted ) );
        std::sort( m_sorted->begin(), m_sorted->end() );
    }

    Type head() const {
        sort();
        return (*m_sorted)[ m_pos ];
    }

    Sorted tail() const {
        sort();
        Sorted s = *this;
        s.m_pos ++;
        return s;
    }

    bool empty() const {
        sort();
        return m_pos == m_sorted->size();
    }

    Sorted( const Sorted &o ) : m_list( o.m_list ), m_sorted( o.m_sorted ),
                                m_pos( o.m_pos ) {}

    Sorted &operator=( const Sorted &o ) {
        m_sorted = o.m_sorted;
        m_list = o.m_list;
        m_pos = o.m_pos;
        return *this;
    }

    Sorted( List l ) : m_list( l ), m_sorted( 0 ), m_pos( 0 ) {}
};

template< typename List, typename Predicate >
struct Filtered
{
    typedef typename List::Type Type;
    mutable List m_list;
    Predicate m_pred;

    bool empty() const {
        seek();
        return m_list.empty();
    }

    Type head() const {
        seek();
        return m_list.head();
    }

    void seek() const
    {
        while ( !m_list.empty() && !m_pred( m_list.head() ) )
            m_list = m_list.tail();
    }

    Filtered tail() const
    {
        Filtered r = *this;
        r.seek();
        r.m_list = r.m_list.tail();
        return r;
    }

    Filtered( List l, Predicate p )
        : m_list( l ), m_pred( p )
    {
    }
};

template< typename List >
struct Take {
    List l;
    int remaining;

    typedef typename List::Type Type;

    Type head() {
        return l.head();
    }

    bool empty() {
        return l.empty() || remaining == 0;
    }

    Take tail() {
        Take t;
        t.remaining = remaining - 1;
        t.l = l.tail();
        return t;
    }

    Take( List _l, int m ) : l( _l ), remaining( m ) {}
    Take() : remaining( 0 ) {}
};

template< typename List, typename F >
struct Map {
    List l;
    F f;
    typedef typename F::result_type Type;

    Type head() {
        return f( l.head() );
    }

    Map tail() {
        Map m;
        m.l = l.tail();
        m.f = f;
        return m;
    }

    bool empty() {
        return l.empty();
    }

    Map() {}
    Map( const List &_l, const F &_f ) 
        : l( _l ), f( _f )
    {}
};

template< typename List >
size_t count( List l ) {
    size_t count = 0;
    while ( !l.empty() ) {
        l = l.tail();
        ++ count;
    }
    return count;
}

#undef foreach // Work around Qt braindamage.

template< typename List, typename F >
void foreach( List l, F f ) {
    size_t count = 0;
    while ( !l.empty() ) {
        f( l.head() );
        l = l.tail();
    }
}

template< typename List, template< typename > class F >
void foreach( List l, F< typename List::Type > f ) {
    size_t count = 0;
    while ( !l.empty() ) {
        f( l.head() );
        l = l.tail();
    }
}

template< typename List, typename Pred >
Filtered< List, Pred > filter( List l, Pred p )
{
    return Filtered< List, Pred >( l, p );
}

template< typename List, typename F >
Map< List, F > map( const List &l, const F &f )
{
    return Map< List, F >( l, f );
}

template< typename List >
Sorted< List > sort( List l )
{
    return Sorted< List >( l );
}

template< typename List >
Take< List > take( int t, List l )
{
    return Take< List >( l, t );
}

template< typename List >
ListIterator< List > begin( List l ) {
    return ListIterator< List >( l );
}

template< typename List >
ListIterator< List > end( List ) {
    return ListIterator< List >();
}

}
}
}

#endif
