/* Copyright (C) 1995 Bjoern Beutel. */

/* Description. =============================================================*/

/* Tools for a command line interpreter. */

/* Includes. ================================================================*/

#include <stdio.h>
#include <stdlib.h>
#include <setjmp.h>
#include <string.h>
#include <glib.h>
#include "basic.h"
#include "input.h"
#include "files.h"
#include "commands.h"

/* Types. ===================================================================*/

typedef struct /* An alias. */
{ 
  list_node_t *next;
  string_t name;
  string_t line; /* Command line this alias stands for. */
} alias_t;

/* Global variables. ========================================================*/

command_t **options;
/* The options that can be set by the set command.
 * This has to be set before executing "set_command". */

bool_t leave_program; /* Set to TRUE if program is to leave. */
bool_t leave_command_loop; /* Set to TRUE if command loop is to leave. */
bool_t in_command_loop; /* Set to TRUE if program is in command loop. */

/* Variables. ===============================================================*/

static command_t **commands_local; 
/* Local copy of command table for "help_command". */

static list_t alias_list;

/* Functions for user interrupts. ===========================================*/

void 
check_user_break( void )
/* Abort command execution if user has sent an interrupt signal. */
{ 
  if (user_break_requested)
  {
    user_break_requested = FALSE;
    complain( "User interrupt." );
  }
}

/* Functions for reading and executing commands. ============================*/

static command_t *
find_command( string_t command_name, command_t *commands[] )
/* Find the entry for COMMAND_NAME in COMMANDS and return it.
 * Return NULL if COMMAND_NAME can't be found. */
{ 
  int_t i;
  string_t names, name;
  bool_t found_command;
      
  for (i = 0; commands[i] != NULL; i++) 
  { 
    names = commands[i]->names;
    while (*names != EOS) 
    { 
      name = parse_word( &names );
      found_command = (strcmp_no_case( command_name, name ) == 0);
      free_mem( &name);
      if (found_command) 
	return commands[i];
    }
  }
  return NULL;
}

/*---------------------------------------------------------------------------*/

void 
execute_command( string_t command_line, string_t command_type,
                 command_t *commands[], bool_t use_aliases )
/* Execute COMMAND_LINE using COMMANDS.
 * If there is no such command, print an error.
 * COMMAND_TYPE may be "command" or "option" and is used for error messages.
 * If USE_ALIASES == TRUE, look for the command in the list of aliases. */
{ 
  string_t line, command_name;
  string_t alias_line, line_p, name_readable; 
  command_t *command;
  bool_t command_found;
  alias_t *alias;
      
  /* Divide command line into strings COMMAND and ARGUMENTS. */
  line = command_name = NULL;
  TRY 
  { 
    line_p = line = replace_vars_in_string( command_line );
    parse_whitespace( &line_p );
    if (*line_p != EOS) 
    { 
      command_name = parse_word( &line_p );
      command_found = FALSE;
      user_break_requested = FALSE;
      commands_local = commands;
      command = find_command( command_name, commands );
      if (command != NULL) /* Execute the command. */
      {
	command_found = TRUE;
	command->command( line_p );
      } 
      else if (use_aliases) 
      { 
	FOREACH( alias, alias_list ) 
	{ 
	  if (strcmp_no_case( alias->name, command_name ) == 0) 
	  { 
	    command_found = TRUE;
	    alias_line = replace_arguments( alias->line, "a", line_p );
	    execute_command( alias_line, command_type, commands, FALSE );
	    free_mem( &alias_line );
	    break;
	  }
	}
      }
      if (! command_found) 
      { 
	name_readable = new_string_readable( command_name, NULL );
	free_mem( &command_name );
	command_name = name_readable;
	complain( "Unknown %s %s.", command_type, command_name );
      }
    }
  } 
  FINALLY 
  { 
    free_mem( &command_name );
    free_mem( &line );
  }
  END_TRY;
}

/*---------------------------------------------------------------------------*/

void 
execute_command_file( string_t command_file, string_t line_header, 
                      command_t *commands[] )
/* Execute commands in COMMAND_FILE using COMMANDS.
 * If LINE_HEADER != NULL, command lines must start with LINE_HEADER. */
{ 
  FILE *command_stream;
  string_t include_file, header, command_line_p;
  char_t *command_line;
  bool_t is_command_line;
  volatile int_t line_count;
  volatile bool_t leave_loop;

  command_stream = open_stream( command_file, "r" );
  leave_loop = FALSE;
  line_count = 0;
  while (! leave_loop) 
  { 
    header = command_line = NULL;
    TRY 
    { 
      /* Read a command line */
      command_line = read_line( command_stream );
      if (command_line == NULL) 
	leave_loop = TRUE;
      else 
      { 
	cut_comment( command_line );
	line_count++;
        command_line_p = command_line;
        if (*command_line_p == EOS) 
	  is_command_line = FALSE;
	else if (line_header == NULL) 
	  is_command_line = TRUE;
	else 
        { 
	  /* Check if current line begins with LINE_HEADER. */
	  header = parse_word( &command_line_p );
          is_command_line = FALSE;
          if (strcmp_no_case( header, line_header ) == 0) 
	    is_command_line = TRUE;
	  else if (strcmp_no_case( header, "include:" ) == 0) 
	  { 
	    include_file = parse_absolute_path( &command_line_p, 
						command_file );
            parse_end( &command_line_p );
            execute_command_file( include_file, line_header, commands );
            free_mem( &include_file );
          }
        }
        if (is_command_line) 
	  execute_command( command_line_p, "command", commands, TRUE );
      }
    } 
    IF_ERROR 
    {
      print_text( error_text, " (\"%s\", line %d)", 
		  name_in_path( command_file ), line_count );
    } 
    FINALLY 
    { 
      free_mem( &header );
      free_mem( &command_line );
    }
    END_TRY;
  }
  close_stream( &command_stream, command_file );
}

/*---------------------------------------------------------------------------*/

void 
free_aliases( void )
/* Free all currently defined aliases. */
{ 
  alias_t *alias;

  FOREACH_FREE( alias, alias_list ) 
  { 
    free_mem( &alias->name );
    free_mem( &alias->line );
  }
}

/* General commands. ========================================================*/

static void 
print_help_summary( command_t *commands[] )
/* Print names of all commands in COMMANDS. */
{ 
  int i, j;
  string_t names, name;

  for (i = 0; commands[i] != NULL; i++) 
  { 
    names = commands[i]->names;
    name = parse_word( &names );
    if (i > 0 && i % 5 == 0) 
      printf( "\n" );
    printf( "%s", name );
    for (j = g_utf8_strlen( name, -1 ); j < 15; j++) 
      printf( " " );
    free_mem( &name );
  }
  printf( "\n" );
}

/*---------------------------------------------------------------------------*/

static void 
print_help( string_t command_type, command_t *command )
/* Print help about COMMAND, 
 * which is of COMMAND_TYPE (either "command" or "option"). */
{ 
  string_t names, name;

  names = command->names;

  /* Print full name. */
  name = parse_word( &names );
  printf( "%s \"%s\"", command_type, name );
  free_mem( &name );

  /* Print shortcuts. */
  while (*names != EOS) 
  { 
    name = parse_word( &names );
    printf( ", \"%s\"", name );
    free_mem( &name );
  }

  /* Print help text. */
  printf( ":\n%s", command->help );
}

/*---------------------------------------------------------------------------*/

static void 
do_help( string_t arguments )
/* Give help on commands. */
{ 
  string_t command_name;
  command_t *command, *option;

  if (*arguments == EOS) 
  { 
    printf( "Commands available: "
            "(Use \"help COMMAND\" to get help about COMMAND.)\n" );
    print_help_summary( commands_local );
    printf( "\nOptions available: "
            "(Use \"help OPTION\" to get help about OPTION.)\n" );
    print_help_summary( options );
  } 
  else 
  { 
    command_name = parse_word( &arguments );
    if (strcmp_no_case( command_name, "command" ) == 0) 
    { 
      free_mem( &command_name );
      command_name = parse_word( &arguments );
      parse_end( &arguments );
      command = find_command( command_name, commands_local );
      option = NULL;
      if (command == NULL) 
	complain( "\"%s\" is no command.", command_name );
    } 
    else if (strcmp_no_case( command_name, "option" ) == 0) 
    { 
      free_mem( &command_name );
      command_name = parse_word( &arguments );
      command = NULL;
      option = find_command( command_name, options );
      if (option == NULL) 
	complain( "\"%s\" is no option.", command_name );
    } 
    else 
    { 
      command = find_command( command_name, commands_local );
      option = find_command( command_name, options );
      if (command != NULL && option != NULL) 
      { 
	complain( "Use \"help command %s\" or \"help option %s\",", 
		  command_name, command_name );
      } 
      else if (command == NULL && option == NULL) 
	complain( "\"%s\" is neither a command nor an option.", command_name );
    }
    parse_end( &arguments );
    if (command != NULL) 
      print_help( "Command", command );
    if (option != NULL) 
      print_help( "Option", option );
    free_mem( &command_name );
  }
}

command_t help_command = 
{ 
  "help h ?", do_help,
  "Usage:\n"
  "  help command COMMAND -- Print help about COMMAND.\n"
  "  help option OPTION -- Print help about OPTION.\n"
  "  help COMMAND_OR_OPTION -- Print help about COMMAND_OR_OPTION if unique.\n"
  "  help -- get a list of all available commands and options\n"
};

/*---------------------------------------------------------------------------*/

static void 
do_quit( string_t arguments )
/* Quit the program. */
{ 
  parse_end( &arguments );
  leave_program = TRUE;
}

command_t quit_command = 
{ 
  "quit exit q", do_quit,
  "Leave the program.\n"
  "Usage: quit\n"
};

/*---------------------------------------------------------------------------*/

static void 
do_get( string_t arguments )
/* Get setting for ARGUMENTS. */
{ 
  int_t i;
  string_t option;
      
  if (*arguments == EOS) 
  { 
    for (i = 0; options[i] != NULL; i++) 
      options[i]->command( "" );
  } 
  else
  { 
    option = parse_word( &arguments );
    parse_end( &arguments );
    execute_command( option, "option", options, FALSE );
    free_mem( &option );
  }
}

command_t get_command = 
{ 
  "get", do_get,
  "Query program settings.\n"
  "Usage:\n"
  "  get OPTION -- Print the setting of OPTION.\n"
  "  get -- Print all settings.\n"
};

/*---------------------------------------------------------------------------*/

static void 
do_set( string_t arguments )
/* Set an option. */
{ 
  string_t option_arguments, option;

  option_arguments = arguments;
  option = parse_word( &arguments );
  if (*arguments == EOS) 
    complain( "Missing arguments to set \"%s\".", option );
  execute_command( option_arguments, "option", options, FALSE );
  free_mem( &option );
}

command_t set_command = 
{ 
  "set", do_set,
  "Change program settings.\n"
  "Usage:\n"
  "  set OPTION ARGUMENT -- Change value of OPTION to ARGUMENT.\n"
};

/*---------------------------------------------------------------------------*/

static command_t *set_commands[] = {&set_command, NULL};
/* The commands that can be called during initialisation. */

void 
execute_set_commands( string_t file_name, string_t prefix )
/* Execute set commands in file FILE_NAME that are prefixed with PREFIX. */
{ 
  execute_command_file( file_name, prefix, set_commands );
}

/*---------------------------------------------------------------------------*/

static void 
do_alias_option( string_t arguments )
{ 
  alias_t *alias;
  string_t name, line;

  if (*arguments == EOS) 
  { 
    if (alias_list.first == NULL) 
      printf( "alias: (none)\n" );
    else 
    { 
      FOREACH( alias, alias_list ) 
      { 
	line = new_string_readable( alias->line, NULL );
        printf( "alias \"%s\": %s\n", alias->name, line );
        free_mem( &line );
      }
    }
  } 
  else 
  { 
    name = parse_word( &arguments );

    /* Find the alias NAME. */
    FOREACH( alias, alias_list ) 
    { 
      if (strcmp_no_case( alias->name, name ) == 0) 
	break;
    }

    if (*arguments == EOS) 
    { 
      /* Delete the alias. */
      if (alias == NULL) 
	complain( "Alias \"%s\" not defined." );
      free_mem( &alias->name );
      free_mem( &alias->line );
      free_node( &alias_list, (list_node_t *) alias );
    } 
    else 
    { 
      /* Define an alias. */
      line = parse_word( &arguments );
      parse_end( &arguments );
      if (alias == NULL) /* Create alias entry if it doesn't exist. */
      { 
	alias = new_node( &alias_list, sizeof( alias_t ), LIST_END );
        alias->name = new_string( name, NULL );
      } 
      else 
	free_mem( &alias->line );
      alias->line = line;
    }
    free_mem( &name );
  }
}

command_t alias_option = 
{ 
  "alias", do_alias_option,
  "Define command line aliases.\n"
  "Usage:\n"
  "  alias NAME LINE -- Define alias NAME for LINE.\n"
  "  alias NAME -- Undefine alias NAME.\n"
  "The LINE may contain the following special sequence:\n"
  "  %a -- The command line arguments when invoking the alias.\n"

};

/* End of file. =============================================================*/
