/* Copyright (C) 2002 Bjoern Beutel. */

/* Description. =============================================================*/

/* Read in and display Malaga Variables. */

/* Includes. ================================================================*/

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <setjmp.h>
#include <string.h>
#include <gtk/gtk.h>
#include <glib.h>
#include "basic.h"
#include "scanner.h"
#include "input.h"
#include "canvas.h"
#include "variables.h"

/* Types. ===================================================================*/

typedef struct 
{ 
  list_node_t *next;
  string_t string;
  pos_string_t *name;
  pos_value_t *value;
  bool_t is_shown;
} variable_t;

typedef struct 
{ 
  list_node_t *next;
  string_t name;
} hidden_var_t;

/* Global variables. ========================================================*/

rectangle_t variables_geometry;
string_t variables_font_family;
int_t variables_font_size;

/* Variables. ===============================================================*/

static list_t variables;
static list_t hidden_vars;
static canvas_t *variables_canvas;
static pos_string_t *equal, *dots;

/* Functions. ===============================================================*/

static void
configure_variables( canvas_t *canvas, int_t *width_p, int_t *height_p )
{
  int_t width, height;
  variable_t *variable;
  int_t font_height = get_font_height( canvas );
  int_t space_width = get_space_width( canvas );
  int_t border_width = get_border_width( canvas );

  config_pos_string( equal, canvas );
  config_pos_string( dots, canvas );

  width = height = border_width;
  FOREACH( variable, variables ) 
  { 
    if (variable != (variable_t *) variables.first) 
      height += font_height;

    config_pos_string( variable->name, canvas );
    variable->name->x = border_width;
    variable->name->y = height;
    equal->x = variable->name->x + variable->name->width + space_width;
    if (variable->is_shown)
    {
      config_pos_value( variable->value, canvas );
      variable->value->x = equal->x + equal->width + space_width;
      variable->value->y = height;
      variable->name->y += (variable->value->height - font_height) / 2;
      width = MAX( width, variable->value->x + variable->value->width );
      height += variable->value->height;
    }
    else
    {
      dots->x = equal->x + equal->width + space_width;
      width = MAX( width, dots->x + dots->width );
      height += font_height;
    }
  }

  *width_p = width + border_width;
  *height_p = height + border_width;
}

/*---------------------------------------------------------------------------*/

static void
expose_variables( canvas_t *canvas, rectangle_t *area )
{
  int_t space_width = get_space_width( canvas );
  variable_t *variable;
  
  set_color( BLACK );
  FOREACH( variable, variables ) 
  { 
    draw_pos_string( variable->name, canvas );
    equal->x = variable->name->x + variable->name->width + space_width;
    equal->y = variable->name->y;
    draw_pos_string( equal, canvas );
    if (variable->is_shown) 
      draw_pos_value( variable->value, canvas );
    else 
    {
      dots->x = equal->x + equal->width + space_width;
      dots->y = equal->y;
      draw_pos_string( dots, canvas );
    }
  }
}

/*---------------------------------------------------------------------------*/

static void
show_variable( variable_t *variable, bool_t do_show )
{
  hidden_var_t *hidden_var;

  if (variable->is_shown == do_show) 
    return;
  if (do_show) 
  { 
    /* Remove the variable in the list of hidden variables. */
    FOREACH( hidden_var, hidden_vars )
    {
      if (strcmp_no_case( hidden_var->name, variable->string ) == 0) 
	break;
    }
    free_mem( &hidden_var->name );
    free_node( &hidden_vars, (list_node_t *) hidden_var );
  }
  else 
  { 
    /* Add the variable to the list of hidden variables. */
    hidden_var = new_node( &hidden_vars, sizeof( hidden_var_t ), LIST_END );
    hidden_var->name = new_string( variable->string, NULL );
  }
  variable->is_shown = do_show;
}

/*---------------------------------------------------------------------------*/

static bool_t
mouse_event( canvas_t *canvas, int_t x, int_t y, int_t button )
/* Called if pointer has entered CANVAS at position (X,Y), 
 * if mouse has been moved to position (X,Y),
 * or if BUTTON has been pressed at position (X,Y). */
{
  variable_t *variable;
  int_t font_height = get_font_height( canvas );

  FOREACH( variable, variables )
  {
    if (x >= variable->name->x && x < variable->name->x + variable->name->width
	&& y >= variable->name->y && y < variable->name->y + font_height)
    {
      if (button != 0)
      {
	show_variable( variable, ! variable->is_shown );
	configure_canvas( canvas );
      }
      else 
	set_cursor( canvas, TRUE );
      return TRUE;
    }
  }
  set_cursor( canvas, FALSE );
  return FALSE;
}

/*---------------------------------------------------------------------------*/

static void
show_all_variables( canvas_t *canvas, guint do_show )
{
  variable_t *variable;

  FOREACH( variable, variables ) 
    show_variable( variable, do_show );
  configure_canvas( canvas );
}

/*---------------------------------------------------------------------------*/

static GtkItemFactoryEntry menu_items[] = 
{
  { "/Variables", NULL, NULL, 0, "<Branch>" },
  { "/Variables/Show All", NULL, show_all_variables, TRUE, NULL },
  { "/Variables/Hide All", NULL, show_all_variables, FALSE, NULL },
};

/*---------------------------------------------------------------------------*/

static void free_variables( canvas_t *canvas )
{
  variable_t *variable;

  free_pos_string( &equal );
  free_pos_string( &dots );
  FOREACH_FREE( variable, variables ) 
  { 
    free_mem( &variable->string );
    free_pos_string( &variable->name );
    free_pos_value( &variable->value );
  }
}

/*---------------------------------------------------------------------------*/

void 
read_variables( void )
/* Read new variables from STDIN. */
{
  string_t line;
  variable_t *variable;
  hidden_var_t *hidden_var;

  free_variables( NULL );

  while (TRUE) 
  { 
    line = read_line( stdin );
    if (line == NULL) 
      complain( "Premature EOF." );
    if (strcmp_no_case( line, "end" ) == 0) 
      break;

    /* Read a new variable. */
    variable = new_node( &variables, sizeof( variable_t ), LIST_END );
    set_scanner_input( line );

    /* Read variable name. */
    test_token( TOK_STRING );
    variable->name = new_pos_string( token_string );
    variable->string = new_string( token_string, NULL );
    read_next_token();

    /* Read variable value. */
    parse_token( '{' );
    variable->value = parse_pos_value();
    parse_token( '}' );
    parse_token( EOF );
    set_scanner_input( NULL );
    free_mem( &line );

    /* Check if variable is to be shown. */
    FOREACH( hidden_var, hidden_vars )
    {
      if (strcmp_no_case( hidden_var->name, variable->string ) == 0) 
	break;
    }
    variable->is_shown = (hidden_var == NULL);
  }
  free_mem( &line );
  
  equal = new_pos_string( "=" );
  dots = new_pos_string( "..." );

  if (variables_canvas == NULL) 
  { 
    variables_canvas = create_canvas(
      "Malaga Variables", "variables.eps", &variables_geometry, 
      configure_variables, expose_variables, free_variables, mouse_event,
      TRUE, menu_items, ARRAY_LENGTH( menu_items ) );
  } 
  else
  {
    configure_canvas( variables_canvas );
    show_canvas( variables_canvas );
  }
}

/* End of file. =============================================================*/
