/**
  @file btcond-bt.c

  @author Johan Hedberg <johan.hedberg@nokia.com>

  Copyright (C) 2006 Nokia Corporation. All rights reserved.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License, version 2, as
  published by the Free Software Foundation.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA

*/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <time.h>
#include <errno.h>
#include <stdint.h>
#include <glib.h>

#include "config.h"

#include "log.h"
#include "dbus.h"
#include "bt-dbus.h"
#include "dbus-helper.h"
#include "bt-utils.h"
#include "state.h"
#include "bt-error.h"
#include "btcond-signals.h"
#include "btcond-rfcomm.h"
#include "btcond-bt.h"

#define STREQ(a,b) (strcasecmp((a), (b)) == 0)

#define SAP_MIN_KLEN 64

#define HCITOOL "hcitool"

/* How long to wait for btsdp reply in ms. -1 means default timeout */
#define BTSDP_TIMEOUT 22000

/** Bluetooth device information */
typedef struct {
    char            *bda;            /**< BT address */

    GSList          *services;       /**< List of BtService elements */

    GSList          *bound;          /**< List of RfcommDev elements */
    GSList          *connected;      /**< List of RfcommDev elements */

    gboolean         need_update;    /**< Update at next chance */
    gboolean         auto_update;    /**< If FALSE, skip in process_queue() */

    GSList          *wanted_svcs;    /**< Services to be queried via SDP */
    DBusPendingCall *pending_update; /**< Waiting for SDP reply */

    GSList          *notify_list;    /**< List of functions to call on update */
} BdInfo;

typedef struct {
    BdInfo     *bd;        /**< BT Device that the service belongs to */
    BtService  *svc;       /**< Service that the device node is bound to */
    gchar      *owner;     /**< Owner D-BUS service of this node */
    uint8_t     id;        /**< Device number (/dev/rfcommN) */
    GIOChannel *io;        /**< NULL for bind-only services */
    guint       io_id;     /**< GSource ID  of GIOChannel */
} RfcommDev;

typedef struct {
    BdInfo      *dev;
    BtService   *svc;
    uint8_t      channel;
    gchar       *owner;      /**< D-BUS service which requested the connection */
    btcon_cb_t   func;
    gpointer     data;
    GFreeFunc    free_data;
} ConnectData;

typedef struct {
    btsdp_cb_t  func;
    gpointer    user_data;
    GFreeFunc   free_data;
} PendingSdpInfo;

static GSList *devs = NULL;

static gboolean sdp_in_progress;

static void _schedule_capability_update(BdInfo *dev, btsdp_cb_t func, const char *svc,
                                        gpointer user_data, GFreeFunc free_data);

static void bt_owner_disconnect(const char *name, RfcommDev *node);

static void free_pending_info(BdInfo *dev)
{
    GSList *l;

    for (l = dev->notify_list; l != NULL; l = l->next) {
        PendingSdpInfo *info;
        info = l->data;
        if (info->user_data && info->free_data)
            info->free_data(info->user_data);
        g_free(info);
    }

    g_slist_free(dev->notify_list);
    dev->notify_list = NULL;
}

static void bt_service_ref(BtService *svc)
{
    svc->__ref += 1;
}

static void bt_service_unref(BtService *svc)
{
    svc->__ref -= 1;
    if (!svc->__ref) {
        g_free(svc->name);
        g_free(svc->dscr);
        g_free(svc);
    }
}
                
static void free_rfcomm_dev(RfcommDev *dev, gboolean complete)
{
    if (dev->io) {
        rfcomm_disconnect(dev->io, dev->id, NULL);
        g_source_remove(dev->io_id);
        send_dbus_rfcomm_status(dev->bd->bda, dev->svc->name, "disconnected");
    }
    else if (complete)
        rfcomm_release(dev->id, NULL);
    bt_service_unref(dev->svc);
    if (dev->owner) {
        if (complete)
            remove_name_listener(get_dbus_connection(), dev->owner,
                    (name_cb)bt_owner_disconnect, dev);
        g_free(dev->owner);
    }
    g_free(dev);
}

static void free_service_list(GSList *services)
{
    g_slist_foreach(services, (GFunc)bt_service_unref, NULL);
    g_slist_free(services);
}

static void free_rfcomm_dev_list(GSList *devlist)
{
    g_slist_foreach(devlist, (GFunc)free_rfcomm_dev, (gpointer)TRUE);
    g_slist_free(devlist);
}

static gint cmp_dev(BdInfo *dev, const gchar *bda)
{
    return g_ascii_strcasecmp(dev->bda, bda);
}

static BdInfo *create_new_device(const char *bda)
{
    BdInfo *bd;

    if (!bda_ok(bda)) {
        error("Ignoring invalid BDA: %s", bda);
        return NULL;
    }

    debug("Adding %s to list of known devices", bda);

    bd = g_new0(BdInfo, 1);
    bd->bda = g_strdup(bda);
    bd->need_update = TRUE;
    bd->auto_update = TRUE;
    devs = g_slist_append(devs, bd);

    return bd;
}

static BdInfo *get_dev(const gchar *bda)
{
    GSList *element;

    element = g_slist_find_custom(devs, bda, (GCompareFunc)cmp_dev);
    if (element == NULL)
        return create_new_device(bda);
    else
        return element->data;
}

static void bt_owner_disconnect(const char *name, RfcommDev *node)
{
    debug("D-BUS owner for (%s, %s) exited, disconnecting service.",
            node->svc->name, node->bd->bda);
    node->bd->connected = g_slist_remove(node->bd->connected, node);
    free_rfcomm_dev(node, FALSE);
}

static void _expire_dev_capability(BdInfo *dev)
{
    dev->need_update = TRUE;
}

static gint cmp_service_name(BtService *svc, const char *name) {
    return strcmp(svc->name, name);
}

static gint cmp_service_chan(BtService *svc, gpointer data)
{
    long int channel = GPOINTER_TO_INT(data);
    return svc->channel - (uint8_t)channel;
}

static gint cmp_dev_service_name(RfcommDev *dev, const char *name)
{
    g_assert(dev->svc != NULL);
    return strcmp(dev->svc->name, name);
}

static gint cmp_dev_service_chan(RfcommDev *dev, long int *channel)
{
    g_assert(dev->svc != NULL);
    return dev->svc->channel - (uint8_t)(*channel);
}

static gint cmp_dev_id(RfcommDev *dev, uint8_t *id)
{
    return dev->id - *id;
}

static RfcommDev *find_rfcomm_dev(GSList *devlist,
                                  const char *name, long int ch)
{
    GSList *l;

    if (ch > 255 || ch < 0)
        return NULL;

    for (l = devlist; l != NULL; l = l->next) {
        RfcommDev *dev = l->data;
        if (STREQ(name, dev->svc->name) && dev->svc->channel == (uint8_t)ch)
            return dev;
    }

    return NULL;
}

static RfcommDev *find_rfcomm_dev_id(GSList *devlist, uint8_t id)
{
    GSList *match;

    match = g_slist_find_custom(devlist, &id,
                                (GCompareFunc)cmp_dev_id);

    if (match)
        return match->data;
    else
        return NULL;
}

static RfcommDev *find_rfcomm_dev_ch(GSList *devlist, long int ch)
{
    GSList *match;

    if (ch > 255 || ch < 0)
        return NULL;

    match = g_slist_find_custom(devlist, &ch,
                                (GCompareFunc)cmp_dev_service_chan);

    if (match)
        return match->data;
    else
        return NULL;
}

static RfcommDev *find_rfcomm_dev_name(GSList *devlist, const char *name)
{
    GSList *match;

    if (isdigit(name[0]) && strlen(name) < 4)
        return find_rfcomm_dev_ch(devlist, strtol(name, NULL, 0));
    else
        match = g_slist_find_custom(devlist, name,
                                    (GCompareFunc)cmp_dev_service_name);

    if (match)
        return match->data;
    else
        return NULL;
}

static time_t calculate_expiry_time(uint32_t ttl)
{
    /* Give minimum 10s lifetime for services */
    if (ttl == 0)
        ttl = 10;

    return time(NULL) + ttl;
}

/** Add service to list of services of a device
 * @param dev     Device to operate on
 * @param type    'r' for RFCOMM 'l' for L2CAP
 * @param name    Name of the service (eg. "DUN")
 * @param channel RFCOMM channel the service is on
 * @returns The newly created BtService
 */
static BtService *add_service(BdInfo *dev, uint8_t type, const char *name,
                              uint8_t channel, uint32_t ttl, const char *dscr)
{
    BtService *service;

    debug("%s: adding %s, channel %u, ttl %d, \"%s\"", dev->bda, name, channel, ttl, dscr);

    service = g_new0(BtService, 1);
    service->bda     = dev->bda;
    service->type    = type;
    service->expiry  = calculate_expiry_time(ttl);
    service->name    = g_strdup(name);
    service->channel = channel;
    service->dscr    = g_strdup(dscr);

    dev->services = g_slist_append(dev->services, service);

    bt_service_ref(service);

    return service;
}

static BtService *bt_get_service(BdInfo *dev, const char *name, long int ch)
{
    GSList *l;

    if (ch < 0 || ch > 255)
        return NULL;

    for (l = dev->services; l != NULL; l = l->next) {
        BtService *svc = l->data;

        if (STREQ(svc->name, name) && (uint8_t)ch == svc->channel)
            return svc;
    }

    return NULL;
}

static BtService *bt_get_service_ch(BdInfo *dev, long int ch, gboolean create)
{
    GSList *match;

    if (ch < 0 || ch > 255)
        return NULL;

    match = g_slist_find_custom(dev->services, GINT_TO_POINTER(ch),
                                (GCompareFunc)cmp_service_chan);

    if (match)
        return match->data;

    if (create) {
        char svc[4];
        snprintf(svc, sizeof(svc), "%d", (int)ch);
        return add_service(dev, 'r', svc, ch, 0, "");
    }

    return NULL;
}

static BtService *get_rfcomm_service_by_name(BdInfo *dev, const char *name)
{
    BtService *svc = NULL;

    if (isdigit(name[0]) && strlen(name) < 4)
        svc = bt_get_service_ch(dev, strtol(name, NULL, 0), TRUE);
    else {
        GSList *match;
        match = g_slist_find_custom(dev->services, name,
                (GCompareFunc)cmp_service_name);
        if (match)
            svc = match->data;
    }

    if (svc && svc->type != 'r')
        svc = NULL;

    return svc;
}

static void _reset_capability(BdInfo *dev)
{
    if (dev->services != NULL) {
        free_service_list(dev->services);
        dev->services = NULL;
        free_rfcomm_dev_list(dev->bound);
        dev->bound = NULL;
        free_rfcomm_dev_list(dev->connected);
        dev->connected = NULL;
        dev->need_update = TRUE;
        dev->auto_update = TRUE;
    }
}

static gboolean disconnect_cb(GIOChannel *io, GIOCondition cond, RfcommDev *node)
{
    debug("Service %s for device %s was disconnected",
            node->svc->name, node->bd->bda);
    node->bd->connected = g_slist_remove(node->bd->connected, node);
    free_rfcomm_dev(node, TRUE);
    return FALSE;
}

static void connect_cb(gint dev_id, GIOChannel *io, GError *err, ConnectData *data)
{
    GError *local_err = NULL;

    g_assert((dev_id < 0 && io == NULL) || (dev_id >= 0 && io != NULL));

    debug("connect_cb: (%s, %s): %s", data->dev->bda, data->svc->name,
          dev_id < 0 ? "connect failed" : "connected"); 

    /* Check if the calling D-BUS name is still present */
    if (data->owner &&
            !dbus_bus_name_has_owner(get_dbus_connection(), data->owner, NULL)) {
        error("Connection initiator (%s) exited", data->owner);
        if (dev_id >= 0) {
            rfcomm_disconnect(io, dev_id, NULL);
            dev_id = -1;
            g_set_error(&local_err, BT_COND_ERROR,
                    BT_COND_ERROR_FAILED,
                    "Connection initiator (%s) exited", data->owner);
            err = local_err;
        }
    }

    if (dev_id >= 0) {
        RfcommDev *node;

        node = g_new0(RfcommDev, 1);
        node->bd  = data->dev;
        node->svc = data->svc;
        bt_service_ref(node->svc);
        node->id  = dev_id;
        node->io  = io;
        if (data->owner) {
            node->owner = g_strdup(data->owner);
            add_name_listener(get_dbus_connection(), node->owner,
                    (name_cb)bt_owner_disconnect, node);
        }

        node->io_id = g_io_add_watch(io, G_IO_ERR | G_IO_HUP, (GIOFunc)disconnect_cb, node);

        data->dev->connected = g_slist_append(data->dev->connected, node);

        send_dbus_rfcomm_status(node->bd->bda, node->svc->name, "connected");
    }

    data->func(dev_id, err, data->data);

    if (data->data && data->free_data)
        data->free_data(data->data);

    if (local_err)
        g_error_free(local_err);

    bt_service_unref(data->svc);
    g_free(data->owner);
    g_free(data);
} 

static void _bt_disconnect(BdInfo *dev, gboolean clear_bd_info)
{
    if (connection_status(dev->bda) == CONN_STATUS_CONNECTED) {
        gchar *cmd;

        free_rfcomm_dev_list(dev->connected);
        dev->connected = NULL;

        free_rfcomm_dev_list(dev->bound);
        dev->bound = NULL;

        /* After this check we should be able to safely use system() */
        if (!bda_ok(dev->bda)) {
            error("Invalid BDA '%s'", dev->bda);
            return;
        }
        cmd = g_strdup_printf("%s dc %s", HCITOOL, dev->bda);
        (void) system((char *)cmd);
        g_free(cmd);
    }

    if (clear_bd_info)
        _reset_capability(dev);
}

void bt_disconnect(const gchar *bda, gboolean clear_bd_info)
{
    BdInfo *dev;

    /* If bda is NULL, disconnect all devices */
    if (bda == NULL) {
        GSList *list;
        for (list = devs; list != NULL; list = list->next) {
            dev = list->data;
            g_assert(dev != NULL);
            _bt_disconnect(dev, clear_bd_info);
        }
    }
    else {
        dev = get_dev(bda);

        /* This should mean the we have no devices seleted. No error. Just return. */
        if (dev == NULL)
            return;

        _bt_disconnect(dev, clear_bd_info);
    }
}

static RfcommDev *bind_service(BdInfo *dev, BtService *service, GError **err)
{
    int dev_id;
    RfcommDev *node;

    if (!rfcomm_bind(dev->bda, service->channel, &dev_id, err))
        return NULL;
    
    node = g_new0(RfcommDev, 1);
    node->bd  = dev;
    node->id  = dev_id;
    node->svc = service;
    bt_service_ref(service);

    dev->bound = g_slist_append(dev->bound, node);

    return node;
}

static void handle_btsdp_reply(DBusMessage *reply, BdInfo *dev)
{
    const char *bda;
    DBusMessageIter iter;

    debug("handle_btsdp_reply()");

    free_service_list(dev->services);
    dev->services = NULL;

    dbus_message_iter_init(reply, &iter);

    if (dbus_message_iter_get_arg_type(&iter) != DBUS_TYPE_STRING)
        return;

    dbus_message_iter_get_basic(&iter, &bda);

    if (dbus_message_iter_next(&iter)) {
        DBusMessageIter list_iter;

        if (dbus_message_iter_get_arg_type(&iter) != DBUS_TYPE_ARRAY)
            return;

        dbus_message_iter_recurse(&iter, &list_iter);

        for (dbus_message_iter_recurse(&iter, &list_iter);
                dbus_message_iter_get_arg_type(&list_iter) != DBUS_TYPE_INVALID;
                dbus_message_iter_next(&list_iter)) {
            uint32_t ttl;
            uint8_t channel, type;
            const char *service, *name;
            BtService *svc;
            DBusMessageIter svc_iter;

            dbus_message_iter_recurse(&list_iter, &svc_iter);

            if (!get_basic_args(&svc_iter,
                                DBUS_TYPE_BYTE, &type,
                                DBUS_TYPE_STRING, &service,
                                DBUS_TYPE_BYTE, &channel,
                                DBUS_TYPE_UINT32, &ttl,
                                DBUS_TYPE_STRING, &name,
                                DBUS_TYPE_INVALID))
                continue;

            svc = bt_get_service(dev, service, channel);
            if (svc)
                svc->expiry = calculate_expiry_time(ttl);
            else
                add_service(dev, type, service, channel, ttl, name);
        }
    }
}

static void process_update_queue(void)
{
    GSList *l;

    debug("process_update_queue()");

    for (l = devs; l != NULL; l = l->next) {
        BdInfo *dev;
        dev = l->data;
        if (dev->auto_update && dev->need_update) {
            _schedule_capability_update(dev, NULL, NULL, NULL, NULL);
            return;
        }
    }
}

static void btsdp_notify(DBusPendingCall *pending, BdInfo *dev)
{
    DBusMessage *reply;
    GError *err = NULL;
    DBusError derr;

    debug("btsdp_notify()");

    sdp_in_progress = FALSE;

    dev->pending_update = NULL;

    dbus_error_init(&derr);

    reply = dbus_pending_call_steal_reply(pending);
    if (reply == NULL) {
        debug("btsdp_reply_notify: reply timed out for %s", dev->bda);
        g_set_error(&err, BT_COND_ERROR,
                    BT_COND_ERROR_CONNECT_FAILED,
                    "Timeout while waiting for reply from btsdp");
        dev->auto_update = FALSE;
    }
    else if (dbus_set_error_from_message(&derr, reply)) {
        debug("Capability query to btsdp with BDA %s failed: %s",
                dev->bda, derr.name);
        g_set_error(&err, BT_COND_ERROR,
                BT_COND_ERROR_CONNECT_FAILED,
                "%s: %s", derr.name, derr.message);
        dbus_error_free(&derr);
        dev->auto_update = FALSE;
        dbus_message_unref(reply);
    }
    else {
        handle_btsdp_reply(reply, dev);
        dev->need_update = FALSE;
        dev->auto_update = TRUE;
        dbus_message_unref(reply);
    }

    if (dev->notify_list) {
        GSList *l;
        for (l = dev->notify_list; l != NULL; l = l->next) {
            PendingSdpInfo *info;
            info = l->data;
            g_assert(info->func != NULL);
            info->func(err, info->user_data);
        }

        /* Since dbus_pending_call_unref doesn't seem to call this */
        free_pending_info(dev);
    }

    g_clear_error(&err);

    /* This should call free_pending_info() */
    dbus_pending_call_unref(pending);

    g_assert(dev->notify_list == NULL);

    process_update_queue();
}

static DBusMessage *new_sdp_request(BdInfo *dev)
{
    DBusMessage *request;
    GSList *l;

    request = new_dbus_method_call(BTSDP_SERVICE,
                                   BTSDP_REQ_PATH,
                                   BTSDP_REQ_INTERFACE,
                                   BTSDP_GET_SERVICES_REQ);

    append_dbus_args(request,
                     DBUS_TYPE_STRING, &dev->bda,
                     DBUS_TYPE_INVALID);

    /* Some BT stacks (e.g. the Widcomm windows stack) don't support the public
     * browse group (for some reason), so we ask for each UUID separately */
    for (l = dev->wanted_svcs; l != NULL; l = l->next) {
        const char *svc = l->data;
        append_dbus_args(request,
                         DBUS_TYPE_STRING, &svc,
                         DBUS_TYPE_INVALID);
    }

    return request;
}

static void _schedule_capability_update(BdInfo *dev, btsdp_cb_t func, const char *svc,
                                        gpointer user_data, GFreeFunc free_data)
{
    DBusConnection *conn = get_dbus_connection();
    DBusPendingCall *pending;
    DBusMessage *request;

    if (svc && !g_slist_find_custom(dev->wanted_svcs, svc, (GCompareFunc)strcmp)) {
        dev->wanted_svcs = g_slist_append(dev->wanted_svcs, g_strdup(svc));

        /* Search for both NFTP and FTP when NFTP is given */
        if (g_str_equal(svc, "NFTP") && !g_slist_find_custom(dev->wanted_svcs, "FTP", (GCompareFunc)strcmp))
                dev->wanted_svcs = g_slist_append(dev->wanted_svcs, g_strdup("FTP"));
    }
                

    if (func) {
        PendingSdpInfo *info;
        info = g_new0(PendingSdpInfo, 1);
        info->func      = func;
        info->user_data = user_data;
        info->free_data = free_data;
        dev->notify_list = g_slist_append(dev->notify_list, info);
    }

    dev->need_update = TRUE;

    if (sdp_in_progress)
        return;

    debug("Sending SDP query to %s", dev->bda);

    request = new_sdp_request(dev);
    if (!dbus_connection_send_with_reply(conn, request, &pending, BTSDP_TIMEOUT))
        die("Out of memory during dbus_connection_send_with_reply()");
    dbus_connection_flush(conn);
    dbus_message_unref(request);

    g_slist_foreach(dev->wanted_svcs, (GFunc)g_free, NULL);
    g_slist_free(dev->wanted_svcs);
    dev->wanted_svcs = NULL;

    dev->pending_update = pending;

    sdp_in_progress = TRUE;

    if (!dbus_pending_call_set_notify(pending,
                                      (DBusPendingCallNotifyFunction)btsdp_notify,
                                      dev, (DBusFreeFunction)free_pending_info))
        die("Out of memory during dbus_pending_call_set_notify()");
}

gboolean schedule_capability_update(const gchar *bda, btsdp_cb_t func, const char *svc,
                                    gpointer user_data, GFreeFunc free_data,
                                    GError **err)
{
    BdInfo *dev;

    debug("schedule_capability_update(%s, %s)", bda ? bda : "[def]", svc ? svc : "[all]");

    if (bt_disabled()) {
        g_set_error(err, BT_COND_ERROR,
                    BT_COND_ERROR_BT_DISABLED,
                    "Bluetooth is disabled");
        return FALSE;
    }

    dev = get_dev(bda);

    if (dev == NULL) {
        g_set_error(err, BT_COND_ERROR,
                    BT_COND_ERROR_INVALID_DEV,
                    "Invalid BT address");
        return FALSE;
    }

    /* Allow automatic updates */
    dev->auto_update = TRUE;

    _schedule_capability_update(dev, func, svc, user_data, free_data);

    return TRUE;
}

gboolean connect_service(const char *bda, const char *name, const char *owner,
                         gboolean auth, gboolean encr, const char *role,
                         btcon_cb_t func, gpointer user_data, GFreeFunc free_data,
                         GError **err)
{
    ConnectData *data;
    BtService *svc;
    BdInfo *bd;
    RfcommDev *dev;
    rfcomm_params params;

    debug("connect_service(\"%s\", \"%s\", \"%s\", %s, %s, \"%s\")",
            bda ? bda : "", name, owner ? owner : "",
            auth ? "auth" : "noauth", encr ? "encrypt" : "noencrypt",
            role ? role : "any");

    if (bt_disabled()) {
        g_set_error(err, BT_COND_ERROR,
                    BT_COND_ERROR_BT_DISABLED,
                    "Bluetooth is disabled");
        return FALSE;
    }

    bd = get_dev(bda);

    if (bd == NULL) {
        g_set_error(err, BT_COND_ERROR,
                    BT_COND_ERROR_INVALID_DEV,
                    "No matching BT device found");
        return FALSE;
    }

    svc = get_rfcomm_service_by_name(bd, name);
    if (svc == NULL) {
        g_set_error(err, BT_COND_ERROR,
                    BT_COND_ERROR_INVALID_SVC,
                    "%s does not have service %s", bd->bda, name);
        return FALSE;
    }

    dev = find_rfcomm_dev_ch(bd->connected, svc->channel);
    if (dev != NULL) {
        g_set_error(err, BT_COND_ERROR,
                    BT_COND_ERROR_CONNECTED,
                    "BDA %s, channel %u is already connected at rfcomm%d",
                    bda == NULL ? "" : bda, svc->channel, dev->id);
        return FALSE;

    }

    data = g_new0(ConnectData, 1);
    data->svc       = svc;
    bt_service_ref(svc);
    data->channel   = svc->channel;
    data->dev       = bd;
    if (owner)
        data->owner = g_strdup(owner);
    data->func      = func;
    data->data      = user_data;
    data->free_data = free_data;

    memset(&params, 0, sizeof(params));

    params.auth    = auth;
    params.encrypt = encr;

    if (role) {
        if (STREQ(role, "slave"))
            params.role    = ROLE_SLAVE;
        else if (STREQ(role, "master"))
            params.role    = ROLE_MASTER;
        else
            params.role    = ROLE_ANY;
    }
    else
        params.role = ROLE_ANY;
    
    if (STREQ(name, "SAP"))
        params.min_klen = SAP_MIN_KLEN;

    if (!rfcomm_connect(svc->bda, svc->channel, &params,
                        (rfcomm_connect_cb)connect_cb, data, err)) {
        bt_service_unref(data->svc);
        g_free(data->owner);
        g_free(data);
        return FALSE;
    }

    return TRUE;
}

gboolean disconnect_service(const char *bda, const char *name, GError **err)
{
    RfcommDev *dev;
    BdInfo *bd;

    debug("disconnect_service(\"%s\", \"%s\")", bda ? bda : "", name);

    bd = get_dev(bda);

    if (bd == NULL) {
        g_set_error(err, BT_COND_ERROR,
                    BT_COND_ERROR_INVALID_DEV,
                    "No matching BT device found");
        return FALSE;
    }

    dev = find_rfcomm_dev_name(bd->connected, name);
    if (dev == NULL) {
        g_set_error(err, BT_COND_ERROR,
                    BT_COND_ERROR_NOT_CONNECTED,
                    "Service is not connected");
        return FALSE;
    }

    bd->connected = g_slist_remove(bd->connected, dev);
    free_rfcomm_dev(dev, TRUE);

    return TRUE;
}

gboolean release_dev(const char *bda, const char *name, GError **err)
{
    RfcommDev *dev;
    BtService *svc;
    BdInfo *bd;

    debug("release_dev(\"%s\", \"%s\")", bda ? bda : "", name);

    bd = get_dev(bda);

    if (bd == NULL) {
        g_set_error(err, BT_COND_ERROR,
                    BT_COND_ERROR_INVALID_DEV,
                    "No matching BT device found");
        return FALSE;
    }

    svc = get_rfcomm_service_by_name(bd, name);
    if (svc == NULL) {
        g_set_error(err, BT_COND_ERROR,
                    BT_COND_ERROR_INVALID_SVC,
                    "%s does not have service %s", bd->bda, name);
        return FALSE;
    }

    dev = find_rfcomm_dev(bd->bound, svc->name, svc->channel);
    if (dev == NULL) {
        g_set_error(err, BT_COND_ERROR,
                    BT_COND_ERROR_NOT_BOUND,
                    "No matching RFCOMM device found");
        return FALSE;
    }

    if (!rfcomm_release(dev->id, err))
        return FALSE;

    bd->bound = g_slist_remove(bd->bound, dev);
    free_rfcomm_dev(dev, FALSE);

    return TRUE;
}

int bind_dev(const char *bda, const char *name, GError **err)
{
    RfcommDev *dev;
    BtService *svc;
    BdInfo *bd;

    debug("bind_dev(\"%s\", \"%s\")", bda ? bda : "", name);

    bd = get_dev(bda);

    if (bd == NULL) {
        g_set_error(err, BT_COND_ERROR,
                    BT_COND_ERROR_INVALID_DEV,
                    "No matching BT device found");
        return -1;
    }

    svc = get_rfcomm_service_by_name(bd, name);
    if (svc == NULL) {
        g_set_error(err, BT_COND_ERROR,
                    BT_COND_ERROR_INVALID_SVC,
                    "%s does not have service %s", bd->bda, name);
        return -1;
    }

    dev = find_rfcomm_dev(bd->bound, svc->name, svc->channel);
    if (dev) {
        g_set_error(err, BT_COND_ERROR,
                    BT_COND_ERROR_BOUND,
                    "Service is already bound at /dev/rfcomm%d", dev->id);
        return -1;
    }

    dev = bind_service(bd, svc, err);
    if (dev == NULL)
        return -1;

    return dev->id;
}


GSList *get_service_name_list(const gchar *bda, GError **err)
{
    GSList *svc_list, *l;
    BtService *svc;
    BdInfo *dev;

    dev = get_dev(bda);

    if (dev == NULL) {
        g_set_error(err, BT_COND_ERROR,
                    BT_COND_ERROR_INVALID_DEV,
                    "No matching BT device found");
        return NULL;
    }

    svc_list = NULL;
    for (l = dev->services; l != NULL; l = l->next) {
        svc = (BtService *)(l->data);
        svc_list = g_slist_append(svc_list, svc->name);
    }

    return svc_list;
}

void bt_hw_error(uint8_t code)
{
    error("Bluetooth HW error: 0x%02x", code);
    bt_disconnect(NULL, TRUE);
    hci_bt_down();
    hci_bt_up();
}

void expire_device_capability(const gchar *bda)
{
    BdInfo *dev;

    dev = get_dev(bda);

    if (dev == NULL)
        return;

    _expire_dev_capability(dev);
}

gboolean need_capability_update(const gchar *bda, const gchar *name)
{
    BtService *svc;
    BdInfo *dev;

    dev = get_dev(bda);

    if (!dev)
        return TRUE;

    svc = get_rfcomm_service_by_name(dev, name);
    if (!svc)
        return TRUE;

    if (time(NULL) < svc->expiry)
        return FALSE;

    /* Remove expired service */
    dev->services = g_slist_remove(dev->services, svc);
    bt_service_unref(svc);

    return TRUE;
}

GSList *get_bda_list(void)
{
    GSList *l, *bda_list = NULL;
    BdInfo *dev;

    for (l = devs; l != NULL; l = l->next) {
        dev = l->data;
        bda_list = g_slist_append(bda_list, dev->bda);
    }

    return bda_list;
}

const gchar *bt_service_status(const char *bda, const char *name, GError **err)
{
    BdInfo *dev;

    dev = get_dev(bda);

    if (dev == NULL) {
        g_set_error(err, BT_COND_ERROR,
                    BT_COND_ERROR_INVALID_DEV,
                    "Invalid BT address");
        return NULL;
    }

    if (find_rfcomm_dev_name(dev->connected, name) != NULL)
        return "connected";
    else
        return "disconnected";
}

BtService *get_bound_service(uint8_t dev_id)
{
    GSList *l;

    for (l = devs; l != NULL; l = l->next) {
        BdInfo *bd;
        RfcommDev *node;

        g_assert(l->data != NULL);
        bd = l->data;

        node = find_rfcomm_dev_id(bd->bound, dev_id);
        if (node)
            return node->svc;
    }

    return NULL;
}

BtService *get_service(const char *bda, const char *name, GError **err)
{
    BdInfo *dev;
    BtService *svc;

    dev = get_dev(bda);

    if (dev == NULL) {
        g_set_error(err, BT_COND_ERROR,
                    BT_COND_ERROR_INVALID_DEV,
                    "Invalid BT address");
        return NULL;
    }

    svc = get_rfcomm_service_by_name(dev, name);
    if (svc == NULL)
        g_set_error(err, BT_COND_ERROR,
                    BT_COND_ERROR_INVALID_SVC,
                    "No such \"%s\" service on %s", name, bda);

    return svc;
}

BtService *get_connected_service(uint8_t dev_id)
{
    GSList *l;

    for (l = devs; l != NULL; l = l->next) {
        BdInfo *bd;
        RfcommDev *node;

        g_assert(l->data != NULL);
        bd = l->data;

        node = find_rfcomm_dev_id(bd->connected, dev_id);
        if (node)
            return node->svc;
    }

    return NULL;
}

char *bt_devname_from_id(gint dev_num, char *dev, int max_len)
{
    snprintf(dev, max_len, "/dev/rfcomm%d", dev_num);
    return dev;
}

