/* -*- Mode: C; c-basic-offset: 2; indent-tabs-mode: nil -*-
 *
 * Pigment image example
 *
 * Copyright © 2006, 2007, 2008 Fluendo Embedded S.L.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author: Loïc Molinari <loic@fluendo.com>
 */

#include <pgm/pgm.h>
#include <gdk-pixbuf/gdk-pixbuf.h>

static gboolean
drawable_pressed_cb (PgmDrawable *drawable,
                     gfloat x,
                     gfloat y,
                     gfloat z,
                     PgmButtonType button,
                     guint32 time,
                     gpointer data)
{
  g_print ("%s \"pressed\"\n", GST_OBJECT_NAME (drawable));

  return FALSE;
}

static gboolean
drawable_released_cb (PgmDrawable *drawable,
                      PgmButtonType button,
                      guint32 time,
                      gpointer data)
{
  g_print ("%s \"released\"\n", GST_OBJECT_NAME (drawable));

  return FALSE;
}

static gboolean
drawable_clicked_cb (PgmDrawable *drawable,
                     gfloat x,
                     gfloat y,
                     gfloat z,
                     PgmButtonType button,
                     guint32 time,
                     gpointer data)
{
  g_print ("%s \"clicked\"\n", GST_OBJECT_NAME (drawable));

  return FALSE;
}

static gboolean
drawable_double_clicked_cb (PgmDrawable *drawable,
                            gfloat x,
                            gfloat y,
                            gfloat z,
                            PgmButtonType button,
                            guint32 time,
                            gpointer data)
{
  g_print ("%s \"double-clicked\"\n", GST_OBJECT_NAME (drawable));

  return FALSE;
}

static gboolean
drawable_scrolled_cb (PgmDrawable *drawable,
                      gfloat x,
                      gfloat y,
                      gfloat z,
                      PgmScrollDirection direction,
                      guint32 time,
                      gpointer data)
{
  g_print ("%s \"scrolled %s\"\n", GST_OBJECT_NAME (drawable),
           direction == PGM_SCROLL_DOWN ? "down" : "up");

  return TRUE;
}

static gboolean
drawable_drag_begin_cb (PgmDrawable *drawable,
                        gfloat x,
                        gfloat y,
                        gfloat z,
                        PgmButtonType button,
                        guint32 time,
                        gpointer data)
{
  g_print ("%s \"drag-begin\", (%.2f, %.2f, %.2f), %d, %u\n",
           GST_OBJECT_NAME (drawable), x, y, z, button, time);

  return TRUE;
}

static gboolean
drawable_drag_motion_cb (PgmDrawable *drawable,
                         gfloat x,
                         gfloat y,
                         gfloat z,
                         PgmButtonType button,
                         guint32 time,
                         gpointer data)
{
  g_print ("%s \"drag-motion\", (%.2f, %.2f, %.2f), %d, %u\n",
           GST_OBJECT_NAME (drawable), x, y, z, button, time);

  return FALSE;
}

static gboolean
drawable_drag_end_cb (PgmDrawable *drawable,
                      gfloat x,
                      gfloat y,
                      gfloat z,
                      PgmButtonType button,
                      guint32 time,
                      gpointer data)
{
  g_print ("%s \"drag-end\", (%.2f, %.2f, %.2f), %d, %u\n",
           GST_OBJECT_NAME (drawable), x, y, z, button, time);

  return TRUE;
}

static void
keypress_event_cb (PgmViewport *viewport,
                   PgmEventKey *event,
                   gpointer data)
{
  static gboolean fullscreen = FALSE;

  if (event->type == PGM_KEY_PRESS)
    {
      switch (event->keyval)
        {
          /* Fullscreen */
        case PGM_f:
          fullscreen = !fullscreen;
          pgm_viewport_set_fullscreen (viewport, fullscreen);
          break;

          /* Quit */
        case PGM_q:
        case PGM_Escape:
          pgm_main_quit ();
          break;

        default:
          break;
        }
    }
}

int
main (int argc,
      char *argv[])
{
  const gchar *img_path = "examples/pictures/fluendo.png";
  PgmViewport *viewport = NULL;
  PgmCanvas *canvas;
  PgmDrawable *img, *img2, *img3;
  GdkPixbuf *icon;

  /* Init */
  pgm_init (&argc, &argv);

  /* OpenGL viewport creation */
  pgm_viewport_factory_make ("opengl", &viewport);
  if (!viewport)
    {
      g_print ("Cannot create the 'opengl' viewport\n");
      return -1;
    }

  pgm_viewport_set_title (viewport, "Image");

  /* Icon definition */
  icon = gdk_pixbuf_new_from_file (img_path, NULL);
  if (icon)
    {
      pgm_viewport_set_icon (viewport, icon);
      g_object_unref (icon);
    }
  else
    g_print ("Couldn't load icon\n");

  /* First image */
  img = pgm_image_new_from_file (img_path, 0);
  pgm_drawable_set_size (img, 1.0f, 1.0f);
  pgm_drawable_set_position (img, 1.25f, 0.75f, -50.0f);
  pgm_drawable_set_bg_color (img, 150, 150, 255, 150);
  pgm_drawable_show (img);
  g_signal_connect (G_OBJECT (img), "pressed",
                    G_CALLBACK (drawable_pressed_cb), NULL);
  g_signal_connect (G_OBJECT (img), "released",
                    G_CALLBACK (drawable_released_cb), NULL);
  g_signal_connect (G_OBJECT (img), "clicked",
                    G_CALLBACK (drawable_clicked_cb), NULL);
  g_signal_connect (G_OBJECT (img), "double-clicked",
                    G_CALLBACK (drawable_double_clicked_cb), NULL);
  g_signal_connect (G_OBJECT (img), "scrolled",
                    G_CALLBACK (drawable_scrolled_cb), NULL);
  g_signal_connect (G_OBJECT (img), "drag-begin",
                    G_CALLBACK (drawable_drag_begin_cb), NULL);
  g_signal_connect (G_OBJECT (img), "drag-motion",
                    G_CALLBACK (drawable_drag_motion_cb), NULL);
  g_signal_connect (G_OBJECT (img), "drag-end",
                    G_CALLBACK (drawable_drag_end_cb), NULL);

  /* Second image */
  img2 = pgm_image_new_from_image (PGM_IMAGE (img));
  pgm_drawable_set_size (img2, 1.0f, 1.0f);
  pgm_drawable_set_position (img2, 1.5f, 1.0f, 0.0f);
  pgm_drawable_set_bg_color (img2, 255, 150, 150, 150);
  pgm_drawable_show (img2);
  g_signal_connect (G_OBJECT (img2), "pressed",
                    G_CALLBACK (drawable_pressed_cb), NULL);
  g_signal_connect (G_OBJECT (img2), "released",
                    G_CALLBACK (drawable_released_cb), NULL);
  g_signal_connect (G_OBJECT (img2), "clicked",
                    G_CALLBACK (drawable_clicked_cb), NULL);
  g_signal_connect (G_OBJECT (img2), "double-clicked",
                    G_CALLBACK (drawable_double_clicked_cb), NULL);
  g_signal_connect (G_OBJECT (img2), "scrolled",
                    G_CALLBACK (drawable_scrolled_cb), NULL);
  g_signal_connect (G_OBJECT (img2), "drag-begin",
                    G_CALLBACK (drawable_drag_begin_cb), NULL);
  g_signal_connect (G_OBJECT (img2), "drag-motion",
                    G_CALLBACK (drawable_drag_motion_cb), NULL);
  g_signal_connect (G_OBJECT (img2), "drag-end",
                    G_CALLBACK (drawable_drag_end_cb), NULL);

  /* Third image */
  img3 = pgm_image_new_from_image (PGM_IMAGE (img));
  pgm_drawable_set_size (img3, 1.0f, 1.0f);
  pgm_drawable_set_position (img3, 1.75f, 1.25f, 50.0f);
  pgm_drawable_set_bg_color (img3, 150, 255, 150, 150);
  pgm_drawable_show (img3);
  g_signal_connect (G_OBJECT (img3), "pressed",
                    G_CALLBACK (drawable_pressed_cb), NULL);
  g_signal_connect (G_OBJECT (img3), "released",
                    G_CALLBACK (drawable_released_cb), NULL);
  g_signal_connect (G_OBJECT (img3), "clicked",
                    G_CALLBACK (drawable_clicked_cb), NULL);
  g_signal_connect (G_OBJECT (img3), "double-clicked",
                    G_CALLBACK (drawable_double_clicked_cb), NULL);
  g_signal_connect (G_OBJECT (img3), "scrolled",
                    G_CALLBACK (drawable_scrolled_cb), NULL);
  g_signal_connect (G_OBJECT (img3), "drag-begin",
                    G_CALLBACK (drawable_drag_begin_cb), NULL);
  g_signal_connect (G_OBJECT (img3), "drag-motion",
                    G_CALLBACK (drawable_drag_motion_cb), NULL);
  g_signal_connect (G_OBJECT (img3), "drag-end",
                    G_CALLBACK (drawable_drag_end_cb), NULL);

  /* Canvas handling */
  canvas = pgm_canvas_new ();
  pgm_viewport_set_canvas (viewport, canvas);
  pgm_canvas_add_many (canvas, PGM_DRAWABLE_MIDDLE, img, img2, img3, NULL);

  /* Main loop */
  g_signal_connect (G_OBJECT (viewport), "key-press-event",
                    G_CALLBACK (keypress_event_cb), NULL);
  g_signal_connect (G_OBJECT (viewport), "delete-event",
                    G_CALLBACK (pgm_main_quit), NULL);
  pgm_viewport_show (viewport);
  pgm_main ();

  /* Deinit */
  gst_object_unref (canvas);
  gst_object_unref (viewport);
  pgm_deinit ();

  return 0;
}
