/* -*- Mode: C; c-basic-offset: 2; indent-tabs-mode: nil -*-
 *
 * Pigment OpenGL plugin
 *
 * Copyright © 2006, 2007, 2008 Fluendo Embedded S.L.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author: Loïc Molinari <loic@fluendo.com>
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

#include "pgmgldrawable.h"

GST_DEBUG_CATEGORY_STATIC (pgm_gl_drawable_debug);
#define GST_CAT_DEFAULT pgm_gl_drawable_debug

static GstObjectClass *parent_class = NULL;

/* Private methods */

/* Update the position of the duplicated GL drawable. Since these values are
   accessed quite often, storing it in the GL drawable avoids putting locks
   on the generic drawable. */
static void
update_position (PgmGlDrawable *gldrawable)
{
  PgmDrawable *drawable = gldrawable->drawable;

  GST_OBJECT_LOCK (drawable);
  gldrawable->x = drawable->x;
  gldrawable->y = drawable->y;
  gldrawable->z = drawable->z;
  GST_OBJECT_UNLOCK (drawable);
}

/* Update the size of the duplicated GL drawable. Since these values are
   accessed quite often, storing it in the GL drawable avoids putting locks
   on the generic drawable. */
static void
update_size (PgmGlDrawable *gldrawable)
{
  PgmDrawable *drawable = gldrawable->drawable;

  GST_OBJECT_LOCK (drawable);
  gldrawable->width = drawable->width;
  gldrawable->height = drawable->height;
  GST_OBJECT_UNLOCK (drawable);
}

/* GObject stuff */

PGM_DEFINE_DYNAMIC_TYPE_EXTENDED (PgmGlDrawable, pgm_gl_drawable,
                                  GST_TYPE_OBJECT, G_TYPE_FLAG_ABSTRACT, {});

void
pgm_gl_drawable_register (GTypeModule *module)
{
  pgm_gl_drawable_register_type (module);
}


static void
pgm_gl_drawable_dispose (GObject *object)
{
  PgmGlDrawable *gldrawable = PGM_GL_DRAWABLE (object);

  GST_DEBUG_OBJECT (gldrawable, "dispose");

  gldrawable->drawable = NULL;
  gldrawable->glviewport = NULL;

  GST_CALL_PARENT (G_OBJECT_CLASS, dispose, (object));
}

static void
pgm_gl_drawable_class_init (PgmGlDrawableClass *klass)
{
  GObjectClass *gobject_class;

  GST_DEBUG_CATEGORY_INIT (pgm_gl_drawable_debug, "pgm_gl_drawable", 0,
                           "OpenGL plugin: PgmGlDrawable");

  parent_class = g_type_class_peek_parent (klass);

  gobject_class = G_OBJECT_CLASS (klass);

  /* GObject virtual table */
  gobject_class->dispose = GST_DEBUG_FUNCPTR (pgm_gl_drawable_dispose);
}

static void
pgm_gl_drawable_class_finalize (PgmGlDrawableClass *klass)
{
  return;
}

static void
pgm_gl_drawable_init (PgmGlDrawable *gldrawable)
{
  GST_DEBUG_OBJECT (gldrawable, "init");

  gldrawable->drawable = NULL;
  gldrawable->glviewport = NULL;
}

/* Public methods */

void
pgm_gl_drawable_sync (PgmGlDrawable *gldrawable)
{
  PgmGlDrawableClass *klass;

  g_return_if_fail (PGM_IS_GL_DRAWABLE (gldrawable));

  update_size (gldrawable);
  pgm_gl_drawable_set_position (gldrawable);
  pgm_gl_drawable_set_bg_color (gldrawable);

  klass = PGM_GL_DRAWABLE_GET_CLASS (gldrawable);

  if (klass->sync)
    klass->sync (gldrawable);
}

void
pgm_gl_drawable_draw (PgmGlDrawable *gldrawable)
{
  PgmGlDrawableClass *klass;
  PgmDrawable *drawable;
  PgmContextProcAddress *gl;
  gboolean visible;

  g_return_if_fail (PGM_IS_GL_DRAWABLE (gldrawable));
  g_return_if_fail (PGM_IS_DRAWABLE (gldrawable->drawable));

  GST_LOG_OBJECT (gldrawable, "draw");

  drawable = gldrawable->drawable;

  /* Don't draw if not visible, or if the opacity is 0 */
  GST_OBJECT_LOCK (drawable);
  visible = (GST_OBJECT_FLAG_IS_SET (drawable, PGM_DRAWABLE_FLAG_VISIBLE));
  if (visible)
    visible = drawable->opacity;
  GST_OBJECT_UNLOCK (drawable);
  if (!visible)
    return;

  gl = gldrawable->glviewport->context->gl;

  /* Batch background array if it's visible */
  if (gldrawable->bg_color[3] != 0.0f)
    {
      gl->bind_texture (PGM_GL_TEXTURE_2D, 0);
      gl->enable_client_state (PGM_GL_VERTEX_ARRAY);
      gl->vertex_pointer (3, PGM_GL_FLOAT, 0, gldrawable->bg_vertex);
      gl->color_4fv (gldrawable->bg_color);
      gl->draw_arrays (PGM_GL_QUADS, 0, 4);
      gl->disable_client_state (PGM_GL_VERTEX_ARRAY);
    }

  klass = PGM_GL_DRAWABLE_GET_CLASS (gldrawable);

  if (klass && klass->draw)
    klass->draw (gldrawable);
}

void
pgm_gl_drawable_regenerate (PgmGlDrawable *gldrawable)
{
  PgmGlDrawableClass *klass;

  g_return_if_fail (PGM_IS_GL_DRAWABLE (gldrawable));

  klass = PGM_GL_DRAWABLE_GET_CLASS (gldrawable);

  if (klass->regenerate)
    klass->regenerate (gldrawable);
}

void
pgm_gl_drawable_update_projection (PgmGlDrawable *gldrawable)
{
  PgmGlDrawableClass *klass;

  g_return_if_fail (PGM_IS_GL_DRAWABLE (gldrawable));

  pgm_gl_drawable_set_position (gldrawable);
  pgm_gl_drawable_set_size (gldrawable);

  klass = PGM_GL_DRAWABLE_GET_CLASS (gldrawable);

  if (klass->update_projection)
    klass->update_projection (gldrawable);
}

void
pgm_gl_drawable_set_visibility (PgmGlDrawable *gldrawable)
{
  PgmGlDrawableClass *klass;

  g_return_if_fail (PGM_IS_GL_DRAWABLE (gldrawable));

  GST_DEBUG_OBJECT (gldrawable, "set_visibility");

  klass = PGM_GL_DRAWABLE_GET_CLASS (gldrawable);

  if (klass->set_visibility)
    klass->set_visibility (gldrawable);
}

void
pgm_gl_drawable_set_size (PgmGlDrawable *gldrawable)
{
  PgmGlDrawableClass *klass;

  g_return_if_fail (PGM_IS_GL_DRAWABLE (gldrawable));

  GST_DEBUG_OBJECT (gldrawable, "set_size");

  update_size (gldrawable);

  gldrawable->bg_vertex[3] = gldrawable->bg_vertex[0] + gldrawable->width;
  gldrawable->bg_vertex[6] = gldrawable->bg_vertex[9] + gldrawable->width;
  gldrawable->bg_vertex[10] = gldrawable->bg_vertex[1] + gldrawable->height;
  gldrawable->bg_vertex[7] = gldrawable->bg_vertex[4] + gldrawable->height;

  klass = PGM_GL_DRAWABLE_GET_CLASS (gldrawable);

  if (klass->set_size)
    klass->set_size (gldrawable);
}

void
pgm_gl_drawable_set_position (PgmGlDrawable *gldrawable)
{
  PgmGlDrawableClass *klass;

  g_return_if_fail (PGM_IS_GL_DRAWABLE (gldrawable));

  GST_DEBUG_OBJECT (gldrawable, "set_position");

  update_position (gldrawable);

  gldrawable->bg_vertex[0] = gldrawable->x;
  gldrawable->bg_vertex[1] = gldrawable->y;
  gldrawable->bg_vertex[2] = gldrawable->z;
  gldrawable->bg_vertex[3] = gldrawable->x + gldrawable->width;
  gldrawable->bg_vertex[4] = gldrawable->y;
  gldrawable->bg_vertex[5] = gldrawable->z;
  gldrawable->bg_vertex[6] = gldrawable->x + gldrawable->width;
  gldrawable->bg_vertex[7] = gldrawable->y + gldrawable->height;
  gldrawable->bg_vertex[8] = gldrawable->z;
  gldrawable->bg_vertex[9] = gldrawable->x;
  gldrawable->bg_vertex[10] = gldrawable->y + gldrawable->height;
  gldrawable->bg_vertex[11] = gldrawable->z;

  klass = PGM_GL_DRAWABLE_GET_CLASS (gldrawable);

  if (klass->set_position)
    klass->set_position (gldrawable);
}

void
pgm_gl_drawable_set_fg_color (PgmGlDrawable *gldrawable)
{
  PgmGlDrawableClass *klass;

  g_return_if_fail (PGM_IS_GL_DRAWABLE (gldrawable));

  GST_DEBUG_OBJECT (gldrawable, "set_fg_color");

  klass = PGM_GL_DRAWABLE_GET_CLASS (gldrawable);

  if (klass->set_fg_color)
    klass->set_fg_color (gldrawable);

}

void
pgm_gl_drawable_set_bg_color (PgmGlDrawable *gldrawable)
{
  PgmGlDrawableClass *klass;
  PgmDrawable *drawable;

  g_return_if_fail (PGM_IS_GL_DRAWABLE (gldrawable));

  GST_DEBUG_OBJECT (gldrawable, "set_bg_color");

  drawable = gldrawable->drawable;

  GST_OBJECT_LOCK (drawable);
  gldrawable->bg_color[0] = drawable->bg_r * INV_255;
  gldrawable->bg_color[1] = drawable->bg_g * INV_255;
  gldrawable->bg_color[2] = drawable->bg_b * INV_255;
  gldrawable->bg_color[3] = drawable->bg_a * drawable->opacity * SQR_INV_255;
  GST_OBJECT_UNLOCK (drawable);

  klass = PGM_GL_DRAWABLE_GET_CLASS (gldrawable);

  if (klass->set_bg_color)
    klass->set_bg_color (gldrawable);
}

void
pgm_gl_drawable_set_opacity (PgmGlDrawable *gldrawable)
{
  PgmGlDrawableClass *klass;
  PgmDrawable *drawable;

  g_return_if_fail (PGM_IS_GL_DRAWABLE (gldrawable));

  klass = PGM_GL_DRAWABLE_GET_CLASS (gldrawable);

  GST_DEBUG_OBJECT (gldrawable, "set_opacity");

  drawable = gldrawable->drawable;

  GST_OBJECT_LOCK (drawable);
  gldrawable->bg_color[3] = drawable->bg_a * drawable->opacity * SQR_INV_255;
  GST_OBJECT_UNLOCK (drawable);

  if (klass->set_opacity)
    klass->set_opacity (gldrawable);
}
