/* -*- Mode: C; c-basic-offset: 2; indent-tabs-mode: nil -*-
 *
 * Pigment OpenGL plugin
 *
 * Copyright © 2006, 2007, 2008 Fluendo Embedded S.L.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author: Loïc Molinari <loic@fluendo.com>
 */

/*
 * OpenGL plugin backend for Xlib/GLX.
 *
 * It follows several specifications from the X Desktop Group including:
 *
 * o the Extended Window Manager Hints
 * o the XEmbed Protocol Specification
 * o the Drag-and-Drop Protocol for the X Window System
 * o the Startup Notification Protocol
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

#include <string.h>
#include <X11/Xatom.h>
#include <X11/keysym.h>
#include <X11/cursorfont.h>
#include <X11/extensions/Xrender.h>
#include <gdk-pixbuf-xlib/gdk-pixbuf-xlib.h>
#include "pgmglxbackend.h"
#include "pgmglviewport.h"

GST_DEBUG_CATEGORY_STATIC (pgm_gl_glxbackend_debug);
#define GST_CAT_DEFAULT pgm_gl_glxbackend_debug

/* Supported XDND version */
#define XDND_PROTOCOL_VERSION 5

/* Supported XEmbed version */
#define XEMBED_PROTOCOL_VERSION 1

/* Flag(s) for _XEMBED_INFO */
#define XEMBED_MAPPED (1 << 0)

/* XEmbed messages */
typedef enum {
  XEMBED_EMBEDDED_NOTIFY   = 0,
  XEMBED_WINDOW_ACTIVATE   = 1,
  XEMBED_WINDOW_DEACTIVATE = 2,
  XEMBED_REQUEST_FOCUS     = 3,
  XEMBED_FOCUS_IN          = 4,
  XEMBED_FOCUS_OUT         = 5,
  XEMBED_FOCUS_NEXT        = 6,
  XEMBED_FOCUS_PREV        = 7,
  XEMBED_GRAB_KEY          = 8, /* Removed from the spec */
  XEMBED_UNGRAB_KEY        = 9, /* Removed from the spec */
  XEMBED_MODALITY_ON       = 10,
  XEMBED_MODALITY_OFF      = 11,
} XEmbedMessageType;

/* Motif function flags */
typedef enum {
  MWM_FUNC_ALL      = (1L << 0),
  MWM_FUNC_RESIZE   = (1L << 1),
  MWM_FUNC_MOVE     = (1L << 2),
  MWM_FUNC_MINIMIZE = (1L << 3),
  MWM_FUNC_MAXIMIZE = (1L << 4),
  MWM_FUNC_CLOSE    = (1L << 5)
} MwmFunctionFlags;

/* Motif hint flags */
typedef enum {
  MWM_HINTS_DECORATIONS = (1L << 1),
  MWM_HINTS_FUNCTIONS   = (1L << 0)
} MwmHintFlags;

/* Prototypes */
static gboolean event_prepare  (GSource *source,
                                gint *timeout);
static gboolean event_check    (GSource *source);
static gboolean event_dispatch (GSource *source,
                                GSourceFunc callback,
                                gpointer data);

/* Event handling functions */
static GSourceFuncs event_funcs = {
  event_prepare, event_check, event_dispatch, NULL
};

/* Motif window management hints */
typedef struct {
  gulong flags;
  gulong functions;
  glong  decorations;
  glong  input_mode;
  gulong status;
} MwmHints;

/* Map GLX extension names and feature bits */
typedef struct {
  const gchar *name;
  gint         feature_mask;
} ExtensionMap;

/* GLX extensions checked at runtime */
static ExtensionMap glx_extensions_map[] = {
  { "GLX_SGIX_fbconfig",    PGM_GLX_FEAT_FBCONFIG     },
  { "GLX_SGI_video_sync",   PGM_GLX_FEAT_VIDEO_SYNC   },
  { "GLX_SGI_swap_control", PGM_GLX_FEAT_SWAP_CONTROL },
  { NULL,                   0                         }
};

/* GLX functions binding */
static PgmGlxBackendProcAddress glx_proc_address = {
  (PgmGlxChooseFbconfigFunc)        NULL,
  (PgmGlxGetFbconfigsFunc)          NULL,
  (PgmGlxGetFbconfigAttribFunc)     NULL,
  (PgmGlxGetVisualFromFbconfigFunc) NULL,
  (PgmGlxCreateNewContextFunc)      NULL,
  (PgmGlxGetVideoSyncFunc)          NULL,
  (PgmGlxWaitVideoSyncFunc)         NULL,
  (PgmGlxSwapIntervalFunc)          NULL
};

/* X atom index IDs */
typedef enum {
  ATOM_NET_WM_STATE_FULLSCREEN = 0,
  ATOM_NET_WM_STATE            = 1,
  ATOM_NET_WM_PING             = 2,
  ATOM_NET_WM_USER_TIME        = 3,
  ATOM_NET_WM_NAME             = 4,
  ATOM_NET_ACTIVE_WINDOW       = 5,
  ATOM_NET_STARTUP_INFO        = 6,
  ATOM_NET_STARTUP_INFO_BEGIN  = 7,
  ATOM_NET_STARTUP_ID          = 8,
  ATOM_XEMBED                  = 9,
  ATOM_XEMBED_INFO             = 10,
  ATOM_MOTIF_WM_HINTS          = 11,
  ATOM_UTF8_STRING             = 12,
  ATOM_WM_PROTOCOLS            = 13,
  ATOM_WM_DELETE_WINDOW        = 14,
  ATOM_PGM_SELECTION           = 15,
  ATOM_XDND_AWARE              = 16,
  ATOM_XDND_ENTER              = 17,
  ATOM_XDND_POSITION           = 18,
  ATOM_XDND_DROP               = 19,
  ATOM_XDND_LEAVE              = 20,
  ATOM_XDND_TYPE_LIST          = 21,
  ATOM_XDND_SELECTION          = 22,
  ATOM_XDND_STATUS             = 23,
  ATOM_XDND_FINISHED           = 24,
  ATOM_XDND_ACTION_COPY        = 25,
  ATOM_TEXT_URI_LIST           = 26
} XAtomIndexId;

/* X atom names */
static const gchar *atom_name[] = {
  "_NET_WM_STATE_FULLSCREEN", /* 0 */
  "_NET_WM_STATE",            /* 1 */
  "_NET_WM_PING",             /* 2 */
  "_NET_WM_USER_TIME",        /* 3 */
  "_NET_WM_NAME",             /* 4 */
  "_NET_ACTIVE_WINDOW",       /* 5 */
  "_NET_STARTUP_INFO",        /* 6 */
  "_NET_STARTUP_INFO_BEGIN",  /* 7 */
  "_NET_STARTUP_ID",          /* 8 */
  "_XEMBED",                  /* 9 */
  "_XEMBED_INFO",             /* 10 */
  "_MOTIF_WM_HINTS",          /* 11 */
  "UTF8_STRING",              /* 12 */
  "WM_PROTOCOLS",             /* 13 */
  "WM_DELETE_WINDOW",         /* 14 */
  "PGM_SELECTION",            /* 15 */
  "XdndAware",                /* 16 */
  "XdndEnter",                /* 17 */
  "XdndPosition",             /* 18 */
  "XdndDrop",                 /* 19 */
  "XdndLeave",                /* 20 */
  "XdndTypeList",             /* 21 */
  "XdndSelection",            /* 22 */
  "XdndStatus",               /* 23 */
  "XdndFinished",             /* 24 */
  "XdndActionCopy",           /* 25 */
  "text/uri-list"             /* 26 */
};

/* Number of atoms, atom strings and index IDs must be kept synchronised! */
static const guint nb_atoms = G_N_ELEMENTS (atom_name);

static PgmBackendClass *parent_class = NULL;

static int (*old_error_handler) (Display*, XErrorEvent*);
static int trapped_error_code = 0;

/* Private functions */

/* The X error handler */
static int
error_handler (Display *display,
               XErrorEvent *error)
{
  trapped_error_code = error->error_code;

  return 0;
}

/* Push X errors */
static void
trap_x_errors (void)
{
  trapped_error_code = 0;
  old_error_handler = XSetErrorHandler (error_handler);
}

/* Pop X errors */
static gint
untrap_x_errors (void)
{
  XSetErrorHandler (old_error_handler);

  return trapped_error_code;
}

/* Retrieve a function pointer given a procedure name */
static inline gpointer
get_proc_address (PgmGlxBackend *glxbackend,
                  const gchar *proc_name)
{
  return glXGetProcAddressARB ((const guchar *) proc_name);
}

/* Check whether an extension is supported by the GLX implementation given
 * the extension name and the list of supported extensions */
static gboolean
has_glx_extension (const gchar *extensions,
                   const gchar *extension_name)
{
  gchar *end;
  gint ext_name_len = strlen (extension_name);
  gchar *p = (gchar*) extensions;
  size_t size;

  if (!extensions)
    return FALSE;

  end = p + strlen (p);
  while (p < end)
    {
      size = strcspn (p, " ");
      if ((ext_name_len == size)  && (strncmp (extension_name, p, size) == 0))
        return TRUE;
      p += size + 1;
    }

  return FALSE;
}

/* Bind the GLX extension proc addresses depending on the features available
 * at run-time and the GLX version. */
static void
bind_glx_extensions (PgmGlxBackend *glxbackend)
{
  PgmGlxBackendProcAddress *glx = glxbackend->glx;

  /* FBConfig */
  if (glxbackend->feature_mask & PGM_GLX_FEAT_FBCONFIG)
    {
      if (glxbackend->version >= 1.3f)
        {
          glx->choose_fbconfig = (PgmGlxChooseFbconfigFunc)
            get_proc_address (glxbackend, "glXChooseFBConfig");
          glx->get_fbconfigs = (PgmGlxGetFbconfigsFunc)
            get_proc_address (glxbackend, "glXGetFBConfigs");
          glx->get_fbconfig_attrib = (PgmGlxGetFbconfigAttribFunc)
            get_proc_address (glxbackend, "glXGetFBConfigAttrib");
          glx->get_visual_from_fbconfig = (PgmGlxGetVisualFromFbconfigFunc)
            get_proc_address (glxbackend, "glXGetVisualFromFBConfig");
          glx->create_new_context = (PgmGlxCreateNewContextFunc)
            get_proc_address (glxbackend, "glXCreateNewContext");
        }
      else
        {
          glx->choose_fbconfig = (PgmGlxChooseFbconfigFunc)
            get_proc_address (glxbackend, "glXChooseFBConfigSGIX");
          glx->get_fbconfigs = (PgmGlxGetFbconfigsFunc)
            get_proc_address (glxbackend, "glXGetFBConfigsSGIX");
          glx->get_fbconfig_attrib = (PgmGlxGetFbconfigAttribFunc)
            get_proc_address (glxbackend, "glXGetFBConfigAttribSGIX");
          glx->get_visual_from_fbconfig = (PgmGlxGetVisualFromFbconfigFunc)
            get_proc_address (glxbackend, "glXGetVisualFromFBConfigSGIX");
          glx->create_new_context = (PgmGlxCreateNewContextFunc)
            get_proc_address (glxbackend, "glXCreateContextWithConfigSGIX");
        }

      if ((!glx->choose_fbconfig) || (!glx->get_fbconfigs)
          || (!glx->get_fbconfig_attrib) || (!glx->get_visual_from_fbconfig)
          || (!glx->create_new_context))
        {
          glxbackend->feature_mask &= ~PGM_GLX_FEAT_FBCONFIG;
        }
    }

  /* Video sync */
  if (glxbackend->feature_mask & PGM_GLX_FEAT_VIDEO_SYNC)
    {
      glx->get_video_sync = (PgmGlxGetVideoSyncFunc)
        get_proc_address (glxbackend, "glXGetVideoSyncSGI");
      glx->wait_video_sync = (PgmGlxWaitVideoSyncFunc)
        get_proc_address (glxbackend, "glXWaitVideoSyncSGI");

      if ((!glx->get_video_sync) || (!glx->wait_video_sync))
        glxbackend->feature_mask &= ~PGM_GLX_FEAT_VIDEO_SYNC;
    }

  /* Swap control */
  if (glxbackend->feature_mask & PGM_GLX_FEAT_SWAP_CONTROL)
    {
      glx->swap_interval = (PgmGlxSwapIntervalFunc)
        get_proc_address (glxbackend, "glXSwapIntervalSGI");

      if (!glx->swap_interval)
        glxbackend->feature_mask &= ~PGM_GLX_FEAT_SWAP_CONTROL;
    }
}

/* Get the GLX informations and load the extensions */
static gboolean
load_glx_extensions (PgmGlxBackend *glxbackend)
{
  gint error_base, event_base;
  gint major, minor;
  gint i = 0;

  /* Are GLX extensions supported? */
  if (!glXQueryExtension (glxbackend->dpy, &error_base, &event_base))
    {
      GST_ERROR_OBJECT (glxbackend, "GLX extensions not supported");
      return FALSE;
    }

  /* Get the GLX server version */
  if (!glXQueryVersion (glxbackend->dpy, &major, &minor))
    {
      GST_ERROR_OBJECT (glxbackend, "couldn't query GLX version");
      return FALSE;
    }

  /* Check minimum GLX version */
  glxbackend->version = major + minor / 10.0f;
  if (major < 1 || (major == 1 && minor < 2))
    {
      GST_ERROR_OBJECT (glxbackend, "GLX version %.1f (1.2 minimum required)",
                        glxbackend->version);
      return FALSE;
    }

  glxbackend->vendor = (const gchar *)
    glXGetClientString (glxbackend->dpy, GLX_VENDOR);

  /* ATI's driver emulates GLX 1.3 support */
  if (glxbackend->vendor)
    if (glxbackend->version < 1.3f)
      if (!strncmp ("ATI", glxbackend->vendor, 3))
        glxbackend->version = 1.3f;

  /* Retrieve and bind supported extensions */
  glxbackend->extensions = (const gchar *)
    glXQueryExtensionsString (glxbackend->dpy, glxbackend->screen);

  glxbackend->glx = &glx_proc_address;

  while (glx_extensions_map[i].name)
    {
      if (has_glx_extension (glxbackend->extensions,
                             glx_extensions_map[i].name))
        glxbackend->feature_mask |= glx_extensions_map[i].feature_mask;
      i++;
    }

  bind_glx_extensions (glxbackend);

  return TRUE;
}

/* Retrieve the best visual supported by the OpenGL implementation that
 * matches our requirements. An ARGB visual could be searched on demand with
 * the "PGM_GL_ARGB_VISUAL" environment variable. This is sad but ARGB
 * visuals are not correctly supported by the DRI (the issue will be
 * fixed with redirected direct rendering and DRI2) and with the NVIDIA
 * proprietary driver there are vertical synchronization issues. */
static XVisualInfo*
get_visual_info (PgmGlxBackend *glxbackend)
{
  PgmGlxBackendProcAddress *glx = glxbackend->glx;
  XVisualInfo *visual = NULL;
  gboolean use_argb_visual = FALSE;
  const gchar *argb_visual;

  /* Check for PGM_GL_ARGB_VISUAL in the environment */
  argb_visual = g_getenv ("PGM_GL_ARGB_VISUAL");
  if (argb_visual && *argb_visual != '0')
    use_argb_visual = TRUE;

  /* 'GLX_SGIX_fbconfig' is available, let's use FBConfigs */
  if (glxbackend->feature_mask & PGM_GLX_FEAT_FBCONFIG)
    {
      gint render_event_base, render_error_base, nbconfigs;
      GLXFBConfig *fbconfig = NULL;

      gint attrib[] = {
        GLX_DOUBLEBUFFER, True, GLX_RED_SIZE, 1, GLX_GREEN_SIZE, 1,
        GLX_BLUE_SIZE, 1, GLX_ALPHA_SIZE, 1, GLX_DEPTH_SIZE, 0,
        GLX_RENDER_TYPE, GLX_RGBA_BIT, GLX_DRAWABLE_TYPE, GLX_WINDOW_BIT,
        None };

      GST_DEBUG_OBJECT (glxbackend, "'GLX_SGIX_fbconfig' is available");

      /* Get the FBConfigs available on the system */
      fbconfig = glx->choose_fbconfig (glxbackend->dpy, glxbackend->screen,
                                       attrib, &nbconfigs);
      if (!fbconfig)
        {
          GST_ERROR_OBJECT (glxbackend, "couldn't retrieve the fbconfigs");
          goto error;
        }
      GST_DEBUG_OBJECT (glxbackend, "retrieved %d fbconfigs", nbconfigs);

      /* The user requested an ARGB visual and we have the RENDER extension,
       * sounds like good conditions to search for such a visual */
      if (use_argb_visual && XRenderQueryExtension (glxbackend->dpy,
                                                    &render_event_base,
                                                    &render_error_base))
        {
          XRenderPictFormat *format = NULL;
          guint i;

          /* Search for an ARGB visual */
          for (i = 0; i < nbconfigs; i++)
            {
              visual = glx->get_visual_from_fbconfig (glxbackend->dpy,
                                                      fbconfig[i]);
              if (!visual)
                continue;

              format = XRenderFindVisualFormat (glxbackend->dpy,
                                                visual->visual);
              if (!format)
                {
                  XFree (visual);
                  continue;
                }

              if (format->direct.alphaMask > 0)
                {
                  GST_INFO_OBJECT (glxbackend, "found an ARGB visual", visual);
                  glxbackend->feature_mask |= PGM_GLX_FEAT_ARGB_VISUAL;
                  break;
                }

              XFree (visual);
            }

          /* If there's no ARGB visual, we use the first visual retrieved */
          if (i == nbconfigs)
            {
              GST_DEBUG_OBJECT (glxbackend, "couldn't find an ARGB visual");

              visual = glx->get_visual_from_fbconfig (glxbackend->dpy,
                                                      fbconfig[0]);
              if (visual)
                GST_INFO_OBJECT (glxbackend, "found a RGB visual");
              else
                GST_ERROR_OBJECT (glxbackend, "couldn't find any visual");
            }
        }
      else
        {
          visual = glx->get_visual_from_fbconfig (glxbackend->dpy, fbconfig[0]);
          if (visual)
            GST_INFO_OBJECT (glxbackend, "found a RGB visual");
          else
            GST_ERROR_OBJECT (glxbackend, "couldn't find any visual");
        }

      XFree (fbconfig);
    }

  /* 'GLX_SGIX_fbconfig' not available, let the implementation choose a RGB
   * visual for us */
  else
    {
      gint attrib[] = {
        GLX_DOUBLEBUFFER, 1, GLX_RED_SIZE, 1, GLX_GREEN_SIZE, 1,
        GLX_BLUE_SIZE, 1, GLX_DEPTH_SIZE, 0, GLX_RGBA, None };

      GST_DEBUG_OBJECT (glxbackend, "'GLX_SGIX_fbconfig' is not available");

      visual = glXChooseVisual (glxbackend->dpy, glxbackend->screen, attrib);
      if (visual)
        GST_INFO_OBJECT (glxbackend, "found a RGB visual");
      else
        GST_ERROR_OBJECT (glxbackend, "couldn't find any visual");
    }

 error:
  return visual;
}

/* Setup the method used for the VBlank syncing */
static void
setup_vblank (PgmGlxBackend *glxbackend)
{
  const gchar *variable;

  /* NVIDIA proprietary driver has an environment variable to automatically
   * enable VBlank syncing, in the case this variable is defined we just
   * disable it on Pigment side. */
  if (g_getenv ("__GL_SYNC_TO_VBLANK"))
    {
      GST_INFO_OBJECT (glxbackend, "__GL_SYNC_TO_VBLANK defined in the "
                       "environment, disabling vblank on pigment side");
      glxbackend->vblank_mode = PGM_VBLANK_NONE;

      return;
    }

  /* The PGM_GL_VBLANK environment variable overrides the automatic VBlank
   * method choice done by Pigment. */
  variable = g_getenv ("PGM_GL_VBLANK");
  if (variable)
    {
      /* VBlank disabled */
      if (variable[0] == '0')
        {
          GST_INFO_OBJECT (glxbackend, "PGM_GL_VBLANK disables vblank syncing");
          glxbackend->vblank_mode = PGM_VBLANK_NONE;
        }

      /* VBlank using the GLX_SGI_video_sync extension */
      else if (variable[0] == '1')
        {
          if (glxbackend->feature_mask & PGM_GLX_FEAT_VIDEO_SYNC)
            {
              GST_INFO_OBJECT (glxbackend, "PGM_GL_VBLANK forces use of the "
                               "video_sync extension for vblank syncing");
              glxbackend->vblank_mode = PGM_VBLANK_VIDEO_SYNC;
            }
          else
            GST_WARNING_OBJECT (glxbackend, "PGM_GL_VBLANK mode not supported "
                                "by the GLX implementation");
        }

      /* VBlank using the GLX_SGI_swap_control extension */
      else if (variable[0] == '2')
        {
          if (glxbackend->feature_mask & PGM_GLX_FEAT_SWAP_CONTROL)
            {
              GST_INFO_OBJECT (glxbackend, "PGM_GL_VBLANK forces use of the "
                               "swap_control extension for vblank syncing");
              glxbackend->vblank_mode = PGM_VBLANK_SWAP_CONTROL;
              glxbackend->glx->swap_interval (1);
            }
          else
            GST_WARNING_OBJECT (glxbackend, "PGM_GL_VBLANK mode not supported "
                                "by the GLX implementation");
        }

      return;
    }

  /* With the NVIDIA proprietary driver, with the GeForce 8000 GPUs, the
   * video_sync extension works much better than for the previous one, that's
   * good since the swap_control one takes all the CPU in fullscreen mode.
   * Let's choose that extension for this class of GPUs. */
  if (glxbackend->feature_mask & PGM_GLX_FEAT_VIDEO_SYNC)
    {
      pgm_gl_get_string get_string = get_proc_address
        (glxbackend, "glGetString");
      const gchar *renderer = (const gchar*) get_string (GL_RENDERER);
      if (!strncmp ("GeForce 8", renderer, 9))
        {
          GST_DEBUG_OBJECT (glxbackend, "Using the video_sync extension for "
                            "vblank syncing");
          glxbackend->vblank_mode = PGM_VBLANK_VIDEO_SYNC;
          return;
        }
    }

  /* Use the swap_control extension by default */
  if (glxbackend->feature_mask & PGM_GLX_FEAT_SWAP_CONTROL)
    {
      GST_INFO_OBJECT (glxbackend, "Using the swap_control extension for "
                        "vblank syncing");
      glxbackend->vblank_mode = PGM_VBLANK_SWAP_CONTROL;
      glxbackend->glx->swap_interval (1);
      return;
    }

  /* Use the video_sync extension if swap_control's not available */
  if (glxbackend->feature_mask & PGM_GLX_FEAT_VIDEO_SYNC)
    {
      GST_INFO_OBJECT (glxbackend, "Using the video_sync extension for "
                        "vblank syncing");
      glxbackend->vblank_mode = PGM_VBLANK_VIDEO_SYNC;
      return;
    }

  /* Fallback to ... nothing */
  GST_WARNING_OBJECT (glxbackend, "No extensions found to enable vblank "
                      "syncing");
  glxbackend->vblank_mode = PGM_VBLANK_NONE;
}

/* Specifies the WM_PROTOCOLS property of the window */
static void
set_wm_protocols (PgmGlxBackend *glxbackend)
{
  Atom protocols[2];

  protocols[0] = glxbackend->atom[ATOM_WM_DELETE_WINDOW];
  protocols[1] = glxbackend->atom[ATOM_NET_WM_PING];

  XSetWMProtocols (glxbackend->dpy, glxbackend->win, protocols, 2);
}

/* Updates the _NET_WM_USER_TIME property of the window */
static void
set_wm_user_time (PgmGlxBackend *glxbackend,
                  glong timestamp)
{
  if (timestamp)
    XChangeProperty (glxbackend->dpy, glxbackend->win,
                     glxbackend->atom[ATOM_NET_WM_USER_TIME], XA_CARDINAL, 32,
                     PropModeReplace, (guchar *) &timestamp, 1);
}

/* Handles the protocols event and returns an event to push if any */
static PgmEvent *
handle_wm_protocols_event (PgmGlxBackend *glxbackend,
                           XEvent *xevent)
{
  PgmEvent *pgmevent = NULL;

  /* Delete request */
  if (xevent->xclient.data.l[0] == glxbackend->atom[ATOM_WM_DELETE_WINDOW]
      && xevent->xany.window == glxbackend->win)
    {
      pgmevent = pgm_event_new (PGM_DELETE);

      ((PgmEventDelete *) pgmevent)->time = xevent->xclient.data.l[1];
      set_wm_user_time (glxbackend, xevent->xclient.data.l[1]);
    }

  /* Ping request */
  else if (xevent->xclient.data.l[0] == glxbackend->atom[ATOM_NET_WM_PING]
           && xevent->xany.window == glxbackend->win)
    {
      xevent->xclient.window = glxbackend->root;

      XSendEvent (glxbackend->dpy, glxbackend->root, False,
                  SubstructureRedirectMask | SubstructureNotifyMask,
                  (XEvent *) &xevent->xclient);
    }

  return pgmevent;
}

/* Setup XEmbed informations */
static void
set_xembed_info (PgmGlxBackend *glxbackend,
                 guint32 xembed_info)
{
  gint32 info[2];

  info[0] = XEMBED_PROTOCOL_VERSION;
  info[1] = xembed_info;

  trap_x_errors ();
  XChangeProperty (glxbackend->dpy, glxbackend->win,
                   glxbackend->atom[ATOM_XEMBED_INFO],
                   glxbackend->atom[ATOM_XEMBED_INFO],
                   32, PropModeReplace, (guchar*) info, 2);
  untrap_x_errors ();
}

/* Send an XEmbed message */
static void
send_xembed_message (PgmGlxBackend *glxbackend,
                     glong message,
                     glong detail,
                     glong data1,
                     glong data2)
{
  XClientMessageEvent xclient;

  memset (&xclient, 0, sizeof (xclient));
  xclient.window = glxbackend->embedder;
  xclient.type = ClientMessage;
  xclient.message_type = glxbackend->atom[ATOM_XEMBED];
  xclient.format = 32;
  xclient.data.l[0] = CurrentTime;
  xclient.data.l[1] = message;
  xclient.data.l[2] = detail;
  xclient.data.l[3] = data1;
  xclient.data.l[4] = data2;

  trap_x_errors ();
  XSendEvent (glxbackend->dpy, glxbackend->embedder, False, NoEventMask,
              (XEvent*) &xclient);
  XSync (glxbackend->dpy, False);
  untrap_x_errors ();
}

/* Handle an XEmbed event */
static PgmEvent*
handle_xembed_event (PgmGlxBackend *glxbackend,
                     XEvent *xevent)
{
  PgmEvent *pgmevent = NULL;

  switch (xevent->xclient.data.l[1])
    {
      /* The viewport has just been embedded, let's store the embedded Window
       * and create an Expose event */
    case XEMBED_EMBEDDED_NOTIFY:
      glxbackend->embedder = xevent->xclient.data.l[3];
      set_xembed_info (glxbackend, XEMBED_MAPPED);
      pgmevent = pgm_event_new (PGM_EXPOSE);
      break;

      /* Unused */
    case XEMBED_WINDOW_ACTIVATE:
    case XEMBED_WINDOW_DEACTIVATE:
    case XEMBED_REQUEST_FOCUS:
    case XEMBED_FOCUS_IN:
    case XEMBED_FOCUS_OUT:
    case XEMBED_FOCUS_NEXT:
    case XEMBED_FOCUS_PREV:
    case XEMBED_MODALITY_ON:
    case XEMBED_MODALITY_OFF:
      break;

    default:
      break;
    }

  return pgmevent;
}

/* Retrieve the drag uri list from the selection */
static gchar**
get_xdnd_uri_list (PgmGlxBackend *glxbackend,
                   XEvent *xevent)
{
  guchar *data = NULL;
  gchar **uri = NULL;
  gulong nitems, nbytes;
  gint format;
  Atom type;

  XGetWindowProperty (glxbackend->dpy, xevent->xselection.requestor,
                      xevent->xselection.property, 0, 0x1fffffff, False,
                      AnyPropertyType, &type, &format, &nitems, &nbytes, &data);

  uri = g_uri_list_extract_uris ((gchar*) data);
  XFree (data);

  GST_DEBUG_OBJECT (glxbackend, "drag uri list retrieved");

  return uri;
}

/* Clean up xdnd data */
static void
cleanup_xdnd_data (PgmGlxBackend *glxbackend)
{
  glxbackend->drag_data_has_uri = FALSE;
  glxbackend->drag_data_received = FALSE;
  glxbackend->drag_status = FALSE;
  glxbackend->drag_source = None;
  glxbackend->drag_x = -1.0f;
  glxbackend->drag_y = -1.0f;

  if (glxbackend->drag_uri)
    {
      g_strfreev (glxbackend->drag_uri);
      glxbackend->drag_uri = NULL;
    }
}

/* Send an Xdnd message */
static void
send_xdnd_message (PgmGlxBackend *glxbackend,
                   Window dest_window,
                   Atom message,
                   glong data1,
                   glong data2,
                   glong data3,
                   glong data4)
{
  XEvent xevent;

  /* Build it */
  memset (&xevent, 0, sizeof (XEvent));
  xevent.xclient.type = ClientMessage;
  xevent.xclient.message_type = message;
  xevent.xclient.format = 32;
  xevent.xclient.window = dest_window;
  xevent.xclient.data.l[0] = glxbackend->win;
  xevent.xclient.data.l[1] = data1;
  xevent.xclient.data.l[2] = data2;
  xevent.xclient.data.l[3] = data3;
  xevent.xclient.data.l[4] = data4;

  /* Then send it */
  trap_x_errors ();
  XSendEvent (glxbackend->dpy, dest_window, False, NoEventMask,
              (XEvent*) &xevent);
  XSync (glxbackend->dpy, False);
  untrap_x_errors ();
}

/* Indicate the drag source if we accept or refuse a drag */
static void
update_xdnd_status (PgmGlxBackend *glxbackend)
{
  /* Accept and request the next XdndPosition */
  if (glxbackend->drag_status)
    {
      GST_DEBUG_OBJECT (glxbackend, "sending message accepting xdnd drag");
      send_xdnd_message (glxbackend, glxbackend->drag_source,
                         glxbackend->atom[ATOM_XDND_STATUS], 1 | 2, 0, 0,
                         glxbackend->atom[ATOM_XDND_ACTION_COPY]);
    }
  /* Refuse but request next XdndPosition message */
  else
    {
      GST_DEBUG_OBJECT (glxbackend, "sending message refusing xdnd drag");
      send_xdnd_message (glxbackend, glxbackend->drag_source,
                         glxbackend->atom[ATOM_XDND_STATUS], 2, 0, 0, None);
    }
}

/* Handle an XdndEnter event */
static void
handle_xdnd_enter_event (PgmGlxBackend *glxbackend,
                         XEvent *xevent)
{
  const gchar *uri_type = "text/uri-list";
  guint uri_type_length = 13;
  gboolean found = FALSE;
  gchar *data_type;
  guint i;

  GST_DEBUG_OBJECT (glxbackend, "initiated xdnd protocol %d with window 0x%x",
                    (xevent->xclient.data.l[1] & 0xff000000) >> 24,
                    xevent->xclient.data.l[0]);

  /* Search for the "text/uri-list" data type in the three available one */
  for (i = 2; i < 5; i++)
    {
      if (xevent->xclient.data.l[i] == None)
        continue;

      data_type = XGetAtomName (glxbackend->dpy, xevent->xclient.data.l[i]);
      if (!strncmp (data_type, uri_type, uri_type_length))
        {
          found = TRUE;
          XFree (data_type);
          break;
        }
      XFree (data_type);
    }

  /* If we've not found the data type, we check if the source has more than
   * three data types, then retrieve them and trigger a new search */
  if (!found && (xevent->xclient.data.l[1] & 1))
    {
      gulong nitems, after;
      Atom type, *atoms;
      guchar *data;
      gint format;
      guint i;

      /* Get the atoms */
      XGetWindowProperty (glxbackend->dpy, xevent->xclient.data.l[0],
                          glxbackend->atom[ATOM_XDND_TYPE_LIST], 0, 65536, False,
                          XA_ATOM, &type, &format, &nitems, &after, &data);
      XSync (glxbackend->dpy, False);
      atoms = (Atom*) data;

      /* Search for the good one */
      for (i = 0; i < nitems; i++)
        {
          if (xevent->xclient.data.l[i] == None)
            continue;

          data_type = XGetAtomName (glxbackend->dpy, atoms[i]);
          if (!strncmp (data_type, uri_type, uri_type_length))
            {
              found = TRUE;
              XFree (data_type);
              break;
            }
          XFree (data_type);
        }

      XFree (atoms);
    }

  /* If the uri-list data type is available, we request the data and set up
   * the drag session correctly */
  if (found)
    {
      GST_DEBUG_OBJECT (glxbackend, "\"text/uri-list\" drag type available, "
                        "requesting drag data");

      XConvertSelection (glxbackend->dpy,
                         glxbackend->atom[ATOM_XDND_SELECTION],
                         glxbackend->atom[ATOM_TEXT_URI_LIST],
                         glxbackend->atom[ATOM_PGM_SELECTION],
                         glxbackend->win, CurrentTime);

      glxbackend->drag_source = xevent->xclient.data.l[0];
      glxbackend->drag_data_has_uri = TRUE;
      glxbackend->drag_data_received = FALSE;
    }
  else
    {
      GST_DEBUG_OBJECT (glxbackend, "\"text/uri-list\" drag type unavailable");
      glxbackend->drag_data_has_uri = FALSE;
    }
}

/* Handle an XdndPosition event */
static PgmEvent*
handle_xdnd_position_event (PgmGlxBackend *glxbackend,
                            XEvent *xevent)
{
  PgmEventDnd *pgmevent = NULL;

  /* We handle this dragging */
  if (glxbackend->drag_data_has_uri)
    {
      /* If we already received the uri list from the selection */
      if (glxbackend->drag_data_received)
        {
          gint src_x, src_y, dst_x, dst_y;
          Window child;

          /* Translate the mouse position in window coordinates */
          src_x = xevent->xclient.data.l[2] >> 16;
          src_y = xevent->xclient.data.l[2] & 0xffff;
          XTranslateCoordinates (glxbackend->dpy, glxbackend->root,
                                 glxbackend->win, src_x, src_y, &dst_x, &dst_y,
                                 &child);

          /* Store position so that we have it to handle the drop */
          glxbackend->drag_x = (gfloat) dst_x;
          glxbackend->drag_y = (gfloat) dst_y;

          /* Then build a drag motion event */
          pgmevent = (PgmEventDnd*) pgm_event_new (PGM_DRAG_MOTION);
          pgmevent->time = xevent->xclient.data.l[3];
          pgmevent->uri = g_strdupv (glxbackend->drag_uri);
          pgmevent->x = (gfloat) dst_x;
          pgmevent->y = (gfloat) dst_y;
        }

      update_xdnd_status (glxbackend);
    }
  /* We don't handle this dragging, refuse the xdnd session */
  else
    {
      GST_DEBUG_OBJECT (glxbackend, "sending message refusing xdnd session");
      send_xdnd_message (glxbackend, glxbackend->drag_source,
                         glxbackend->atom[ATOM_XDND_STATUS], glxbackend->win,
                         0, None, None);
    }

  return (PgmEvent*) pgmevent;
}

/* Handle an XdndDrop event */
static PgmEvent*
handle_xdnd_drop_event (PgmGlxBackend *glxbackend,
                        XEvent *xevent)
{
  PgmEventDnd *pgmevent = NULL;

  if (glxbackend->drag_data_received && glxbackend->drag_status)
    {
      GST_DEBUG_OBJECT (glxbackend, "drop accepted, finishing xdnd protocol");

      pgmevent = (PgmEventDnd*) pgm_event_new (PGM_DRAG_DROP);
      pgmevent->time = xevent->xclient.data.l[1];
      pgmevent->uri = g_strdupv (glxbackend->drag_uri);
      pgmevent->x = glxbackend->drag_x;
      pgmevent->y = glxbackend->drag_y;

      send_xdnd_message (glxbackend, glxbackend->drag_source,
                         glxbackend->atom[ATOM_XDND_FINISHED], 1, None, None,
                         glxbackend->atom[ATOM_XDND_ACTION_COPY]);
    }
  else
    {
      GST_DEBUG_OBJECT (glxbackend, "drop refused, finishing xdnd protocol");
      send_xdnd_message (glxbackend, glxbackend->drag_source,
                         glxbackend->atom[ATOM_XDND_FINISHED], 0, None, None,
                         None);
    }

  cleanup_xdnd_data (glxbackend);

  return (PgmEvent*) pgmevent;
}

/* Handle an XdndLeave event */
static PgmEvent*
handle_xdnd_leave_event (PgmGlxBackend *glxbackend,
                         XEvent *xevent)
{
  PgmEventDnd *pgmevent = NULL;

  GST_DEBUG_OBJECT (glxbackend, "drag leave, finishing xdnd protocol");

  pgmevent = (PgmEventDnd*) pgm_event_new (PGM_DRAG_LEAVE);
  pgmevent->time = CurrentTime;
  pgmevent->uri = g_strdupv (glxbackend->drag_uri);
  pgmevent->x = glxbackend->drag_x;
  pgmevent->y = glxbackend->drag_y;

  send_xdnd_message (glxbackend, glxbackend->drag_source,
                     glxbackend->atom[ATOM_XDND_FINISHED], 0, None, None, None);
  cleanup_xdnd_data (glxbackend);

  return (PgmEvent*) pgmevent;
}

/* Escape string for X messages */
static gchar*
escape_for_x_message (const gchar *str)
{
  GString *retval;
  const char *p;

  retval = g_string_new (NULL);

  p = str;
  while (*p)
    {
      switch (*p)
        {
        case ' ':
        case '"':
        case '\\':
          g_string_append_c (retval, '\\');
          break;
        }

      g_string_append_c (retval, *p);
      ++p;
    }

  return g_string_free (retval, FALSE);
}

/* Initialize the startup notification protocol, filling up the startup id from
 * the DESKTOP_STARTUP_ID environment variable */
static void
init_startup_notification (PgmGlxBackend *glxbackend)
{
  const gchar *startup_id;

  startup_id = g_getenv ("DESKTOP_STARTUP_ID");

  if (startup_id && *startup_id != '\0')
    {
      GST_INFO_OBJECT (glxbackend, "DESKTOP_STARTUP_ID=\"%s\"", startup_id);

      if (!g_utf8_validate (startup_id, -1, NULL))
        GST_WARNING_OBJECT (glxbackend, "DESKTOP_STARTUP_ID contains "
                            "invalid UTF-8");
      else
        glxbackend->startup_notification_id = g_strdup (startup_id);

      /* Clear the environment variable so it won't be inherited by child
       * processes and confuse things */
      g_unsetenv ("DESKTOP_STARTUP_ID");

      /* Set the startup id on the leader window so it applies to all windows
       * we create on this display */
      XChangeProperty (glxbackend->dpy, glxbackend->win,
                       glxbackend->atom[ATOM_NET_STARTUP_ID],
                       glxbackend->atom[ATOM_UTF8_STRING], 8, PropModeReplace,
                       (guchar*) startup_id, strlen (startup_id));
    }
}

/* Creates and returns a void cursor from X */
static Cursor
get_none_cursor (PgmGlxBackend *backend)
{
  Cursor none = None;
  gchar none_bits[32];
  Pixmap none_pixmap;
  XColor color;

  memset (none_bits, 0, sizeof (none_bits));
  memset (&color, 0, sizeof (color));

  none_pixmap = XCreateBitmapFromData (backend->dpy, backend->win, none_bits,
                                       16, 16);
  if (none_pixmap != None)
    {
      none = XCreatePixmapCursor (backend->dpy, none_pixmap, none_pixmap,
                                  &color, &color, 0, 0);
      XFreePixmap (backend->dpy, none_pixmap);
    }

  return none;
}

/* Frees the current X icon */
static gboolean
free_icon (PgmBackend *backend)
{
  PgmGlxBackend *glxbackend = PGM_GLX_BACKEND (backend);
  XWMHints *old_wm_hints = NULL;

  old_wm_hints = XGetWMHints (glxbackend->dpy, glxbackend->win);

  if (old_wm_hints)
    {
      if ((old_wm_hints->flags & IconPixmapHint) && old_wm_hints->icon_pixmap)
        XFreePixmap (glxbackend->dpy, old_wm_hints->icon_pixmap);
      if ((old_wm_hints->flags & IconMaskHint) && old_wm_hints->icon_mask)
        XFreePixmap (glxbackend->dpy, old_wm_hints->icon_mask);
      XFree (old_wm_hints);
    }

  return TRUE;
}

/* Compresses events of the same type */
static void
compress_events (PgmGlxBackend *glxbackend,
                 XEvent *xevent)
{
  while (XCheckTypedWindowEvent (glxbackend->dpy, glxbackend->win,
                                 xevent->type, xevent));
}

/* Update the viewport size and the projection */
static void
update_viewport_size (PgmGlxBackend *glxbackend,
                      gint width,
                      gint height)
{
  PgmBackend *backend = PGM_BACKEND (glxbackend);
  PgmGlViewport *glviewport = backend->context->glviewport;
  PgmViewport *viewport = PGM_VIEWPORT (glviewport);

  GST_OBJECT_LOCK (viewport);

  if (viewport->width != width || viewport->height != height)
    {
      viewport->width = width;
      viewport->height = height;

      GST_OBJECT_UNLOCK (viewport);
      pgm_viewport_update_projection (viewport);

      return;
    }

  GST_OBJECT_UNLOCK (viewport);
}

/* Fill a PgmEventKey giving the corresponding keyboard xevent */
static void
translate_key_event (PgmGlxBackend *glxbackend,
                     PgmEventKey *event,
                     XEvent *xevent)
{
  gint index;

  event->time = xevent->xkey.time;
  event->modifier = (PgmModifierType) xevent->xkey.state;

  /* FIXME: This is a really basic support, a lot of keys are not handled */
  index = (event->modifier & (PGM_SHIFT_MASK | PGM_CAPSLOCK_MASK)) ? 1 : 0;
  event->keyval = XKeycodeToKeysym (glxbackend->dpy, xevent->xkey.keycode,
                                    index);

  event->hardware_keycode = xevent->xkey.keycode;
}

/* Fill a PgmEventMotion giving the corresponding motion xevent */
static void
translate_motion_event (PgmGlxBackend *glxbackend,
                        PgmEventMotion *event,
                        XEvent *xevent)
{
  event->time = xevent->xmotion.time;
  event->x = xevent->xmotion.x;
  event->y = xevent->xmotion.y;
}

/* Fill a PgmEventButton giving the corresponding button xevent */
static void
translate_button_press_event (PgmGlxBackend *glxbackend,
                              PgmEventButton *event,
                              XEvent *xevent)
{
  event->time = xevent->xbutton.time;
  event->x = xevent->xbutton.x;
  event->y = xevent->xbutton.y;
}

/* Fill a PgmEventScroll giving the corresponding button xevent */
static void
translate_scroll_event (PgmGlxBackend *glxbackend,
                        PgmEventScroll *event,
                        XEvent *xevent)
{
  event->time = xevent->xbutton.time;
  event->x = xevent->xbutton.x;
  event->y = xevent->xbutton.y;
}

/* Fill a PgmEventConfigure giving the corresponding configure xevent */
static void
translate_configure_event (PgmGlxBackend *glxbackend,
                           PgmEventConfigure *event,
                           XEvent *xevent)
{
  event->x = xevent->xconfigure.x;
  event->y = xevent->xconfigure.y;
  event->width = xevent->xconfigure.width;
  event->height = xevent->xconfigure.height;
}

/* Create a button event (Release) giving the corresponding button xevent */
static PgmEvent *
create_button_release_event (PgmGlxBackend *glxbackend,
                             XEvent *xevent)
{
  PgmEventButton *event;

  switch (xevent->xbutton.button)
    {
    case 1:
      event = (PgmEventButton *) pgm_event_new (PGM_BUTTON_RELEASE);
      event->button = PGM_BUTTON_LEFT;
      break;

    case 2:
      event = (PgmEventButton *) pgm_event_new (PGM_BUTTON_RELEASE);
      event->button = PGM_BUTTON_MIDDLE;
      break;

    case 3:
      event = (PgmEventButton *) pgm_event_new (PGM_BUTTON_RELEASE);
      event->button = PGM_BUTTON_RIGHT;
      break;

    /* No release */
    default:
      return NULL;
    }

  event->time = xevent->xbutton.time;
  event->x = xevent->xbutton.x;
  event->y = xevent->xbutton.y;

  return (PgmEvent *) event;
}

/* Create a button event (Scroll, Press) given an XEvent */
static PgmEvent *
create_button_event (PgmGlxBackend *glxbackend,
                     XEvent *xevent)
{
  PgmEvent *event;

  switch (xevent->xbutton.button)
    {
    case 1:
      event = pgm_event_new (PGM_BUTTON_PRESS);
      ((PgmEventButton *) event)->button = PGM_BUTTON_LEFT;
      translate_button_press_event (glxbackend, (PgmEventButton *) event,
                                    xevent);
      break;

    case 2:
      event = pgm_event_new (PGM_BUTTON_PRESS);
      ((PgmEventButton *) event)->button = PGM_BUTTON_MIDDLE;
      translate_button_press_event (glxbackend, (PgmEventButton *) event,
                                    xevent);
      break;

    case 3:
      event = pgm_event_new (PGM_BUTTON_PRESS);
      ((PgmEventButton *) event)->button = PGM_BUTTON_RIGHT;
      translate_button_press_event (glxbackend, (PgmEventButton *) event,
                                    xevent);
      break;

    case 4:
      event = pgm_event_new (PGM_SCROLL);
      ((PgmEventScroll *) event)->direction = PGM_SCROLL_UP;
      translate_scroll_event (glxbackend, (PgmEventScroll *) event, xevent);
      break;

    case 5:
      event = pgm_event_new (PGM_SCROLL);
      ((PgmEventScroll *) event)->direction = PGM_SCROLL_DOWN;
      translate_scroll_event (glxbackend, (PgmEventScroll *) event, xevent);
      break;

    default:
      event = NULL;
      break;
    }

  return event;
}

/* Dispatches events pushing them on the GlViewport queue */
static gboolean
dispatch_x_events (PgmBackend *backend)
{
  static guint8 button_mask = 0;
  PgmGlxBackend *glxbackend = PGM_GLX_BACKEND (backend);
  PgmGlViewport *glviewport = backend->context->glviewport;
  PgmViewport *viewport = PGM_VIEWPORT (glviewport);

  GST_LOG_OBJECT (glxbackend, "dispatch_x_events");

  if (!XPending (glxbackend->dpy))
    return TRUE;

  while (XEventsQueued (glxbackend->dpy, QueuedAlready))
    {
      PgmEvent *pgmevent = NULL;
      XEvent xevent;

      XNextEvent (glxbackend->dpy, &xevent);

      switch (xevent.type)
        {
          /* --- Expose event --- */
        case Expose:
          GST_DEBUG_OBJECT (glxbackend, "Expose event");

          compress_events (glxbackend, &xevent);
          pgmevent = pgm_event_new (PGM_EXPOSE);
          pgm_context_update (backend->context);
          pgm_viewport_push_event (viewport, pgmevent);
          break;

          /* --- Key press event --- */
        case KeyPress:
          GST_DEBUG_OBJECT (glxbackend, "KeyPress event: (time=%u, state=0x%x, "
                            "keycode=0x%x)", xevent.xkey.time, xevent.xkey.state,
                            xevent.xkey.keycode);

          pgmevent = pgm_event_new (PGM_KEY_PRESS);
          translate_key_event (glxbackend, (PgmEventKey *) pgmevent, &xevent);

          /* Handles TAB when the Window is embedded, we request a focus
           * switch to the prev/next widget through the XEmbed protocol */
          if (glxbackend->embedder != None)
            {
              switch (((PgmEventKey *) pgmevent)->keyval)
                {
                case PGM_Tab:
                  send_xembed_message (glxbackend, XEMBED_FOCUS_NEXT, 0, 0, 0);
                  pgm_event_free (pgmevent);
                  continue;

                case PGM_ISO_Left_Tab:
                  send_xembed_message (glxbackend, XEMBED_FOCUS_PREV, 0, 0, 0);
                  pgm_event_free (pgmevent);
                  continue;

                default:
                  break;
                }
            }
          set_wm_user_time (glxbackend, xevent.xkey.time);
          pgm_viewport_push_event (viewport, pgmevent);
          break;

          /* --- Key release event --- */
        case KeyRelease:
          GST_DEBUG_OBJECT (glxbackend, "KeyRelease event: (time=%u, "
                            "state=0x%x, keycode=0x%x)", xevent.xkey.time,
                            xevent.xkey.state, xevent.xkey.keycode);

          pgmevent = pgm_event_new (PGM_KEY_RELEASE);
          translate_key_event (glxbackend, (PgmEventKey *) pgmevent, &xevent);
          pgm_viewport_push_event (viewport, pgmevent);
          break;

          /* --- Button press event --- */
        case ButtonPress:
          GST_DEBUG_OBJECT (glxbackend, "ButtonPress event: (time=%u, "
                            "button=%d x=%d y=%d)", xevent.xbutton.time,
                            xevent.xbutton.button, xevent.xbutton.x,
                            xevent.xbutton.y);

          /* If the Window is embedded, we request the focus through the
           * XEmbed protocol */
          if (glxbackend->embedder != None)
            send_xembed_message (glxbackend, XEMBED_REQUEST_FOCUS, 0, 0, 0);

          pgmevent = create_button_event (glxbackend, &xevent);
          if (pgmevent)
            {
              set_wm_user_time (glxbackend, xevent.xbutton.time);
              pgm_viewport_push_event (viewport, pgmevent);

              /* Events here could also be scroll events, so we check it's not
               * and then grab the pointer */
              if (pgmevent->type == PGM_BUTTON_PRESS)
                {
                  button_mask |= ((PgmEventButton *) pgmevent)->button;
                  XGrabPointer (glxbackend->dpy, glxbackend->win, False,
                                ButtonPressMask | ButtonReleaseMask
                                | PointerMotionMask,
                                GrabModeAsync, GrabModeAsync,
                                None, None, CurrentTime);
                }
            }
          break;

          /* --- Button release event --- */
        case ButtonRelease:
          GST_DEBUG_OBJECT (glxbackend, "ButtonRelease event: (time=%u, "
                            "button=%d x=%d y=%d)", xevent.xbutton.time,
                            xevent.xbutton.button, xevent.xbutton.x,
                            xevent.xbutton.y);

          pgmevent = create_button_release_event (glxbackend, &xevent);
          if (pgmevent)
            {
              pgm_viewport_push_event (viewport, pgmevent);
              button_mask &= ~((PgmEventButton *) pgmevent)->button;
              if (button_mask == 0)
                XUngrabPointer(glxbackend->dpy, CurrentTime);
            }
          break;

          /* --- Motion notify event --- */
        case MotionNotify:
          GST_DEBUG_OBJECT (glxbackend, "MotionNotify event: (time=%u, x=%d, "
                            "y=%d)", xevent.xmotion.time, xevent.xmotion.x,
                            xevent.xmotion.y);

          compress_events (glxbackend, &xevent);
          pgmevent = pgm_event_new (PGM_MOTION_NOTIFY);
          translate_motion_event (glxbackend, (PgmEventMotion *) pgmevent,
                                  &xevent);
          pgm_viewport_push_event (viewport, pgmevent);
          break;

          /* --- Configure notify event --- */
        case ConfigureNotify:
          GST_DEBUG_OBJECT (glxbackend, "ConfigureNotify event: (x=%d, y=%d, "
                            "width=%d, height=%d)", xevent.xconfigure.x,
                            xevent.xconfigure.y, xevent.xconfigure.width,
                            xevent.xconfigure.height);

          pgmevent = pgm_event_new (PGM_CONFIGURE);
          translate_configure_event (glxbackend, (PgmEventConfigure *) pgmevent,
                                     &xevent);
          update_viewport_size (glxbackend,
                                ((PgmEventConfigure *) pgmevent)->width,
                                ((PgmEventConfigure *) pgmevent)->height);
          pgm_viewport_push_event (viewport, pgmevent);
          break;

          /* --- Selection notify event --- */
        case SelectionNotify:
          GST_DEBUG_OBJECT (glxbackend, "SelectionNotify event: "
                            "(requestor=0x%x)", xevent.xselection.requestor);

          glxbackend->drag_uri = get_xdnd_uri_list (glxbackend, &xevent);
          if (glxbackend->drag_uri)
            glxbackend->drag_data_received = TRUE;
          break;

          /* --- Client message event --- */
        case ClientMessage:
          GST_DEBUG_OBJECT (glxbackend, "ClientMessage event: "
                            "message_type='%s')", XGetAtomName
                            (glxbackend->dpy, xevent.xclient.message_type));
          {
            Atom atom = xevent.xclient.message_type;

            if (atom == glxbackend->atom[ATOM_WM_PROTOCOLS])
              pgmevent = handle_wm_protocols_event (glxbackend, &xevent);
            else if (atom ==  glxbackend->atom[ATOM_XEMBED])
              pgmevent = handle_xembed_event (glxbackend, &xevent);
            else if (atom == glxbackend->atom[ATOM_XDND_ENTER])
              handle_xdnd_enter_event (glxbackend, &xevent);
            else if (atom == glxbackend->atom[ATOM_XDND_POSITION])
              pgmevent = handle_xdnd_position_event (glxbackend, &xevent);
            else if (atom == glxbackend->atom[ATOM_XDND_DROP])
              pgmevent = handle_xdnd_drop_event (glxbackend, &xevent);
            else if (atom == glxbackend->atom[ATOM_XDND_LEAVE])
              pgmevent = handle_xdnd_leave_event (glxbackend, &xevent);

            if (pgmevent)
              pgm_viewport_push_event (viewport, pgmevent);
          }

        default:
          break;
        }
    }

  return TRUE;
}

/* Events preparing */
static gboolean
event_prepare (GSource *source,
               gint *timeout)
{
  PgmGlxBackendSource *glxbackend_source = (PgmGlxBackendSource *) source;
  PgmGlxBackend *glxbackend = glxbackend_source->glxbackend;
  PgmBackend *backend = PGM_BACKEND (glxbackend);

  if (XEventsQueued (glxbackend->dpy, QueuedAlready))
    return TRUE;

  *timeout = backend->context->update_tag ? 0 : -1;

  return FALSE;
}

/* Events checking */
static gboolean
event_check (GSource *source)
{
  PgmGlxBackendSource *glxbackend_source = (PgmGlxBackendSource *) source;

  return (glxbackend_source->poll_fd.revents & G_IO_IN) != 0;
}

/* Events dispatching */
static gboolean
event_dispatch (GSource *source,
                GSourceFunc callback,
                gpointer data)
{
  if (callback)
    return callback (data);

  return FALSE;
}

/* Add event handling source to the rendering main context */
static gboolean
add_event_source (PgmGlxBackend *glxbackend)
{
  PgmBackend *backend = PGM_BACKEND (glxbackend);
  GMainContext *render_context = backend->context->render_context;
  PgmGlxBackendSource *source;

  /* Filter handled events */
  XSelectInput (glxbackend->dpy, glxbackend->win,
                KeyPressMask | KeyReleaseMask | ButtonPressMask
                | ButtonReleaseMask | StructureNotifyMask | ExposureMask
                | PointerMotionMask | PropertyChangeMask);

  /* Create and initialize the event handling dedicated source */
  source = (PgmGlxBackendSource *)
    g_source_new (&event_funcs, sizeof (PgmGlxBackendSource));
  source->poll_fd.fd = ConnectionNumber (glxbackend->dpy);
  source->poll_fd.events = G_IO_IN;
  source->glxbackend = glxbackend;
  g_source_add_poll ((GSource *) source, &source->poll_fd);

  /* Attach it */
  g_source_set_callback ((GSource *) source, (GSourceFunc) dispatch_x_events,
                         glxbackend, NULL);
  glxbackend->event_id = g_source_attach ((GSource *) source, render_context);
  g_source_unref ((GSource *) source);

  return TRUE;
}

/* Switches the window to fullscreen depending on the fullscreen value */
static void
set_fullscreen (PgmGlxBackend *glxbackend,
                gboolean fullscreen)
{
  PgmBackend *backend = PGM_BACKEND (glxbackend);
  PgmViewport *viewport = PGM_VIEWPORT (backend->context->glviewport);
  XClientMessageEvent xclient;

  /* Adapt the viewport size */
  if (glxbackend->fullscreen != fullscreen)
    {
      if (fullscreen)
        {
          glxbackend->windowed_width = viewport->width;
          glxbackend->windowed_height = viewport->height;
          update_viewport_size (glxbackend, glxbackend->resolution_width,
                                glxbackend->resolution_height);
        }
      else
        update_viewport_size (glxbackend, glxbackend->windowed_width,
                              glxbackend->windowed_height);
    }

  /* And switch the fullscreen state */
  xclient.type = ClientMessage;
  xclient.window = glxbackend->win;
  xclient.display = glxbackend->dpy;
  xclient.message_type = glxbackend->atom[ATOM_NET_WM_STATE];
  xclient.format = 32;

  xclient.data.l[0] = fullscreen;
  xclient.data.l[1] = glxbackend->atom[ATOM_NET_WM_STATE_FULLSCREEN];
  xclient.data.l[2] = None;
  xclient.data.l[3] = 0;
  xclient.data.l[4] = 0;

  XSendEvent (glxbackend->dpy, glxbackend->root, False,
              SubstructureRedirectMask | SubstructureNotifyMask,
              (XEvent *) &xclient);
  XSync (glxbackend->dpy, True);
}

/* PgmBackend methods */

static gboolean
pgm_glx_backend_create_window (PgmBackend *backend)
{
  PgmGlxBackend *glxbackend = PGM_GLX_BACKEND (backend);
  XVisualInfo *visual = NULL;
  guchar xdnd_version = XDND_PROTOCOL_VERSION;
  const gchar *indirect;
  XSetWindowAttributes attr;
  PgmViewport *viewport;
  gint width, height;

  GST_DEBUG_OBJECT (glxbackend, "create_window");

  viewport = PGM_VIEWPORT (backend->context->glviewport);

  glxbackend->dpy = XOpenDisplay (NULL);
  if (!glxbackend->dpy)
    {
      GST_ERROR_OBJECT (glxbackend, "couldn't open default display");
      return FALSE;
    }

  glxbackend->screen = DefaultScreen (glxbackend->dpy);
  glxbackend->root = DefaultRootWindow (glxbackend->dpy);

  if (!load_glx_extensions (glxbackend))
    {
      GST_ERROR_OBJECT (glxbackend, "couldn't load GLX extensions");
      return FALSE;
    }

  /* Visual retrieving */
  visual = get_visual_info (glxbackend);
  if (!visual)
    {
      GST_ERROR_OBJECT (glxbackend, "couldn't get visual");
      return FALSE;
    }

  /* Create the window */
  pgm_viewport_get_size (viewport, &width, &height);
  memset (&attr, 0, sizeof (XSetWindowAttributes));
  attr.colormap = XCreateColormap (glxbackend->dpy, glxbackend->root,
                                   visual->visual, AllocNone);
  glxbackend->win = XCreateWindow (glxbackend->dpy, glxbackend->root,
                                   0, 0, width, height, 0, visual->depth,
                                   InputOutput, visual->visual,
                                   CWBackPixel | CWBorderPixel | CWColormap,
                                   &attr);
  if (!glxbackend->win)
    {
      GST_ERROR_OBJECT (glxbackend, "couldn't create window");
      return FALSE;
    }
  GST_INFO_OBJECT (glxbackend, "created window (id 0x%x)", glxbackend->win);

  /* Event source */
  if (!add_event_source (glxbackend))
    {
      GST_ERROR_OBJECT (glxbackend, "couldn't add event handling source\n");
      return FALSE;
    }

  /* Fill X atoms arrays */
  glxbackend->atom = g_malloc (nb_atoms * sizeof (Atom));
  XInternAtoms (glxbackend->dpy, (gchar**) atom_name, nb_atoms, False,
                glxbackend->atom);

  /* X protocols initialization */
  set_wm_protocols (glxbackend);
  set_xembed_info (glxbackend, 0);
  XChangeProperty (glxbackend->dpy, glxbackend->win,
                   glxbackend->atom[ATOM_XDND_AWARE], XA_ATOM, 32,
                   PropModeReplace, &xdnd_version, 1);

  /* Create the OpenGL context */
  indirect = g_getenv ("PGM_GL_INDIRECT_RENDERING");
  if (indirect && *indirect == '1')
    glxbackend->ctx = glXCreateContext (glxbackend->dpy, visual, NULL,
                                        PGM_GL_FALSE);
  else
    glxbackend->ctx = glXCreateContext (glxbackend->dpy, visual, NULL,
                                        PGM_GL_TRUE);
  if (!glxbackend->ctx)
    {
      GST_ERROR_OBJECT (glxbackend, "couldn't create OpenGL context");
      return FALSE;
    }
  glXMakeCurrent (glxbackend->dpy, glxbackend->win, glxbackend->ctx);
  XFree (visual);

  /* Is it a direct rendering context? */
  if (glXIsDirect (glxbackend->dpy, glxbackend->ctx))
    {
      GST_INFO_OBJECT (glxbackend, "created OpenGL context (direct rendering)");
      glxbackend->feature_mask |= PGM_GLX_FEAT_DIRECT_RENDERING;
    }
  else
    GST_INFO_OBJECT (glxbackend, "created opengl context (indirect rendering)");

  /* Init the startup notification protocol */
  if (G_LIKELY (!glxbackend->startup_notification_id))
    init_startup_notification (glxbackend);

  setup_vblank (glxbackend);

  glxbackend->resolution_width =
    DisplayWidth (glxbackend->dpy, glxbackend->screen);
  glxbackend->resolution_height =
    DisplayHeight (glxbackend->dpy, glxbackend->screen);
  glxbackend->size_mm_width =
    DisplayWidthMM (glxbackend->dpy, glxbackend->screen);
  glxbackend->size_mm_height =
    DisplayHeightMM (glxbackend->dpy, glxbackend->screen);

  glxbackend->none_cursor = get_none_cursor (glxbackend);

  gdk_pixbuf_xlib_init (glxbackend->dpy, glxbackend->screen);

  glxbackend->created = TRUE;

  return TRUE;
}

static gboolean
pgm_glx_backend_destroy_window (PgmBackend *backend)
{
  PgmGlxBackend *glxbackend = PGM_GLX_BACKEND (backend);

  GST_DEBUG_OBJECT (glxbackend, "destroy_window");

  if (glxbackend->created)
    {
      /* Remove event handling source */
      g_source_remove (glxbackend->event_id);

      /* Clean up X related data */
      XFreeCursor (glxbackend->dpy, glxbackend->none_cursor);
      glXDestroyContext (glxbackend->dpy, glxbackend->ctx);
      XDestroyWindow (glxbackend->dpy, glxbackend->win);
      XCloseDisplay (glxbackend->dpy);
      glxbackend->created = FALSE;

      g_free (glxbackend->atom);
      glxbackend->atom = NULL;
    }

  return TRUE;
}

static void
pgm_glx_backend_set_title (PgmBackend *backend,
                           const gchar *title)
{
  PgmGlxBackend *glxbackend = PGM_GLX_BACKEND (backend);

  GST_DEBUG_OBJECT (glxbackend, "set_title");

  XChangeProperty (glxbackend->dpy, glxbackend->win,
                   glxbackend->atom[ATOM_NET_WM_NAME],
                   glxbackend->atom[ATOM_UTF8_STRING], 8, PropModeReplace,
                   (guchar *) title, strlen (title));
  XSync (glxbackend->dpy, False);
}

static gboolean
pgm_glx_backend_set_visibility (PgmBackend *backend,
                                gboolean visible)
{
  PgmGlxBackend *glxbackend = PGM_GLX_BACKEND (backend);
  PgmViewport *viewport = PGM_VIEWPORT (backend->context->glviewport);

  GST_DEBUG_OBJECT (glxbackend, "set_visibility");

  if (visible)
    {
      XMapWindow (glxbackend->dpy, glxbackend->win);

      /* If the fullscreen state has changed to FALSE during the hidden period */

      /* we need to resize the window to avoid showing it on the whole screen */
      if (!glxbackend->fullscreen)
        {
          gint width, height;
          pgm_viewport_get_size (viewport, &width, &height);
          XResizeWindow (glxbackend->dpy, glxbackend->win, width, height);
        }

      /* and then change accordingly the fullscreen state */
      set_fullscreen (glxbackend, glxbackend->fullscreen);
    }
  else
    {
      XUnmapWindow (glxbackend->dpy, glxbackend->win);
      XSync (glxbackend->dpy, False);
    }

  return TRUE;
}

static gboolean
pgm_glx_backend_set_decorated (PgmBackend *backend,
                               gboolean decorated)
{
  PgmGlxBackend *glxbackend = PGM_GLX_BACKEND (backend);
  XClientMessageEvent xclient;
  MwmHints *hints, new_hints;
  gulong nitems, bytes_after;
  guchar *data;
  gint format;
  Atom type;

  GST_DEBUG_OBJECT (glxbackend, "set_decorated");

  /* Create the new hints */
  memset (&new_hints, 0, sizeof (new_hints));
  new_hints.flags = MWM_HINTS_DECORATIONS;
  new_hints.decorations = decorated;

  /* Get the current hints */
  XGetWindowProperty (glxbackend->dpy, glxbackend->win,
                      glxbackend->atom[ATOM_MOTIF_WM_HINTS], 0,
                      sizeof (MwmHints) / sizeof (glong), False,
                      AnyPropertyType, &type, &format, &nitems, &bytes_after,
                      &data);

  /* Set hints with our new hints */
  if (type == None)
    hints = &new_hints;
  else
    {
      hints = (MwmHints*) data;
      hints->flags |= MWM_HINTS_DECORATIONS;
      hints->decorations = new_hints.decorations;
    }

  /* Change the decorations */
  XChangeProperty (glxbackend->dpy, glxbackend->win,
                   glxbackend->atom[ATOM_MOTIF_WM_HINTS],
                   glxbackend->atom[ATOM_MOTIF_WM_HINTS],
                   32, PropModeReplace, (guchar*) hints,
                   sizeof (MwmHints) / sizeof (long));

  if (hints != &new_hints)
    XFree (hints);

  /* Send a focus request message so that the window keeps the focus after
   * the decorations change */
  memset (&xclient, 0, sizeof (xclient));
  xclient.type = ClientMessage;
  xclient.window = glxbackend->win;
  xclient.message_type = glxbackend->atom[ATOM_NET_ACTIVE_WINDOW];
  xclient.format = 32;
  xclient.data.l[0] = 1;
  xclient.data.l[1] = CurrentTime;
  xclient.data.l[2] = None;
  xclient.data.l[3] = 0;
  xclient.data.l[4] = 0;
  XSendEvent (glxbackend->dpy, glxbackend->root, False,
              SubstructureRedirectMask | SubstructureNotifyMask,
              (XEvent*) &xclient);
  XSync (glxbackend->dpy, False);

  return TRUE;
}

static void
pgm_glx_backend_swap_buffers (PgmBackend *backend)
{
  PgmGlxBackend *glxbackend = PGM_GLX_BACKEND (backend);

  GST_LOG_OBJECT (glxbackend, "swap_buffers");

  glXSwapBuffers (glxbackend->dpy, glxbackend->win);
}

static gpointer
pgm_glx_backend_get_proc_address (PgmBackend *backend,
                                  const gchar *proc_name)
{
  PgmGlxBackend *glxbackend = PGM_GLX_BACKEND (backend);

  return get_proc_address (glxbackend, proc_name);
}

static gboolean
pgm_glx_backend_set_size (PgmBackend *backend,
                          gint width,
                          gint height)
{
  PgmGlxBackend *glxbackend = PGM_GLX_BACKEND (backend);

  GST_DEBUG_OBJECT (glxbackend, "set_size");

  XResizeWindow (glxbackend->dpy, glxbackend->win, width, height);
  XSync (glxbackend->dpy, False);

  return FALSE;
}

static gboolean
pgm_glx_backend_set_fullscreen (PgmBackend *backend,
                                gboolean fullscreen)
{
  PgmGlxBackend *glxbackend = PGM_GLX_BACKEND (backend);

  GST_DEBUG_OBJECT (glxbackend, "set_fullscreen");

  set_fullscreen (glxbackend, fullscreen);
  glxbackend->fullscreen = fullscreen;

  return TRUE;
}

static void
pgm_glx_backend_get_screen_size_mm (PgmBackend *backend,
                                    gint *width,
                                    gint *height)
{
  PgmGlxBackend *glxbackend = PGM_GLX_BACKEND (backend);

  GST_LOG_OBJECT (glxbackend, "get_screen_size_mm");

  *width = glxbackend->size_mm_width;
  *height = glxbackend->size_mm_height;
}

static gboolean
pgm_glx_backend_set_screen_resolution (PgmBackend *backend,
                                       gint width,
                                       gint height)
{
  PgmGlxBackend *glxbackend = PGM_GLX_BACKEND (backend);

  GST_DEBUG_OBJECT (glxbackend, "set_screen_resolution");

  /* FIXME */

  return FALSE;
}

static void
pgm_glx_backend_get_screen_resolution (PgmBackend *backend,
                                       gint *width,
                                       gint *height)
{
  PgmGlxBackend *glxbackend = PGM_GLX_BACKEND (backend);

  GST_LOG_OBJECT (glxbackend, "get_screen_resolution");

  *width = glxbackend->resolution_width;
  *height = glxbackend->resolution_height;
}

static gboolean
pgm_glx_backend_build_text_lists (PgmBackend *backend)
{
  PgmGlxBackend *glxbackend = PGM_GLX_BACKEND (backend);
  PgmContextProcAddress *gl = backend->context->gl;
  XFontStruct *font;
  gint first;
  gint last;

  GST_DEBUG_OBJECT (glxbackend, "build_text_lists");

  glxbackend->text_lists = gl->gen_lists (256);
  if (!gl->is_list (glxbackend->text_lists))
    {
      GST_WARNING_OBJECT (glxbackend, "unable to build text display lists\n");
      return FALSE;
    }

  font = XLoadQueryFont (glxbackend->dpy, "7x13bold");
  if (!font)
    {
      GST_WARNING_OBJECT (glxbackend, "unable to load X font \"7x13bold\"\n");
      font = XLoadQueryFont (glxbackend->dpy, "fixed");
      if (!font)
        {
          GST_ERROR_OBJECT (glxbackend, "unable to load X font \"fixed\"\n");
          return FALSE;
        }
    }

  first = font->min_char_or_byte2;
  last = font->max_char_or_byte2;

  /* FIXME: Set up an error handler */
  glXUseXFont (font->fid, first, last - first + 1,
               glxbackend->text_lists + first);

  return TRUE;
}

static gboolean
pgm_glx_backend_destroy_text_lists (PgmBackend *backend)
{
  PgmGlxBackend *glxbackend = PGM_GLX_BACKEND (backend);
  PgmContextProcAddress *gl = backend->context->gl;

  GST_DEBUG_OBJECT (glxbackend, "destroy_text_lists");

  if (gl->is_list (glxbackend->text_lists))
    gl->delete_lists (glxbackend->text_lists, 256);

  return TRUE;
}

static void
pgm_glx_backend_raster_text (PgmBackend *backend,
                             const gchar *text,
                             gfloat x,
                             gfloat y,
                             gfloat r,
                             gfloat g,
                             gfloat b)
{
  PgmGlxBackend *glxbackend = PGM_GLX_BACKEND (backend);
  PgmContextProcAddress *gl = backend->context->gl;

  GST_DEBUG_OBJECT (glxbackend, "raster_text");

  gl->load_identity ();
  gl->bind_texture (PGM_GL_TEXTURE_2D, 0);
  gl->push_attrib (PGM_GL_LIST_BIT);
  gl->color_4f (r, g, b, 1.0f);
  gl->raster_pos_2f (x, y);
  gl->list_base (glxbackend->text_lists);
  gl->call_lists (strlen (text), PGM_GL_UNSIGNED_BYTE, (PgmGlUbyte *) text);
  gl->pop_attrib ();
}

static void
pgm_glx_backend_wait_for_vblank (PgmBackend *backend)
{
  PgmGlxBackend *glxbackend = PGM_GLX_BACKEND (backend);

  GST_LOG_OBJECT (glxbackend, "wait_for_vblank");

  if (glxbackend->vblank_mode == PGM_VBLANK_VIDEO_SYNC)
    {
      PgmGlxBackendProcAddress *glx = glxbackend->glx;
      guint retrace_count;

      glx->get_video_sync (&retrace_count);
      glx->wait_video_sync (2, (retrace_count + 1) % 2, &retrace_count);
    }
}

static void
pgm_glx_backend_notify_startup_complete (PgmBackend *backend)
{
  PgmGlxBackend *glxbackend = PGM_GLX_BACKEND (backend);
  gchar *dest, *dest_end, *escaped_id, *message;
  const gchar *src, *src_end;
  XSetWindowAttributes attrs;
  XClientMessageEvent xclient;
  Window win;

  GST_LOG_OBJECT (glxbackend, "notify_startup_complete");

  if (!glxbackend->startup_notification_id)
    return;

  /* Create the message string ending the startup sequence */
  escaped_id = escape_for_x_message (glxbackend->startup_notification_id);
  message = g_strdup_printf ("remove: ID=%s", escaped_id);
  g_free (escaped_id);

  /* Creates an X window to be used to identify the message uniquely */
  attrs.override_redirect = True;
  attrs.event_mask = PropertyChangeMask | StructureNotifyMask;
  win = XCreateWindow (glxbackend->dpy, glxbackend->root, -100, -100, 1, 1, 0,
                       CopyFromParent, CopyFromParent, (Visual*) CopyFromParent,
                       CWOverrideRedirect | CWEventMask, &attrs);

  /* Send the X message, following sample code in the freedesktop startup
   * notification specification */
  memset (&xclient, 0, sizeof (xclient));
  xclient.type = ClientMessage;
  xclient.message_type = glxbackend->atom[ATOM_NET_STARTUP_INFO_BEGIN];
  xclient.display = glxbackend->dpy;
  xclient.window = win;
  xclient.format = 8;

  src = message;
  src_end = message + strlen (message) + 1;

  while (src != src_end)
    {
      dest = &xclient.data.b[0];
      dest_end = dest + 20;

      while (dest != dest_end && src != src_end)
        {
          *dest = *src;
          ++dest;
          ++src;
        }
      while (dest != dest_end)
        {
          *dest = 0;
          ++dest;
        }

      XSendEvent (glxbackend->dpy, glxbackend->root, False, PropertyChangeMask,
                  (XEvent*) &xclient);
      xclient.message_type = glxbackend->atom[ATOM_NET_STARTUP_INFO];
    }

  g_free (message);

  /* The dedicated window can be destroyed now */
  XDestroyWindow (glxbackend->dpy, win);
  XSync (glxbackend->dpy, False);
}

static gboolean
pgm_glx_backend_set_cursor (PgmBackend *backend,
                            PgmViewportCursor cursor)
{
  PgmGlxBackend *glxbackend = PGM_GLX_BACKEND (backend);

  GST_DEBUG_OBJECT (glxbackend, "set_cursor");

  switch (cursor)
    {
    case PGM_VIEWPORT_LEFT_ARROW:
      XDefineCursor (glxbackend->dpy, glxbackend->win,
                     XCreateFontCursor (glxbackend->dpy, XC_top_left_arrow));
      break;

    case PGM_VIEWPORT_INHERIT:
      XDefineCursor (glxbackend->dpy, glxbackend->win,
                     XCreateFontCursor (glxbackend->dpy, XC_top_left_arrow));
      break;

    case PGM_VIEWPORT_NONE:
      XDefineCursor (glxbackend->dpy, glxbackend->win, glxbackend->none_cursor);
      break;

    default:
      break;
    }

  XSync (glxbackend->dpy, False);

  return TRUE;
}

static gboolean
pgm_glx_backend_set_icon (PgmBackend *backend,
                          GdkPixbuf *icon)
{
  PgmGlxBackend *glxbackend = PGM_GLX_BACKEND (backend);
  XWMHints wm_hints = { 0 };
  Pixmap icon_pixmap = None;
  Pixmap icon_mask = None;

  GST_DEBUG_OBJECT (glxbackend, "set_icon");

  if (icon)
    gdk_pixbuf_xlib_render_pixmap_and_mask (icon, &icon_pixmap, &icon_mask, 128);
  else
    {
      icon_pixmap = None;
      icon_mask = None;
    }

  free_icon (backend);

  wm_hints.flags = IconPixmapHint | IconMaskHint;
  wm_hints.icon_pixmap = icon_pixmap;
  wm_hints.icon_mask = icon_mask;

  XSetWMHints (glxbackend->dpy, glxbackend->win, &wm_hints);
  XSync (glxbackend->dpy, False);

  return TRUE;
}

static void
pgm_glx_backend_set_drag_status (PgmBackend *backend,
                                 gboolean accept)
{
  PgmGlxBackend *glxbackend = PGM_GLX_BACKEND (backend);

  GST_LOG_OBJECT (backend, "set_drag_status");

  /* Avoid useless computations */
  if (accept == glxbackend->drag_status)
    return;

  glxbackend->drag_status = accept;

  if (G_LIKELY (glxbackend->drag_data_received))
    update_xdnd_status (glxbackend);
  else
    GST_DEBUG_OBJECT (glxbackend, "no on-going drag");
}

static gboolean
pgm_glx_backend_is_accelerated (PgmBackend *backend)
{
  PgmGlxBackend *glxbackend = PGM_GLX_BACKEND (backend);

  GST_LOG_OBJECT (backend, "is_accelerated");

  return glxbackend->feature_mask & PGM_GLX_FEAT_DIRECT_RENDERING;
}

static gboolean
pgm_glx_backend_is_embeddable (PgmBackend *backend)
{
  GST_LOG_OBJECT (backend, "is_embeddable");

  /* Supported through the XEmbed implementation */
  return TRUE;
}

static void
pgm_glx_backend_get_embedding_id (PgmBackend *backend,
                                  gulong *embedding_id)
{
  PgmGlxBackend *glxbackend = PGM_GLX_BACKEND (backend);

  GST_LOG_OBJECT (backend, "get_embedding_id");

  *embedding_id = glxbackend->win;
}

static gboolean
pgm_glx_backend_has_alpha_component (PgmBackend *backend)
{
  PgmGlxBackend *glxbackend = PGM_GLX_BACKEND (backend);

  GST_LOG_OBJECT (backend, "has_alpha_component");

  return glxbackend->feature_mask & PGM_GLX_FEAT_ARGB_VISUAL;
}

/* GObject stuff */

PGM_DEFINE_DYNAMIC_TYPE (PgmGlxBackend, pgm_glx_backend, PGM_TYPE_BACKEND);

void
pgm_glx_backend_register (GTypeModule *module)
{
  pgm_glx_backend_register_type (module);
}

static void
pgm_glx_backend_dispose (GObject *object)
{
  PgmGlxBackend *glxbackend = PGM_GLX_BACKEND (object);

  GST_DEBUG_OBJECT (glxbackend, "dispose");

  pgm_glx_backend_destroy_text_lists (PGM_BACKEND (glxbackend));

  if (glxbackend->created)
    {
      free_icon (PGM_BACKEND (glxbackend));
      pgm_glx_backend_destroy_window (PGM_BACKEND (glxbackend));
    }

  if (glxbackend->startup_notification_id)
    {
      g_free (glxbackend->startup_notification_id);
      glxbackend->startup_notification_id = NULL;
    }

  GST_CALL_PARENT (G_OBJECT_CLASS, dispose, (object));
}

static void
pgm_glx_backend_class_init (PgmGlxBackendClass *klass)
{
  GObjectClass *gobject_class;
  PgmBackendClass *backend_class;

  GST_DEBUG_CATEGORY_INIT (pgm_gl_glxbackend_debug, "pgm_gl_glxbackend", 0,
                           "OpenGL plugin: PgmGlxBackend");

  parent_class = g_type_class_peek_parent (klass);

  gobject_class = G_OBJECT_CLASS (klass);
  backend_class = PGM_BACKEND_CLASS (klass);

  /* GObject virtual table */
  gobject_class->dispose = GST_DEBUG_FUNCPTR (pgm_glx_backend_dispose);

  /* PgmBackend virtual table */
  backend_class->create_window =
    GST_DEBUG_FUNCPTR (pgm_glx_backend_create_window);
  backend_class->destroy_window =
    GST_DEBUG_FUNCPTR (pgm_glx_backend_destroy_window);
  backend_class->set_title = GST_DEBUG_FUNCPTR (pgm_glx_backend_set_title);
  backend_class->set_decorated =
    GST_DEBUG_FUNCPTR (pgm_glx_backend_set_decorated);
  backend_class->swap_buffers =
    GST_DEBUG_FUNCPTR (pgm_glx_backend_swap_buffers);
  backend_class->get_proc_address =
    GST_DEBUG_FUNCPTR (pgm_glx_backend_get_proc_address);
  backend_class->set_size = GST_DEBUG_FUNCPTR (pgm_glx_backend_set_size);
  backend_class->set_fullscreen =
    GST_DEBUG_FUNCPTR (pgm_glx_backend_set_fullscreen);
  backend_class->set_visibility =
    GST_DEBUG_FUNCPTR (pgm_glx_backend_set_visibility);
  backend_class->get_screen_size_mm =
    GST_DEBUG_FUNCPTR (pgm_glx_backend_get_screen_size_mm);
  backend_class->set_screen_resolution =
    GST_DEBUG_FUNCPTR (pgm_glx_backend_set_screen_resolution);
  backend_class->get_screen_resolution =
    GST_DEBUG_FUNCPTR (pgm_glx_backend_get_screen_resolution);
  backend_class->build_text_lists =
    GST_DEBUG_FUNCPTR (pgm_glx_backend_build_text_lists);
  backend_class->destroy_text_lists =
    GST_DEBUG_FUNCPTR (pgm_glx_backend_destroy_text_lists);
  backend_class->raster_text = GST_DEBUG_FUNCPTR (pgm_glx_backend_raster_text);
  backend_class->wait_for_vblank =
    GST_DEBUG_FUNCPTR (pgm_glx_backend_wait_for_vblank);
   backend_class->notify_startup_complete =
     GST_DEBUG_FUNCPTR (pgm_glx_backend_notify_startup_complete);
  backend_class->set_cursor = GST_DEBUG_FUNCPTR (pgm_glx_backend_set_cursor);
  backend_class->set_icon = GST_DEBUG_FUNCPTR (pgm_glx_backend_set_icon);
  backend_class->set_drag_status =
    GST_DEBUG_FUNCPTR (pgm_glx_backend_set_drag_status);
  backend_class->is_accelerated =
    GST_DEBUG_FUNCPTR (pgm_glx_backend_is_accelerated);
  backend_class->is_embeddable =
    GST_DEBUG_FUNCPTR (pgm_glx_backend_is_embeddable);
  backend_class->get_embedding_id =
    GST_DEBUG_FUNCPTR (pgm_glx_backend_get_embedding_id);
  backend_class->has_alpha_component =
    GST_DEBUG_FUNCPTR (pgm_glx_backend_has_alpha_component);
}

static void
pgm_glx_backend_class_finalize (PgmGlxBackendClass *klass)
{
  return;
}

static void
pgm_glx_backend_init (PgmGlxBackend *glxbackend)
{
  GST_DEBUG_OBJECT (glxbackend, "init");

  glxbackend->dpy = NULL;
  glxbackend->glx = NULL;
  glxbackend->atom = NULL;
  glxbackend->startup_notification_id = NULL;
  glxbackend->embedder = None;
  glxbackend->drag_source = None;
  glxbackend->drag_data_has_uri = FALSE;
  glxbackend->drag_data_received = FALSE;
  glxbackend->drag_x = -1.0f;
  glxbackend->drag_y = -1.0f;
  glxbackend->drag_uri = NULL;
  glxbackend->created = FALSE;
  glxbackend->fullscreen = FALSE;
  glxbackend->windowed_width = 800;
  glxbackend->windowed_height = 600;
  glxbackend->vendor = 0;
  glxbackend->version = 0.0f;
  glxbackend->extensions = 0;
  glxbackend->feature_mask = 0;
  glxbackend->vblank_mode = PGM_VBLANK_NONE;
}

/* Public methods */

PgmBackend *
pgm_glx_backend_new (PgmContext *context)
{
  PgmBackend *backend;

  backend = g_object_new (PGM_TYPE_GLX_BACKEND, NULL);
  GST_DEBUG_OBJECT (PGM_GLX_BACKEND (backend), "created new glxbackend");

  backend->context = context;

  return backend;
}
