/*  Screem:  skel-plugin.c
 *
 *  Copyright (C) 2004 David A Knight
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
 */
#include <config.h>

#include <glib/gi18n.h>
#include <gtk/gtk.h>

#include "screem-skel-plugin.h"

/* setup your plugin here */

/* plugin name should only include a-zA-Z */
static const gchar *plugin_name = "ScreemColorWizard";

static const gchar *authors[] = {
	"David A Knight <david@screem.org>",
	NULL
};
static const gchar *displayed_name = N_( "Screem Color Wizard" );
static const gchar *description = N_( "Color selector for inserting color codes" );
static const gchar *version = "2.0.0";

/* add any per instance data items here */
struct ScreemSkelPluginPrivate {
	GtkWidget *dialog;
	GtkWidget *colorsel;
};

static void color_wizard_display( GtkAction *action, 
		gpointer user_data );
static void clicked( GtkWidget *widget, gint button, gpointer data );
static void init_dialog( ScreemPlugin *plugin );

/**
 * setup:
 * 
 * this function will be called once for each window,
 * you should add any actions / ui here, eg.
 * 
 * screem_plugin_add_action( plugin, name, label, tip, stock_id,
 * 				callback, error );
 * screem_plugin_add_menu( plugin, path, action, error );
 * screem_plugin_add_toolbar( plugin, path, action, error );
 *
 *
 * to insert text into the current page being edited your callbacks
 * should make use of
 * screem_plugin_get_cursor_position( plugin )
 * screem_plugin_set_cursor_position( plugin, pos )
 * screem_plugin_insert( plugin, pos, text, length, indent )
 * 
 **/
static gboolean setup( ScreemPlugin *plugin )
{
	GError *error;
	gboolean ret;

	error = NULL;
	
	ret = screem_plugin_add_interface( plugin, "ColorWizard",
				_( "Color Wizard" ),
				_( "Select a Color to Insert" ),
				GTK_STOCK_SELECT_COLOR,
				G_CALLBACK( color_wizard_display ),
				&error  );
	if( ! ret ) {
		g_print( "Add interface error: %s\n", error->message );
		g_error_free( error );
	} 	
	return ret;
}

/**
 * cleanup:
 *
 * this function will be called once for each window when
 * it is closed, you should cleanup any data items you
 * have in ScreemSkelPluginPrivate here
 **/
static void cleanup( ScreemSkelPluginPrivate *priv )
{
	gtk_widget_destroy( priv->dialog );
}

static void color_wizard_display( GtkAction *action, 
		gpointer user_data )
{
	ScreemPlugin *plugin;
	ScreemSkelPluginPrivate *priv;
	ScreemPage *page;

	plugin = SCREEM_PLUGIN( user_data );
	priv = SCREEM_SKEL_PLUGIN( plugin )->priv;
	
	page = screem_plugin_get_current_document( plugin );
	
	if( page ) {
		init_dialog( plugin );
		if( ! GTK_WIDGET_VISIBLE( priv->dialog ) ) {
			screem_plugin_restore_from_session( plugin, 
				priv->dialog );
		}
		gtk_widget_show_all( priv->dialog );
		gdk_window_raise( priv->dialog->window ); 
	}
}

static void clicked( GtkWidget *widget, gint button, gpointer data )
{
	ScreemPlugin *plugin;
	ScreemSkelPluginPrivate *priv;
	gint r, g, b;
	gchar *text;
	guint pos;

	GdkColor colour;

	plugin = SCREEM_PLUGIN( data );
	priv = SCREEM_SKEL_PLUGIN( plugin )->priv;
	
	if( button == GTK_RESPONSE_OK || button == GTK_RESPONSE_APPLY ) {
		gtk_color_selection_get_current_color( GTK_COLOR_SELECTION( priv->colorsel ),
				&colour );
		
		/* we want 8 bit values, not 16 */
		r = colour.red >> 8;
		g = colour.green >> 8;
		b = colour.blue >> 8;

		/* FIXME: handle selection */
			
		text = g_strdup_printf( "#%.2x%.2x%.2x", r, g, b );
		pos = screem_plugin_get_cursor_position( plugin );
		screem_plugin_insert( plugin, pos, text, 7, FALSE );
		g_free( text );
	} else { 
		/* FIXME: handle restoring original selection */
	}

	screem_plugin_store_in_session( plugin, widget );
	gtk_widget_hide( widget );
}

static void init_dialog( ScreemPlugin *plugin )
{
	ScreemSkelPluginPrivate *priv;
	GtkWidget *widget;
	
	/* setup color wizard dialog */
	priv = SCREEM_SKEL_PLUGIN( plugin )->priv;
	
	if( ! priv->dialog ) {
		priv->dialog = gtk_dialog_new_with_buttons( _( "Insert a color" ),
				NULL, 0,
				GTK_STOCK_CLOSE, GTK_RESPONSE_CLOSE,
				GTK_STOCK_APPLY, GTK_RESPONSE_APPLY,
				GTK_STOCK_OK, GTK_RESPONSE_OK,
				NULL );
      
		priv->colorsel = widget = gtk_color_selection_new();
		gtk_color_selection_set_has_palette( GTK_COLOR_SELECTION( priv->colorsel ), TRUE );
		gtk_color_selection_set_has_opacity_control( GTK_COLOR_SELECTION( priv->colorsel ), FALSE );

		gtk_box_pack_start( GTK_BOX( GTK_DIALOG( priv->dialog )->vbox ), 
				widget, TRUE, TRUE, 6 );

		g_signal_connect( G_OBJECT( priv->dialog ), "response", 
			  G_CALLBACK( clicked ), plugin );

		gtk_window_set_wmclass( GTK_WINDOW( priv->dialog ), 
				"color_wizard", "Screem" );
	
		gtk_window_set_role( GTK_WINDOW( priv->dialog ), 
				"screem_color_wisard" );

		g_signal_connect( G_OBJECT( priv->dialog ), 
				"delete_event", 
				G_CALLBACK( gtk_widget_hide ), NULL );
	}
}

/* There should be no need to change any code below here */
enum {
	ARG_0
};

static void screem_skel_plugin_class_init( ScreemSkelPluginClass *klass );
static void screem_skel_plugin_init( ScreemSkelPlugin *skel_plugin );
static void screem_skel_plugin_finalize( GObject *object );

/* G Object stuff */
#define PARENT_TYPE SCREEM_TYPE_PLUGIN

static gpointer parent_class;

static void screem_skel_plugin_class_init( ScreemSkelPluginClass *klass )
{
	GObjectClass *object_class;

	object_class = G_OBJECT_CLASS( klass );

	object_class->finalize = screem_skel_plugin_finalize;
	parent_class = g_type_class_peek_parent( klass );
}

static void screem_skel_plugin_init( ScreemSkelPlugin *skel_plugin )
{
	skel_plugin->priv = g_new0( ScreemSkelPluginPrivate, 1 );
	SCREEM_PLUGIN( skel_plugin )->setup = setup;
}

static void screem_skel_plugin_finalize( GObject *object )
{
	ScreemSkelPlugin *skel_plugin;
	ScreemSkelPluginPrivate *priv;
	
	skel_plugin = SCREEM_SKEL_PLUGIN( object );
	priv = skel_plugin->priv;

	cleanup( priv );
	
	g_free( priv );
	
	G_OBJECT_CLASS( parent_class )->finalize( object );
}

static GType screem_skel_plugin_get_type()
{
	static GType type = 0;
	
	if( ! type ) {
		static const GTypeInfo info = {
			sizeof( ScreemSkelPluginClass ),
			NULL, /* base init */
			NULL, /* base finalise */
			(GClassInitFunc)screem_skel_plugin_class_init,
			NULL, /* class finalise */
			NULL, /* class data */
			sizeof( ScreemSkelPlugin ),
			0, /* n_preallocs */
			(GInstanceInitFunc)screem_skel_plugin_init
		};

		type = g_type_register_static( PARENT_TYPE,
					       plugin_name,
					       &info, 0 );
	}

	return type;
}

static ScreemSkelPlugin *screem_skel_plugin_new( void )
{
	ScreemSkelPlugin *skel_plugin;

	skel_plugin = SCREEM_SKEL_PLUGIN( g_object_new( SCREEM_TYPE_SKEL_PLUGIN, 
				"name", plugin_name,
				NULL ) );

	return skel_plugin;
}

G_MODULE_EXPORT void get_details( ScreemPluginDetails **ret )
{
	ScreemPluginDetails *details;

	details = g_new0( ScreemPluginDetails, 1 );
	details->name = plugin_name;
	details->displayed_name = displayed_name;
	details->authors = authors;
	details->description = description;
	details->version = version;
	details->create = screem_skel_plugin_new;
	details->api_version = SCREEM_PLUGIN_REQUIRED_VERSION;

	*ret = details;
}

