/*  Screem:  skel-plugin.c
 *
 *  Copyright (C) 2004 David A Knight
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
 */
#include <config.h>

#include <glib/gi18n.h>
#include <gtk/gtk.h>

#include <glade/glade.h>
#include <glib/gunicode.h>

#include <string.h>

#include "screem-skel-plugin.h"

#include "entities.h"

/* setup your plugin here */

/* plugin name should only include a-zA-Z */
static const gchar *plugin_name = "ScreemEntityWizard";

static const gchar *authors[] = {
	"David A Knight <david@screem.org>",
	NULL
};
static const gchar *displayed_name = N_( "Screem Entity Wizard" );
static const gchar *description = N_( "A wizard for inserting entities for special characters" );
static const gchar *version = "2.0.0";

/* add any per instance data items here */
struct ScreemSkelPluginPrivate {
	GladeXML *xml;

	GtkWidget *dialog;
	GtkListStore *iso_store;
	GtkListStore *math_store;
	GtkListStore *markup_store;
};

static void entity_wizard_display( GtkAction *action, 
		gpointer user_data );
static void populate_lists( ScreemPlugin *plugin );
static void entity_row_selected( GtkTreeView *treeview,
			  GtkTreePath *path, GtkTreeViewColumn *column,
			  gpointer data );
static void response( GtkWidget *widget, gint id, gpointer data );
static void init_dialog( ScreemPlugin *plugin );

/**
 * setup:
 * 
 * this function will be called once for each window,
 * you should add any actions / ui here, eg.
 * 
 * screem_plugin_add_action( plugin, name, label, tip, stock_id,
 * 				callback, error );
 * screem_plugin_add_menu( plugin, path, action, error );
 * screem_plugin_add_toolbar( plugin, path, action, error );
 *
 *
 * to insert text into the current page being edited your callbacks
 * should make use of
 * screem_plugin_get_cursor_position( plugin )
 * screem_plugin_set_cursor_position( plugin, pos )
 * screem_plugin_insert( plugin, pos, text, length, indent )
 * 
 **/
static gboolean setup( ScreemPlugin *plugin )
{
	GError *error;
	gboolean ret;

	error = NULL;
	
	ret = screem_plugin_add_interface( plugin, "EntityWizard",
				_( "Entity Wizard" ),
				_( "Select an Entity to Insert" ),
				"Screem_Entity",
				G_CALLBACK( entity_wizard_display ),
				&error  );
	if( ! ret ) {
		g_print( "Add interface error: %s\n", error->message );
		g_error_free( error );
	}	
	return ret;

}

/**
 * cleanup:
 *
 * this function will be called once for each window when
 * it is closed, you should cleanup any data items you
 * have in ScreemSkelPluginPrivate here
 **/
static void cleanup( ScreemSkelPluginPrivate *priv )
{
	gtk_widget_destroy( priv->dialog );
	g_object_unref( priv->xml );
}

static void entity_wizard_display( GtkAction *action, gpointer user_data )
{
	ScreemPlugin *plugin;
	ScreemSkelPluginPrivate *priv;
	ScreemPage *page;
 
	plugin = SCREEM_PLUGIN( user_data );
	priv = SCREEM_SKEL_PLUGIN( plugin )->priv;
	
	page = screem_plugin_get_current_document( plugin );
        
	if( page ) {
		init_dialog( plugin );
		if( ! GTK_WIDGET_VISIBLE( priv->dialog ) ) { 
			screem_plugin_restore_from_session( plugin,
				priv->dialog );
		}
	}
	gtk_widget_show_all( priv->dialog );
	gdk_window_raise( priv->dialog->window );
}

static void populate_lists( ScreemPlugin *plugin )
{
	ScreemSkelPluginPrivate *priv;
	GtkWidget *list;
	gint num;
	GladeXML *xml;
	GtkCellRenderer *renderer;
	GtkTreeViewColumn *col;
	
	priv = SCREEM_SKEL_PLUGIN( plugin )->priv;
	
	xml = priv->xml;

	list = glade_xml_get_widget( xml, "iso_list" );
	renderer = gtk_cell_renderer_text_new();
	col = gtk_tree_view_column_new();
	gtk_tree_view_column_set_title( col, "Character" );
	gtk_tree_view_column_pack_start( col, renderer, TRUE );
	gtk_tree_view_append_column( GTK_TREE_VIEW( list ), col );
	gtk_tree_view_column_set_attributes( col, renderer, "text",
					     0, NULL );
	renderer = gtk_cell_renderer_text_new();
	col = gtk_tree_view_column_new();
	gtk_tree_view_column_set_title( col, "Entity" );
	gtk_tree_view_column_pack_start( col, renderer, TRUE );
	gtk_tree_view_append_column( GTK_TREE_VIEW( list ), col );
	gtk_tree_view_column_set_attributes( col, renderer, "text", 1, NULL );

	/* insert the entities */
	priv->iso_store = gtk_list_store_new( 2,
			G_TYPE_STRING, G_TYPE_STRING, NULL );
	for( num = 0; ISO_8859_1[ num ]; num ++ ) {
		GtkTreeIter it;

		/* ucs for ISO-8859-1 entity is 160 + num */
		gunichar c = (gunichar)( 160 + num );
		gchar outbuf[ 6 ];
		gint len;
		
		memset( outbuf, 0, 6 );
		len = g_unichar_to_utf8( c, outbuf );
		
		gtk_list_store_append( priv->iso_store, &it );

		gtk_list_store_set( priv->iso_store, &it,
				    0, outbuf,
				    1, ISO_8859_1[ num ], -1 );
	}
	gtk_tree_view_set_model( GTK_TREE_VIEW( list ),
				 GTK_TREE_MODEL( priv->iso_store ) );
	g_object_unref( priv->iso_store );
	g_signal_connect( G_OBJECT( list ), "row_activated",
			G_CALLBACK( entity_row_selected ),
			plugin );

	list = glade_xml_get_widget( xml, "math_list" );
	renderer = gtk_cell_renderer_text_new();
	col = gtk_tree_view_column_new();
	gtk_tree_view_column_set_title( col, "Character" );
	gtk_tree_view_column_pack_start( col, renderer, TRUE );
	gtk_tree_view_append_column( GTK_TREE_VIEW( list ), col );
	gtk_tree_view_column_set_attributes( col, renderer, "text",
					     0, NULL );
	renderer = gtk_cell_renderer_text_new();
	col = gtk_tree_view_column_new();
	gtk_tree_view_column_set_title( col, "Entity" );
	gtk_tree_view_column_pack_start( col, renderer, TRUE );
	gtk_tree_view_append_column( GTK_TREE_VIEW( list ), col );
	gtk_tree_view_column_set_attributes( col, renderer, "text", 1, NULL );

	/* insert the entities */
	priv->math_store = gtk_list_store_new( 2,
			G_TYPE_STRING, G_TYPE_STRING, NULL );
	for( num = 0; MATH_GREEK_SYMBOLIC[ num ]; num ++ ) {
		GtkTreeIter it;
		gunichar c = MATH_GREEK_SYMBOLIC_ucs[ num ];
		gchar outbuf[ 6 ];
		gint len;
		
		memset( outbuf, 0, 6 );
		len = g_unichar_to_utf8( c, outbuf );
		gtk_list_store_append( priv->math_store, &it );
		gtk_list_store_set( priv->math_store, &it,
				    0, outbuf,
				    1, MATH_GREEK_SYMBOLIC[ num ], -1 );
	}
	gtk_tree_view_set_model( GTK_TREE_VIEW( list ),
				 GTK_TREE_MODEL( priv->math_store ) );
	g_object_unref( priv->math_store );
	g_signal_connect( G_OBJECT( list ), "row_activated",
			G_CALLBACK( entity_row_selected ),
			plugin );

	list = glade_xml_get_widget( xml, "markup_list" );
	renderer = gtk_cell_renderer_text_new();
	col = gtk_tree_view_column_new();
	gtk_tree_view_column_set_title( col, "Character" );
	gtk_tree_view_column_pack_start( col, renderer, TRUE );
	gtk_tree_view_append_column( GTK_TREE_VIEW( list ), col );
	gtk_tree_view_column_set_attributes( col, renderer, "text",
					     0, NULL );
	renderer = gtk_cell_renderer_text_new();
	col = gtk_tree_view_column_new();
	gtk_tree_view_column_set_title( col, "Entity" );
	gtk_tree_view_column_pack_start( col, renderer, TRUE );
	gtk_tree_view_append_column( GTK_TREE_VIEW( list ), col );
	gtk_tree_view_column_set_attributes( col, renderer, "text", 1, NULL );

	/* insert the entities */
	priv->markup_store = gtk_list_store_new( 2,
			G_TYPE_STRING, G_TYPE_STRING,NULL );
	for( num = 0; MARKUP_INTL[ num ]; num ++ ) {
		GtkTreeIter it;
		gunichar c = MARKUP_INTL_ucs[ num ];
		gchar outbuf[ 6 ];
		gint len;
		
		memset( outbuf, 0, 6 );
		len = g_unichar_to_utf8( c, outbuf );

		gtk_list_store_append( priv->markup_store, &it );
		gtk_list_store_set( priv->markup_store, &it,
				    0, outbuf,
				    1, MARKUP_INTL[ num ], -1 );
	}
	gtk_tree_view_set_model( GTK_TREE_VIEW( list ),
				 GTK_TREE_MODEL( priv->markup_store ) );
	g_object_unref( priv->markup_store );
	g_signal_connect( G_OBJECT( list ), "row_activated",
			G_CALLBACK( entity_row_selected ),
			plugin );
}

static void entity_row_selected( GtkTreeView *treeview,
			  GtkTreePath *path, GtkTreeViewColumn *column,
			  gpointer data )
{
	ScreemPlugin *plugin;
	ScreemSkelPluginPrivate *priv;

	GtkTreeModel *model;
	GtkTreeSelection *sel;
	GtkTreeIter iter;
   
	gchar *ent;
	guint pos;
	gchar *text;
 
	plugin = SCREEM_PLUGIN( data );
	priv = SCREEM_SKEL_PLUGIN( plugin )->priv;
	
	sel = gtk_tree_view_get_selection( GTK_TREE_VIEW( treeview ) );
	model = gtk_tree_view_get_model( GTK_TREE_VIEW( treeview ) );

	if( sel && screem_plugin_get_current_document( plugin ) ) {
		gtk_tree_selection_get_selected( sel, &model, &iter );
		gtk_tree_model_get( model, &iter, 1, &ent, -1 );
	
		pos = screem_plugin_get_cursor_position( plugin );
		text = g_strdup_printf( "&%s;", ent );
		screem_plugin_insert( plugin, pos, text,
				strlen( text ), FALSE );
		pos += strlen( text );
		screem_plugin_set_cursor_position( plugin, pos );
		g_free( text );
		g_free( ent );
	}
}

static void response( GtkWidget *widget, gint id, gpointer data )
{
	ScreemPlugin *plugin;

	plugin = SCREEM_PLUGIN( data );
	
	screem_plugin_store_in_session( plugin, widget );
}

static void init_dialog( ScreemPlugin *plugin )
{
	ScreemSkelPluginPrivate *priv;

	priv = SCREEM_SKEL_PLUGIN( plugin )->priv;
	if( ! priv->xml ) {	
		priv->xml = glade_xml_new( GLADE_PATH"/entityWizard.glade",
				"ent_wizard", NULL );
		priv->dialog = glade_xml_get_widget( priv->xml,
				"ent_wizard" );

		populate_lists( plugin );
		
		g_signal_connect( G_OBJECT( priv->dialog ), "response",
				G_CALLBACK( response ), plugin );

		glade_xml_signal_autoconnect( priv->xml );
		
	}
}

/* There should be no need to change any code below here */
enum {
	ARG_0
};

static void screem_skel_plugin_class_init( ScreemSkelPluginClass *klass );
static void screem_skel_plugin_init( ScreemSkelPlugin *skel_plugin );
static void screem_skel_plugin_finalize( GObject *object );

/* G Object stuff */
#define PARENT_TYPE SCREEM_TYPE_PLUGIN

static gpointer parent_class;

static void screem_skel_plugin_class_init( ScreemSkelPluginClass *klass )
{
	GObjectClass *object_class;

	object_class = G_OBJECT_CLASS( klass );

	object_class->finalize = screem_skel_plugin_finalize;
	parent_class = g_type_class_peek_parent( klass );
}

static void screem_skel_plugin_init( ScreemSkelPlugin *skel_plugin )
{
	skel_plugin->priv = g_new0( ScreemSkelPluginPrivate, 1 );
	SCREEM_PLUGIN( skel_plugin )->setup = setup;
}

static void screem_skel_plugin_finalize( GObject *object )
{
	ScreemSkelPlugin *skel_plugin;
	ScreemSkelPluginPrivate *priv;
	
	skel_plugin = SCREEM_SKEL_PLUGIN( object );
	priv = skel_plugin->priv;

	cleanup( priv );
	
	g_free( priv );
	
	G_OBJECT_CLASS( parent_class )->finalize( object );
}

static GType screem_skel_plugin_get_type()
{
	static GType type = 0;
	
	if( ! type ) {
		static const GTypeInfo info = {
			sizeof( ScreemSkelPluginClass ),
			NULL, /* base init */
			NULL, /* base finalise */
			(GClassInitFunc)screem_skel_plugin_class_init,
			NULL, /* class finalise */
			NULL, /* class data */
			sizeof( ScreemSkelPlugin ),
			0, /* n_preallocs */
			(GInstanceInitFunc)screem_skel_plugin_init
		};

		type = g_type_register_static( PARENT_TYPE,
					       plugin_name,
					       &info, 0 );
	}

	return type;
}

static ScreemSkelPlugin *screem_skel_plugin_new( void )
{
	ScreemSkelPlugin *skel_plugin;

	skel_plugin = SCREEM_SKEL_PLUGIN( g_object_new( SCREEM_TYPE_SKEL_PLUGIN, 
				"name", plugin_name,
				NULL ) );

	return skel_plugin;
}

G_MODULE_EXPORT void get_details( ScreemPluginDetails **ret )
{
	ScreemPluginDetails *details;

	details = g_new0( ScreemPluginDetails, 1 );
	details->name = plugin_name;
	details->displayed_name = displayed_name;
	details->authors = authors;
	details->description = description;
	details->version = version;
	details->create = screem_skel_plugin_new;
	details->api_version = SCREEM_PLUGIN_REQUIRED_VERSION;

	*ret = details;
}

