/*  Screem:  screem-file.c
 *
 *  ScreemFile wraps up file operations
 *
 *  Copyright (C) 2004 David A Knight
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
 *
 *  For contact information with the author of this source code please see
 *  the AUTHORS file.  If there is no AUTHORS file present then check the
 *  about box under the help menu for a contact address
 */

#include <config.h>

#include <libgnomevfs/gnome-vfs.h>

#include <errno.h>
#include <fcntl.h>
#include <string.h>

#include <sys/mman.h>
#include <sys/stat.h>
#include <sys/types.h>

#include "screem-file.h"
#include "screem-errors.h"

#include "screemmarshal.h"

struct ScreemFilePrivate {
	gchar *uri;
	GnomeVFSFileInfo *info;
	ScreemFileCompressType compress;
	
	GnomeVFSAsyncHandle *handle;
	GnomeVFSResult res;
	GError *error;

	GString *data;
	gchar buffer[ 8192 ];

	gchar *mmaped;
	gint fd;
	
	guint64 total;
	guint64 current;

	guint timeout_duration;
	guint timeout;

	gboolean cancelled;
};

static void screem_file_close_cb( GnomeVFSAsyncHandle *handle,
		GnomeVFSResult result, ScreemFile *file );
static void screem_file_read_cb( GnomeVFSAsyncHandle *handle,
		GnomeVFSResult result, gchar *buffer,
		GnomeVFSFileSize requested,
		GnomeVFSFileSize read, ScreemFile *file );
static void screem_file_load_cb( GnomeVFSAsyncHandle *handle,
		GnomeVFSResult result, ScreemFile *file );


/* gobject stuff */
enum {
	PROP_0,
	PROP_URI
};

enum {
	LONG_JOB,
	ERROR,
	PROGRESS,
	COMPLETE,
	LAST_SIGNAL
};
static guint screem_file_signals[ LAST_SIGNAL ] = { 0 };

G_DEFINE_TYPE( ScreemFile, screem_file, G_TYPE_OBJECT )
	
static void screem_file_finalize( GObject *object );
static void screem_file_set_prop( GObject *object, guint prop_id,
		const GValue *value, GParamSpec *spec );
static void screem_file_get_prop( GObject *object, guint prop_id,
		GValue *value, GParamSpec *spec );

static void screem_file_class_init( ScreemFileClass *klass )
{
	GObjectClass *obj_class;
	GParamSpec *spec;
	
	obj_class = G_OBJECT_CLASS( klass );
	obj_class->finalize = screem_file_finalize;
	obj_class->get_property = screem_file_get_prop;
	obj_class->set_property = screem_file_set_prop;

	spec = g_param_spec_string( "uri", "file uri",
			"The uri for the file to operate on",
			NULL,
			G_PARAM_READWRITE | G_PARAM_CONSTRUCT );
	g_object_class_install_property( obj_class,
			PROP_URI, spec );

	screem_file_signals[ LONG_JOB ] =
		g_signal_new( "long_job",
			      G_OBJECT_CLASS_TYPE( obj_class ),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET( ScreemFileClass, long_job ),
			      NULL, NULL,
			      g_cclosure_marshal_VOID__VOID,
			      G_TYPE_NONE, 0 );
	screem_file_signals[ ERROR ] =
		g_signal_new( "error",
			      G_OBJECT_CLASS_TYPE( obj_class ),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET( ScreemFileClass, error ),
			      NULL, NULL,
			      g_cclosure_marshal_VOID__STRING,
			      G_TYPE_NONE, 1,
			      G_TYPE_STRING );
	screem_file_signals[ PROGRESS ] =
		g_signal_new( "progress",
			      G_OBJECT_CLASS_TYPE( obj_class ),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET( ScreemFileClass, progress ),
			      NULL, NULL,
			      screem_marshal_VOID__STRING_UINT64_UINT64,
			      G_TYPE_NONE, 3,
			      G_TYPE_STRING,
			      G_TYPE_UINT64,
			      G_TYPE_UINT64 );
	screem_file_signals[ COMPLETE ] =
		g_signal_new( "complete",
			      G_OBJECT_CLASS_TYPE( obj_class ),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET( ScreemFileClass, complete ),
			      NULL, NULL,
			      screem_marshal_VOID__STRING_BOOLEAN,
			      G_TYPE_NONE, 2,
			      G_TYPE_STRING,
			      G_TYPE_BOOLEAN );
}

static void screem_file_init( ScreemFile *file )
{
	ScreemFilePrivate *priv;
	
	priv = file->priv = g_new0( ScreemFilePrivate, 1 );
		
	priv->info = gnome_vfs_file_info_new();
	
	priv->timeout_duration = 2000;
}

static void screem_file_finalize( GObject *object )
{
	ScreemFile *file;
	ScreemFilePrivate *priv;
		
	g_return_if_fail( object != NULL );
	g_return_if_fail( SCREEM_IS_FILE( object ) );

	file = SCREEM_FILE( object );

	priv = file->priv;

	g_free( priv->uri );
	priv->uri = NULL;

	if( priv->timeout ) {
		g_source_remove( priv->timeout );
	}
	
	if( priv->handle ) {
		screem_file_cancel( file );
	}

	if( priv->error ) {
		g_error_free( priv->error );
	}

	if( priv->data ) {
		g_string_free( priv->data, TRUE );
	} else if( priv->mmaped ) {
		munmap( priv->mmaped, priv->total );
		priv->mmaped = NULL;	
	}
	if( priv->fd != -1 ) {
		close( priv->fd );
	}

	gnome_vfs_file_info_unref( priv->info );
	
	g_free( priv );
	
	G_OBJECT_CLASS( screem_file_parent_class )->finalize( object );
}

static void screem_file_set_prop( GObject *object, guint prop_id,
				const GValue *value, GParamSpec *spec )
{
	ScreemFile *file;
	ScreemFilePrivate *priv;
	
	file = SCREEM_FILE( object );
	priv = file->priv;

	switch( prop_id ) {
		case PROP_URI:
			g_free( priv->uri );
			priv->uri = g_value_dup_string( value );
			break;
		default:
			break;
	}
}

static void screem_file_get_prop( GObject *object, guint prop_id,
				GValue *value, GParamSpec *spec )
{
	ScreemFile *file;
	ScreemFilePrivate *priv;
	
	file = SCREEM_FILE( object );
	priv = file->priv;

	switch( prop_id ) {
		case PROP_URI:
			g_value_set_string( value, priv->uri );
			break;
		default:
			break;
	}
}

/* private */
static void screem_file_emit_error( ScreemFile *file )
{
	ScreemFilePrivate *priv;

	priv = file->priv;

	if( priv->data ) {
		g_string_free( priv->data, TRUE );
		priv->data = NULL;
	}
	g_set_error( &priv->error, 
			g_quark_from_string( SCREEM_FILE_ERROR ),
			(gint)priv->res,
			gnome_vfs_result_to_string( priv->res ) );

	g_signal_emit( G_OBJECT( file ),
			screem_file_signals[ ERROR ], 0,
			priv->uri );

	if( priv->handle ) {
		screem_file_cancel( file );
	}
}

static gboolean screem_file_provide_feedback( ScreemFile *file )
{
	file->priv->timeout = 0;

	g_signal_emit( G_OBJECT( file ),
			screem_file_signals[ LONG_JOB ], 0 );
	
	return FALSE;
}

static void screem_file_close_cb( GnomeVFSAsyncHandle *handle,
		GnomeVFSResult result, ScreemFile *file )
{
	ScreemFilePrivate *priv;

	priv = file->priv;
	
	priv->handle = NULL;

	if( ! priv->uri ) {
		g_object_unref( file );
		return;
	}

	if( priv->timeout ) {
		g_source_remove( priv->timeout );
		priv->timeout = 0;
	}
	
	/* emit complete, can't do much if the close fails */
	g_signal_emit( G_OBJECT( file ),
		       screem_file_signals[ COMPLETE ], 0,
		       priv->uri, ( priv->error == NULL ) );
	g_object_unref( file );
	priv->cancelled = FALSE;
}

static void screem_file_read_cb( GnomeVFSAsyncHandle *handle,
		GnomeVFSResult result, gchar *buffer,
		GnomeVFSFileSize requested,
		GnomeVFSFileSize read, ScreemFile *file )
{
	ScreemFilePrivate *priv;

	priv = file->priv;

	if( ! priv->uri ) {
		g_object_unref( file );
		return;
	}
	
	if( ! priv->handle ) {
		g_return_if_fail( priv->handle != NULL );
	}
	
	/* gzip method never gives EOF, assume 0 bytes is EOF,
	 * see http://bugzilla.gnome.org/show_bug.cgi?id=41173 */
	if( read == 0 && result == GNOME_VFS_OK ) {
		result = GNOME_VFS_ERROR_EOF;
	}
	
	priv->res = result;
	
	switch( result ) {
		case GNOME_VFS_OK:
			priv->current += read;
			g_string_append_len( priv->data, buffer,
					read );
			g_signal_emit( G_OBJECT( file ),
			       screem_file_signals[ PROGRESS ], 0,
			       priv->uri, priv->total, priv->current );
			gnome_vfs_async_read( handle,
					priv->buffer, 8192,
					(GnomeVFSAsyncReadCallback)screem_file_read_cb,
					file );
			break;
		case GNOME_VFS_ERROR_EOF:
			priv->current += read;
			g_string_append_len( priv->data, buffer,
					read );
			g_signal_emit( G_OBJECT( file ),
			       screem_file_signals[ PROGRESS ], 0,
			       priv->uri, priv->total, priv->current );
			gnome_vfs_async_close( handle,
					(GnomeVFSAsyncCloseCallback)screem_file_close_cb,
					file );
			break;
		default:
			/* emit error signal */
			screem_file_emit_error( file );
			g_object_unref( file );
			break;
	}
}

static void screem_file_load_cb( GnomeVFSAsyncHandle *handle,
		GnomeVFSResult result, ScreemFile *file )
{
	ScreemFilePrivate *priv;

	priv = file->priv;

	if( priv->cancelled || ! priv->uri ) {
		g_object_unref( file );
		return;
	}
	
	priv->res = result;

	switch( result ) {
		case GNOME_VFS_OK:
			/* priv->handle seems to be null in some
			 * case where the open has taken some time,
			 * despite the result being GNOME_VFS_OK,
			 * assign handle to it here to make sure */
			priv->handle = handle;
			
			priv->data = g_string_new( NULL );
			gnome_vfs_async_read( handle,
					priv->buffer, 8192,
					(GnomeVFSAsyncReadCallback)screem_file_read_cb,
					file );
			break;
		default:
			/* emit error signal */
			priv->handle = NULL;
			screem_file_emit_error( file );
			g_object_unref( file );
			break;
	}
}

static gboolean screem_file_get_info( ScreemFile *file, 
		GnomeVFSFileInfoOptions options )
{
	ScreemFilePrivate *priv;
	GnomeVFSResult res;
	gboolean ret;
	
	g_return_val_if_fail( SCREEM_IS_FILE( file ), FALSE );
	
	priv = file->priv;
	
	g_return_val_if_fail( priv->uri != NULL, FALSE );
	
	gnome_vfs_file_info_clear( priv->info );
	res = gnome_vfs_get_file_info( priv->uri, priv->info, options );

	ret = FALSE;
	if( res == GNOME_VFS_OK ) {
		GnomeVFSHandle *handle;
		gchar buffer[ 16 ];
		GnomeVFSFileSize size;
		ScreemFileCompressType compress;
		
		ret = TRUE;
		compress = SCREEM_FILE_UNCOMPRESSED;
		
		/* got file size info, open non async and read
		 * first few bytes to id gzipped files etc */
		handle = NULL;
		res = gnome_vfs_open( &handle, priv->uri,
				GNOME_VFS_OPEN_READ );
		if( res == GNOME_VFS_OK ) {
			res = gnome_vfs_read( handle, 
					buffer, 16, &size );
		}
		if( res == GNOME_VFS_OK ) {
			if( size >= 2 && 
				buffer[ 0 ] == '\037' &&
				buffer[ 1 ] == '\213' ) {
				/* gzip */
				compress = SCREEM_FILE_GZIPPED;
			} else if( size >= 3 &&
				buffer[ 0 ] == 'B' &&
				buffer[ 1 ] == 'Z' &&
				buffer[ 2 ] == 'h' ) {
				/* bzip, ignore for now */
			}
		}
		if( handle ) {
			gnome_vfs_close( handle );
		}
		priv->compress = compress;
	}
	
	return ret;
}

/* public */

ScreemFile *screem_file_new()
{
	ScreemFile *file;
	
	file = g_object_new( SCREEM_TYPE_FILE, NULL );

	return file;
}

ScreemFile *screem_file_new_with_uri( const gchar *uri )
{
	ScreemFile *file;

	file = g_object_new( SCREEM_TYPE_FILE, 
			"uri", uri, NULL );

	return file;
}

void screem_file_set_uri( ScreemFile *file, const gchar *uri )
{
	g_return_if_fail( SCREEM_IS_FILE( file ) );
	g_return_if_fail( uri != NULL );
	
	g_object_set( G_OBJECT( file ), "uri", uri, NULL );
}

const gchar *screem_file_get_uri( ScreemFile *file )
{
	g_return_val_if_fail( SCREEM_IS_FILE( file ), NULL );
	
	return file->priv->uri;
}

void screem_file_cancel( ScreemFile *file )
{
	ScreemFilePrivate *priv;
	
	g_return_if_fail( SCREEM_IS_FILE( file ) );
	
	priv = file->priv;
	
	g_return_if_fail( priv->handle != NULL );
	
	if( priv->timeout ) {
		g_source_remove( priv->timeout );
		priv->timeout = 0;
	}

	if( priv->data ) {
		g_string_free( priv->data, TRUE );
		priv->data = NULL;
	}

	if( ! priv->error ) {
		g_set_error( &priv->error, 
			g_quark_from_string( SCREEM_FILE_ERROR ),
			(gint)GNOME_VFS_ERROR_CANCELLED,
			gnome_vfs_result_to_string( GNOME_VFS_ERROR_CANCELLED ) );
	}
	
	if( priv->handle ) {
		//g_object_ref( file );
		priv->cancelled = TRUE;
		gnome_vfs_async_cancel( priv->handle );
		gnome_vfs_async_close( priv->handle,
				(GnomeVFSAsyncCloseCallback)screem_file_close_cb, 
				file );
		priv->handle = NULL;
	} else {
		screem_file_close_cb( NULL, GNOME_VFS_OK, file );
	}
}

void screem_file_load( ScreemFile *file )
{
	ScreemFilePrivate *priv;
	const gchar *uri;
	GnomeVFSFileInfoOptions options;
	gchar *openuri;
	const gchar *ptype;
	
	g_return_if_fail( SCREEM_IS_FILE( file ) );
		
	priv = file->priv;

	g_return_if_fail( priv->handle == NULL );
	
	uri = priv->uri;
	
	g_return_if_fail( uri != NULL );

	if( priv->data ) {
		g_string_free( priv->data, TRUE );
		priv->data = NULL;
	}
	
	priv->total = 0;
	priv->current = 0;

	/* get file size + compression type, non async */
	options = GNOME_VFS_FILE_INFO_DEFAULT;
	if( screem_file_get_info( file, options ) ) {
		priv->total = priv->info->size;
	
	}

	switch( priv->compress ) {
		case SCREEM_FILE_GZIPPED:
			ptype = "#ugzip:";
			break;
		case SCREEM_FILE_BZIPPED:
			ptype = "#ubzip:";
			break;
		default:
			ptype = "";
			break;
	}

	priv->handle = NULL;
	g_object_ref( file );

	if( g_str_has_prefix( uri, "file://" ) ) {
		uri += strlen( "file://" );
	}
	
	openuri = NULL;
	if( *ptype == '\0' && *uri == '/' && priv->total != 0 )  {
		/* mmap local files */
		openuri = gnome_vfs_unescape_string( uri, "/" );
		priv->fd = open( openuri, O_RDONLY );
		if( priv->fd != -1 ) {

			priv->mmaped = mmap( NULL, 
					priv->total, PROT_READ,
					MAP_SHARED, priv->fd, 0 );
			if( priv->mmaped == MAP_FAILED ) {
				/* error */
				priv->mmaped = NULL;
				close( priv->fd );
				priv->fd = -1;

			}
		} else {
			/* error */
		}
		g_signal_emit( G_OBJECT( file ),
				screem_file_signals[ COMPLETE ], 
				0, priv->uri, ( priv->error == NULL ) );
		g_object_unref( file );
	} else {
		priv->timeout = g_timeout_add_full( 
			G_PRIORITY_HIGH,
			priv->timeout_duration,
			(GSourceFunc)screem_file_provide_feedback,
			file, NULL );

		openuri = g_strconcat( uri, ptype, NULL );
		
		/* async open */
		gnome_vfs_async_open( &priv->handle,
				openuri, GNOME_VFS_OPEN_READ,
				GNOME_VFS_PRIORITY_DEFAULT,
				(GnomeVFSAsyncOpenCallback)screem_file_load_cb,
				file );
	}
	g_free( openuri );
}

GError *screem_file_get_error( ScreemFile *file )
{
	ScreemFilePrivate *priv;
	GError *error;
	
	g_return_val_if_fail( SCREEM_IS_FILE( file ), NULL );
	
	priv = file->priv;
	error = NULL;
	if( priv->error ) {
		error = g_error_copy( priv->error );
	}
	
	return error;
}

ScreemFileCompressType screem_file_get_compression_type( ScreemFile *file )
{
	g_return_val_if_fail( SCREEM_IS_FILE( file ),
			SCREEM_FILE_UNCOMPRESSED );

	return file->priv->compress;
}

const gchar *screem_file_get_data( ScreemFile *file )
{
	ScreemFilePrivate *priv;
	const gchar *ret;
	
	g_return_val_if_fail( SCREEM_IS_FILE( file ), NULL );
	
	priv = file->priv;

	g_return_val_if_fail( priv->uri != NULL, NULL );

	if( priv->data ) {
		ret = priv->data->str;
	} else if( priv->mmaped ) {
		ret = priv->mmaped;
	} else {
		ret = NULL;
	}
	
	return ret;
}

const GString *screem_file_get_data_as_string( ScreemFile *file )
{
	ScreemFilePrivate *priv;
	
	g_return_val_if_fail( SCREEM_IS_FILE( file ), NULL );
	
	priv = file->priv;

	g_return_val_if_fail( priv->uri != NULL, NULL );
	g_return_val_if_fail( priv->mmaped == NULL, NULL );

	return priv->data;
}

guint64 screem_file_get_size( const ScreemFile *file )
{
	ScreemFilePrivate *priv;
	guint64 ret;
	
	g_return_val_if_fail( SCREEM_IS_FILE( file ), 0 );
	
	priv = file->priv;
	
	if( priv->data ) {
		ret = priv->data->len;
	} else {
		g_return_val_if_fail( priv->mmaped != NULL, 0 );

		ret = priv->total;
	}

	return ret;
}

gboolean screem_file_completed( const ScreemFile *file )
{
	ScreemFilePrivate *priv;
	
	g_return_val_if_fail( SCREEM_IS_FILE( file ), TRUE );
	
	priv = file->priv;

	g_return_val_if_fail( priv->uri != NULL, TRUE );
	
	return ( priv->handle == NULL );
}

const gchar *screem_file_get_mime_type( const ScreemFile *file )
{
	ScreemFilePrivate *priv;
	const gchar *ret;
	
	g_return_val_if_fail( SCREEM_IS_FILE( file ), NULL );
	
	priv = file->priv;

	g_return_val_if_fail( priv->uri != NULL, NULL );

	ret = NULL;
	
	if( priv->info->valid_fields & GNOME_VFS_FILE_INFO_FIELDS_MIME_TYPE ) {
		ret = priv->info->mime_type;
	}

	return ret;
}

