/*  Screem:  screem-hint.c
 *
 *  The startup hint of the day class
 *
 *  Copyright (C) 2001  David A Knight
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
 *
 *  For contact information with the author of this source code please see
 *  the AUTHORS file.  If there is no AUTHORS file present then check the
 *  about box under the help menu for a contact address
 */

#include <config.h>

#include <stdio.h>
#include <string.h>

#include <gtk/gtklabel.h>
#include <gtk/gtktogglebutton.h>
#include <gtk/gtkimage.h>
#include <gtk/gtktable.h>
#include <gtk/gtkcheckbutton.h>

#include <gconf/gconf-client.h>

#include <glade/glade.h>

#include <glib/gi18n.h>

#include "screem-hint.h"

#include "support.h"

static void screem_hint_show( GtkWidget *widget );
static void screem_hint_load( ScreemHint *hint );
static void screem_hint_store( ScreemHint *hint, gchar *str );

ScreemHint *screem_hint_new()
{
	ScreemHint *hint;
	GType type;

	type = screem_hint_get_type();

	hint = SCREEM_HINT( g_object_new( type, NULL ) );

	return hint;
}

gboolean screem_hint_next_hint( GtkWidget *widget )
{
	ScreemHint *hint;
	GtkWidget *label;
	GConfClient *client;

	client = gconf_client_get_default();

	while( gtk_widget_get_parent( widget ) ) {
		widget = gtk_widget_get_parent( widget );
	}

	g_assert( SCREEM_IS_HINT( widget ) );
	hint = SCREEM_HINT( widget );

	label = hint->label;

	hint->last_hint ++;

	if( hint->last_hint >= hint->hint_count || hint->last_hint < 0 )
		hint->last_hint = 0;

	gconf_client_set_int( client, "/apps/screem/hint/number", 
			      hint->last_hint, NULL );

	gtk_label_set_text( GTK_LABEL( label ),
			hint->hint_text[ hint->last_hint ]);

	g_object_unref( client );

	return TRUE;
}

gboolean screem_hint_prev_hint( GtkWidget *widget )
{
	ScreemHint *hint;
	GtkWidget *label;
	GConfClient *client;

	client = gconf_client_get_default();

	while( gtk_widget_get_parent( widget ) ) {
		widget = gtk_widget_get_parent( widget );
	}

	g_assert( SCREEM_IS_HINT( widget ) );
	hint = SCREEM_HINT( widget );

	label = hint->label;

	hint->last_hint --;

	if( hint->last_hint < 0 )
		hint->last_hint = hint->hint_count - 1;

	gconf_client_set_int( client, "/apps/screem/hint/number", 
			      hint->last_hint, NULL );

	gtk_label_set_text( GTK_LABEL( label ), 
			hint->hint_text[ hint->last_hint ]);

	g_object_unref( client );

	return TRUE;
}

gboolean screem_hint_hide_hint( GtkWidget *widget )
{
	ScreemHint *hint;
	gint x;
	gint y;
	
	while( gtk_widget_get_parent( widget ) ) {
		widget = gtk_widget_get_parent( widget );
	}

	g_assert( SCREEM_IS_HINT( widget ) );
	hint = SCREEM_HINT( widget );
	
	gtk_window_get_position( GTK_WINDOW( hint ), &x, &y );
	gtk_widget_hide( GTK_WIDGET( hint ) );
	gtk_window_move( GTK_WINDOW( hint ), x, y );

	return TRUE;
}

gboolean screem_hint_hide_on_startup( GtkWidget *widget )
{
	GConfClient *client;
	gboolean flag;

	flag = gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) );

	client = gconf_client_get_default();
       	gconf_client_set_bool( client, "/apps/screem/general/show_tips", flag, NULL );
	g_object_unref( client );

	return TRUE;
}


/* G Object stuff */

G_DEFINE_TYPE( ScreemHint, screem_hint, GTK_TYPE_WINDOW )

static void screem_hint_finalize( GObject *hint );

static void screem_hint_class_init( ScreemHintClass *klass )
{
	GObjectClass *object_class;

	object_class = G_OBJECT_CLASS( klass );

	object_class->finalize = screem_hint_finalize;

	GTK_WIDGET_CLASS( klass )->show = screem_hint_show;
}

static void screem_hint_init( ScreemHint *hint )
{
	GladeXML *xml;
	GConfClient *client;
	GtkWidget *table;
	GtkWidget *toggle;
	GtkWidget *image;

	gchar *temp;

	gtk_window_set_title( GTK_WINDOW( hint ),
			      _( "Tip of the day - Screem" ) );

	xml = glade_xml_new( GLADE_PATH"/screem.glade", "hint_table", NULL );

	client = gconf_client_get_default();
	hint->show_hint = gconf_client_get_bool( client,
						 "/apps/screem/general/show_tips",
						 NULL );

	toggle = glade_xml_get_widget( xml, "show_tips" );
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( toggle ),
				      hint->show_hint );
	glade_xml_signal_autoconnect( xml );

	table = glade_xml_get_widget( xml, "hint_table" );
	g_object_set_data( G_OBJECT( table ), "hint", hint );
	gtk_container_add( GTK_CONTAINER( hint ), table );
	gtk_widget_show_all( table );

	hint->label = glade_xml_get_widget( xml, "hint_label" );

	image = glade_xml_get_widget( xml, "hint_icon" );
	gtk_image_set_from_file( GTK_IMAGE( image ), 
			DATADIR"/pixmaps/screem.png" );
	
	/* load hints */
	hint->hint_file = gconf_client_get_string( client,
						   "/apps/screem/hint/path",
						   NULL );
	if( ! hint->hint_file ) {
		hint->hint_file = g_strdup( HINTS_PATH );
		gconf_client_set_string( client, "/apps/screem/hint/path",
					HINTS_PATH, NULL );
	}
	temp = hint->hint_file;

	hint->hint_file = g_strconcat( temp, G_DIR_SEPARATOR_S,
					/* filename for translated hints file */
				       _( "screem_hints.txt" ),
				       NULL );

	g_free( temp );

	screem_hint_load( hint );

	gtk_window_set_default_size( GTK_WINDOW( hint ),
					450, 150 );
	gtk_window_set_type_hint( GTK_WINDOW( hint ), GDK_WINDOW_TYPE_HINT_DIALOG );
	gtk_window_set_wmclass( GTK_WINDOW( hint ),
				"Screem", "hint_window" );

	hint->last_hint = gconf_client_get_int( client,
						"/apps/screem/hint/number",
						NULL );

	g_signal_connect( G_OBJECT( hint ), "delete_event",
			G_CALLBACK( gtk_widget_hide_on_delete ),
			NULL );

	g_object_unref( G_OBJECT( client ) );
	g_object_unref( xml );
}

static void screem_hint_finalize( GObject *hint )
{
	ScreemHint *h;
	gint i;

	h = SCREEM_HINT( hint );
	
	g_free( h->hint_file );
	for( i = 0; i < h->hint_count; ++ i ) {
		g_free( h->hint_text[ i ] );
	}
	g_free( h->hint_text );

	G_OBJECT_CLASS( screem_hint_parent_class )->finalize( hint );
}

static void screem_hint_show( GtkWidget *widget )
{
	ScreemHint *hint;

	hint = SCREEM_HINT( widget );

	g_object_set_data( G_OBJECT( widget ), "hint", hint );
	screem_hint_next_hint( GTK_WIDGET( hint ) );

	gtk_window_set_position( GTK_WINDOW( hint ), 
				GTK_WIN_POS_CENTER_ON_PARENT );
	
	GTK_WIDGET_CLASS( screem_hint_parent_class )->show( widget );
}


static void screem_hint_load( ScreemHint *hint )
{
	FILE *fp;
	gchar *tip = NULL;
	gchar *str = NULL;
	gchar *temp = NULL;

	hint->hint_count = 0;
	hint->hint_text = NULL; 
	
	fp = fopen( hint->hint_file, "rt" );
	if( !fp ) {
		fp = fopen( HINTS_PATH"/screem_hints.txt", "rt" );
	}
	if( ! fp ) {
		screem_hint_store( hint, 
				   _( "\
Your SCREEM tips file appears to be missing!\n\
There should be a file called screem_hints.txt in the\n\
SCREEM data directory.  Please check your installation.") );
		return;
	}

	str = g_new( gchar, BUFSIZ + 1 );
	while( ! feof( fp ) ) {
		if( ! fgets( str, BUFSIZ, fp ) )
			continue;
   
		if( str[ 0 ] == '#' || str[ 0 ] == '\n' ) {
			if( tip ) {
				screem_hint_store( hint, tip );
				tip = NULL;
			}
		} else 	if( ! tip ) {
			tip = g_strdup( str );
		} else {
			temp = tip;
			tip = g_strconcat( tip, str, NULL );
			g_free( temp );
		}
	}
	if( tip ) {
		screem_hint_store( hint, tip );
	}
	g_free( str );
	fclose( fp );
}

static void screem_hint_store( ScreemHint *hint, gchar *str )
{
	hint->hint_count ++;
	hint->hint_text = (gchar**) g_realloc( hint->hint_text, 
					       sizeof( gchar * ) * hint->hint_count );
	hint->hint_text[ hint->hint_count - 1 ] = str;
}
