/*  Screem:  screem-tag_file.c
 *
 *  Copyright (C) 2003 David A Knight
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
 *
 *  For contact information with the author of this source code please see
 *  the AUTHORS file.  If there is no AUTHORS file present then check the
 *  about box under the help menu for a contact address
 */

#include "config.h"

#include <glib/gi18n.h>

#include "screem-tagfile.h"

#include <string.h>
#include <gtk/gtktreestore.h>
#include <gtk/gtktreemodelsort.h>
#include <gdk/gdk.h>

#include "fileops.h"

#include "support.h"

#include "screem-file.h"

#include "screem-application.h"
#include "screem-icon-cache.h"

extern ScreemApplication *app;

static gint screem_tag_file_compare_func( GtkTreeModel *model, 
				    	  GtkTreeIter *a, 
					  GtkTreeIter *b,
				    	  gpointer data );
static void screem_tag_file_icon_change( ScreemIconCache *cache, 
					gpointer data );
static void screem_tag_file_free_node_info( NodeInfo *info );
static void screem_tag_file_clear( ScreemTagFile *tagfile );
static gboolean screem_tag_file_clear_foreach( GtkTreeModel *model,
		GtkTreePath *path, GtkTreeIter *iter, gpointer data );


static void screem_tag_file_start_element( GMarkupParseContext *context,
					const gchar *element,
					const gchar **attributes,
					const gchar **values,
					gpointer data,
					GError **error );
static void screem_tag_file_end_element( GMarkupParseContext *context,
					const gchar *element,
					gpointer data,
					GError **error );
static void screem_tag_file_text( GMarkupParseContext *context,
				const gchar *text,
				gsize len,
				gpointer data,
				GError **error );
static void screem_tag_file_error( GMarkupParseContext *context,
				GError *error,
				gpointer data );
static void screem_tag_file_parse_node( ScreemTagFilePrivate *priv,
					const gchar *element,
					const gchar **attributes,
					const gchar **values );
static void screem_tag_file_parse_insert( ScreemTagFilePrivate *priv,
					  const gchar **attributes,
					  const gchar **values );

static void screem_tag_file_parse_param( ScreemTagFilePrivate *priv,
					 const gchar **attributes,
					 const gchar **values );


enum {
	PROP_0,
	PROP_ACTIVE
};

static GMarkupParser screem_parser_cbs = {
	screem_tag_file_start_element,
	screem_tag_file_end_element,
	screem_tag_file_text,
	NULL,
	screem_tag_file_error
};

typedef enum {
	PARSER_STATE_NONE = 0,
	PARSER_STATE_REFS,
	PARSER_STATE_REF,
	PARSER_STATE_GROUP,
	PARSER_STATE_ENTRY,
	PARSER_STATE_FUNCTION,
	PARSER_STATE_INSERT,
	PARSER_STATE_DESCRIPTION,
	PARSER_STATE_PARAM,
	PARSER_STATE_PROPERTY
} ScreemTagTreeParserState;

static const gchar *tag_file_tags[] = {
	"",
	"refs",
	"ref",
	"group",
	"entry",
	"function",
	"insert",
	"description",
	"param",
	"property",
};

struct ScreemTagFilePrivate {
	ScreemTagFile *tfile;
	ScreemFile *file;

	GStringChunk *chunk;
	
	gchar *filename;

	gchar *name;
	gchar *description;

	gchar *type;

	ScreemIconCache *cache;
	
	/* the unsorted browser model */
	GtkTreeModel *model;
	
	gboolean active;

	gboolean loaded;

	/* current auto complete prefix */
	const gchar *prefix;
	guint prefix_len;

	ScreemTagFileFormat format;

	GQueue *stack;
	/* current parent path */
	GtkTreePath *path;

	/* current node, so we can avoid having to look it up */
	NodeInfo *info;
	
	/* current state */
	ScreemTagTreeParserState state;
	
	/* tmp pixbuf for folder icon to avoid lots of
	 * cache lookups */
	GdkPixbuf *pixbuf;

};

G_DEFINE_TYPE( ScreemTagFile, screem_tag_file, G_TYPE_OBJECT )

static void screem_tag_file_finalize( GObject *object );
static void screem_tag_file_set_prop( GObject *object, guint prop_id,
				const GValue *value, GParamSpec *spec );
static void screem_tag_file_get_prop( GObject *object, guint prop_id,
				GValue *value, GParamSpec *spec );


static void screem_tag_file_class_init( ScreemTagFileClass *klass )
{
	GObjectClass *obj_class;
	GParamSpec *spec;

	obj_class = G_OBJECT_CLASS( klass );
	obj_class->finalize = screem_tag_file_finalize;
	obj_class->get_property = screem_tag_file_get_prop;
	obj_class->set_property = screem_tag_file_set_prop;

	spec = g_param_spec_boolean( "active", "active",
				_( "Is the Tag File active or not" ),
				TRUE, 
				G_PARAM_READABLE | G_PARAM_WRITABLE );
	g_object_class_install_property( obj_class, PROP_ACTIVE,
					spec );
}

static void screem_tag_file_init( ScreemTagFile *tag_file )
{
	ScreemTagFilePrivate *priv;
	GtkTreeStore *store;
	priv = tag_file->priv = g_new0( ScreemTagFilePrivate, 1 );
	priv->tfile = tag_file;

	/* used for attribute names function names etc.
	 * alloc a reasonable size */
	priv->chunk = g_string_chunk_new( 20 );
	
	priv->cache = screem_application_get_icon_cache( app );

	store = gtk_tree_store_new( SCREEM_TAG_FILE_COLS,
			GDK_TYPE_PIXBUF, G_TYPE_STRING, G_TYPE_POINTER );
	priv->model = GTK_TREE_MODEL( store );
	
	g_object_set_data( G_OBJECT( priv->model ), "tagfile", tag_file );
	
	gtk_tree_sortable_set_sort_func( GTK_TREE_SORTABLE( store ), 
			SCREEM_TAG_FILE_NAME,
			(GtkTreeIterCompareFunc)screem_tag_file_compare_func, 
			tag_file, NULL );
	
	g_signal_connect( G_OBJECT( priv->cache ), "changed",
		G_CALLBACK( screem_tag_file_icon_change ), tag_file );


}

static void screem_tag_file_finalize( GObject *object )
{
	ScreemTagFile *tag_file;
	ScreemTagFilePrivate *priv;
		
	g_return_if_fail( object != NULL );
	g_return_if_fail( SCREEM_IS_TAGFILE( object ) );

	tag_file = SCREEM_TAGFILE( object );

	priv = tag_file->priv;

	screem_tag_file_clear( tag_file );
	g_string_chunk_free( priv->chunk );

	g_free( priv->filename );
	g_free( priv->name );
	g_free( priv->description );
	g_free( priv->type );
	
	g_object_unref( priv->cache );

	g_object_unref( priv->model );
	
	g_free( priv );
	
	G_OBJECT_CLASS( screem_tag_file_parent_class )->finalize( object );
}

static void screem_tag_file_set_prop( GObject *object, guint prop_id,
		const GValue *value, GParamSpec *spec )
{
	ScreemTagFile *tag_file;
	ScreemTagFilePrivate *priv;
	
	tag_file = SCREEM_TAGFILE( object );
	priv = tag_file->priv;

	switch( prop_id ) {
		case PROP_ACTIVE:
			priv->active = g_value_get_boolean( value );
			if( priv->active && ! priv->loaded ) {
				screem_tag_file_load( tag_file,
						priv->format );
			}
			break;
		default:
			break;
	}
}

static void screem_tag_file_get_prop( GObject *object, guint prop_id,
		GValue *value, GParamSpec *spec )
{
	ScreemTagFile *tag_file;
	ScreemTagFilePrivate *priv;
	
	tag_file = SCREEM_TAGFILE( object );
	priv = tag_file->priv;

	switch( prop_id ) {
		case PROP_ACTIVE:
			g_value_set_boolean( value, priv->active );
			break;
		default:
			break;
	}
}

/* static stuff */
static gint screem_tag_file_compare_func( GtkTreeModel *model, 
				    	  GtkTreeIter *a, 
					  GtkTreeIter *b,
				    	  gpointer data )
{
        NodeInfo *ainfo;
        NodeInfo *binfo;
	int ret = 0;

	gtk_tree_model_get( model, a, 
			SCREEM_TAG_FILE_DATA, &ainfo, -1 );
	gtk_tree_model_get( model, b, 
			SCREEM_TAG_FILE_DATA, &binfo, -1 );

	if( ainfo && ! binfo ) {
		ret = 1;
	} else if( binfo && !ainfo ) {
		ret = -1;
	} else if( ! ainfo && ! binfo ) {
		ret = 0;
	} else {
		ret = strcmp( ainfo->name, binfo->name );
	}
	
	return ret;
}

static void screem_tag_file_foreach( GtkTreeModel *model, 
		GtkTreePath *path, GtkTreeIter *it,
		ScreemTagFile *tfile )
{
	NodeInfo *info;
	ScreemTagFilePrivate *priv;
	ScreemIconCache *cache;
	GdkPixbuf *pixbuf;
	
	priv = tfile->priv;
	cache = priv->cache;
	
	gtk_tree_model_get( model, it, 
			SCREEM_TAG_FILE_DATA, &info, -1 );
	if( info ) {
		pixbuf = NULL;
		if( info->file ) {
			pixbuf = screem_icon_cache_get_pixbuf( cache,
					info->file, NULL, 
					SCREEM_ICON_CACHE_DEFAULT_SIZE );

		} else if( gtk_tree_model_iter_has_child( model, it ) ) {
			pixbuf = priv->pixbuf;
		}
		
		gtk_tree_store_set( GTK_TREE_STORE( model ), it,
				SCREEM_TAG_FILE_ICON, pixbuf, -1 );
		if( pixbuf && info->file ) {
			g_object_unref( pixbuf );
		}
	}

}

static void screem_tag_file_icon_change( ScreemIconCache *cache, 
					gpointer data )
{
	ScreemTagFile *tfile;
	ScreemTagFilePrivate *priv;
	GtkTreeModel *model;
	
	tfile = SCREEM_TAGFILE( data );
	priv = tfile->priv;
	model = tfile->priv->model;

	priv->pixbuf = screem_icon_cache_get_pixbuf( cache,
			"/", "x-directory/normal",
			SCREEM_ICON_CACHE_DEFAULT_SIZE );

	gtk_tree_model_foreach( model, 
			(GtkTreeModelForeachFunc)screem_tag_file_foreach,
			tfile );

	if( priv->pixbuf ) {
		g_object_unref( priv->pixbuf );
		priv->pixbuf = NULL;
	}
}

static void screem_tag_file_free_node_info( NodeInfo *info )
{
	GSList *list;
	NodeAttribute *attr;

	/* info->name is in the string chunk */
	g_free( info->open );
	g_free( info->close );
	g_free( info->file );
	g_free( info->text );
	g_free( info->description );

	for( list = info->attributes; list; list = list->next ) {
		attr = (NodeAttribute*)list->data;

		/* attr->name is in the string chunk */
		
		g_free( attr->type );
		g_free( attr->defvalue );
		
		/* attr->description is in the string chunk */

		g_free( attr );
	}
	if( info->attributes ) {
		g_slist_free( info->attributes );
	}
	
	g_free( info );
}

static void screem_tag_file_clear( ScreemTagFile *tagfile )
{
	ScreemTagFilePrivate *priv;
	
	g_return_if_fail( SCREEM_IS_TAGFILE( tagfile ) );

	priv = tagfile->priv;
	
	gtk_tree_model_foreach( priv->model,
			screem_tag_file_clear_foreach,
			NULL );

	gtk_tree_store_clear( GTK_TREE_STORE( priv->model ) );

	g_string_chunk_free( priv->chunk );
	priv->chunk = g_string_chunk_new( 20 );
}

static gboolean screem_tag_file_clear_foreach( GtkTreeModel *model,
						GtkTreePath *path,
						GtkTreeIter *iter,
						gpointer data )
{
	NodeInfo *info;
	
	gtk_tree_model_get( model, iter, 
			SCREEM_TAG_FILE_DATA, &info, -1 );
	if( info ) {
		screem_tag_file_free_node_info( info );
		gtk_tree_store_set( GTK_TREE_STORE( model ), iter,
				SCREEM_TAG_FILE_DATA, NULL, -1 );
	} else {
//		g_warning( "NULL DATA COL in Tag Tree\n" );
	}

	return FALSE;
}

static void screem_tag_file_start_element( GMarkupParseContext *context,
					const gchar *element,
					const gchar **attributes,
					const gchar **values,
					gpointer data,
					GError **error )
{
	ScreemTagFile *tag_file;
	ScreemTagFilePrivate *priv;
	ScreemTagTreeParserState state;
	GQueue *stack;
	gchar *tmp;
	
	tag_file = SCREEM_TAGFILE( data );
	priv = tag_file->priv;

	stack = priv->stack;
	
	switch( priv->state ) {

		/* no parser state, valid elements == <ref> or <refs> */
		case PARSER_STATE_NONE:
			if( ! strcmp( tag_file_tags[ PARSER_STATE_REFS ],
						element ) ) {
				/* got refs state */
				state = PARSER_STATE_REFS;
				g_queue_push_head( stack,
						GINT_TO_POINTER( state ) );
				priv->state = state;
			} else if( ! strcmp( tag_file_tags[ PARSER_STATE_REF ],
					element ) ) {
				/* got ref state */


				state = PARSER_STATE_REF;
				g_queue_push_head( stack,
						GINT_TO_POINTER( state ) );
				priv->state = state;
				screem_tag_file_parse_node( priv,
						element,
						attributes,
						values );
			} else {
				/* error */
				tmp = g_strconcat( G_STRLOC, "\t", 
						element, NULL );
				g_set_error( error, 
					G_MARKUP_ERROR, 
					G_MARKUP_ERROR_UNKNOWN_ELEMENT,
					tmp );
				g_free( tmp );
			}
			
			break;

		/* <refs> allows <ref> */
		case PARSER_STATE_REFS:
			if( ! strcmp( tag_file_tags[ PARSER_STATE_REF ],
					element ) ) {
				/* got ref state */


				state = PARSER_STATE_REF;
				g_queue_push_head( stack,
						GINT_TO_POINTER( state ) );
				priv->state = state;
				screem_tag_file_parse_node( priv,
						element,
						attributes,
						values );
			} else {
				/* error */
				tmp = g_strconcat( G_STRLOC, "\t", 
						element, NULL );
				g_set_error( error, 
					G_MARKUP_ERROR, 
					G_MARKUP_ERROR_UNKNOWN_ELEMENT,
					tmp );
				g_free( tmp );
			}

			break;
			
		/* <ref> allows <group> */
		case PARSER_STATE_REF:
			if( ! strcmp( tag_file_tags[ PARSER_STATE_GROUP ],
					element ) ) {
				/* got group state */

				state = PARSER_STATE_GROUP;
				g_queue_push_head( stack,
						GINT_TO_POINTER( state ) );
				priv->state = state;
				screem_tag_file_parse_node( priv,
						element,
						attributes,
						values );
			} 
			/* not in our DTD, but simplifies the
			 * current doctype node */
			else if( ! strcmp( tag_file_tags[ PARSER_STATE_ENTRY ],
						element ) ) {
				state = PARSER_STATE_ENTRY;
				g_queue_push_head( stack,
						GINT_TO_POINTER( state ) );
				priv->state = state;
				screem_tag_file_parse_node( priv, 
						element,
						attributes,
						values );
			} else {
				/* error */
				tmp = g_strconcat( G_STRLOC, "\t", 
						element, NULL );
				g_set_error( error, 
					G_MARKUP_ERROR, 
					G_MARKUP_ERROR_PARSE,
					tmp );
				g_free( tmp );
			}

			break;
	
		/* <group> allows <group> <entry> or <function> */
		case PARSER_STATE_GROUP:
			if( ! strcmp( tag_file_tags[ PARSER_STATE_GROUP ],
					element ) ) {
				/* got group state */

				state = PARSER_STATE_GROUP;
				priv->state = state;
				g_queue_push_head( stack,
						GINT_TO_POINTER( state ) );
			} else 	if( ! strcmp( tag_file_tags[ PARSER_STATE_ENTRY ],
						element ) ) {
				/* got entry state */

				state = PARSER_STATE_ENTRY;
				priv->state = state;
				g_queue_push_head( stack,
						GINT_TO_POINTER( state ) );
			} else if( ! strcmp( tag_file_tags[ PARSER_STATE_FUNCTION ],
						element ) ) {
				/* got function state */

				state = PARSER_STATE_FUNCTION;
				priv->state = state;
				g_queue_push_head( stack,
						GINT_TO_POINTER( state ) );
			} else if( ! strcmp( tag_file_tags[ PARSER_STATE_DESCRIPTION ],
						element ) ) {
				/* we don't allow this,
				 * bluefish does it though on python
				 * ref */
				state = PARSER_STATE_DESCRIPTION;
				priv->state = state;
				g_queue_push_head( stack,
						GINT_TO_POINTER( state ) );

			} else {
				/* error */
				tmp = g_strconcat( G_STRLOC, "\t", 
						element, NULL );
				g_set_error( error, 
					G_MARKUP_ERROR, 
					G_MARKUP_ERROR_PARSE,
					tmp );
				g_free( tmp );
				break;
			}
			screem_tag_file_parse_node( priv, element,
						attributes,
						values );
			break;
	
		/* <entry> allows <insert> <description> <property> */
		case PARSER_STATE_ENTRY:
			if( ! strcmp( tag_file_tags[ PARSER_STATE_INSERT ],
					element ) ) {
				/* got insert state */
				screem_tag_file_parse_insert( priv,
						attributes,
						values );

				state = PARSER_STATE_INSERT;
				priv->state = state;
				g_queue_push_head( stack,
						GINT_TO_POINTER( state ) );
			} else 	if( ! strcmp( tag_file_tags[ PARSER_STATE_DESCRIPTION ],
						element ) ) {
				/* got description state */

				state = PARSER_STATE_DESCRIPTION;
				priv->state = state;
				g_queue_push_head( stack,
						GINT_TO_POINTER( state ) );
			} else if( ! strcmp( tag_file_tags[ PARSER_STATE_PROPERTY ],
						element ) ) {
				/* got property state */

				state = PARSER_STATE_PROPERTY;
				priv->state = state;
				g_queue_push_head( stack,
						GINT_TO_POINTER( state ) );
				screem_tag_file_parse_param( priv,
						attributes,
						values );
			} else {
				/* error */
				tmp = g_strconcat( G_STRLOC, "\t", 
						element, NULL );
				g_set_error( error, 
					G_MARKUP_ERROR, 
					G_MARKUP_ERROR_PARSE,
					tmp );
				g_free( tmp );
			}
			break;

		/* <function> allows <insert> <description> <param> */
		case PARSER_STATE_FUNCTION:
			if( ! strcmp( tag_file_tags[ PARSER_STATE_INSERT ],
					element ) ) {
				/* got insert state */
				screem_tag_file_parse_insert( priv,
						attributes,
						values );

				state = PARSER_STATE_INSERT;
				priv->state = state;
				g_queue_push_head( stack,
						GINT_TO_POINTER( state ) );
			} else 	if( ! strcmp( tag_file_tags[ PARSER_STATE_DESCRIPTION ],
						element ) ) {
				/* got description state */

				state = PARSER_STATE_DESCRIPTION;
				priv->state = state;
				g_queue_push_head( stack,
						GINT_TO_POINTER( state ) );
			} else if( ! strcmp( tag_file_tags[ PARSER_STATE_PARAM ],
						element ) ) {
				/* got param state */

				state = PARSER_STATE_PARAM;
				priv->state = state;
				g_queue_push_head( stack,
						GINT_TO_POINTER( state ) );
				screem_tag_file_parse_param( priv,
						attributes,
						values );
			} else {
				/* error, does comply with out
				 * DTD, however that is because
				 * we don't support <return> <dialog>
				 * <info> in the Bluefish reference
				 * files, ignore the error
				 * 
				g_set_error( error, 
					G_MARKUP_ERROR, 
					G_MARKUP_ERROR_PARSE,
					element );
				*/

			}
			break;

		/* <insert> allows only #PCDATA 
		 * <description> allows only #PCDATA
		 * <param> allows only #PCDATA
		 * <property> allows only #PCDATA */
		case PARSER_STATE_INSERT:
		case PARSER_STATE_DESCRIPTION:
		case PARSER_STATE_PARAM:
		case PARSER_STATE_PROPERTY:
			/* error */
			tmp = g_strconcat( G_STRLOC, "\t", element,
					NULL );
			g_set_error( error, 
				G_MARKUP_ERROR, 
				G_MARKUP_ERROR_PARSE,
				tmp );
			g_free( tmp );
			break;
	}
}

static void screem_tag_file_end_element( GMarkupParseContext *context,
					const gchar *element,
					gpointer data,
					GError **error )
{
	ScreemTagFile *tag_file;
	ScreemTagFilePrivate *priv;
	ScreemTagTreeParserState state;
	GQueue *stack;
	
	tag_file = SCREEM_TAGFILE( data );
	priv = tag_file->priv;

	stack = priv->stack;
	state = priv->state;
	
	if( ! strcmp( tag_file_tags[ state ], element ) ) {

		switch( state ) {
			case PARSER_STATE_REF:
				gtk_tree_path_free( priv->path );
				priv->path = NULL;
				break;
			case PARSER_STATE_GROUP:
			case PARSER_STATE_ENTRY:
			case PARSER_STATE_FUNCTION:
				gtk_tree_path_up( priv->path );
				break;
			default:
				/* don't move up for <refs>
				 * as no node is added to the
				 * tree for it */
				break;
		}
		g_queue_pop_head( stack );
		priv->state = GPOINTER_TO_INT( g_queue_peek_head( stack ) );
	} else {
		/* error, but as we accept Bluefish function
		 * files it is possible to receive an end
		 * tag for one of the tags we don't support,
		 * ignore the error
		g_set_error( error, 
			G_MARKUP_ERROR, 
			G_MARKUP_ERROR_PARSE,
			element );
			*/
	}
}

static void screem_tag_file_text( GMarkupParseContext *context,
				const gchar *text,
				gsize len,
				gpointer data,
				GError **error )
{
	ScreemTagFile *tag_file;
	ScreemTagFilePrivate *priv;
	NodeInfo *info;
	NodeAttribute *attr;
	gchar *tmp;

	
	tag_file = SCREEM_TAGFILE( data );
	priv = tag_file->priv;

	if( priv->path ) {
		info = priv->info;
		switch( priv->state ) {
			case PARSER_STATE_INSERT:
				info->text = g_strdup( text );
				info->text = g_strstrip( info->text );
				break;
			case PARSER_STATE_DESCRIPTION:
				info->description = g_strdup( gettext( text ) );
				info->description = g_strstrip( info->description );
				break;
			case PARSER_STATE_PARAM:
			case PARSER_STATE_PROPERTY:
				if( info->attributes ) {
					attr = (NodeAttribute*)info->attributes->data;
					if( attr ) {
						tmp = g_strdup( gettext( text ) );
						tmp = g_strstrip( tmp );
						attr->description = g_string_chunk_insert_const( priv->chunk, tmp );
						g_free( tmp );

					}
				}
				break;
			default:
				/* ignore */
				break;
		}
	}
}

static void screem_tag_file_error( GMarkupParseContext *context,
				GError *error,
				gpointer data )
{
	ScreemTagFile *tag_file;
	ScreemTagFilePrivate *priv;
	ScreemTagTreeParserState state;
	GQueue *stack;
	
	tag_file = SCREEM_TAGFILE( data );
	priv = tag_file->priv;

	stack = priv->stack;
	
	state = priv->state;
	
}

static void screem_tag_file_parse_node( ScreemTagFilePrivate *priv,
					const gchar *element,
					const gchar **attributes,
					const gchar **values )
{
	ScreemTagTreeParserState state;
	GQueue *stack;

	GtkTreeIter it;
	GtkTreeIter *parent;

	GdkPixbuf *pixbuf;
	const gchar *name;
	const gchar *description;
	const gchar *type;
	gint i;
	NodeInfo *info;
	
	stack = priv->stack;
	state = priv->state;
	
	parent = NULL;
	if( priv->path ) {
		if( ! gtk_tree_model_get_iter( priv->model, 
					&it, priv->path ) ) {
			/* error */
			return;
		}
		parent = &it;
	} else if( state != PARSER_STATE_REF ) {
		/* error, can only not have a path for <ref> */
		return;
	}

	gtk_tree_store_append( GTK_TREE_STORE( priv->model ),
					&it, parent );
	pixbuf = NULL;
	name = description = type = NULL;
	for( i = 0; attributes[ i ]; ++ i ) {
		if( ! strcmp( "name", attributes[ i ] ) ) {
			name = values[ i ];
		} else if( ! strcmp( "description", attributes[ i ] ) ) {
			description = values[ i ];
		} else if( ! strcmp( "type", attributes[ i ] ) ) {
			type = values[ i ];
		}
	}
	if( ! name ) {
		name = _( "Unknown Resource" );
	}
	
	switch( state ) {
		case PARSER_STATE_REF:
			pixbuf = priv->pixbuf;
			g_object_ref( pixbuf );
			
			g_free( priv->name );
			g_free( priv->description );
			g_free( priv->type );
			priv->name = g_strdup( gettext( name ) );
			priv->description = NULL;
			if( description ) {
				priv->description = g_strdup( gettext( description ) );
			}
			if( type ) {
				priv->type = g_strdup( type );
			}
			break;
		case PARSER_STATE_GROUP:
			pixbuf = priv->pixbuf;
			g_object_ref( pixbuf );
			break;
		case PARSER_STATE_ENTRY:
		case PARSER_STATE_FUNCTION:
			description = NULL;
			break;
		default:
			/* error */
			return;
			break;
	}
	if( priv->path ) {
		gtk_tree_path_free( priv->path );
		priv->path = NULL;

	}
	priv->path = gtk_tree_model_get_path( priv->model, &it );

	info = g_new0( NodeInfo, 1 );
	info->tfile = priv->tfile;
	info->name = g_string_chunk_insert_const( priv->chunk,
			gettext( name ) );
	if( description ) {
		info->description = g_strdup( gettext( description ) );
	}
	
	gtk_tree_store_set( GTK_TREE_STORE( priv->model ),
			&it,
			SCREEM_TAG_FILE_NAME, name,
			SCREEM_TAG_FILE_ICON, pixbuf,
			SCREEM_TAG_FILE_DATA, info,
			-1 );
	if( pixbuf ) {
		g_object_unref( pixbuf );
	}

	priv->info = info;
}

static void screem_tag_file_parse_insert( ScreemTagFilePrivate *priv,
					const gchar **attributes,
					const gchar **values )
{
	GtkTreeIter it;
	gint i;
	NodeInfo *info;
	const gchar *attr;
	const gchar *val;
	GdkPixbuf *pixbuf;
	gchar *pathname;
	
	info = NULL;
	/* get parent iter */
	if( priv->path && gtk_tree_model_get_iter( priv->model, 
				&it, priv->path ) ) {
		info = priv->info;

		info->isfunc = ( priv->state == PARSER_STATE_FUNCTION );
	
		for( i = 0; attributes[ i ]; ++ i ) {
			attr = attributes[ i ];
			val = values[ i ];
		
			if( ! strcmp( "href", attr ) ) {
				/* external reference */
				pathname = relative_to_full( val,
							RESOURCE_PATH );
				info->file = pathname;
				pixbuf = screem_icon_cache_get_pixbuf( priv->cache,
						pathname, NULL,
						SCREEM_ICON_CACHE_DEFAULT_SIZE );
				gtk_tree_store_set( GTK_TREE_STORE( priv->model ),
						&it,
						SCREEM_TAG_FILE_ICON,
						pixbuf, -1 );
			} else if( ! strcmp( "open", attr ) ) {
				/* open text */
				info->open = g_strdup( val );	
			} else if( ! strcmp( "close", attr ) ) {
				/* close text */
				info->close = g_strdup( val );
			}
		}
	}

}

static void screem_tag_file_parse_param( ScreemTagFilePrivate *priv,
					 const gchar **attributes,
					 const gchar **values )
{
	gint i;
	NodeInfo *info;
	NodeAttribute *attribute;
	const gchar *attr;
	const gchar *val;

	info = NULL;
	/* get parent iter */
	if( priv->path ) {
		info = priv->info;
	
		attribute = g_new0( NodeAttribute, 1 );
	
		for( i = 0; attributes[ i ]; ++ i ) {
			attr = attributes[ i ];
			val = values[ i ];

			if( ! strcmp( "name", attr ) ) {
				attribute->name = g_string_chunk_insert_const( priv->chunk,
						val );
			} else if( ! strcmp( "required", attr ) ) {
				attribute->required = ( *val == '1' );
			} else if( ! strcmp( "vallist", attr ) ) {
				attribute->vallist = ( *val == '1' );
			} else if( ! strcmp( "type", attr ) ) {
				attribute->type = g_strdup( val );
			} else if( ! strcmp( "default", attr ) ) {
				attribute->defvalue = g_strdup( val );
				attribute->defvalue = g_strstrip( attribute->defvalue );
			}
		}
		info->attributes = g_slist_prepend( info->attributes, 
						attribute );
	}
}

static gboolean screem_tag_file_autocomplete_fill( GtkTreeModel *model,
						GtkTreePath *path,
						GtkTreeIter *iter,
						gpointer data )
{
	ScreemTagFilePrivate *priv;
	GSList **ret;
	ScreemTagFile *tfile;
	NodeInfo *info;
	const gchar *str;
	const gchar *prefix;
	guint len;

	tfile = SCREEM_TAGFILE( g_object_get_data( G_OBJECT( model ),
						"tagfile" ) );
	priv = tfile->priv;
	prefix = priv->prefix;
	len = priv->prefix_len;
	ret = (GSList**)data;

	gtk_tree_model_get( model, iter,
			SCREEM_TAG_FILE_DATA, &info, -1 );

	str = info->open;
	if( ! str ) {
		str = info->close;
	}
	if( ! str ) {
		str = info->text;
	}
	if( str && 
	   ! strncmp( str, prefix, len ) ) {
		*ret = g_slist_prepend( *ret, g_strdup( str ) );
	}

	return FALSE;
}

static gboolean screem_tag_file_autocomplete_tip_fill( GtkTreeModel *model,
						GtkTreePath *path,
						GtkTreeIter *iter,
						gpointer data )
{
	ScreemTagFilePrivate *priv;
	GSList **ret;
	ScreemTagFile *tfile;
	NodeInfo *info;
	const gchar *str;
	const gchar *prefix;
	guint len;
	
	tfile = SCREEM_TAGFILE( g_object_get_data( G_OBJECT( model ),
						"tagfile" ) );
	priv = tfile->priv;
	prefix = priv->prefix;
	len = priv->prefix_len;
	ret = (GSList**)data;

	g_return_val_if_fail( ret != NULL, FALSE );
	
	gtk_tree_model_get( model, iter, 
			SCREEM_TAG_FILE_DATA, &info, -1 );

	str = info->tip;
	if( ! str ) {
		str = info->open;
	}
	if( ! str ) {
		str = info->close;
	}
	if( ! str ) {
		str = info->text;
	}
	if( str && 
	   ! strncmp( str, prefix, len ) ) {
		*ret = g_slist_prepend( *ret, (gchar*)str );
	}

	return FALSE;
}

static gboolean parse_from_string( const gchar *buffer, guint len,
		gpointer userdata )
{
	ScreemTagFile *tfile;
	ScreemTagFilePrivate *priv;
	GMarkupParseContext *ctx;
	gboolean ret;
	GError *error;

	tfile = SCREEM_TAGFILE( userdata );
	priv = tfile->priv;
	
	ctx = g_markup_parse_context_new( &screem_parser_cbs, 0,
			userdata, NULL );
	priv->stack = g_queue_new();
	priv->path = NULL;
	g_queue_push_head( priv->stack,
			GINT_TO_POINTER( PARSER_STATE_NONE ) );
	priv->state = PARSER_STATE_NONE;
	priv->pixbuf = screem_icon_cache_get_pixbuf( priv->cache,
			"/", "x-directory/normal",
			SCREEM_ICON_CACHE_DEFAULT_SIZE );

	ret = FALSE;
	if( ctx ) {
		error = NULL;
		ret = g_markup_parse_context_parse( ctx, buffer, len,
				&error );
		if( ret ) {
			ret = g_markup_parse_context_end_parse( ctx,
					NULL );
		} else if( error ) {
			g_print( "ERROR: %s\n",
					error->message );
		}
		g_markup_parse_context_free( ctx );
	}
	
	priv->loaded = ret;

	g_queue_free( priv->stack );
	priv->stack = NULL;
	gtk_tree_path_free( priv->path );
	priv->path = NULL;

	if( priv->pixbuf ) {
		g_object_unref( priv->pixbuf );
		priv->pixbuf = NULL;
	}
	
	return ret;
}

static void complete( ScreemFile *file, const gchar *uri, 
		gboolean success, ScreemTagFile *tagfile )
{
	const gchar *data;
	guint64 size;

	if( success ) {
		data = screem_file_get_data( file );
		if( data ) {
			size = screem_file_get_size( file );
			success = screem_tag_file_set_from_string( tagfile,
					data, size );
		}
	} else {
		GError *error;

		error = screem_file_get_error( file );
		g_print( "Error %s = %s\n", uri, error->message );
		g_error_free( error );
	}
	g_object_set( G_OBJECT( tagfile ), "active", success, NULL );
//	g_object_unref( file );
}

/* public stuff */

ScreemTagFile *screem_tag_file_new( void )
{
	ScreemTagFile *tag_file;

	tag_file = g_object_new( SCREEM_TYPE_TAGFILE, NULL );

	return tag_file;
}

void screem_tag_file_load( ScreemTagFile *tagfile,
		ScreemTagFileFormat format )
{
	ScreemTagFilePrivate *priv;
	const gchar *type;
	
	g_return_if_fail( SCREEM_IS_TAGFILE( tagfile ) );

	priv = tagfile->priv;
	
	if( format == SCREEM_TAG_UNKNOWN_FORMAT ) {
		type = screem_get_mime_type( priv->filename,
				FALSE ); 
		if( ! strcmp( "application/x-screem-tag-tree",
					type ) ) {
			format = SCREEM_TAG_TREE_FORMAT;
		} else if( ! strcmp( "application/x-weml+xml",
					type ) ) {
			format = SCREEM_TAG_WEML_FORMAT;
		} else if( ! strcmp( "text/xml", type ) ) {
			/* treat text/xml as screem tag trees,
			 * this is as bluefish uses .xml and
			 * we support those files */
			format = SCREEM_TAG_TREE_FORMAT;
		} else {
			return;
		}
	}
	priv->format = format;

	priv->file = screem_file_new_with_uri( priv->filename );
	g_signal_connect( G_OBJECT( priv->file ), "complete",
			G_CALLBACK( complete ), tagfile );
	screem_file_load( priv->file );
	
	/* unref so it will be destroyed on completion */
	g_object_unref( priv->file );
}

void screem_tag_file_load_with_uri( ScreemTagFile *tagfile,
				const gchar *uri,
				ScreemTagFileFormat format )
{
	screem_tag_file_set_uri( tagfile, uri );

	screem_tag_file_load( tagfile, format );
}

gboolean screem_tag_file_set_from_string( ScreemTagFile *tagfile,
					const gchar *str,
					guint length )
{
	ScreemTagFilePrivate *priv;
	gboolean ret;

	g_return_val_if_fail( SCREEM_IS_TAGFILE( tagfile ), FALSE );
	g_return_val_if_fail( str != NULL, FALSE );
	
	priv = tagfile->priv;
	
	if( length == 0 ) {
		length = strlen( str );
	}
	
	ret = FALSE;

	screem_tag_file_clear( tagfile );
	
	switch( priv->format ) {
		case SCREEM_TAG_WEML_FORMAT:

			break;
		default:
			ret = parse_from_string( str, length, tagfile );
			break;
	}
	
	priv->loaded = ret;
	
	return ret;	
}

GtkTreeModel *screem_tag_file_get_model( ScreemTagFile *tagfile )
{
	g_return_val_if_fail( SCREEM_IS_TAGFILE( tagfile ), NULL );

	return tagfile->priv->model;
}

const gchar *screem_tag_file_get_name( const ScreemTagFile *tagfile )
{
	g_return_val_if_fail( SCREEM_IS_TAGFILE( tagfile ), NULL );

	return tagfile->priv->name;
}

const gchar *screem_tag_file_get_desc( const ScreemTagFile *tagfile )
{
	g_return_val_if_fail( SCREEM_IS_TAGFILE( tagfile ), NULL );

	return tagfile->priv->description;
}

const gchar *screem_tag_file_get_uri( const ScreemTagFile *tagfile )
{
	g_return_val_if_fail( SCREEM_IS_TAGFILE( tagfile ), NULL );

	return tagfile->priv->filename;
}

const gchar *screem_tag_file_for_type( const ScreemTagFile *tagfile )
{
	g_return_val_if_fail( SCREEM_IS_TAGFILE( tagfile ), NULL );

	return tagfile->priv->type;
}

ScreemTagFileFormat screem_tag_file_get_format( const ScreemTagFile *tagfile )
{
	g_return_val_if_fail( SCREEM_IS_TAGFILE( tagfile ), 0 );

	return tagfile->priv->format;
}

void screem_tag_file_set_name( ScreemTagFile *tagfile, 
				const gchar *name )
{
	g_return_if_fail( SCREEM_IS_TAGFILE( tagfile ) );
	g_return_if_fail( name != NULL );

	g_free( tagfile->priv->name );
	tagfile->priv->name = g_strdup( name );
}

void screem_tag_file_set_desc( ScreemTagFile *tagfile,
				const gchar *desc )
{
	g_return_if_fail( SCREEM_IS_TAGFILE( tagfile ) );
	g_return_if_fail( desc != NULL  );

	g_free( tagfile->priv->description );
	tagfile->priv->description = g_strdup( desc );
}

void screem_tag_file_set_uri( ScreemTagFile *tagfile,
				const gchar *uri )
{
	g_return_if_fail( SCREEM_IS_TAGFILE( tagfile ) );
	g_return_if_fail( uri != NULL );

	g_free( tagfile->priv->filename );
	tagfile->priv->filename = g_strdup( uri );
}

void screem_tag_file_set_format( ScreemTagFile *tagfile,
		ScreemTagFileFormat format )
{
	g_return_if_fail( SCREEM_IS_TAGFILE( tagfile ) );

	tagfile->priv->format = format;
}

GSList *screem_tag_file_autocomplete( ScreemTagFile *tagfile,
		const gchar *prefix, gboolean fortip )
{
	ScreemTagFilePrivate *priv;
	GSList *ret;
	GtkTreeModelForeachFunc func;
	
	g_return_val_if_fail( SCREEM_IS_TAGFILE( tagfile ), NULL );
	g_return_val_if_fail( prefix != NULL, NULL );

	priv = tagfile->priv;
	
	ret = NULL;
	priv->prefix = prefix;
	priv->prefix_len = strlen( prefix );
	if( ! fortip ) {
		func = screem_tag_file_autocomplete_fill;
	} else {
		func = screem_tag_file_autocomplete_tip_fill;
	}
	gtk_tree_model_foreach( priv->model, func, &ret );

	return ret;			
}

