/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * OCL Editor                                                        *
 * Copyright (C) 2001 Steffen Zschaler.                              *
 *                                                                   *
 * This work is free software; you can redistribute it and/or        *
 * modify it under the terms of the GNU Library General Public       *
 * License as published by the Free Software Foundation; either      *
 * version 2 of the License, or (at your option) any later version.  *
 *                                                                   *
 * This work is distributed in the hope that it will be useful,      *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of    *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU *
 * Library General Public License for more details.                  *
 *                                                                   *
 * You should have received a copy of the GNU Library General Public *
 * License along with this library; if not, write to the             *
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,      *
 * Boston, MA  02111-1307, USA.                                      *
 *                                                                   *
 * See CREDITS file for further details.                             *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

// JOCLEditorApp.java -- standalone implementation of an OCL editor
//
// 04/17/2001  [sz9 ]  Created.
//
package tudresden.ocl.gui;

import tudresden.ocl.gui.events.*;
import tudresden.ocl.parser.*;
import tudresden.ocl.check.*;

import java.util.List;
import java.util.ArrayList;

import javax.swing.event.EventListenerList;

/**
 * A sample application for the OCLEditor.
 *
 * @author  sz9
 */
public class JOCLEditorApp extends javax.swing.JFrame {

  /** Creates new form JOCLEditorApp */
  public JOCLEditorApp() {
    initComponents ();
    
    m_ocleEditor.setModel (new DefaultOCLEditorModel());
  }

  /** This method is called from within the constructor to
   * initialize the form.
   * WARNING: Do NOT modify this code. The content of this method is
   * always regenerated by the FormEditor.
   */
  private void initComponents () {//GEN-BEGIN:initComponents
    m_ocleEditor = new tudresden.ocl.gui.OCLEditor ();
    setTitle ("Dresden OCL Toolkit - Sample Standalone OCL Editor");
    setDefaultCloseOperation (javax.swing.WindowConstants.DISPOSE_ON_CLOSE);
    addWindowListener (new java.awt.event.WindowAdapter () {
      public void windowClosing (java.awt.event.WindowEvent evt) {
        exitForm (evt);
      }
    }
    );
    
    
    getContentPane ().add (m_ocleEditor, java.awt.BorderLayout.CENTER);
    
    pack ();
    java.awt.Dimension screenSize = java.awt.Toolkit.getDefaultToolkit ().getScreenSize ();
    java.awt.Dimension dialogSize = getSize ();
    setSize (new java.awt.Dimension (300, 200));
    setLocation ((screenSize.width-300)/2,(screenSize.height-200)/2);
  }//GEN-END:initComponents

  /** Exit the Application */
    private void exitForm(java.awt.event.WindowEvent evt) {//GEN-FIRST:event_exitForm
      System.exit (0);
    }//GEN-LAST:event_exitForm

  /**
  * @param args the command line arguments
  */
  public static void main (String args[]) {
    new JOCLEditorApp ().setVisible (true);
  }
  
  public static class DefaultOCLEditorModel implements OCLEditorModel {
    
    public class DefaultConstraintRepresentation
        implements ConstraintRepresentation {

      private String m_sName;
      private String m_sData;
          
      public DefaultConstraintRepresentation (String sName, String sData) {
        super();
        
        m_sName = sName;
        m_sData = sData;
      }
      
      /**
       * Get the constraint's body text.
       */
      public String getData() {
        return ((m_sData != null)?
                (m_sData):(""));
      }

      /**
       * Get the constraint's name.
       */
      public String getName() {
        return ((m_sName != null)?
                (m_sName):("new_constraint"));
      }

      /**
       * Set the constraint's body text. For the exceptions the detailed message must
       * be human readable.
       *
       * @param sData the new body of the constraint
       *
       * @exception IllegalStateException if the constraint is not in a state to
       *    accept body changes.
       * @exception OclParserException if the specified constraint is not
       *    syntactically correct.
       * @exception OclTypeException if the specified constraint does not adhere by
       *    OCL type rules.
       */
      public void setData(String sData,EditingUtilities euHelper) throws IllegalStateException, OclParserException, OclTypeException {
            
        String sOldData = m_sData;
        m_sData = sData;
        fireConstraintDataChanged (m_lcrConstraints.indexOf (this),
                                   sOldData, this);
      }

      /**
       * Set the constraint's name. For the exceptions the detailed message must
       * be human readable.
       *
       * @param sName the new name of the constraint
       *
       * @exception IllegalStateException if the constraint is not in a state to
       *    accept name changes.
       * @exception IllegalArgumentException if the specified name is not a legal
       *    name for the constraint.
       */
      public void setName (final String sName,
                           final EditingUtilities euHelper)
          throws IllegalStateException, IllegalArgumentException {
        if (m_sData == null) {
          throw new IllegalStateException ("Please specify a constraint body first.");
        }
        
        // Check name for consistency with spec
        if (!euHelper.isValidConstraintName (sName)) {
          throw new IllegalArgumentException ("Please specify a valid name.");
        }

        // Set name
        String sOldName = getName();
        m_sName = sName;
        
        fireConstraintNameChanged (m_lcrConstraints.indexOf (this),
                                   sOldName, this);
      }
    }
    
    private List m_lcrConstraints = new ArrayList();
    
    /**
     * List of listeners.
     */
    private EventListenerList m_ellListeners = new EventListenerList();

    /**
     * Return the number of constraints in this model.
     */
    public int getConstraintCount() {
      return m_lcrConstraints.size();
    }
    
    /**
     * Return the constraint with the specified index.
     *
     * @param nIdx the index of the constraint to be returned. 0 <= nIdx < {@link #getConstraintCount}
     */
    public ConstraintRepresentation getConstraintAt (int nIdx) {
      return (ConstraintRepresentation) m_lcrConstraints.get (nIdx);
    }
    
    /**
     * Add a fresh constraint to the model.
     */
    public void addConstraint() {
      m_lcrConstraints.add (new DefaultConstraintRepresentation (null, null));
      fireConstraintAdded();
    }
    
    /**
     * Remove the specified constraint from the model.
     *
     * @param nIdx the index of the constraint to be removed. 0 <= nIdx < {@link #getConstraintCount}
     */
    public void removeConstraintAt (int nIdx) {
      ConstraintRepresentation cr = (ConstraintRepresentation)
          m_lcrConstraints.remove (nIdx);
      
      if (cr != null) {
        fireConstraintRemoved (cr, nIdx);
      }
    }
    
    /**
      * Add a listener to be informed of changes in the model.
      *
      * @param ccl the new listener
      */
    public void addConstraintChangeListener (ConstraintChangeListener ccl) {
      m_ellListeners.add (ConstraintChangeListener.class, ccl);
    }
    
    /**
      * Remove a listener to be informed of changes in the model.
      *
      * @param ccl the listener to be removed
      */
    public void removeConstraintChangeListener (ConstraintChangeListener ccl) {
      m_ellListeners.remove (ConstraintChangeListener.class, ccl);
    }
    
    protected void fireConstraintRemoved (ConstraintRepresentation cr,
                                          int nIdx) {
      // Guaranteed to return a non-null array
      Object[] listeners = m_ellListeners.getListenerList();
    
      ConstraintChangeEvent cce = null;
      
      // Process the listeners last to first, notifying
      // those that are interested in this event
      for (int i = listeners.length - 2; i >= 0; i -= 2) {
        if (listeners[i] == ConstraintChangeListener.class) {
          // Lazily create the event:
          if (cce == null) {
            cce = new ConstraintChangeEvent (this,
                                             nIdx,
                                             cr,
                                             null);
          }
          
          ((ConstraintChangeListener) listeners[i + 1]).constraintRemoved (cce);
        }
      }
    }

    protected void fireConstraintAdded() {
      // Guaranteed to return a non-null array
      Object[] listeners = m_ellListeners.getListenerList();
    
      ConstraintChangeEvent cce = null;
      
      // Process the listeners last to first, notifying
      // those that are interested in this event
      for (int i = listeners.length - 2; i >= 0; i -= 2) {
        if (listeners[i] == ConstraintChangeListener.class) {
          // Lazily create the event:
          if (cce == null) {
            int nIdx = m_lcrConstraints.size() - 1;
            cce = new ConstraintChangeEvent (this,
                                             nIdx,
                                             null,
                                             getConstraintAt (nIdx));
          }
          
          ((ConstraintChangeListener) listeners[i + 1]).constraintAdded (cce);
        }
      }
    }

    protected void fireConstraintDataChanged (int nIdx,
                                              String sOldConstraint,
                                              ConstraintRepresentation crNew) {
      // Guaranteed to return a non-null array
      Object[] listeners = m_ellListeners.getListenerList();
    
      ConstraintChangeEvent cce = null;
      
      // Process the listeners last to first, notifying
      // those that are interested in this event
      for (int i = listeners.length - 2; i >= 0; i -= 2) {
        if (listeners[i] == ConstraintChangeListener.class) {
          // Lazily create the event:
          if (cce == null) {
            cce = new ConstraintChangeEvent (this,
                                             nIdx,
                                             new DefaultConstraintRepresentation
                                               (crNew.getName(), sOldConstraint),
                                             crNew);
          }
          
          ((ConstraintChangeListener) listeners[i + 1]).constraintDataChanged (cce);
        }
      }
    }
          
    protected void fireConstraintNameChanged (int nIdx,
                                              String sOldName,
                                              ConstraintRepresentation crNew) {
      // Guaranteed to return a non-null array
      Object[] listeners = m_ellListeners.getListenerList();
    
      ConstraintChangeEvent cce = null;
      
      // Process the listeners last to first, notifying
      // those that are interested in this event
      for (int i = listeners.length - 2; i >= 0; i -= 2) {
        if (listeners[i] == ConstraintChangeListener.class) {
          // Lazily create the event:
          if (cce == null) {
            cce = new ConstraintChangeEvent (this,
                                             nIdx,
                                             new DefaultConstraintRepresentation
                                                (sOldName, crNew.getData()),
                                             crNew);
          }
          
          ((ConstraintChangeListener) listeners[i + 1]).constraintNameChanged (cce);
        }
      }
    }
  }

  // Variables declaration - do not modify//GEN-BEGIN:variables
  private tudresden.ocl.gui.OCLEditor m_ocleEditor;
  // End of variables declaration//GEN-END:variables
}