// ==============================================================
//	This file is part of Glest (www.glest.org)
//
//	Copyright (C) 2001-2005 Martio Figueroa
//
//	You can redistribute this code and/or modify it under 
//	the terms of the GNU General Public License as published 
//	by the Free Software Foundation; either version 2 of the 
//	License, or (at your option) any later version
// ==============================================================

#include "menu_state_join_game.h"

#include "renderer.h"
#include "sound_renderer.h"
#include "core_data.h"
#include "config.h"
#include "menu_state_root.h"
#include "metrics.h"
#include "network_manager.h"
#include "network_message.h"
#include "client_interface.h"
#include "conversion.h"
#include "game.h"
#include "socket.h"

#include "leak_dumper.h"

namespace Glest{ namespace Game{

using namespace Shared::Util;

// ===============================
// 	class MenuStateJoinGame  
// ===============================

MenuStateJoinGame::MenuStateJoinGame(Program *program, MainMenu *mainMenu):
	MenuState(program, mainMenu, "join-game")
{
	Lang &lang= Lang::getInstance();
	Config &config= Config::getInstance();
	NetworkManager &networkManager= NetworkManager::getInstance();

	buttonReturn.init(325, 300, 125);
	buttonReturn.setText(lang.get("Return"));

	buttonConnect.init(475, 300, 125);
	buttonConnect.setText(lang.get("Connect"));

	labelServerIp.init(480, 430);
	labelServerIp.setText(config.getString("ServerIp") + "_");

	labelServer.init(400, 430);
	labelServer.setText(lang.get("Server") + ": " + serverName);

	labelStatus.init(400, 400);
	labelStatus.setText("");

	labelInfo.init(400, 370);
	labelInfo.setText("");

	networkManager.init(nrClient);
	connected= false;
	playerIndex= -1;
}

void MenuStateJoinGame::mouseClick(int x, int y, MouseButton mouseButton){

	CoreData &coreData= CoreData::getInstance();
	SoundRenderer &soundRenderer= SoundRenderer::getInstance();
	NetworkManager &networkManager= NetworkManager::getInstance();
	Config &config= Config::getInstance();

	if(buttonReturn.mouseClick(x, y)){
		soundRenderer.playFx(coreData.getClickSoundA());
		mainMenu->setState(new MenuStateRoot(program, mainMenu));
    }  
	else if(buttonConnect.mouseClick(x, y)){
		ClientInterface* clientInterface= networkManager.getClientInterface();

		soundRenderer.playFx(coreData.getClickSoundA());
		labelInfo.setText("");
			
		if(clientInterface->isConnected()){
			clientInterface->reset();
		}
		else{
			Ip serverIp(labelServerIp.getText());

			clientInterface->connect(serverIp, GameConstants::serverPort);
			labelServerIp.setText(serverIp.getString()+'_');
			labelInfo.setText("");

			// save server ip
			config.setString("ServerIp", serverIp.getString());
			config.save();
		}
	}
}

void MenuStateJoinGame::mouseMove(int x, int y, const MouseState *ms){
	buttonReturn.mouseMove(x, y);
	buttonConnect.mouseMove(x, y);
}

void MenuStateJoinGame::render(){
	Renderer &renderer= Renderer::getInstance();

	renderer.renderButton(&buttonReturn);
	renderer.renderLabel(&labelServer);
	renderer.renderLabel(&labelServerIp);
	renderer.renderLabel(&labelStatus);
	renderer.renderLabel(&labelInfo);
	renderer.renderButton(&buttonConnect);
}

void MenuStateJoinGame::update(){
	ClientInterface* clientInterface= NetworkManager::getInstance().getClientInterface();
	Lang &lang= Lang::getInstance();

	//update status label
	if(clientInterface->isConnected()){
		buttonConnect.setText(lang.get("Disconnect"));
		if(!serverName.empty()){
			labelStatus.setText(lang.get("ConnectedToServer") + " " + serverName);
		}
		else{
			labelStatus.setText(lang.get("ConnectedToServer"));
		}
	}
	else{
		buttonConnect.setText(lang.get("Connect"));
		labelStatus.setText(lang.get("NotConnected"));
		labelInfo.setText("");
	}

	//process network messages
	if(clientInterface->isConnected()){
		NetworkMessageType networkMessageType= clientInterface->getNextMessageType();
		
		if(networkMessageType!=nmtInvalid){
			//intro message
			switch(networkMessageType){
				case nmtIntro:{
					NetworkMessageIntro networkMessageIntro;

					if(clientInterface->receiveMessage(&networkMessageIntro)){
						
						//check consistency
						if(networkMessageIntro.getVersionString()!=getNetworkVersionString()){
							throw runtime_error("Server and client versions do not match (" + networkMessageIntro.getVersionString() + ")");
						}	

						//send intro message
						NetworkMessageIntro sendNetworkMessageIntro(glestVersionString, clientInterface->getHostName(), -1);

						labelInfo.setText(lang.get("WaitingHost"));
						playerIndex= networkMessageIntro.getPlayerIndex();
						serverName= networkMessageIntro.getName();
						clientInterface->sendMessage(&sendNetworkMessageIntro);
							
						assert(playerIndex>=0 && playerIndex<GameConstants::maxPlayers);
					}
				}
				break;

				case nmtLaunch:{
					NetworkMessageLaunch networkMessageLaunch;
					GameSettings gameSettings;

					if(clientInterface->receiveMessage(&networkMessageLaunch)){
						networkMessageLaunch.buildGameSettings(&gameSettings);

						//replace server player by network
						for(int i= 0; i<gameSettings.getFactionCount(); ++i){
							
							//replace by network
							if(gameSettings.getFactionControl(i)==ctHuman){
								gameSettings.setFactionControl(i, ctNetwork);
							}

							//set the faction index
							if(gameSettings.getStartLocationIndex(i)==playerIndex){
								gameSettings.setThisFactionIndex(i);
							}
						}
						program->setState(new Game(program, &gameSettings)); 
					}
					
				}
				break;
				
				//throw exception if we receive an unexpected message
				default:
					throw runtime_error("Invalid network message: " + intToStr(networkMessageType));
			}
		}
	}
}

void MenuStateJoinGame::keyDown(char key){
	ClientInterface* clientInterface= NetworkManager::getInstance().getClientInterface();
	
	if(!clientInterface->isConnected()){
		int maxTextSize= 16;

		if(key>='0' && key<='9'){

			if(labelServerIp.getText().size()<maxTextSize){
				string text= labelServerIp.getText();

				text.insert(text.end()-1, key);

				labelServerIp.setText(text);
			}
		}
		else if(key==vkBack){
			string text= labelServerIp.getText();

			if(text.size()>1){
				text.erase(text.end()-2);
			}

			labelServerIp.setText(text);
		}
		else{
			if(labelServerIp.getText().size()<maxTextSize){
				string text= labelServerIp.getText();

				text.insert(text.end()-1, '.');

				labelServerIp.setText(text);
			}
		}
	}
}

}}//end namespace
