/* -*- Mode: C; indent-tabs-mode: t; c-basic-offset: 4; tab-width: 4 -*- */
/*
 * playlist.c
 * Copyright (C) Kevin DeKorte 2006 <kdekorte@gmail.com>
 * 
 * playlist.c is free software.
 * 
 * You may redistribute it and/or modify it under the terms of the
 * GNU General Public License, as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option)
 * any later version.
 * 
 * playlist.c is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with playlist.c.  If not, write to:
 * 	The Free Software Foundation, Inc.,
 * 	51 Franklin Street, Fifth Floor
 * 	Boston, MA  02110-1301, USA.
 */

#include "playlist.h"
#include "common.h"
#include "gui.h"
#include "support.h"


void update_gui()
{
    if (gtk_tree_model_iter_n_children(GTK_TREE_MODEL(playliststore), NULL) < 2) {
        gtk_widget_hide(prev_event_box);
        gtk_widget_hide(next_event_box);
        gtk_widget_set_sensitive(GTK_WIDGET(menuitem_edit_random), FALSE);
    } else {
        gtk_widget_show(prev_event_box);
        gtk_widget_show(next_event_box);
        gtk_widget_set_sensitive(GTK_WIDGET(menuitem_edit_random), TRUE);
    }
}

gboolean playlist_popup_handler(GtkWidget * widget, GdkEvent * event, void *data)
{
    GtkMenu *menu;
    GdkEventButton *event_button;

    menu = GTK_MENU(widget);
    gtk_widget_grab_focus(fixed);
    if (event->type == GDK_BUTTON_PRESS) {

        event_button = (GdkEventButton *) event;

        if (event_button->button == 3) {
            gtk_menu_popup(menu, NULL, NULL, NULL, NULL, event_button->button, event_button->time);
            return TRUE;
        }
    }
    return FALSE;
}

void playlist_set_subtitle_callback(GtkMenuItem * menuitem, void *data)
{
    GtkTreeSelection *sel;
    GtkTreeView *view = (GtkTreeView *) data;
    GtkTreeIter localiter;
    gchar *subtitle;
    GtkWidget *dialog;

    sel = gtk_tree_view_get_selection(view);

    if (gtk_tree_selection_get_selected(sel, NULL, &localiter)) {

        dialog = gtk_file_chooser_dialog_new(_("Set Subtitle"),
                                             GTK_WINDOW(window),
                                             GTK_FILE_CHOOSER_ACTION_OPEN,
                                             GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
                                             GTK_STOCK_OPEN, GTK_RESPONSE_ACCEPT, NULL);
        if (gtk_dialog_run(GTK_DIALOG(dialog)) == GTK_RESPONSE_ACCEPT) {
            subtitle = gtk_file_chooser_get_filename(GTK_FILE_CHOOSER(dialog));
            gtk_list_store_set(playliststore, &localiter, SUBTITLE_COLUMN, subtitle, -1);
        }
        gtk_widget_destroy(dialog);
    }

}


gboolean playlist_drop_callback(GtkWidget * widget, GdkDragContext * dc,
                                gint x, gint y, GtkSelectionData * selection_data,
                                guint info, guint t, gpointer data)
{
    gchar *filename;
    GtkTreeIter localiter;
    gchar **list;
    gint i = 0;
    gint playlist;

    /* Important, check if we actually got data.  Sometimes errors
     * occure and selection_data will be NULL.
     */
    if (selection_data == NULL)
        return FALSE;
    if (selection_data->length < 0)
        return FALSE;

    if ((info == DRAG_INFO_0) || (info == DRAG_INFO_1) || (info == DRAG_INFO_2)) {
        filename = g_filename_from_uri((const gchar *) selection_data->data, NULL, NULL);
        list = g_strsplit(filename, "\n", 0);

        while (list[i] != NULL) {
            g_strchomp(list[i]);
            if (strlen(list[i]) > 0) {
printf("filename = '%s'\n",list[i]);
                playlist = detect_playlist(list[i]);

                if (!playlist) {
                    add_item_to_playlist(list[i], playlist);
                } else {
                    if (!parse_playlist(list[i])) {
                        localiter = add_item_to_playlist(list[i], playlist);
                    }
                }

            }
            i++;
        }

        if (gtk_tree_model_iter_n_children(GTK_TREE_MODEL(playliststore), NULL) < 2) {
            gtk_tree_model_get_iter_first(GTK_TREE_MODEL(playliststore), &iter);
            gtk_tree_model_get(GTK_TREE_MODEL(playliststore), &iter, ITEM_COLUMN, &filename,
                               PLAYLIST_COLUMN, &playlist, -1);

            shutdown();
            play_file(filename, playlist);
            g_free(filename);

        } else {
            gtk_widget_set_sensitive(GTK_WIDGET(menuitem_edit_random), TRUE);
        }
        g_strfreev(list);
    }
    update_gui();
    return FALSE;
}

gboolean playlist_motion_callback(GtkWidget * widget, GdkEventMotion * event, gpointer user_data)
{
    GtkTreePath *localpath;
    GtkTreeIter localiter;
    gchar *iterfilename;
    gchar *itersubtitle;
    gchar *tip;

    gtk_tree_view_get_path_at_pos(GTK_TREE_VIEW(widget), event->x, event->y, &localpath, NULL, NULL,
                                  NULL);

    if (localpath != NULL) {
        if (gtk_tree_model_get_iter(GTK_TREE_MODEL(playliststore), &localiter, localpath)) {
            gtk_tree_model_get(GTK_TREE_MODEL(playliststore), &localiter, ITEM_COLUMN,
                               &iterfilename, SUBTITLE_COLUMN, &itersubtitle, -1);
            if (itersubtitle == NULL) {
                tip = g_strdup_printf("%s", iterfilename);
            } else {
                tip = g_strdup_printf(_("Filename: %s\nSubtitle: %s"), iterfilename, itersubtitle);
            }
            gtk_tooltips_set_tip(playlisttip, widget, tip, NULL);
        }
    }
    return FALSE;

}


void save_playlist(GtkWidget * widget, void *data)
{
    GtkWidget *dialog;
    gchar *filename;
    gchar *new_filename;
    GtkFileFilter *filter;
    GConfClient *gconf;
    gchar *last_dir;

    dialog = gtk_file_chooser_dialog_new(_("Save Playlist"),
                                         GTK_WINDOW(window),
                                         GTK_FILE_CHOOSER_ACTION_SAVE,
                                         GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
                                         GTK_STOCK_SAVE, GTK_RESPONSE_ACCEPT, NULL);
    gconf = gconf_client_get_default();
    last_dir = gconf_client_get_string(gconf, LAST_DIR, NULL);
    if (last_dir != NULL)
        gtk_file_chooser_set_current_folder(GTK_FILE_CHOOSER(dialog), last_dir);

    gtk_file_chooser_set_do_overwrite_confirmation(GTK_FILE_CHOOSER(dialog), TRUE);
    filter = gtk_file_filter_new();
    gtk_file_filter_set_name(filter, _("Playlist (*.pls)"));
    gtk_file_filter_add_pattern(filter, "*.pls");
    gtk_file_chooser_add_filter(GTK_FILE_CHOOSER(dialog), filter);

    filter = gtk_file_filter_new();
    gtk_file_filter_set_name(filter, _("MP3 Playlist (*.m3u)"));
    gtk_file_filter_add_pattern(filter, "*.m3u");
    gtk_file_chooser_add_filter(GTK_FILE_CHOOSER(dialog), filter);

    if (gtk_dialog_run(GTK_DIALOG(dialog)) == GTK_RESPONSE_ACCEPT) {

        filename = gtk_file_chooser_get_filename(GTK_FILE_CHOOSER(dialog));
        last_dir = gtk_file_chooser_get_current_folder(GTK_FILE_CHOOSER(dialog));
        gconf_client_set_string(gconf, LAST_DIR, last_dir, NULL);
        g_free(last_dir);

        if (g_strrstr(filename, ".m3u") != NULL) {
            save_playlist_m3u(filename);
        }
        if (g_strrstr(filename, ".pls") != NULL) {
            save_playlist_pls(filename);
        }

        if (g_strrstr(filename, ".") == NULL) {
            new_filename = g_strdup_printf("%s.pls", filename);
            save_playlist_pls(new_filename);
            g_free(new_filename);
        }
        g_free(filename);
    }

    gtk_widget_destroy(dialog);

}

void load_playlist(GtkWidget * widget, void *data)
{
    GtkWidget *dialog;
    gchar *filename;
    GtkFileFilter *filter;
    GConfClient *gconf;
    gchar *last_dir;

    dialog = gtk_file_chooser_dialog_new(_("Open Playlist"),
                                         GTK_WINDOW(window),
                                         GTK_FILE_CHOOSER_ACTION_OPEN,
                                         GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
                                         GTK_STOCK_OPEN, GTK_RESPONSE_ACCEPT, NULL);

    gconf = gconf_client_get_default();
    last_dir = gconf_client_get_string(gconf, LAST_DIR, NULL);
    if (last_dir != NULL)
        gtk_file_chooser_set_current_folder(GTK_FILE_CHOOSER(dialog), last_dir);

    filter = gtk_file_filter_new();
    gtk_file_filter_set_name(filter, _("Playlist (*.pls)"));
    gtk_file_filter_add_pattern(filter, "*.pls");
    gtk_file_chooser_add_filter(GTK_FILE_CHOOSER(dialog), filter);

    filter = gtk_file_filter_new();
    gtk_file_filter_set_name(filter, _("Reference Playlist (*.ref)"));
    gtk_file_filter_add_pattern(filter, "*.ref");
    gtk_file_chooser_add_filter(GTK_FILE_CHOOSER(dialog), filter);

    filter = gtk_file_filter_new();
    gtk_file_filter_set_name(filter, _("MP3 Playlist (*.m3u)"));
    gtk_file_filter_add_pattern(filter, "*.m3u");
    gtk_file_chooser_add_filter(GTK_FILE_CHOOSER(dialog), filter);

    if (gtk_dialog_run(GTK_DIALOG(dialog)) == GTK_RESPONSE_ACCEPT) {

        filename = gtk_file_chooser_get_filename(GTK_FILE_CHOOSER(dialog));
        last_dir = gtk_file_chooser_get_current_folder(GTK_FILE_CHOOSER(dialog));
        gconf_client_set_string(gconf, LAST_DIR, last_dir, NULL);
        g_free(last_dir);

        gtk_list_store_clear(playliststore);
        gtk_list_store_clear(nonrandomplayliststore);

        if (!parse_playlist(filename)) {
            add_item_to_playlist(filename, 1);
        }

    }
    update_gui();
    gtk_widget_destroy(dialog);
}

void add_item_to_playlist_callback(gpointer data, gpointer user_data)
{
    gchar *filename = (gchar *) data;
    gint playlist;

    playlist = detect_playlist(filename);
    if (!playlist) {
        add_item_to_playlist(filename, playlist);
    } else {
        if (!parse_playlist(filename)) {
            add_item_to_playlist(filename, playlist);
        }
    }

}

gint compar(gpointer a, gpointer b)
{
    return strcasecmp((char *) a, (char *) b);
}

void add_folder_to_playlist_callback(gpointer data, gpointer user_data)
{
    gchar *filename = (gchar *) data;
    GDir *dir;
    gchar *name;
    gchar *subdir;
    GSList *list = NULL;

    dir = g_dir_open(filename, 0, NULL);
    if (dir != NULL) {
        do {
            name = g_strdup(g_dir_read_name(dir));
            if (name != NULL) {
                subdir = g_strdup_printf("%s/%s", filename, name);
                if (g_file_test(subdir, G_FILE_TEST_IS_DIR)) {
                    add_folder_to_playlist_callback((gpointer) subdir, NULL);
                } else {
                    list = g_slist_prepend(list, subdir);
                    filecount++;
                }
                g_free(name);
            } else {
                break;
            }
        } while (TRUE);
        list = g_slist_sort(list, (GCompareFunc) compar);
        g_slist_foreach(list, &add_item_to_playlist_callback, NULL);
        g_slist_free(list);
        g_dir_close(dir);
    }

}

void add_to_playlist(GtkWidget * widget, void *data)
{

    GtkWidget *dialog;
    GSList *filename;
    GConfClient *gconf;
    gchar *last_dir;
    GtkTreeViewColumn *column;
    gchar *coltitle;
	gint count;
	
    dialog = gtk_file_chooser_dialog_new(_("Open File"),
                                         GTK_WINDOW(window),
                                         GTK_FILE_CHOOSER_ACTION_OPEN,
                                         GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
                                         GTK_STOCK_OPEN, GTK_RESPONSE_ACCEPT, NULL);

    /*allow multiple files to be selected */
    gtk_file_chooser_set_select_multiple(GTK_FILE_CHOOSER(dialog), TRUE);

    gconf = gconf_client_get_default();
    last_dir = gconf_client_get_string(gconf, LAST_DIR, NULL);
    if (last_dir != NULL) {
        gtk_file_chooser_set_current_folder(GTK_FILE_CHOOSER(dialog), last_dir);
        g_free(last_dir);
    }

    if (gtk_dialog_run(GTK_DIALOG(dialog)) == GTK_RESPONSE_ACCEPT) {

        filename = gtk_file_chooser_get_filenames(GTK_FILE_CHOOSER(dialog));

        last_dir = gtk_file_chooser_get_current_folder(GTK_FILE_CHOOSER(dialog));
        gconf_client_set_string(gconf, LAST_DIR, last_dir, NULL);
        g_free(last_dir);

        g_slist_foreach(filename, &add_item_to_playlist_callback, NULL);
        g_slist_free(filename);
    }
    update_gui();
    g_object_unref(G_OBJECT(gconf));
    column = gtk_tree_view_get_column(GTK_TREE_VIEW(list), 0);
	count = gtk_tree_model_iter_n_children(GTK_TREE_MODEL(playliststore), NULL);
    coltitle = g_strdup_printf(ngettext("Item to Play","Items to Play",count));
    gtk_tree_view_column_set_title(column, coltitle);
    g_free(coltitle);
	
    gtk_widget_destroy(dialog);
}

void add_folder_to_playlist(GtkWidget * widget, void *data)
{

    GtkWidget *dialog;
    GSList *filename;
    GConfClient *gconf;
    gchar *last_dir;
    gchar *message;

    dialog = gtk_file_chooser_dialog_new(_("Choose Directory"),
                                         GTK_WINDOW(window),
                                         GTK_FILE_CHOOSER_ACTION_SELECT_FOLDER,
                                         GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
                                         GTK_STOCK_OPEN, GTK_RESPONSE_ACCEPT, NULL);

    /*allow multiple files to be selected */
    gtk_file_chooser_set_select_multiple(GTK_FILE_CHOOSER(dialog), TRUE);

    gconf = gconf_client_get_default();
    last_dir = gconf_client_get_string(gconf, LAST_DIR, NULL);
    if (last_dir != NULL) {
        gtk_file_chooser_set_current_folder(GTK_FILE_CHOOSER(dialog), last_dir);
        g_free(last_dir);
    }

    if (gtk_dialog_run(GTK_DIALOG(dialog)) == GTK_RESPONSE_ACCEPT) {

        filename = gtk_file_chooser_get_filenames(GTK_FILE_CHOOSER(dialog));

        last_dir = gtk_file_chooser_get_current_folder(GTK_FILE_CHOOSER(dialog));
        gconf_client_set_string(gconf, LAST_DIR, last_dir, NULL);
        g_free(last_dir);

        filecount = 0;
        g_slist_foreach(filename, &add_folder_to_playlist_callback, NULL);
        g_slist_free(filename);
    }
    update_gui();
    g_object_unref(G_OBJECT(gconf));
    gtk_widget_destroy(dialog);
    message = g_markup_printf_escaped(ngettext("\n\tFound %i file\n","\n\tFound %i files\n",filecount), filecount);
    g_strlcpy(idledata->media_info, message, 1024);
    g_free(message);
    g_idle_add(set_media_label, idledata);

}

void remove_from_playlist(GtkWidget * widget, gpointer data)
{
    GtkTreeSelection *sel;
    GtkTreeView *view = (GtkTreeView *) data;
    GtkTreeIter localiter;
    GtkTreePath *path;

    sel = gtk_tree_view_get_selection(view);

    if (gtk_tree_selection_get_selected(sel, NULL, &iter)) {
        localiter = iter;
        gtk_tree_model_iter_next(GTK_TREE_MODEL(playliststore), &localiter);
        if (gtk_list_store_remove(playliststore, &iter)) {
            iter = localiter;
            if (!gtk_list_store_iter_is_valid(playliststore, &iter)) {
                gtk_tree_model_get_iter_first(GTK_TREE_MODEL(playliststore), &iter);
            }
            if (GTK_IS_TREE_SELECTION(selection)) {
                path = gtk_tree_model_get_path(GTK_TREE_MODEL(playliststore), &iter);
                gtk_tree_selection_select_path(selection, path);
				if (GTK_IS_WIDGET(list))
					gtk_tree_view_scroll_to_cell(GTK_TREE_VIEW(list),path,NULL,FALSE,0,0);
                gtk_tree_path_free(path);
            }
        }
    }
    update_gui();

}

void clear_playlist(GtkWidget * widget, void *data)
{

    dontplaynext = TRUE;
    shutdown();
    gtk_list_store_clear(playliststore);
    gtk_list_store_clear(nonrandomplayliststore);
	gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(menuitem_edit_random),FALSE);
}


gboolean playlist_select_callback(GtkTreeView * view, GtkTreePath * path,
                                  GtkTreeViewColumn * column, gpointer data)
{

    gchar *filename;
    gint count;
    gint playlist;

    if (gtk_tree_model_get_iter(GTK_TREE_MODEL(playliststore), &iter, path)) {
        gtk_tree_model_get(GTK_TREE_MODEL(playliststore), &iter, ITEM_COLUMN, &filename,
                           COUNT_COLUMN, &count, PLAYLIST_COLUMN, &playlist, -1);
        dontplaynext = TRUE;
        shutdown();
        set_media_info_name(filename);
        play_file(filename, playlist);
        gtk_list_store_set(playliststore, &iter, COUNT_COLUMN, count + 1, -1);
        g_free(filename);
        if (state == 3) {
            dontplaynext = FALSE;
        } else {
            dontplaynext = TRUE;
        }
    }
    return FALSE;
}

void playlist_close(GtkWidget * widget, void *data)
{
	menuitem_view_playlist_callback(NULL, NULL);	
}

void menuitem_view_playlist_callback(GtkMenuItem * menuitem, void *data)
{

    GtkWidget *close;
    GtkWidget *scrolled;
    GtkCellRenderer *renderer;
    GtkTreeViewColumn *column;
    GtkWidget *box;
    GtkWidget *ctrlbox;
    GtkWidget *closebox;
    GtkWidget *hbox;
    GtkWidget *savelist;
    GtkWidget *loadlist;
    GtkWidget *add;
    GtkWidget *remove;
    GtkWidget *add_folder;
    GtkWidget *clear;
    GtkTreePath *path;
    GtkAccelGroup *accel_group;
    GValue value = { 0, };
    GtkTooltips *tooltip;
    //GtkRequisition plreq;
    gchar *coltitle;
    gint x, y, depth;
	gint count;

    //if (GTK_IS_TREE_SELECTION(selection)){
    //      return;
    //}
    g_value_init(&value, G_TYPE_BOOLEAN);

    if (GTK_IS_WIDGET(plvbox)) {
        if (GTK_WIDGET_VISIBLE(plvbox)) {
            if (idledata->videopresent == FALSE) {
                gdk_window_get_geometry(window->window, &x, &y, &window_width, &window_height,
                                        &depth);
				stored_window_width = -1;
				stored_window_height = -1;
                gtk_window_set_resizable(GTK_WINDOW(window), FALSE);
                gtk_widget_hide(GTK_WIDGET(fixed));
                gtk_widget_show_all(media_label);
				gtk_widget_show(vbox);				
                gtk_widget_set_size_request(window, -1, -1);
            } else {
                gtk_window_set_resizable(GTK_WINDOW(window), TRUE);
                gtk_widget_set_size_request(window, -1, -1);
                gtk_window_set_policy(GTK_WINDOW(window), TRUE, TRUE, TRUE);
				window_width = -1;
				window_height = -1;
				if (stored_window_width != -1)
	            	gtk_window_resize(GTK_WINDOW(window), stored_window_width, stored_window_height);
            }
            gtk_container_remove(GTK_CONTAINER(pane), plvbox);
            plvbox = NULL;
            selection = NULL;
        }

    } else {
        gtk_window_set_resizable(GTK_WINDOW(window), TRUE);
        if (idledata->videopresent == FALSE) {
			if (window_width != -1)
	            gtk_window_resize(GTK_WINDOW(window), window_width, window_height);
			gtk_widget_hide(vbox);
        } else {
			gdk_window_get_geometry(window->window, &x, &y, &stored_window_width, &stored_window_height, &depth);		
		}
		plvbox = gtk_vbox_new(FALSE, 12);
        hbox = gtk_hbox_new(FALSE, 12);
        gtk_box_set_homogeneous(GTK_BOX(hbox), FALSE);
        box = gtk_hbox_new(FALSE, 10);
        ctrlbox = gtk_hbox_new(FALSE, 0);
        closebox = gtk_hbox_new(FALSE, 0);

        list = gtk_tree_view_new_with_model(GTK_TREE_MODEL(playliststore));
        gtk_widget_add_events(list, GDK_BUTTON_PRESS_MASK);
        gtk_widget_set_size_request(GTK_WIDGET(list), -1, -1);

        selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(list));
        if (gtk_list_store_iter_is_valid(playliststore, &iter)) {
            path = gtk_tree_model_get_path(GTK_TREE_MODEL(playliststore), &iter);
            gtk_tree_selection_select_path(selection, path);
			gtk_tree_view_scroll_to_cell(GTK_TREE_VIEW(list),path,NULL,FALSE,0,0);
            gtk_tree_path_free(path);
        }

        g_signal_connect(G_OBJECT(list), "row_activated", G_CALLBACK(playlist_select_callback),
                         NULL);
        g_signal_connect(G_OBJECT(list), "motion-notify-event",
                         G_CALLBACK(playlist_motion_callback), NULL);

        playlisttip = gtk_tooltips_new();

		count = gtk_tree_model_iter_n_children(GTK_TREE_MODEL(playliststore), NULL);
        renderer = gtk_cell_renderer_text_new();
        column = gtk_tree_view_column_new_with_attributes(ngettext("Item to play","Items to Play",count),
                                                          renderer,
                                                          "text", DESCRIPTION_COLUMN, NULL);
        if (playlistname != NULL) {
            coltitle = g_strdup_printf(_("%s items"), playlistname);
            gtk_tree_view_column_set_title(column, coltitle);
            g_free(coltitle);
        }
        gtk_tree_view_column_set_expand(column, TRUE);
        gtk_tree_view_column_set_max_width(column, 40);
        gtk_tree_view_append_column(GTK_TREE_VIEW(list), column);

        renderer = gtk_cell_renderer_text_new();
        column = gtk_tree_view_column_new_with_attributes(_("Artist"),
                                                          renderer, "text", ARTIST_COLUMN, NULL);
        gtk_tree_view_column_set_expand(column, TRUE);
        gtk_tree_view_column_set_max_width(column, 20);
        gtk_tree_view_append_column(GTK_TREE_VIEW(list), column);


        renderer = gtk_cell_renderer_text_new();
        column = gtk_tree_view_column_new_with_attributes(_("Length"),
                                                          renderer, "text", LENGTH_COLUMN, NULL);
        gtk_tree_view_column_set_expand(column, FALSE);
        gtk_tree_view_column_set_alignment(column, 1.0);
        gtk_tree_view_append_column(GTK_TREE_VIEW(list), column);


        renderer = gtk_cell_renderer_text_new();
        column = gtk_tree_view_column_new_with_attributes("", renderer, "text", COUNT_COLUMN, NULL);
        gtk_tree_view_column_set_expand(column, FALSE);
        gtk_tree_view_append_column(GTK_TREE_VIEW(list), column);


        close = gtk_button_new();
        tooltip = gtk_tooltips_new();
        gtk_tooltips_set_tip(tooltip, close, _("Close Playlist View"), NULL);
        gtk_container_add(GTK_CONTAINER(close),
                          gtk_image_new_from_stock(GTK_STOCK_CLOSE, GTK_ICON_SIZE_MENU));

        g_signal_connect_swapped(GTK_OBJECT(close), "clicked",
                                 GTK_SIGNAL_FUNC(playlist_close), NULL);


        loadlist = gtk_button_new();
        tooltip = gtk_tooltips_new();
        gtk_tooltips_set_tip(tooltip, loadlist, _("Open Playlist"), NULL);
        gtk_container_add(GTK_CONTAINER(loadlist),
                          gtk_image_new_from_stock(GTK_STOCK_OPEN, GTK_ICON_SIZE_MENU));
        gtk_box_pack_start(GTK_BOX(ctrlbox), loadlist, FALSE, FALSE, 0);
        g_signal_connect(GTK_OBJECT(loadlist), "clicked", GTK_SIGNAL_FUNC(load_playlist), NULL);

        savelist = gtk_button_new();
        tooltip = gtk_tooltips_new();
        gtk_tooltips_set_tip(tooltip, savelist, _("Save Playlist"), NULL);
        gtk_container_add(GTK_CONTAINER(savelist),
                          gtk_image_new_from_stock(GTK_STOCK_SAVE, GTK_ICON_SIZE_MENU));
        gtk_box_pack_start(GTK_BOX(ctrlbox), savelist, FALSE, FALSE, 0);
        g_signal_connect(GTK_OBJECT(savelist), "clicked", GTK_SIGNAL_FUNC(save_playlist), NULL);

        add = gtk_button_new();
        tooltip = gtk_tooltips_new();
        gtk_tooltips_set_tip(tooltip, add, _("Add Item to Playlist"), NULL);
        gtk_button_set_image(GTK_BUTTON(add),
                             gtk_image_new_from_stock(GTK_STOCK_ADD, GTK_ICON_SIZE_MENU));
        gtk_box_pack_start(GTK_BOX(ctrlbox), add, FALSE, FALSE, 0);
        g_signal_connect(GTK_OBJECT(add), "clicked", GTK_SIGNAL_FUNC(add_to_playlist), NULL);

        remove = gtk_button_new();
        tooltip = gtk_tooltips_new();
        gtk_tooltips_set_tip(tooltip, remove, _("Remove Item from Playlist"), NULL);
        gtk_button_set_image(GTK_BUTTON(remove),
                             gtk_image_new_from_stock(GTK_STOCK_REMOVE, GTK_ICON_SIZE_MENU));
        gtk_box_pack_start(GTK_BOX(ctrlbox), remove, FALSE, FALSE, 0);
        g_signal_connect(GTK_OBJECT(remove), "clicked", GTK_SIGNAL_FUNC(remove_from_playlist),
                         list);

        add_folder = gtk_button_new();
        tooltip = gtk_tooltips_new();
        gtk_tooltips_set_tip(tooltip, add_folder, _("Add Items in Folder to Playlist"), NULL);
        gtk_button_set_image(GTK_BUTTON(add_folder),
                             gtk_image_new_from_stock(GTK_STOCK_DIRECTORY, GTK_ICON_SIZE_MENU));
        gtk_box_pack_start(GTK_BOX(ctrlbox), add_folder, FALSE, FALSE, 0);
        g_signal_connect(GTK_OBJECT(add_folder), "clicked", GTK_SIGNAL_FUNC(add_folder_to_playlist),
                         list);

        clear = gtk_button_new();
        tooltip = gtk_tooltips_new();
        gtk_tooltips_set_tip(tooltip, clear, _("Clear Playlist"), NULL);
        gtk_button_set_image(GTK_BUTTON(clear),
                             gtk_image_new_from_stock(GTK_STOCK_CLEAR, GTK_ICON_SIZE_MENU));
        gtk_box_pack_start(GTK_BOX(ctrlbox), clear, FALSE, FALSE, 0);
        g_signal_connect(GTK_OBJECT(clear), "clicked", GTK_SIGNAL_FUNC(clear_playlist), list);


        accel_group = gtk_accel_group_new();
        gtk_window_add_accel_group(GTK_WINDOW(window), accel_group);
        gtk_widget_add_accelerator(GTK_WIDGET(remove), "clicked",
                                   accel_group, GDK_Delete, 0, GTK_ACCEL_VISIBLE);

        GTK_WIDGET_SET_FLAGS(close, GTK_CAN_DEFAULT);
        gtk_box_pack_end(GTK_BOX(closebox), close, FALSE, FALSE, 0);

        scrolled = gtk_scrolled_window_new(NULL, NULL);
        gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled),
                                       GTK_POLICY_NEVER, GTK_POLICY_AUTOMATIC);
        gtk_container_add(GTK_CONTAINER(scrolled), list);

        gtk_box_pack_start(GTK_BOX(plvbox), scrolled, TRUE, TRUE, 0);
        gtk_box_pack_start(GTK_BOX(hbox), ctrlbox, FALSE, FALSE, 0);
        gtk_box_pack_end(GTK_BOX(hbox), closebox, FALSE, FALSE, 0);
        gtk_box_pack_start(GTK_BOX(plvbox), hbox, FALSE, FALSE, 0);


        gtk_paned_pack2(GTK_PANED(pane), plvbox, FALSE, FALSE);

		if (vertical_layout) {
        	gtk_widget_set_size_request(plvbox, -1, 150);
			if (idledata->videopresent)
				gtk_window_resize(GTK_WINDOW(window),stored_window_width,stored_window_height+150);
		} else {	
	        gtk_widget_set_size_request(plvbox, 300, -1);
			if (idledata->videopresent)
				gtk_window_resize(GTK_WINDOW(window),stored_window_width+300,stored_window_height);
		}
        gtk_widget_show_all(plvbox);
        if (idledata->videopresent == FALSE) {
            gtk_widget_show(GTK_WIDGET(fixed));
        } else {
			gtk_window_set_policy(GTK_WINDOW(window), TRUE, TRUE, TRUE);
		}
		
        playlist_popup_menu = GTK_MENU(gtk_menu_new());
        playlist_set_subtitle =
            GTK_MENU_ITEM(gtk_image_menu_item_new_with_mnemonic(_("_Set Subtitle")));
        g_signal_connect(GTK_OBJECT(playlist_set_subtitle), "activate",
                         G_CALLBACK(playlist_set_subtitle_callback), list);

        gtk_menu_append(playlist_popup_menu, GTK_WIDGET(playlist_set_subtitle));
        g_signal_connect_swapped(G_OBJECT(list),
                                 "button_press_event",
                                 G_CALLBACK(playlist_popup_handler),
                                 G_OBJECT(playlist_popup_menu));
        gtk_widget_show_all(GTK_WIDGET(playlist_popup_menu));


        gtk_widget_grab_default(close);
    }
}
