/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright 2000-2007 Sun Microsystems, Inc. All rights reserved. 
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common Development
 * and Distribution License ("CDDL") (collectively, the "License").  You may
 * not use this file except in compliance with the License.  You can obtain
 * a copy of the License at https://glassfish.dev.java.net/public/CDDL+GPL.html
 * or mq/legal/LICENSE.txt.  See the License for the specific language
 * governing permissions and limitations under the License.
 * 
 * When distributing the software, include this License Header Notice in each
 * file and include the License file at mq/legal/LICENSE.txt.  Sun designates
 * this particular file as subject to the "Classpath" exception as provided by
 * Sun in the GPL Version 2 section of the License file that accompanied this
 * code.  If applicable, add the following below the License Header, with the
 * fields enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Contributor(s):
 * 
 * If you wish your version of this file to be governed by only the CDDL or
 * only the GPL Version 2, indicate your decision by adding "[Contributor]
 * elects to include this software in this distribution under the [CDDL or GPL
 * Version 2] license."  If you don't indicate a single choice of license, a
 * recipient has the option to distribute your version of this file under
 * either the CDDL, the GPL Version 2 or  to extend the choice of license to
 * its licensees as provided above.  However, if you add GPL Version 2 code
 * and therefore, elected the GPL Version 2 license, then the option applies
 * only if the new code is made subject to such option by the copyright holder. 
 */

/*
 * @(#)BrokerAdmin.java	1.85 06/27/07
 */ 

package com.sun.messaging.jmq.admin.bkrutil;

import java.util.Vector;
import java.util.Hashtable;
import java.util.Enumeration;
import java.util.Properties;
import java.io.EOFException;
import java.net.MalformedURLException;
import javax.jms.*;

import com.sun.messaging.AdministeredObject;
import com.sun.messaging.ConnectionConfiguration;
import com.sun.messaging.QueueConnectionFactory;
import com.sun.messaging.jmq.util.DestState;
import com.sun.messaging.jmq.util.admin.*;
import com.sun.messaging.jmq.ClientConstants;
import com.sun.messaging.jmq.io.MetricCounters;
import com.sun.messaging.jmq.io.DestMetricsCounters;
import com.sun.messaging.jmq.io.MQAddress;
import com.sun.messaging.jmq.io.MQAddressList;

import com.sun.messaging.jmq.jmsclient.resources.ClientResources;
import com.sun.messaging.jmq.admin.apps.console.event.AdminEvent;
import com.sun.messaging.jmq.admin.apps.console.event.AdminEventListener;
import com.sun.messaging.jmq.admin.apps.console.event.BrokerErrorEvent;
import com.sun.messaging.jmq.admin.apps.console.event.BrokerCmdStatusEvent;
import com.sun.messaging.jmq.admin.util.Globals;
import com.sun.messaging.jms.management.server.BrokerClusterInfo;

/**
 * This class provides the convenient methods for sending administration
 * messages to the JMQ broker.
 *
 * <P>
 * The information needed to create this object are:
 * <UL>
 * <LI>connection factory attributes
 * <LI>username/passwd
 * <LI>timeout (for receiving replies)
 * </UL>
 */
public class BrokerAdmin implements ExceptionListener {

    public final static String		DEFAULT_ADMIN_USERNAME	= "admin";
    public final static String		DEFAULT_ADMIN_PASSWD	= "admin";

    public static long			defaultTimeout		= 10000;
    public static int			defaultNumRetries	= 5;

    /*
     * Reconnect attributes.
     * These are used for the restart command.
     */
    public static final int 		RECONNECT_RETRIES = 5;
    public static final long 		RECONNECT_DELAY = 5000;

    private static boolean		debug = false;

    private String			key = null;

    private String			username,
    					passwd;
    private long			timeout;
    private int				numRetries;

    private QueueConnectionFactory	qcf;
    private QueueConnection		connection;
    private QueueSession	       	session;
    private Queue			requestQueue;
    private TemporaryQueue	       	replyQueue;
    private QueueSender	       		sender;
    protected QueueReceiver	       	receiver;

    private MQAddress			address = null;

    /*
     * List of properties that we currently care about i.e. that are
     * saved out if necessary. The reconnect stuff is currently
     * not used. This is used by getBrokerAttrs() which in turn is
     * used only by the admin console.
     */
    private final static String	       	savedQCFProperties[] = {
                                ConnectionConfiguration.imqBrokerHostName,
                                ConnectionConfiguration.imqBrokerHostPort
					};

    private boolean			isConnected = false;
    private boolean			adminKeyUsed = false;
    private boolean			sslTransportUsed = false;

    private boolean			isInitiator = false;
    private boolean			isReconnect = false;


    private Vector eListeners = new Vector();

    private MessageAckThread		msgAckThread = null;
    private boolean			busy = false;
    private BrokerCmdStatusEvent	statusEvent = null;

    private Object			aObj = null;

    /*
     * Temporary convenient constructor.
     */
    public BrokerAdmin(String brokerHost, int brokerPort) throws BrokerAdminException  {
	this(brokerHost, brokerPort, null, null, -1, false, -1, -1);
    }

    public BrokerAdmin(String brokerHost, int brokerPort, 
	               String username, String passwd) 
		       throws BrokerAdminException  {
	this(brokerHost, brokerPort, username, passwd, -1, false, -1, -1);
    }

    public BrokerAdmin(String brokerHost, int brokerPort, 
	               String username, String passwd, int timeout) 
		       throws BrokerAdminException  {
	this(brokerHost, brokerPort, username, passwd, timeout, false, -1, -1);
    }

    public BrokerAdmin(String brokerAddress,
	               String username, String passwd, int timeout, boolean useSSL) 
		       throws BrokerAdminException  {
	this(brokerAddress, username, passwd, timeout, false, -1, -1, useSSL);
    }

    /**
     * Instantiates a BrokerAdmin object. This is a wrapper for
     * this other constructor:
     *
     *  public BrokerAdmin(Properties, String, String, long)
     *
     * @param brokerHost	host name of the broker to administer
     * @param brokerPort 	primary port for broker
     * @param username		username used to authenticate
     * @param passwd		password used to authenticate
     * @param timeout		timeout value (in milliseconds) for receive; 
     *                          0 = never times out and the call blocks 
     *				indefinitely
     * @param reconnect		true if reconnect is enabled; false otherwise
     * @param reconnectRetries	number of reconnect retries
     * @param reconnectDelay	interval of reconnect retries in milliseconds
     */
    public BrokerAdmin(String brokerHost, int brokerPort, 
	               String username, String passwd, long timeout,
		       boolean reconnect, int reconnectRetries, long reconnectDelay) 
		       throws BrokerAdminException  {

	Properties tmpProps = new Properties();

	if (brokerHost != null)  {
	    tmpProps.setProperty(ConnectionConfiguration.imqBrokerHostName, brokerHost);
	}

	if (brokerPort > 0)  {
	    tmpProps.setProperty(ConnectionConfiguration.imqBrokerHostPort, 
				String.valueOf(brokerPort));
	}

	if (reconnect)  {
	    tmpProps.setProperty(ConnectionConfiguration.imqReconnectEnabled, 
		String.valueOf(reconnect));
	    tmpProps.setProperty(ConnectionConfiguration.imqReconnectAttempts, 
		String.valueOf(reconnectRetries));
	    tmpProps.setProperty(ConnectionConfiguration.imqReconnectInterval, 
		String.valueOf(reconnectDelay));
	}

	if (timeout >= 0)  {
	    this.timeout = timeout;
	} else  {
	    this.timeout = defaultTimeout;
	}

	this.numRetries = defaultNumRetries;

	this.username = username;
	this.passwd = passwd;

	createFactory(tmpProps);
    }

    /**
     * Instantiates a BrokerAdmin object. This is a wrapper for
     * this other constructor:
     *
     *  public BrokerAdmin(Properties, String, String, long)
     *
     * @param brokerAddress 	address/url of broker
     * @param username		username used to authenticate
     * @param passwd		password used to authenticate
     * @param timeout		timeout value (in milliseconds) for receive; 
     *                          0 = never times out and the call blocks 
     *				indefinitely
     * @param reconnect		true if reconnect is enabled; false otherwise
     * @param reconnectRetries	number of reconnect retries
     * @param reconnectDelay	interval of reconnect retries in milliseconds
     * @param useSSL		Use encrypted transport via SSL
     */
    public BrokerAdmin(String brokerAddress, 
	               String username, String passwd, 
		       long timeout,
		       boolean reconnect, int reconnectRetries, 
		       long reconnectDelay, boolean useSSL) 
		           throws BrokerAdminException  {

	Properties tmpProps = new Properties();

	if (brokerAddress == null)  {
	    brokerAddress = "";
	}

	try  {
	    if (useSSL)  {
	        address = (MQAddress)SSLAdminMQAddress.createAddress(brokerAddress);
	    } else  {
	        address = (MQAddress)AdminMQAddress.createAddress(brokerAddress);
	    }
	} catch (Exception e)  {
	    BrokerAdminException bae;
	    bae = new BrokerAdminException(BrokerAdminException.BAD_ADDR_SPECIFIED);
	    bae.setBrokerAddress(brokerAddress);
	    bae.setLinkedException(e);
            throw bae;
	}

	tmpProps.setProperty(ConnectionConfiguration.imqAddressList, 
					address.toString());

	if (reconnect)  {
	    tmpProps.setProperty(ConnectionConfiguration.imqReconnectEnabled, 
		String.valueOf(reconnect));
	    tmpProps.setProperty(ConnectionConfiguration.imqReconnectAttempts, 
		String.valueOf(reconnectRetries));
	    tmpProps.setProperty(ConnectionConfiguration.imqReconnectInterval, 
		String.valueOf(reconnectDelay));
	}

	if (timeout >= 0)  {
	    this.timeout = timeout;
	} else  {
	    this.timeout = defaultTimeout;
	}

	this.numRetries = defaultNumRetries;

	this.username = username;
	this.passwd = passwd;

	createFactory(tmpProps);
    }


    /**
     * The constructor for the class.
     *
     * @param brokerAttrs 	Properties object containing
     *				the broker attributes. This is
     *				basically what is used to create
     *				the connection factory.
     * @param username		username used to authenticate
     * @param passwd		password used to authenticate
     * @param timeout		timeout value (in milliseconds) for receive; 
     *                          0 = never times out and the call blocks 
     *				indefinitely
     */
    public BrokerAdmin(Properties brokerAttrs,
			String username, String passwd, 
			long timeout) 
		       throws BrokerAdminException  {

	if (timeout >= 0)  {
	    this.timeout = timeout;
	} else  {
	    this.timeout = defaultTimeout;
	}
	this.numRetries = defaultNumRetries;

	this.username = username;
	this.passwd = passwd;

	createFactory(brokerAttrs);
    }


    public void setBrokerHost(String hostName) throws BrokerAdminException  {
	try  {
	    setFactoryAttr(ConnectionConfiguration.imqBrokerHostName, hostName);
	} catch (JMSException jmse)  {
	    BrokerAdminException bae;
	    bae = new BrokerAdminException(BrokerAdminException.BAD_HOSTNAME_SPECIFIED);
	    bae.setBadValue(hostName);
	    bae.setLinkedException(jmse);
            throw bae;
	}
    }

    public String getBrokerAddress()  {
	if (address != null)  {
	    return (address.toString());
	}

	return (getFactoryAttr(ConnectionConfiguration.imqAddressList));
    }

    public String getBrokerHost()  {
	if (address != null)  {
	    return (address.getHostName());
	}

	return (getFactoryAttr(ConnectionConfiguration.imqBrokerHostName));
    }

    public void setBrokerPort(String port) throws BrokerAdminException  {
	try  {
	    setFactoryAttr(ConnectionConfiguration.imqBrokerHostPort, port);
	} catch (JMSException jmse)  {
	    BrokerAdminException bae;
	    bae = new BrokerAdminException(BrokerAdminException.BAD_PORT_SPECIFIED);
	    bae.setBadValue(port);
	    bae.setLinkedException(jmse);
            throw bae;
	}
    }

    public String getBrokerPort()  {
	if (address != null)  {
	    return ((new Integer(address.getPort())).toString());
	}

	return (getFactoryAttr(ConnectionConfiguration.imqBrokerHostPort));
    }

    public static void setDefaultTimeout(long defaultTimeout)  {
	BrokerAdmin.defaultTimeout = defaultTimeout;
	if (debug)  {
	    Globals.stdOutPrintln("BrokerAdmin defaultTimeout set to: " + BrokerAdmin.defaultTimeout);
	}
    }
    public static long getDefaultTimeout()  {
	return (defaultTimeout);
    }

    public long getTimeout()  {
	return (timeout);
    }

    public static void setDefaultNumRetries(int defaultNumRetries)  {
	BrokerAdmin.defaultNumRetries = defaultNumRetries;
	if (debug)  {
	    Globals.stdOutPrintln("BrokerAdmin defaultNumRetries set to: "
		+ BrokerAdmin.defaultNumRetries);
	}
    }
    public static long getDefaultNumRetries()  {
	return (defaultNumRetries);
    }

    public void setNumRetries(int numRetries)  {
	this.numRetries = numRetries;
	if (debug)  {
	    Globals.stdOutPrintln("BrokerAdmin num retries set to: " + this.numRetries);
        }
    }

    public void setUserName(String userName)  {
	this.username = userName;
    }
    public String getUserName()  {
	return (username);
    }

    public void setPassword(String passwd)  {
	this.passwd = passwd;
    }
    public String getPassword()  {
	return (passwd);
    }

    public void setBusy(boolean b)  {
	busy = b;
	if (debug)  {
	    Globals.stdOutPrintln("***** BrokerAdmin.setBusy(): " + b);
	}

	if (!b && (msgAckThread != null))  {
	    /*
	     * If we are going from busy -> not busy,
	     * nullify the ack thread.
	     */
	    msgAckThread = null;
	}
    }
    public boolean isBusy()  {
	return(busy);
    }

    /*
     * Support for private "-adminkey" option.
     * This is used for authentication when shutting down the
     * broker via the NT service's "Stop" command.
     */
    public void setAdminKeyUsed(boolean b)  {
	this.adminKeyUsed = b;
    }
    public boolean getAdminKeyUsed()  {
	return (adminKeyUsed);
    }

    /*
     * Support for "-ssl" option.
     * This is used to indicate that SSL transport
     * will be used.
     */
    public void setSSLTransportUsed(boolean b)  {
	this.sslTransportUsed = b;
    }
    public boolean getSSLTransportUsed()  {
	return (sslTransportUsed);
    }

    /*
     * Set when this BrokerAdmin has initiated the
     * shutdown operation.
     */
    public void setInitiator(boolean b)  {
        this.isInitiator = b;
    }
    public boolean isInitiator()  {
        return (isInitiator);
    }

    /*
     * Set when this BrokerAdmin is restarted.
     */
    public void setReconnect(boolean b)  {
        this.isReconnect = b;
    }
    public boolean isReconnect()  {
        return (isReconnect);
    }

    /*
     * Return a properties object containing the queue
     * connection configuration properties.
     * We restrict the props to those listed in the
     * savedQCFProperties array - otherwise we'll get
     * the whole load - most of which are defaults.
     */
    public Properties getBrokerAttrs()  {
	Properties tmpProps = new Properties();

	if (qcf == null)  {
	    return (tmpProps);
	}

	for (int i = 0; i < savedQCFProperties.length; ++i)  {
	    String	propName = savedQCFProperties[i],
			propVal = getFactoryAttr(propName);
	    
	    tmpProps.setProperty(propName, propVal);
	}

	return (tmpProps);
    }

    public void connect() throws BrokerAdminException  {
        // We need a flag since username / password can be null.
        connect(null, null, false);
    }

    public void connect(String tempUsername, String tempPasswd) 
	throws BrokerAdminException {
        // We need a flag since username / password can be null.
        connect(tempUsername, tempPasswd, true);
    }

    // REVISIT: should this be synchronized to make sure isConnected is properly set?
    private void connect(String tempUsername, String tempPasswd, boolean useTempValues) 
	throws BrokerAdminException  {
	BrokerAdminException	bae;

        try {
	    if (adminKeyUsed)  {
		/*
		 * turn on using password as special admin key
		 */
		qcf.setConnectionType(ClientConstants.CONNECTIONTYPE_ADMINKEY);
	    }

	    if (sslTransportUsed)  {
		/*
		 * Set connection transport to be SSL
		 */
		try  {
		    qcf.setProperty(ConnectionConfiguration.imqConnectionType, "SSL");
		} catch (Exception e)  {
		}

	        try  {
	            setFactoryAttr(ConnectionConfiguration.imqConnectionType, "SSL");
	        } catch (JMSException jmse)  {
	            bae = new BrokerAdminException(BrokerAdminException.PROB_SETTING_SSL);
	            bae.setBrokerHost(getBrokerHost());
	            bae.setBrokerPort(getBrokerPort());
	            bae.setLinkedException(jmse);
                    throw bae;
	        }
	    }

	    if (debug)  {
		printObjProperties(qcf);
	    }

            if (useTempValues) {
                connection = qcf.createQueueConnection(tempUsername, tempPasswd);
            } else {
                connection = qcf.createQueueConnection(username, passwd);
            }
	    connection.setExceptionListener(this);
	    connection.start();
	    if (debug) Globals.stdOutPrintln("***** Creating queue connection");

	    session = 
	        connection.createQueueSession(false, Session.CLIENT_ACKNOWLEDGE);
	   
	    if (debug) Globals.stdOutPrintln
	    ("***** Creating queue session: not transacted, auto ack");

	    requestQueue = session.createQueue(MessageType.JMQ_ADMIN_DEST);
	    if (debug) 
	    Globals.stdOutPrintln("***** Created requestQueue: " + requestQueue);

	    replyQueue = session.createTemporaryQueue();
	    if (debug) Globals.stdOutPrintln("***** Created replyQueue: " + replyQueue);

	    sender = session.createSender(requestQueue);
	    // making the message delivery mode non-persistent
	    sender.setDeliveryMode(DeliveryMode.NON_PERSISTENT);
	    if (debug) Globals.stdOutPrintln("***** Created a sender: " + sender);

      	    receiver = session.createReceiver(replyQueue);
	    if (debug) Globals.stdOutPrintln("***** Created a receiver: " + receiver);

	} catch (JMSException jmse) {
	    if (jmse instanceof JMSSecurityException) {
		String errorCode = jmse.getErrorCode();
		if ((AdministeredObject.cr.X_INVALID_LOGIN).equals(errorCode))
	    	    bae = new BrokerAdminException(BrokerAdminException.INVALID_LOGIN);
		else
	    	    bae = new BrokerAdminException(BrokerAdminException.SECURITY_PROB);
	    } else {
	        bae = new BrokerAdminException(BrokerAdminException.CONNECT_ERROR);
	    }
	    bae.setBrokerHost(getBrokerHost());
	    bae.setBrokerPort(getBrokerPort());
	    bae.setLinkedException(jmse);
            throw bae;
	    
	} catch (Exception e) {
	    bae = new BrokerAdminException(BrokerAdminException.CONNECT_ERROR);
	    bae.setLinkedException(e);
	    bae.setBrokerHost(getBrokerHost());
	    bae.setBrokerPort(getBrokerPort());
            throw bae;
	}
    }

    private void printObjProperties(AdministeredObject obj) {
        Globals.stdOutPrintln("Connection Factory Object properties:");

        /*
         * Print the properties of the object.
         */
        Properties props = obj.getConfiguration();
        for (Enumeration e = obj.enumeratePropertyNames(); e.hasMoreElements();) {
            String propName = (String)e.nextElement();
            String value = props.getProperty(propName);
            String propLabel;
            try  {
                propLabel = obj.getPropertyLabel(propName);
            } catch (Exception ex)  {
                propLabel = "UNKNOWN";
            }
            String printLabel = propName + " [" + propLabel + "]";

            Globals.stdOutPrintln("\t" + printLabel + " = " + value);
        }
        Globals.stdOutPrintln("");
    }

    public void sendHelloMessage() throws BrokerAdminException  {
	BrokerAdminException bae;

	if (debug) Globals.stdOutPrintln("***** sendHelloMessage *****");
	ObjectMessage mesg = null;

	checkIfBusy();

	try {
	    mesg = session.createObjectMessage();
	    mesg.setJMSReplyTo(replyQueue);		
	    mesg.setIntProperty
		(MessageType.JMQ_MESSAGE_TYPE, MessageType.HELLO);
            statusEvent = createStatusEvent(BrokerCmdStatusEvent.HELLO,
                                            MessageType.HELLO_REPLY,
                                            "HELLO_REPLY");

	    if (debug)  {
		printMsgType(MessageType.HELLO, "HELLO");
		Globals.stdOutPrintln("\t"
			+ MessageType.JMQ_PROTOCOL_LEVEL
			+ "=" 
			+ 102);
	    }
	    sender.send(mesg);	

        } catch (Exception e) {
	    handleSendExceptions(e);
        }
    }


    public void receiveHelloReplyMessage() throws BrokerAdminException {
	BrokerAdminException bae;

        if (debug) Globals.stdOutPrintln("***** receiveHelloReplyMessage() *****");
        Message mesg = null;

        try {
            mesg = (ObjectMessage)receiveCheckMessageTimeout(false);

            mesg.acknowledge();
	    clearStatusEvent();
	    checkReplyTypeStatus(mesg, MessageType.HELLO_REPLY, "HELLO_REPLY");

	    isConnected = true;

        } catch (Exception e) {
	    handleReceiveExceptions(e);
        }
    }


    public void sendGetServicesMessage(String svcName) 
				throws BrokerAdminException {
	BrokerAdminException bae;

        if (debug) Globals.stdOutPrintln("***** sendGetServicesMessage *****");
        ObjectMessage mesg = null;

	checkIfBusy();

        try {
            mesg = session.createObjectMessage();
            mesg.setJMSReplyTo(replyQueue);
            mesg.setIntProperty
		(MessageType.JMQ_MESSAGE_TYPE, MessageType.GET_SERVICES);
	    if (svcName != null)  {
	        mesg.setStringProperty(MessageType.JMQ_SERVICE_NAME, svcName);
	        statusEvent = createStatusEvent(BrokerCmdStatusEvent.QUERY_SVC,
						MessageType.GET_SERVICES_REPLY,
						"GET_SERVICES_REPLY");
	    } else  {
	        statusEvent = createStatusEvent(BrokerCmdStatusEvent.LIST_SVC,
						MessageType.GET_SERVICES_REPLY,
						"GET_SERVICES_REPLY");
	    }

	    statusEvent.setServiceName(svcName);
	    if (debug)  {
		printMsgType(MessageType.GET_SERVICES, "GET_SERVICES");
		if (svcName != null)  {
		    Globals.stdOutPrintln("\t"
			+ MessageType.JMQ_SERVICE_NAME
			+ "=" 
			+ svcName);
		}
	    }
            sender.send(mesg);

        } catch (Exception e) {
	    handleSendExceptions(e);
        }
    }

    public Vector receiveGetServicesReplyMessage() throws BrokerAdminException {
	return receiveGetServicesReplyMessage(true);
    }

    public Vector receiveGetServicesReplyMessage(boolean waitForResponse) 
	throws BrokerAdminException {
	BrokerAdminException bae;

        if (debug) 
	Globals.stdOutPrintln("***** receiveGetServicesReplyMessage *****");
        ObjectMessage mesg = null;

        try {
            mesg = (ObjectMessage)receiveCheckMessageTimeout(false, waitForResponse);

            mesg.acknowledge();
	    clearStatusEvent();

	    checkReplyTypeStatus(mesg, MessageType.GET_SERVICES_REPLY, "GET_SERVICES_REPLY");

            Object obj;

            if ((obj = mesg.getObject()) != null) {
                if (obj instanceof Vector) {
		    if (debug)  {
		        printServiceInfoList((Vector)obj);
		    }
                    return (Vector)obj;
                }
            }

        } catch (Exception e) {
	    handleReceiveExceptions(e);
        }

        return null;
    }


    public void sendPauseMessage(String svcName) 
				throws BrokerAdminException  {

        if (debug) Globals.stdOutPrintln("***** sendPauseMessage *****");
        ObjectMessage mesg = null;

	checkIfBusy();

        try {
            mesg = session.createObjectMessage();
            mesg.setJMSReplyTo(replyQueue);
            mesg.setIntProperty
		(MessageType.JMQ_MESSAGE_TYPE, MessageType.PAUSE);
	    mesg.setStringProperty(MessageType.JMQ_PAUSE_TARGET, MessageType.JMQ_SERVICE_NAME);
	    if (svcName != null)  {
	        mesg.setStringProperty(MessageType.JMQ_SERVICE_NAME, svcName);
	        statusEvent = createStatusEvent(BrokerCmdStatusEvent.PAUSE_SVC,
						MessageType.PAUSE_REPLY,
						"PAUSE_REPLY");
	        statusEvent.setServiceName(svcName);
	    } else  {
	        statusEvent = createStatusEvent(BrokerCmdStatusEvent.PAUSE_BKR,
						MessageType.PAUSE_REPLY,
						"PAUSE_REPLY");
	    }

	    if (debug)  {
		printMsgType(MessageType.PAUSE, "PAUSE");
		if (svcName != null)  {
		    Globals.stdOutPrintln("\t"
			+ MessageType.JMQ_SERVICE_NAME
			+ "=" 
			+ svcName);
		}
	    }
            sender.send(mesg);
        } catch (Exception e) {
	    handleSendExceptions(e);
        }
    }

    public void sendPauseMessage(String dstName, int dstType, int pauseType) 
				throws BrokerAdminException  {

        if (debug) Globals.stdOutPrintln("***** sendPauseMessage *****");
        ObjectMessage mesg = null;

	checkIfBusy();

        try {
            mesg = session.createObjectMessage();
            mesg.setJMSReplyTo(replyQueue);
            mesg.setIntProperty
		(MessageType.JMQ_MESSAGE_TYPE, MessageType.PAUSE);
	    mesg.setStringProperty(MessageType.JMQ_PAUSE_TARGET, MessageType.JMQ_DESTINATION);
	    if (dstName != null)  {
	        mesg.setStringProperty(MessageType.JMQ_DESTINATION, dstName);
	        mesg.setIntProperty(MessageType.JMQ_DEST_TYPE, dstType); 
	    }
	    if (pauseType != DestState.UNKNOWN)  {
	        mesg.setIntProperty(MessageType.JMQ_DEST_STATE, pauseType); 
	    }

	    statusEvent = createStatusEvent(BrokerCmdStatusEvent.PAUSE_DST,
						MessageType.PAUSE_REPLY,
						"PAUSE_REPLY");
	    statusEvent.setDestinationName(dstName);
	    statusEvent.setDestinationType(dstType);


	    if (debug)  {
		printMsgType(MessageType.PAUSE, "PAUSE");
		Globals.stdOutPrintln("\t"
			+ MessageType.JMQ_DESTINATION
			+ "=" 
			+ dstName);

		Globals.stdOutPrintln("\t"
			+ MessageType.JMQ_DEST_TYPE
			+ "=" 
			+ dstType);

		Globals.stdOutPrintln("\t"
			+ MessageType.JMQ_DEST_STATE
			+ "=" 
			+ pauseType);
	    }
            sender.send(mesg);
        } catch (Exception e) {
	    handleSendExceptions(e);
        }
    }


    public void receivePauseReplyMessage() throws BrokerAdminException {

        if (debug) Globals.stdOutPrintln("***** receivePauseReplyMessage *****");
        Message mesg = null;

        try {
            mesg = (ObjectMessage)receiveCheckMessageTimeout(false);

            mesg.acknowledge();
	    clearStatusEvent();
	    checkReplyTypeStatus(mesg, MessageType.PAUSE_REPLY, "PAUSE_REPLY");
        } catch (Exception e) {
	    handleReceiveExceptions(e);
        }
    }

    public void sendResetBrokerMessage(String resetType) 
				throws BrokerAdminException  {

        if (debug) Globals.stdOutPrintln("***** sendResetBrokerMessage *****");
        ObjectMessage mesg = null;

	checkIfBusy();

        try {
            mesg = session.createObjectMessage();
            mesg.setJMSReplyTo(replyQueue);
            mesg.setIntProperty
		(MessageType.JMQ_MESSAGE_TYPE, MessageType.RESET_BROKER);
	    if (resetType != null)  {
	        mesg.setStringProperty(MessageType.JMQ_RESET_TYPE, resetType); 
	    }

	    statusEvent = createStatusEvent(BrokerCmdStatusEvent.RESET_BKR,
						MessageType.RESET_BROKER_REPLY,
						"RESET_BROKER_REPLY");

	    if (debug)  {
		printMsgType(MessageType.RESET_BROKER, "RESET_BROKER");
		Globals.stdOutPrintln("\t"
			+ MessageType.JMQ_RESET_TYPE
			+ "=" 
			+ resetType);
	    }
            sender.send(mesg);
        } catch (Exception e) {
	    handleSendExceptions(e);
        }
    }


    public void receiveResetBrokerReplyMessage() throws BrokerAdminException {

        if (debug) Globals.stdOutPrintln("***** receiveResetBrokerReplyMessage *****");
        Message mesg = null;

        try {
            mesg = (ObjectMessage)receiveCheckMessageTimeout(false);

            mesg.acknowledge();
	    clearStatusEvent();
	    checkReplyTypeStatus(mesg, MessageType.RESET_BROKER_REPLY, "RESET_BROKER_REPLY");
        } catch (Exception e) {
	    handleReceiveExceptions(e);
        }
    }



    public void sendResumeMessage(String svcName) throws BrokerAdminException {

        if (debug) Globals.stdOutPrintln("***** sendResumeMessage *****");
        ObjectMessage mesg = null;

	checkIfBusy();

        try {
            mesg = session.createObjectMessage();
            mesg.setJMSReplyTo(replyQueue);
            mesg.setIntProperty
		(MessageType.JMQ_MESSAGE_TYPE, MessageType.RESUME);
	    mesg.setStringProperty(MessageType.JMQ_PAUSE_TARGET, MessageType.JMQ_SERVICE_NAME);
	    if (svcName != null)  {
	        mesg.setStringProperty(MessageType.JMQ_SERVICE_NAME, svcName);
	        statusEvent = createStatusEvent(BrokerCmdStatusEvent.RESUME_SVC,
						MessageType.RESUME_REPLY,
						"RESUME_REPLY");
	        statusEvent.setServiceName(svcName);
	    } else  {
	        statusEvent = createStatusEvent(BrokerCmdStatusEvent.RESUME_BKR,
						MessageType.RESUME_REPLY,
						"RESUME_REPLY");
	    }

	    if (debug)  {
		printMsgType(MessageType.RESUME, "RESUME");
		if (svcName != null)  {
		    Globals.stdOutPrintln("\t"
			+ MessageType.JMQ_SERVICE_NAME
			+ "=" 
			+ svcName);
		}
	    }
            sender.send(mesg);
        } catch (Exception e) {
	    handleSendExceptions(e);
        }
    }

    public void sendResumeMessage(String dstName, int dstType) 
					throws BrokerAdminException {

        if (debug) Globals.stdOutPrintln("***** sendResumeMessage *****");
        ObjectMessage mesg = null;

	checkIfBusy();

        try {
            mesg = session.createObjectMessage();
            mesg.setJMSReplyTo(replyQueue);
            mesg.setIntProperty
		(MessageType.JMQ_MESSAGE_TYPE, MessageType.RESUME);
	    mesg.setStringProperty(MessageType.JMQ_PAUSE_TARGET, MessageType.JMQ_DESTINATION);
	    if (dstName != null)  {
	        mesg.setStringProperty(MessageType.JMQ_DESTINATION, dstName);
	        mesg.setIntProperty(MessageType.JMQ_DEST_TYPE, dstType); 
	    }

	    statusEvent = createStatusEvent(BrokerCmdStatusEvent.RESUME_DST,
						MessageType.RESUME_REPLY,
						"RESUME_REPLY");
	    statusEvent.setDestinationName(dstName);
	    statusEvent.setDestinationType(dstType);

	    if (debug)  {
		printMsgType(MessageType.RESUME, "RESUME");
		Globals.stdOutPrintln("\t"
			+ MessageType.JMQ_DESTINATION
			+ "=" 
			+ dstName);

		Globals.stdOutPrintln("\t"
			+ MessageType.JMQ_DEST_TYPE
			+ "=" 
			+ dstType);
	    }
            sender.send(mesg);
        } catch (Exception e) {
	    handleSendExceptions(e);
        }
    }


    public void receiveResumeReplyMessage() throws BrokerAdminException {

        if (debug) Globals.stdOutPrintln("***** receiveResumeReplyMessage *****");
        Message mesg = null;

        try {
            mesg = (ObjectMessage)receiveCheckMessageTimeout(false);

            mesg.acknowledge();
	    clearStatusEvent();
	    checkReplyTypeStatus(mesg, MessageType.RESUME_REPLY, "RESUME_REPLY");
        } catch (Exception e) {
	    handleReceiveExceptions(e);
        }
    }


    public void sendQuiesceMessage() throws BrokerAdminException {

        if (debug) Globals.stdOutPrintln("***** sendQuiesceMessage *****");
        ObjectMessage mesg = null;

	checkIfBusy();

        try {
            mesg = session.createObjectMessage();
            mesg.setJMSReplyTo(replyQueue);
            mesg.setIntProperty
		(MessageType.JMQ_MESSAGE_TYPE, MessageType.QUIESCE_BROKER);

	    statusEvent = createStatusEvent(BrokerCmdStatusEvent.QUIESCE_BKR,
						MessageType.QUIESCE_BROKER_REPLY,
						"QUIESCE_BROKER_REPLY");

	    if (debug)  {
		printMsgType(MessageType.QUIESCE_BROKER, "QUIESCE_BROKER");
	    }
            sender.send(mesg);
        } catch (Exception e) {
	    handleSendExceptions(e);
        }
    }

    public void receiveQuiesceReplyMessage() throws BrokerAdminException {

        if (debug) Globals.stdOutPrintln("***** receiveQuiesceReplyMessage *****");
        Message mesg = null;

        try {
            mesg = (ObjectMessage)receiveCheckMessageTimeout(false);

            mesg.acknowledge();
	    clearStatusEvent();
	    checkReplyTypeStatus(mesg, MessageType.QUIESCE_BROKER_REPLY, "QUIESCE_BROKER_REPLY");
        } catch (Exception e) {
	    handleReceiveExceptions(e);
        }
    }


    public void sendUnquiesceMessage() throws BrokerAdminException {

        if (debug) Globals.stdOutPrintln("***** sendUnquiesceMessage *****");
        ObjectMessage mesg = null;

	checkIfBusy();

        try {
            mesg = session.createObjectMessage();
            mesg.setJMSReplyTo(replyQueue);
            mesg.setIntProperty
		(MessageType.JMQ_MESSAGE_TYPE, MessageType.UNQUIESCE_BROKER);

	    statusEvent = createStatusEvent(BrokerCmdStatusEvent.UNQUIESCE_BKR,
						MessageType.UNQUIESCE_BROKER_REPLY,
						"UNQUIESCE_BROKER_REPLY");

	    if (debug)  {
		printMsgType(MessageType.UNQUIESCE_BROKER, "UNQUIESCE_BROKER");
	    }
            sender.send(mesg);
        } catch (Exception e) {
	    handleSendExceptions(e);
        }
    }

    public void receiveUnquiesceReplyMessage() throws BrokerAdminException {

        if (debug) Globals.stdOutPrintln("***** receiveUnquiesceReplyMessage *****");
        Message mesg = null;

        try {
            mesg = (ObjectMessage)receiveCheckMessageTimeout(false);

            mesg.acknowledge();
	    clearStatusEvent();
	    checkReplyTypeStatus(mesg, MessageType.UNQUIESCE_BROKER_REPLY, "UNQUIESCE_BROKER_REPLY");
        } catch (Exception e) {
	    handleReceiveExceptions(e);
        }
    }



    public void sendTakeoverMessage(String brokerID) throws BrokerAdminException {

        if (debug) Globals.stdOutPrintln("***** sendTakeoverMessage *****");
        ObjectMessage mesg = null;

	checkIfBusy();

        try {
            mesg = session.createObjectMessage();
            mesg.setJMSReplyTo(replyQueue);
            mesg.setIntProperty
		(MessageType.JMQ_MESSAGE_TYPE, MessageType.TAKEOVER_BROKER);
	    mesg.setStringProperty(MessageType.JMQ_BROKER_ID, brokerID);

	    statusEvent = createStatusEvent(BrokerCmdStatusEvent.TAKEOVER_BKR,
						MessageType.TAKEOVER_BROKER_REPLY,
						"TAKEOVER_BROKER_REPLY");

	    if (debug)  {
		printMsgType(MessageType.TAKEOVER_BROKER, "TAKEOVER_BROKER");
		Globals.stdOutPrintln("\t"
			+ MessageType.JMQ_BROKER_ID
			+ " = "
			+ brokerID);
	    }
            sender.send(mesg);
        } catch (Exception e) {
	    handleSendExceptions(e);
        }
    }

    public void receiveTakeoverReplyMessage() throws BrokerAdminException {

        if (debug) Globals.stdOutPrintln("***** receiveTakeoverReplyMessage *****");
        Message mesg = null;

        try {
            mesg = (ObjectMessage)receiveCheckMessageTimeout(false);

            mesg.acknowledge();
	    clearStatusEvent();
	    checkReplyTypeStatus(mesg, MessageType.TAKEOVER_BROKER_REPLY, 
				"TAKEOVER_BROKER_REPLY");
        } catch (Exception e) {
	    handleReceiveExceptions(e);
        }
    }

    public void sendGetDestinationsMessage(String dstName, int dstType) 
			throws BrokerAdminException {

        if (debug) Globals.stdOutPrintln ("***** sendGetDestinationsMessage *****");
        ObjectMessage mesg = null;

	checkIfBusy();

        try {
            mesg = session.createObjectMessage();
            mesg.setJMSReplyTo(replyQueue);
            mesg.setIntProperty
                (MessageType.JMQ_MESSAGE_TYPE, MessageType.GET_DESTINATIONS);


	    if (dstName != null)  {
	        mesg.setStringProperty(MessageType.JMQ_DESTINATION, dstName);
	        mesg.setIntProperty(MessageType.JMQ_DEST_TYPE, dstType); 

	        statusEvent = createStatusEvent(BrokerCmdStatusEvent.QUERY_DST,
						MessageType.GET_DESTINATIONS_REPLY,
						"GET_DESTINATIONS_REPLY");
	        statusEvent.setDestinationName(dstName);
	        statusEvent.setDestinationType(dstType);
	    } else  {
	        statusEvent = createStatusEvent(BrokerCmdStatusEvent.LIST_DST,
						MessageType.GET_DESTINATIONS_REPLY,
						"GET_DESTINATIONS_REPLY");
	    }

	    if (debug)  {
		printMsgType(MessageType.GET_DESTINATIONS, "GET_DESTINATIONS");
		if (dstName != null)  {
		    Globals.stdOutPrintln("\t"
			+ MessageType.JMQ_DESTINATION
			+ "=" 
			+ dstName);
		    Globals.stdOutPrintln("\t"
			+ MessageType.JMQ_DEST_TYPE
			+ "=" 
			+ dstType);
		}
	    }
	    sender.send(mesg);
        } catch (Exception e) {
	    handleSendExceptions(e);
        }
    }

    public Vector receiveGetDestinationsReplyMessage() throws BrokerAdminException {
	return receiveGetDestinationsReplyMessage(true);
    }

    public Vector receiveGetDestinationsReplyMessage(boolean waitForResponse)
					throws BrokerAdminException {
        if (debug)
        Globals.stdOutPrintln("***** receiveGetDestinationsReplyMessage *****");
        ObjectMessage mesg = null;

        try {
            mesg = (ObjectMessage)receiveCheckMessageTimeout(false, waitForResponse);

            mesg.acknowledge();
	    clearStatusEvent();

	    checkReplyTypeStatus(mesg, MessageType.GET_DESTINATIONS_REPLY, "GET_DESTINATIONS_REPLY");

            Object obj;

            if ((obj = mesg.getObject()) != null) {
                if (obj instanceof Vector)  {
		    if (debug)  {
		        printDestinationInfoList((Vector)obj);
		    }
                    return (Vector)obj;
		}
            }
        } catch (Exception e) {
	    handleReceiveExceptions(e);
        }

        return null;
    }


    public void sendCreateDestinationMessage(DestinationInfo dstInfo) 
				throws BrokerAdminException {

        if (debug) 
	    Globals.stdOutPrintln("***** sendCreateDestinationMessage *****");
        ObjectMessage mesg = null;

	checkIfBusy();

        try {
            mesg = session.createObjectMessage();
            mesg.setJMSReplyTo(replyQueue);
            mesg.setIntProperty
	    (MessageType.JMQ_MESSAGE_TYPE, MessageType.CREATE_DESTINATION);
            mesg.setObject(dstInfo);

	    statusEvent = createStatusEvent(BrokerCmdStatusEvent.CREATE_DST,
						MessageType.CREATE_DESTINATION_REPLY,
						"CREATE_DESTINATION_REPLY");
	    statusEvent.setDestinationInfo(dstInfo);

	    if (debug)  {
		printMsgType(MessageType.CREATE_DESTINATION, "CREATE_DESTINATION");
		printDestinationInfo(dstInfo);
	    }
            sender.send(mesg);
        } catch (Exception e) {
	    handleSendExceptions(e);
        }
    }


    public void receiveCreateDestinationReplyMessage() 
	throws BrokerAdminException {

        if (debug) 
	Globals.stdOutPrintln("***** receiveCreateDestinationReplyMessage *****");
        Message mesg = null;

        try {
            mesg = (ObjectMessage)receiveCheckMessageTimeout(false);

            mesg.acknowledge();
	    clearStatusEvent();

	    checkReplyTypeStatus(mesg, MessageType.CREATE_DESTINATION_REPLY,
				"CREATE_DESTINATION_REPLY");
        } catch (Exception e) {
	    handleReceiveExceptions(e);
        }
    }


    public void sendDestroyDestinationMessage(String dstName, int dstType) 
			throws BrokerAdminException {

        if (debug) 
	Globals.stdOutPrintln("***** sendDestroyDestinationMessage *****");
        ObjectMessage mesg = null;

	checkIfBusy();

        try {
            mesg = session.createObjectMessage();
            mesg.setJMSReplyTo(replyQueue);
            mesg.setIntProperty
            (MessageType.JMQ_MESSAGE_TYPE, MessageType.DESTROY_DESTINATION);
            mesg.setStringProperty(MessageType.JMQ_DESTINATION, dstName);
            mesg.setIntProperty(MessageType.JMQ_DEST_TYPE, dstType);

	    statusEvent = createStatusEvent(BrokerCmdStatusEvent.DESTROY_DST,
						MessageType.DESTROY_DESTINATION_REPLY,
						"DESTROY_DESTINATION_REPLY");
	    statusEvent.setDestinationName(dstName);
	    statusEvent.setDestinationType(dstType);

	    if (debug)  {
		printMsgType(MessageType.DESTROY_DESTINATION, "DESTROY_DESTINATION");
		Globals.stdOutPrintln("\t"
			+ MessageType.JMQ_DESTINATION
			+ "=" 
			+ dstName);
		Globals.stdOutPrintln("\t"
			+ MessageType.JMQ_DEST_TYPE
			+ "=" 
			+ dstType);
	    }
            sender.send(mesg);
        } catch (Exception e) {
	    handleSendExceptions(e);
        }
    }


    public void receiveDestroyDestinationReplyMessage() 
				throws BrokerAdminException {
        if (debug) 
	Globals.stdOutPrintln("***** receiveDestroyDestinationReplyMessage *****");
        Message mesg = null;

        try {
            mesg = (ObjectMessage)receiveCheckMessageTimeout(false);

            mesg.acknowledge();
	    clearStatusEvent();
	    checkReplyTypeStatus(mesg, MessageType.DESTROY_DESTINATION_REPLY,
				"DESTROY_DESTINATION_REPLY");
        } catch (Exception e) {
	    handleReceiveExceptions(e);
        }
    }


    public void sendPurgeDestinationMessage(String dstName, int dstType) 
				throws BrokerAdminException {

        if (debug) 
	Globals.stdOutPrintln("***** sendPurgeDestinationMessage *****");
        ObjectMessage mesg = null;

	checkIfBusy();

        try {
            mesg = session.createObjectMessage();
            mesg.setJMSReplyTo(replyQueue);
            mesg.setIntProperty
            (MessageType.JMQ_MESSAGE_TYPE, MessageType.PURGE_DESTINATION);
            mesg.setStringProperty(MessageType.JMQ_DESTINATION, dstName);
            mesg.setIntProperty(MessageType.JMQ_DEST_TYPE, dstType);

	    statusEvent = createStatusEvent(BrokerCmdStatusEvent.PURGE_DST,
						MessageType.PURGE_DESTINATION_REPLY,
						"PURGE_DESTINATION_REPLY");
	    statusEvent.setDestinationName(dstName);
	    statusEvent.setDestinationType(dstType);

	    if (debug)  {
		printMsgType(MessageType.PURGE_DESTINATION, "PURGE_DESTINATION");
		Globals.stdOutPrintln("\t"
			+ MessageType.JMQ_DESTINATION 
			+ "=" 
			+ dstName);
		Globals.stdOutPrintln("\t" 
			+ MessageType.JMQ_DEST_TYPE 
			+ "=" 
			+ dstType);
	    }

            sender.send(mesg);
        } catch (Exception e) {
	    handleSendExceptions(e);
        }
    }


    public void receivePurgeDestinationReplyMessage() 
				throws BrokerAdminException {
        if (debug)
	Globals.stdOutPrintln("***** receivePurgeDestinationReplyMessage *****");
        Message mesg = null;

        try {
            mesg = (ObjectMessage)receiveCheckMessageTimeout(false);

            mesg.acknowledge();
	    clearStatusEvent();

	    checkReplyTypeStatus(mesg, MessageType.PURGE_DESTINATION_REPLY,
				"PURGE_DESTINATION_REPLY");
        } catch (Exception e) {
	    handleReceiveExceptions(e);
        }
    }


    public void sendGetBrokerPropsMessage() 
				throws BrokerAdminException {
	BrokerAdminException bae;

        if (debug) Globals.stdOutPrintln("***** sendGetBrokerPropsMessage *****");
        ObjectMessage mesg = null;

	checkIfBusy();

        try {
            mesg = session.createObjectMessage();
            mesg.setJMSReplyTo(replyQueue);
            mesg.setIntProperty
		(MessageType.JMQ_MESSAGE_TYPE, MessageType.GET_BROKER_PROPS);

	    statusEvent = createStatusEvent(BrokerCmdStatusEvent.QUERY_BKR,
						MessageType.GET_BROKER_PROPS_REPLY,
						"GET_BROKER_PROPS_REPLY");

	    if (debug)  {
		printMsgType(MessageType.GET_BROKER_PROPS, "GET_BROKER_PROPS");
	    }
            sender.send(mesg);

        } catch (Exception e) {
	    handleSendExceptions(e);
        }
    }

    public Properties receiveGetBrokerPropsReplyMessage() throws BrokerAdminException {
	return receiveGetBrokerPropsReplyMessage(true);
    }

    public Properties receiveGetBrokerPropsReplyMessage(boolean waitForResponse) 
	throws BrokerAdminException {

        if (debug) 
	Globals.stdOutPrintln("***** receiveGetBrokerPropsReplyMessage *****");
        ObjectMessage mesg = null;

        try {
            mesg = (ObjectMessage)receiveCheckMessageTimeout(false, waitForResponse);
            mesg.acknowledge();
	    clearStatusEvent();

	    checkReplyTypeStatus(mesg, MessageType.GET_BROKER_PROPS_REPLY,
				"GET_BROKER_PROPS_REPLY");

            Object obj;

            if ((obj = mesg.getObject()) != null) {
                if (obj instanceof Properties)
                    return (Properties)obj;
            }

        } catch (Exception e) {
	    handleReceiveExceptions(e);
        }

        return null;
    }


    public void sendUpdateBrokerPropsMessage(Properties props) 
				throws BrokerAdminException {
        if (debug) 
	Globals.stdOutPrintln("***** sendUpdateBrokerPropsMessage *****");
        ObjectMessage mesg = null;

	checkIfBusy();

        try {
            mesg = session.createObjectMessage();
            mesg.setJMSReplyTo(replyQueue);
            mesg.setIntProperty
            (MessageType.JMQ_MESSAGE_TYPE, MessageType.UPDATE_BROKER_PROPS);
	    mesg.setObject(props);

	    statusEvent = createStatusEvent(BrokerCmdStatusEvent.UPDATE_BKR,
						MessageType.UPDATE_BROKER_PROPS_REPLY,
						"UPDATE_BROKER_PROPS_REPLY");
	    statusEvent.setBrokerProperties(props);

	    if (debug)  {
		printMsgType(MessageType.UPDATE_BROKER_PROPS, "UPDATE_BROKER_PROPS");
		Globals.stdOutPrintln("\tProperties=" 
			+ props.toString());
	    }
            sender.send(mesg);
        } catch (Exception e) {
	    handleSendExceptions(e);
        }
    }

    public void receiveUpdateBrokerPropsReplyMessage()
        			throws BrokerAdminException {
        if (debug)
        Globals.stdOutPrintln("***** receiveUpdateBrokerPropsReplyMessage *****");
        Message mesg = null;

        try {
            mesg = (ObjectMessage)receiveCheckMessageTimeout(false);

            mesg.acknowledge();
	    clearStatusEvent();
	    checkReplyTypeStatus(mesg, MessageType.UPDATE_BROKER_PROPS_REPLY,
					"UPDATE_BROKER_PROPS_REPLY");
        } catch (Exception e) {
	    handleReceiveExceptions(e);
        }
    }


    public void sendUpdateDestinationMessage(DestinationInfo dstInfo) 
				throws BrokerAdminException {

        if (debug)
        Globals.stdOutPrintln("***** sendUpdateDestinationMessage *****");
        ObjectMessage mesg = null;

	checkIfBusy();

        try {
            mesg = session.createObjectMessage();
            mesg.setJMSReplyTo(replyQueue);
            mesg.setIntProperty
            (MessageType.JMQ_MESSAGE_TYPE, MessageType.UPDATE_DESTINATION);
            mesg.setObject(dstInfo);

	    statusEvent = createStatusEvent(BrokerCmdStatusEvent.UPDATE_DST,
						MessageType.UPDATE_DESTINATION_REPLY,
						"UPDATE_DESTINATION_REPLY");
	    statusEvent.setDestinationInfo(dstInfo);

	    if (debug)  {
		printMsgType(MessageType.UPDATE_DESTINATION, "UPDATE_DESTINATION");
		printDestinationInfo(dstInfo);
	    }
            sender.send(mesg);
        } catch (Exception e) {
	    handleSendExceptions(e);
        }
    }


    public void receiveUpdateDestinationReplyMessage()
        			throws BrokerAdminException {

        if (debug)
        Globals.stdOutPrintln("***** receiveUpdateDestinationReplyMessage *****");
        Message mesg = null;

        try {
            mesg = (ObjectMessage)receiveCheckMessageTimeout(false);

            mesg.acknowledge();
	    clearStatusEvent();
	    checkReplyTypeStatus(mesg, MessageType.UPDATE_DESTINATION_REPLY,
					"UPDATE_DESTINATION_REPLY");
        } catch (Exception e) {
	    handleReceiveExceptions(e);
        }
    }


    public void sendUpdateServiceMessage(ServiceInfo svcInfo) 
				throws BrokerAdminException {

        if (debug)
        Globals.stdOutPrintln("***** sendUpdateServiceMessage *****");
        ObjectMessage mesg = null;

	checkIfBusy();

        try {
            mesg = session.createObjectMessage();
            mesg.setJMSReplyTo(replyQueue);
            mesg.setIntProperty
            (MessageType.JMQ_MESSAGE_TYPE, MessageType.UPDATE_SERVICE);
            mesg.setObject(svcInfo);

	    statusEvent = createStatusEvent(BrokerCmdStatusEvent.UPDATE_SVC,
						MessageType.UPDATE_SERVICE_REPLY,
						"UPDATE_SERVICE_REPLY");
	    statusEvent.setServiceInfo(svcInfo);

	    if (debug)  {
		printMsgType(MessageType.UPDATE_SERVICE, "UPDATE_SERVICE");
		printServiceInfo(svcInfo);
	    }
            sender.send(mesg);
        } catch (Exception e) {
	    handleSendExceptions(e);
        }
    }


    public void receiveUpdateServiceReplyMessage()
        			throws BrokerAdminException {
        if (debug)
        Globals.stdOutPrintln("***** receiveUpdateServiceReplyMessage *****");
        Message mesg = null;

        try {
            mesg = (ObjectMessage)receiveCheckMessageTimeout(false);

            mesg.acknowledge();
	    clearStatusEvent();
	    checkReplyTypeStatus(mesg, MessageType.UPDATE_SERVICE_REPLY,
					"UPDATE_SERVICE_REPLY");
        } catch (Exception e) {
	    handleReceiveExceptions(e);
        }
    }


    public void sendShutdownMessage(boolean restart) 
				throws BrokerAdminException {
        sendShutdownMessage(restart, false);
    }

    public void sendShutdownMessage(boolean restart, boolean kill) 
				throws BrokerAdminException {
        sendShutdownMessage(restart, kill, false, -1);
    }

    public void sendShutdownMessage(boolean restart, boolean kill,
			boolean noFailover, int time) 
				throws BrokerAdminException {
        if (debug) Globals.stdOutPrintln("***** sendShutdownMessage *****");
        ObjectMessage mesg = null;

	checkIfBusy();

        try {
            mesg = session.createObjectMessage();
            mesg.setJMSReplyTo(replyQueue);
            mesg.setIntProperty
	    (MessageType.JMQ_MESSAGE_TYPE, MessageType.SHUTDOWN);
	    if (restart)  {
	        statusEvent = createStatusEvent(BrokerCmdStatusEvent.RESTART_BKR,
						MessageType.SHUTDOWN_REPLY,
						"SHUTDOWN_REPLY");
                mesg.setBooleanProperty(MessageType.JMQ_RESTART, true);
	    } else  {
	        statusEvent = createStatusEvent(BrokerCmdStatusEvent.SHUTDOWN_BKR,
						MessageType.SHUTDOWN_REPLY,
						"SHUTDOWN_REPLY");

                mesg.setBooleanProperty(MessageType.JMQ_NO_FAILOVER, noFailover);
		if (time > 0)  {
                    mesg.setIntProperty(MessageType.JMQ_TIME, time);
		}
	    }

	    if (kill)  {
                mesg.setBooleanProperty(MessageType.JMQ_KILL, true);
	    }


	    if (debug)  {
		printMsgType(MessageType.SHUTDOWN, "SHUTDOWN");
		if (restart)  {
		    Globals.stdOutPrintln("\t"
			+ MessageType.JMQ_RESTART
			+ "=true");
		} else  {
		    Globals.stdOutPrintln("\t"
			+ MessageType.JMQ_NO_FAILOVER
			+ "="
			+ noFailover);
		    Globals.stdOutPrintln("\t"
			+ MessageType.JMQ_TIME
			+ "="
			+ time);
		}
		if (kill)  {
		    Globals.stdOutPrintln("\t"
			+ MessageType.JMQ_KILL
			+ "=true");
		}
	    }
            sender.send(mesg);
        } catch (Exception e) {
	    handleSendExceptions(e);
        }
    }


    public void receiveShutdownReplyMessage() 
				throws BrokerAdminException {
        if (debug) 
	Globals.stdOutPrintln("***** receiveShutdownReplyMessage *****");
        Message mesg = null;

        try {
            mesg = receiver.receive(timeout);
	    /* 
	     * DO NOT ack shutdown, as the broker is already GONE!
             * mesg.acknowledge();
	     */
	    /* 
	     * Message can be null if receive() times out.
	     * On shutdownReply, it can be null if the broker shuts
	     * down prior to this method receiving the message.  If the 
	     * message is null, simply treat it as successful.  This
	     * is done in checkReplyTypeStatus() method.
	     */
	     checkReplyTypeStatus(mesg, MessageType.SHUTDOWN_REPLY,
	    			 "SHUTDOWN_REPLY");

        } catch (JMSException jmsee) {
	    /* 
	     * One exception that we will most likely encounter is 
	     * javax.jms.IllegalStateException.
	     * We may run into this state when receive() is called
	     * after session is closed.  Similar to the null
	     * message case above, we treat this as successful.
	     * We are ignoring any JMSExceptions, since most likely the
	     * shutdown of the broker is successful when a JMSException is
	     * thrown.
	     */
        } catch (Exception e) {
	    handleReceiveExceptions(e);
        }
    }


    /**
     * Note: The protocol assumes that the destination is of type topic.
     */
    public void sendGetDurablesMessage(String topicName, String durName)
                                throws BrokerAdminException {
        BrokerAdminException bae;

        if (debug) Globals.stdOutPrintln("***** sendGetDurablesMessage *****");
        ObjectMessage mesg = null;

	checkIfBusy();

        try {
            mesg = session.createObjectMessage();
            mesg.setJMSReplyTo(replyQueue);
            mesg.setIntProperty
                (MessageType.JMQ_MESSAGE_TYPE, MessageType.GET_DURABLES);
            mesg.setStringProperty(MessageType.JMQ_DESTINATION, topicName);
            if (durName != null)  {
                mesg.setStringProperty(MessageType.JMQ_DURABLE_NAME, durName);
            }

	    statusEvent = createStatusEvent(BrokerCmdStatusEvent.LIST_DUR,
						MessageType.GET_DURABLES_REPLY,
						"GET_DURABLES_REPLY");
	    statusEvent.setDestinationName(topicName);
	    statusEvent.setDurableName(durName);

	    if (debug)  {
		printMsgType(MessageType.GET_DURABLES, "GET_DURABLES");
		Globals.stdOutPrintln("\t"
			+ MessageType.JMQ_DESTINATION 
			+ "=" 
			+ topicName);
		if (durName != null)  {
		    Globals.stdOutPrintln("\t" 
			+ MessageType.JMQ_DURABLE_NAME 
			+ "=" 
			+ durName);
	        }
	    }
            sender.send(mesg);

        } catch (Exception e) {
            handleSendExceptions(e);
        }
    }

    public Vector receiveGetDurablesReplyMessage() throws BrokerAdminException {
	return receiveGetDurablesReplyMessage(true);
    }

    public Vector receiveGetDurablesReplyMessage(boolean waitForResponse) 
	throws BrokerAdminException {

        if (debug)
        Globals.stdOutPrintln("***** receiveGetDurablesReplyMessage *****");
        ObjectMessage mesg = null;

        try {
            mesg = (ObjectMessage)receiveCheckMessageTimeout(false, waitForResponse);

            mesg.acknowledge();
	    clearStatusEvent();
            checkReplyTypeStatus(mesg, MessageType.GET_DURABLES_REPLY,
					"GET_DURABLES_REPLY");

            Object obj;

            if ((obj = mesg.getObject()) != null) {
                if (obj instanceof Vector)  {
		    if (debug)  {
		        printDurableInfoList((Vector)obj);
		    }
                    return (Vector)obj;
                }
            }

        } catch (Exception e) {
            handleReceiveExceptions(e);
        }

        return null;
    }


    public void sendDestroyDurableMessage(String durName, String clientID)
                                throws BrokerAdminException {
        BrokerAdminException bae;

        if (debug) Globals.stdOutPrintln("***** sendDestroyDurableMessage *****");
        ObjectMessage mesg = null;

	checkIfBusy();

        try {
            mesg = session.createObjectMessage();
            mesg.setJMSReplyTo(replyQueue);
            mesg.setIntProperty
                (MessageType.JMQ_MESSAGE_TYPE, MessageType.DESTROY_DURABLE);
            mesg.setStringProperty(MessageType.JMQ_DURABLE_NAME, durName);
            mesg.setStringProperty(MessageType.JMQ_CLIENT_ID, clientID);

	    statusEvent = createStatusEvent(BrokerCmdStatusEvent.DESTROY_DUR,
						MessageType.DESTROY_DURABLE_REPLY,
						"DESTROY_DURABLE_REPLY");
	    statusEvent.setDurableName(durName);
	    statusEvent.setClientID(clientID);

	    if (debug)  {
		printMsgType(MessageType.DESTROY_DURABLE, "DESTROY_DURABLE");
		Globals.stdOutPrintln("\t"
			+ MessageType.JMQ_DURABLE_NAME 
			+ "=" 
			+ durName);
		Globals.stdOutPrintln("\t" 
			+ MessageType.JMQ_CLIENT_ID 
			+ "=" 
			+ clientID);
	    }
            sender.send(mesg);

        } catch (Exception e) {
            handleSendExceptions(e);
        }
    }

    public void sendPurgeDurableMessage(String durName, String clientID)
                                throws BrokerAdminException {
        BrokerAdminException bae;

        if (debug) Globals.stdOutPrintln("***** sendPurgeDurableMessage *****");
        ObjectMessage mesg = null;

	checkIfBusy();

        try {
            mesg = session.createObjectMessage();
            mesg.setJMSReplyTo(replyQueue);
            mesg.setIntProperty
                (MessageType.JMQ_MESSAGE_TYPE, MessageType.PURGE_DURABLE);
            mesg.setStringProperty(MessageType.JMQ_DURABLE_NAME, durName);
            mesg.setStringProperty(MessageType.JMQ_CLIENT_ID, clientID);

	    statusEvent = createStatusEvent(BrokerCmdStatusEvent.PURGE_DUR,
						MessageType.PURGE_DURABLE_REPLY,
						"PURGE_DURABLE_REPLY");
	    statusEvent.setDurableName(durName);
	    statusEvent.setClientID(clientID);

	    if (debug)  {
		printMsgType(MessageType.PURGE_DURABLE, "PURGE_DURABLE");
		Globals.stdOutPrintln("\t"
			+ MessageType.JMQ_DURABLE_NAME 
			+ "=" 
			+ durName);
		Globals.stdOutPrintln("\t" 
			+ MessageType.JMQ_CLIENT_ID 
			+ "=" 
			+ clientID);
	    }
            sender.send(mesg);

        } catch (Exception e) {
            handleSendExceptions(e);
        }
    }


    public void receiveDestroyDurableReplyMessage()
                                throws BrokerAdminException {
        if (debug)
        Globals.stdOutPrintln("***** receiveDestroyDurableReplyMessage *****");
        Message mesg = null;

        try {
            mesg = (ObjectMessage)receiveCheckMessageTimeout(false);

            mesg.acknowledge();
	    clearStatusEvent();
            checkReplyTypeStatus(mesg, MessageType.DESTROY_DURABLE_REPLY,
					"DESTROY_DURABLE_REPLY");
        } catch (Exception e) {
            handleReceiveExceptions(e);
        }
    }

    public void receivePurgeDurableReplyMessage()
                                throws BrokerAdminException {
        if (debug)
        Globals.stdOutPrintln("***** receivePurgeDurableReplyMessage *****");
        Message mesg = null;

        try {
            mesg = (ObjectMessage)receiveCheckMessageTimeout(false);

            mesg.acknowledge();
	    clearStatusEvent();
            checkReplyTypeStatus(mesg, MessageType.PURGE_DURABLE_REPLY,
					"PURGE_DURABLE_REPLY");
        } catch (Exception e) {
            handleReceiveExceptions(e);
        }
    }


    public void sendGetMetricsMessage(String svcName) 
				throws BrokerAdminException {
	BrokerAdminException bae;

        if (debug) Globals.stdOutPrintln("***** sendGetMetricsMessage *****");
        ObjectMessage mesg = null;

	checkIfBusy();

        try {
            mesg = session.createObjectMessage();
            mesg.setJMSReplyTo(replyQueue);
            mesg.setIntProperty
		(MessageType.JMQ_MESSAGE_TYPE, MessageType.GET_METRICS);
	    if (svcName != null)  {
	        mesg.setStringProperty(MessageType.JMQ_SERVICE_NAME, svcName);

	        statusEvent = createStatusEvent(BrokerCmdStatusEvent.METRICS_SVC,
						MessageType.GET_METRICS_REPLY,
						"GET_METRICS_REPLY");
	        statusEvent.setServiceName(svcName);
	    } else  {
	        statusEvent = createStatusEvent(BrokerCmdStatusEvent.METRICS_BKR,
						MessageType.GET_METRICS_REPLY,
						"GET_METRICS_REPLY");
	    }

	    if (debug)  {
		printMsgType(MessageType.GET_METRICS, "GET_METRICS");
		if (svcName != null)  {
		    Globals.stdOutPrintln("\t"
			+ MessageType.JMQ_SERVICE_NAME 
			+ "=" 
			+ svcName);
		}
	    }
            sender.send(mesg);

        } catch (Exception e) {
	    handleSendExceptions(e);
        }
    }

    public void sendGetMetricsMessage(String dstName, int dstType) 
				throws BrokerAdminException {
	BrokerAdminException bae;

        if (debug) Globals.stdOutPrintln("***** sendGetMetricsMessage *****");
        ObjectMessage mesg = null;

	checkIfBusy();

        try {
            mesg = session.createObjectMessage();
            mesg.setJMSReplyTo(replyQueue);
            mesg.setIntProperty
		(MessageType.JMQ_MESSAGE_TYPE, MessageType.GET_METRICS);
	    mesg.setStringProperty(MessageType.JMQ_DESTINATION, dstName);
	    mesg.setIntProperty(MessageType.JMQ_DEST_TYPE, dstType); 

	    statusEvent = createStatusEvent(BrokerCmdStatusEvent.METRICS_DST,
						MessageType.GET_METRICS_REPLY,
						"GET_METRICS_REPLY");

	    if (debug)  {
		printMsgType(MessageType.GET_METRICS, "GET_METRICS");
		Globals.stdOutPrintln("\t"
			+ MessageType.JMQ_DESTINATION
			+ "=" 
			+ dstName);
		Globals.stdOutPrintln("\t"
			+ MessageType.JMQ_DEST_TYPE
			+ "=" 
			+ dstType);
	    }
            sender.send(mesg);

        } catch (Exception e) {
	    handleSendExceptions(e);
        }
    }


    public Object receiveGetMetricsReplyMessage() 
					throws BrokerAdminException {
	BrokerAdminException bae;

        if (debug) 
	Globals.stdOutPrintln("***** receiveGetMetricsReplyMessage *****");
        ObjectMessage mesg = null;

        try {
            mesg = (ObjectMessage)receiveCheckMessageTimeout(false);

            mesg.acknowledge();
	    clearStatusEvent();

	    checkReplyTypeStatus(mesg, MessageType.GET_METRICS_REPLY,
					"GET_METRICS_REPLY");

	    String metricType = mesg.getStringProperty(MessageType.JMQ_BODY_TYPE);
            Object obj;

            if ((obj = mesg.getObject()) != null) {
		if ("DESTINATION".equals(metricType))  {
                    if (obj instanceof DestMetricsCounters)
                        return (DestMetricsCounters)obj;
		}

		if ((metricType == null) || ("SERVICE".equals(metricType)))  {
                    if (obj instanceof MetricCounters)
                        return (MetricCounters)obj;
		}
            }

        } catch (Exception e) {
	    handleReceiveExceptions(e);
        }

        return null;
    }

    public void sendReloadClusterMessage() 
				throws BrokerAdminException {
	BrokerAdminException bae;

        if (debug) Globals.stdOutPrintln("***** sendReloadClusterMessage *****");
        ObjectMessage mesg = null;

	checkIfBusy();

        try {
            mesg = session.createObjectMessage();
            mesg.setJMSReplyTo(replyQueue);
            mesg.setIntProperty
		(MessageType.JMQ_MESSAGE_TYPE, MessageType.RELOAD_CLUSTER);

	    statusEvent = createStatusEvent(BrokerCmdStatusEvent.RELOAD_CLS,
						MessageType.RELOAD_CLUSTER_REPLY,
						"RELOAD_CLUSTER_REPLY");

	    if (debug)  {
		printMsgType(MessageType.RELOAD_CLUSTER, "RELOAD_CLUSTER");
	    }
            sender.send(mesg);

        } catch (Exception e) {
	    handleSendExceptions(e);
        }
    }

    public void receiveReloadClusterReplyMessage()
                                throws BrokerAdminException {
        if (debug)
        Globals.stdOutPrintln("***** receiveReloadClusterReplyMessage *****");
        Message mesg = null;

        try {
            mesg = (ObjectMessage)receiveCheckMessageTimeout(false);

            mesg.acknowledge();
	    clearStatusEvent();
            checkReplyTypeStatus(mesg, MessageType.RELOAD_CLUSTER_REPLY,
					"RELOAD_CLUSTER_REPLY");
        } catch (Exception e) {
            handleReceiveExceptions(e);
        }
    }

    public void sendGetClusterMessage(boolean listBkr) 
			throws BrokerAdminException {

        if (debug) Globals.stdOutPrintln ("***** sendGetClusterMessage *****");
        ObjectMessage mesg = null;

	checkIfBusy();

        try {
            mesg = session.createObjectMessage();
            mesg.setJMSReplyTo(replyQueue);
            mesg.setIntProperty
                (MessageType.JMQ_MESSAGE_TYPE, MessageType.GET_CLUSTER);


	    if (listBkr)  {
	        statusEvent = createStatusEvent(BrokerCmdStatusEvent.LIST_BKR,
						MessageType.GET_CLUSTER_REPLY,
						"GET_CLUSTER_REPLY");
	    } else  {
	        statusEvent = createStatusEvent(BrokerCmdStatusEvent.QUERY_BKR,
						MessageType.GET_CLUSTER_REPLY,
						"GET_CLUSTER_REPLY");
	    }

	    if (debug)  {
		printMsgType(MessageType.GET_CLUSTER, "GET_CLUSTER");
	    }
	    sender.send(mesg);
        } catch (Exception e) {
	    handleSendExceptions(e);
        }
    }

    public Vector receiveGetClusterReplyMessage() throws BrokerAdminException {
	return receiveGetClusterReplyMessage(true);
    }

    public Vector receiveGetClusterReplyMessage(boolean waitForResponse)
					throws BrokerAdminException {
        if (debug)
        Globals.stdOutPrintln("***** receiveGetClusterReplyMessage *****");
        ObjectMessage mesg = null;

        try {
            mesg = (ObjectMessage)receiveCheckMessageTimeout(false, waitForResponse);

            mesg.acknowledge();
	    clearStatusEvent();

	    checkReplyTypeStatus(mesg, MessageType.GET_CLUSTER_REPLY, "GET_CLUSTER_REPLY");

            Object obj;

            if ((obj = mesg.getObject()) != null) {
                if (obj instanceof Vector)  {
		    if (debug)  {
		        printClusterList((Vector)obj);
		    }
                    return (Vector)obj;
		}
            }
        } catch (Exception e) {
	    handleReceiveExceptions(e);
        }

        return null;
    }

    public void sendGetJMXConnectorsMessage(String name) 
			throws BrokerAdminException {

        if (debug) Globals.stdOutPrintln ("***** sendGetJMXConnectorsMessage *****");
        ObjectMessage mesg = null;

	checkIfBusy();

        try {
            mesg = session.createObjectMessage();
            mesg.setJMSReplyTo(replyQueue);
            mesg.setIntProperty
                (MessageType.JMQ_MESSAGE_TYPE, MessageType.GET_JMX);

	    statusEvent = createStatusEvent(BrokerCmdStatusEvent.LIST_JMX,
						MessageType.GET_JMX_REPLY,
						"GET_JMX_REPLY");

	    if (debug)  {
		printMsgType(MessageType.GET_JMX, "GET_JMX");
	    }
	    sender.send(mesg);
        } catch (Exception e) {
	    handleSendExceptions(e);
        }
    }

    public Vector receiveGetJMXConnectorsReplyMessage() throws BrokerAdminException {
	return receiveGetJMXConnectorsReplyMessage(true);
    }

    public Vector receiveGetJMXConnectorsReplyMessage(boolean waitForResponse)
					throws BrokerAdminException {
        if (debug)
        Globals.stdOutPrintln("***** receiveGetJMXConnectorsReplyMessage *****");
        ObjectMessage mesg = null;

        try {
            mesg = (ObjectMessage)receiveCheckMessageTimeout(false, waitForResponse);

            mesg.acknowledge();
	    clearStatusEvent();

	    checkReplyTypeStatus(mesg, MessageType.GET_JMX_REPLY, "GET_JMX_REPLY");

            Object obj;

            if ((obj = mesg.getObject()) != null) {
                if (obj instanceof Vector)  {
		    if (debug)  {
		        printJMXList((Vector)obj);
		    }
                    return (Vector)obj;
		}
            }
        } catch (Exception e) {
	    handleReceiveExceptions(e);
        }

        return null;
    }

    public void sendGetMessagesMessage(String dstName, int dstType, 
						boolean getBody,
						String msgID,
						Long startMessageIndex, 
						Long maxNumMsgsRetrieved) 
			throws BrokerAdminException {

        if (debug) Globals.stdOutPrintln ("***** sendGetMessagesMessage *****");
        ObjectMessage mesg = null;

	checkIfBusy();

        try {
            mesg = session.createObjectMessage();
            mesg.setJMSReplyTo(replyQueue);
            mesg.setIntProperty
                (MessageType.JMQ_MESSAGE_TYPE, MessageType.GET_MESSAGES);
	    mesg.setStringProperty(MessageType.JMQ_DESTINATION, dstName);
	    mesg.setIntProperty(MessageType.JMQ_DEST_TYPE, dstType); 
            mesg.setBooleanProperty(MessageType.JMQ_GET_MSG_BODY, getBody);
	    if (msgID != null)  {
                mesg.setStringProperty(MessageType.JMQ_MESSAGE_ID, msgID);
	    }

	    if (startMessageIndex != null)  {
                mesg.setLongProperty(MessageType.JMQ_START_MESSAGE_INDEX, 
						startMessageIndex.longValue());
	    }
	    if (maxNumMsgsRetrieved != null)  {
                mesg.setLongProperty(MessageType.JMQ_MAX_NUM_MSGS_RETRIEVED, 
						maxNumMsgsRetrieved.longValue());
	    }

	    statusEvent = createStatusEvent(BrokerCmdStatusEvent.GET_MSGS,
						MessageType.GET_MESSAGES_REPLY,
						"GET_MESSAGE_REPLY");

	    if (debug)  {
		printMsgType(MessageType.GET_MESSAGES, "GET_MESSAGES");
	    }
	    sender.send(mesg);
        } catch (Exception e) {
	    handleSendExceptions(e);
        }
    }

    public Vector receiveGetMessagesReplyMessage() throws BrokerAdminException {
	return receiveGetMessagesReplyMessage(true);
    }

    public Vector receiveGetMessagesReplyMessage(boolean waitForResponse)
					throws BrokerAdminException {
        if (debug)
        Globals.stdOutPrintln("***** receiveGetMessagesReplyMessage *****");
        ObjectMessage mesg = null;

        try {
            mesg = (ObjectMessage)receiveCheckMessageTimeout(false, waitForResponse);

            mesg.acknowledge();
	    clearStatusEvent();

	    checkReplyTypeStatus(mesg, MessageType.GET_MESSAGES_REPLY, 
							"GET_MESSAGES_REPLY");

            Object obj;

            if ((obj = mesg.getObject()) != null) {
                if (obj instanceof Vector)  {
		    /*
		    if (debug)  {
		        printJMXList((Vector)obj);
		    }
		    */
                    return (Vector)obj;
		}
            }
        } catch (Exception e) {
	    handleReceiveExceptions(e);
        }

        return null;
    }

    public void sendDestroyMessagesMessage(String dstName, int dstType, String msgID) 
			throws BrokerAdminException {

        if (debug) Globals.stdOutPrintln ("***** sendDestroyMessagesMessage *****");
        ObjectMessage mesg = null;

	checkIfBusy();

        try {
            mesg = session.createObjectMessage();
            mesg.setJMSReplyTo(replyQueue);
            mesg.setIntProperty
                (MessageType.JMQ_MESSAGE_TYPE, MessageType.DELETE_MESSAGE);
	    mesg.setStringProperty(MessageType.JMQ_DESTINATION, dstName);
	    mesg.setIntProperty(MessageType.JMQ_DEST_TYPE, dstType); 
	    if (msgID != null)  {
                mesg.setStringProperty(MessageType.JMQ_MESSAGE_ID, msgID);
	    }

	    statusEvent = createStatusEvent(BrokerCmdStatusEvent.DELETE_MSG,
						MessageType.DELETE_MESSAGE_REPLY,
						"DELETE_MESSAGE_REPLY");

	    if (debug)  {
		printMsgType(MessageType.DELETE_MESSAGE, "DELETE_MESSAGE");
	    }
	    sender.send(mesg);
        } catch (Exception e) {
	    handleSendExceptions(e);
        }
    }

    public void receiveDestroyMessagesReplyMessage() throws BrokerAdminException {
	receiveDestroyMessagesReplyMessage(true);
    }

    public void receiveDestroyMessagesReplyMessage(boolean waitForResponse)
					throws BrokerAdminException {
        if (debug)
        Globals.stdOutPrintln("***** receiveDestroyMessagesReplyMessage *****");
        ObjectMessage mesg = null;

        try {
            mesg = (ObjectMessage)receiveCheckMessageTimeout(false, waitForResponse);

            mesg.acknowledge();
	    clearStatusEvent();

	    checkReplyTypeStatus(mesg, MessageType.DELETE_MESSAGE_REPLY, 
							"DELETE_MESSAGE_REPLY");
        } catch (Exception e) {
	    handleReceiveExceptions(e);
        }
    }

    public void sendCommitTxnMessage(Long tid) 
				throws BrokerAdminException {
	BrokerAdminException bae;

        if (debug) Globals.stdOutPrintln("***** sendCommitTxnMessage *****");
        ObjectMessage mesg = null;

	checkIfBusy();

        try {
            mesg = session.createObjectMessage();
            mesg.setJMSReplyTo(replyQueue);
            mesg.setIntProperty
		(MessageType.JMQ_MESSAGE_TYPE, MessageType.COMMIT_TRANSACTION);
            mesg.setLongProperty(MessageType.JMQ_TRANSACTION_ID, tid.longValue());

	    statusEvent = createStatusEvent(BrokerCmdStatusEvent.COMMIT_TXN,
						MessageType.COMMIT_TRANSACTION_REPLY,
						"COMMIT_TRANSACTION_REPLY");

	    if (debug)  {
		printMsgType(MessageType.COMMIT_TRANSACTION, "COMMIT_TRANSACTION");
		Globals.stdOutPrintln("\t"
			+ MessageType.JMQ_TRANSACTION_ID
			+ "=" 
			+ tid.longValue());
	    }
            sender.send(mesg);

        } catch (Exception e) {
	    handleSendExceptions(e);
        }
    }

    public void receiveCommitTxnReplyMessage()
                                throws BrokerAdminException {
        if (debug)
        Globals.stdOutPrintln("***** receiveCommitTxnReplyMessage *****");
        Message mesg = null;

        try {
            mesg = (ObjectMessage)receiveCheckMessageTimeout(false);

            mesg.acknowledge();
	    clearStatusEvent();
            checkReplyTypeStatus(mesg, MessageType.COMMIT_TRANSACTION_REPLY,
					"COMMIT_TRANSACTION_REPLY");
        } catch (Exception e) {
            handleReceiveExceptions(e);
        }
    }

    public void sendRollbackTxnMessage(Long tid) 
				throws BrokerAdminException {
	BrokerAdminException bae;

        if (debug) Globals.stdOutPrintln("***** sendRollbackTxnMessage *****");
        ObjectMessage mesg = null;

	checkIfBusy();

        try {
            mesg = session.createObjectMessage();
            mesg.setJMSReplyTo(replyQueue);
            mesg.setIntProperty
		(MessageType.JMQ_MESSAGE_TYPE, MessageType.ROLLBACK_TRANSACTION);
            mesg.setLongProperty(MessageType.JMQ_TRANSACTION_ID, tid.longValue());

	    statusEvent = createStatusEvent(BrokerCmdStatusEvent.ROLLBACK_TXN,
						MessageType.ROLLBACK_TRANSACTION_REPLY,
						"ROLLBACK_TRANSACTION_REPLY");

	    if (debug)  {
		printMsgType(MessageType.ROLLBACK_TRANSACTION, "ROLLBACK_TRANSACTION");
		Globals.stdOutPrintln("\t"
			+ MessageType.JMQ_TRANSACTION_ID
			+ "=" 
			+ tid.longValue());
	    }
            sender.send(mesg);

        } catch (Exception e) {
	    handleSendExceptions(e);
        }
    }

    public void receiveRollbackTxnReplyMessage()
                                throws BrokerAdminException {
        if (debug)
        Globals.stdOutPrintln("***** receiveRollbackTxnReplyMessage *****");
        Message mesg = null;

        try {
            mesg = (ObjectMessage)receiveCheckMessageTimeout(false);

            mesg.acknowledge();
	    clearStatusEvent();
            checkReplyTypeStatus(mesg, MessageType.ROLLBACK_TRANSACTION_REPLY,
					"ROLLBACK_TRANSACTION_REPLY");
        } catch (Exception e) {
            handleReceiveExceptions(e);
        }
    }


    public void sendGetTxnsMessage(Long tid) throws BrokerAdminException {
        sendGetTxnsMessage(true, tid);
    }

    public void sendGetTxnsMessage() throws BrokerAdminException {
        sendGetTxnsMessage(false, null);
    }

    /*
     * We have a flag to indicate whether a long value was passed in or
     * not.
     * This was necessary back when tid was a 'long'. Now that it's a
     * 'Long', this is no longer needed (can check for null), but keeping
     * the same logic until everything is finalized; doesn't hurt
     * to keep it this way...
     */
    private void sendGetTxnsMessage(boolean tid_specified, Long tid) 
			throws BrokerAdminException {

        if (debug) Globals.stdOutPrintln ("***** sendGetTxnsMessage *****");
        ObjectMessage mesg = null;

	checkIfBusy();

        try {
            mesg = session.createObjectMessage();
            mesg.setJMSReplyTo(replyQueue);
            mesg.setIntProperty
                (MessageType.JMQ_MESSAGE_TYPE, MessageType.GET_TRANSACTIONS);


	    if (tid_specified)  {
                mesg.setLongProperty(MessageType.JMQ_TRANSACTION_ID, tid.longValue());

	        statusEvent = createStatusEvent(BrokerCmdStatusEvent.QUERY_TXN,
						MessageType.GET_TRANSACTIONS_REPLY,
						"GET_TRANSACTIONS_REPLY");
	        statusEvent.setTid(tid.longValue());
	    } else  {
	        statusEvent = createStatusEvent(BrokerCmdStatusEvent.LIST_TXN,
						MessageType.GET_TRANSACTIONS_REPLY,
						"GET_TRANSACTIONS_REPLY");
	    }

	    if (debug)  {
		printMsgType(MessageType.GET_TRANSACTIONS, "GET_TRANSACTIONS");
		Globals.stdOutPrintln("\t"
			+ MessageType.JMQ_TRANSACTION_ID
			+ "=");
		if (tid_specified)  {
		    Globals.stdOutPrintln(tid.toString());
		} else  {
	            Globals.stdOutPrintln("NOT SPECIFIED");
	        }
	    }
	    sender.send(mesg);
        } catch (Exception e) {
	    handleSendExceptions(e);
        }
    }

    public Vector receiveGetTxnsReplyMessage() throws BrokerAdminException {
	return receiveGetTxnsReplyMessage(true);
    }

    public Vector receiveGetTxnsReplyMessage(boolean waitForResponse)
					throws BrokerAdminException {
        if (debug)
        Globals.stdOutPrintln("***** receiveGetTxnsReplyMessage *****");
        ObjectMessage mesg = null;

        try {
            mesg = (ObjectMessage)receiveCheckMessageTimeout(false, waitForResponse);

            mesg.acknowledge();
	    clearStatusEvent();

	    checkReplyTypeStatus(mesg, MessageType.GET_TRANSACTIONS_REPLY, 
				"GET_TRANSACTIONS_REPLY");

            Object obj;

            obj = mesg.getObject();

	    if (debug)  {
	        int	quantity = 0;	
	        Globals.stdOutPrintln("obj returned: " + obj);
	        try {
	            quantity = mesg.getIntProperty(MessageType.JMQ_QUANTITY);
	        } catch (JMSException jmse)  {
	            Globals.stdOutPrintln("failed to get JMQ_QUANTITY: " + jmse);
	        }
	        Globals.stdOutPrintln("JMQ_QUANTTY: " + quantity);
	    }

            if (obj != null) {
                if (obj instanceof Vector)  {
		    if (debug)  {
		        printTxnInfoList((Vector)obj);
		    }
                    return (Vector)obj;
		}
            }
        } catch (Exception e) {
	    handleReceiveExceptions(e);
        }

        return null;
    }

    public void sendCompactDestinationMessage(String dstName, int dstType) 
				throws BrokerAdminException {

        if (debug) 
	Globals.stdOutPrintln("***** sendCompactDestinationMessage *****");
        ObjectMessage mesg = null;

	checkIfBusy();

        try {
            mesg = session.createObjectMessage();
            mesg.setJMSReplyTo(replyQueue);
            mesg.setIntProperty
            (MessageType.JMQ_MESSAGE_TYPE, MessageType.COMPACT_DESTINATION);
	    if (dstName != null)
                mesg.setStringProperty(MessageType.JMQ_DESTINATION, dstName);
	    if (dstType != -1)
                mesg.setIntProperty(MessageType.JMQ_DEST_TYPE, dstType);

	    statusEvent = createStatusEvent(BrokerCmdStatusEvent.COMPACT_DST,
						MessageType.COMPACT_DESTINATION_REPLY,
						"COMPACT_DESTINATION_REPLY");
	    statusEvent.setDestinationName(dstName);
	    statusEvent.setDestinationType(dstType);

	    if (debug)  {
		printMsgType(MessageType.COMPACT_DESTINATION, "COMPACT_DESTINATION");
		Globals.stdOutPrintln("\t"
			+ MessageType.JMQ_DESTINATION 
			+ "=" 
			+ dstName);
		Globals.stdOutPrintln("\t" 
			+ MessageType.JMQ_DEST_TYPE 
			+ "=" 
			+ dstType);
	    }

            sender.send(mesg);
        } catch (Exception e) {
	    handleSendExceptions(e);
        }
    }


    public void receiveCompactDestinationReplyMessage() 
				throws BrokerAdminException {
        if (debug)
	Globals.stdOutPrintln("***** receiveCompactDestinationReplyMessage *****");
        Message mesg = null;

        try {
            mesg = (ObjectMessage)receiveCheckMessageTimeout(false);

            mesg.acknowledge();
	    clearStatusEvent();

	    checkReplyTypeStatus(mesg, MessageType.COMPACT_DESTINATION_REPLY,
				"COMPACT_DESTINATION_REPLY");
        } catch (Exception e) {
	    handleReceiveExceptions(e);
        }
    }

    /*
     * Send a GET_CONNECTIONS admin msg
     *
     * NOTE: The current GET_CONNECTIONS protocol only supports
     * only one of {JMQServiceName,JMQConnectionID} being set.
     */
    public void sendGetConnectionsMessage(String svcName, Long cxnId)
			throws BrokerAdminException {

        if (debug) Globals.stdOutPrintln ("***** sendGetConnectionsMessage *****");
        ObjectMessage mesg = null;

	checkIfBusy();

        try {
            mesg = session.createObjectMessage();
            mesg.setJMSReplyTo(replyQueue);
            mesg.setIntProperty
                (MessageType.JMQ_MESSAGE_TYPE, MessageType.GET_CONNECTIONS);

	    if (cxnId != null)  {
	        statusEvent = createStatusEvent(BrokerCmdStatusEvent.QUERY_CXN,
						MessageType.GET_CONNECTIONS_REPLY,
						"GET_CONNECTIONS_REPLY");
                mesg.setLongProperty(MessageType.JMQ_CONNECTION_ID, cxnId.longValue());
	        statusEvent.setCxnid(cxnId.longValue());
	    } else  {
	        statusEvent = createStatusEvent(BrokerCmdStatusEvent.LIST_CXN,
						MessageType.GET_CONNECTIONS_REPLY,
						"GET_CONNECTIONS_REPLY");
	    }


	    if (svcName != null)  {
	        mesg.setStringProperty(MessageType.JMQ_SERVICE_NAME, svcName);
	        statusEvent.setServiceName(svcName);
	    }

	    if (debug)  {
		printMsgType(MessageType.GET_CONNECTIONS, "GET_CONNECTIONS");
		if (svcName != null)  {
		    Globals.stdOutPrintln("\t"
			+ MessageType.JMQ_SERVICE_NAME
			+ "=" 
			+ svcName);
		}
		if (cxnId != null)  {
		    Globals.stdOutPrintln("\t"
			+ MessageType.JMQ_CONNECTION_ID
			+ "=" 
			+ cxnId.longValue());
		}
	    }
	    sender.send(mesg);
        } catch (Exception e) {
	    handleSendExceptions(e);
        }
    }

    public Vector receiveGetConnectionsReplyMessage() throws BrokerAdminException {
	return receiveGetConnectionsReplyMessage(true);
    }

    public Vector receiveGetConnectionsReplyMessage(boolean waitForResponse)
					throws BrokerAdminException {
        if (debug)
        Globals.stdOutPrintln("***** receiveGetConnectionsReplyMessage *****");
        ObjectMessage mesg = null;

        try {
            mesg = (ObjectMessage)receiveCheckMessageTimeout(false, waitForResponse);

            mesg.acknowledge();
	    clearStatusEvent();

	    checkReplyTypeStatus(mesg, 
		MessageType.GET_CONNECTIONS_REPLY, "GET_CONNECTIONS_REPLY");

            Object obj;

            if ((obj = mesg.getObject()) != null) {
                if (obj instanceof Vector)  {
		    if (debug)  {
			printConnectionInfoList((Vector)obj);
		    }
                    return (Vector)obj;
		}
            }
        } catch (Exception e) {
	    handleReceiveExceptions(e);
        }

        return null;
    }


    /*
     * Send a DESTROY_CONNECTION admin msg
     */
    public void sendDestroyConnectionMessage(Long cxnId)
			throws BrokerAdminException {

        if (debug) Globals.stdOutPrintln ("***** sendDestroyConnectionMessage *****");
        ObjectMessage mesg = null;

	checkIfBusy();

        try {
            mesg = session.createObjectMessage();
            mesg.setJMSReplyTo(replyQueue);
            mesg.setIntProperty
                (MessageType.JMQ_MESSAGE_TYPE, MessageType.DESTROY_CONNECTION);

	    if (cxnId != null)  {
	        statusEvent = createStatusEvent(BrokerCmdStatusEvent.DESTROY_CXN,
						MessageType.DESTROY_CONNECTION_REPLY,
						"DESTROY_CONNECTION_REPLY");
                mesg.setLongProperty(MessageType.JMQ_CONNECTION_ID, cxnId.longValue());
	        statusEvent.setCxnid(cxnId.longValue());
	    }


	    if (debug)  {
		printMsgType(MessageType.DESTROY_CONNECTION, "DESTROY_CONNECTION");
		if (cxnId != null)  {
		    Globals.stdOutPrintln("\t"
			+ MessageType.JMQ_CONNECTION_ID
			+ "=" 
			+ cxnId.longValue());
		}
	    }
	    sender.send(mesg);
        } catch (Exception e) {
	    handleSendExceptions(e);
        }
    }

    public void receiveDestroyConnectionReplyMessage()
                                throws BrokerAdminException {
        if (debug)
        Globals.stdOutPrintln("***** receiveDestroyConnectionReplyMessage *****");
        Message mesg = null;

        try {
            mesg = (ObjectMessage)receiveCheckMessageTimeout(false);

            mesg.acknowledge();
	    clearStatusEvent();
            checkReplyTypeStatus(mesg, MessageType.DESTROY_CONNECTION_REPLY,
					"DESTROY_CONNECTION_REPLY");
        } catch (Exception e) {
            handleReceiveExceptions(e);
        }
    }


    /*
     * Send DEBUG message to broker.
     * Parameters are:
     *  operation
     *  type
     *  id
     *  optional properties
     */
    public void sendDebugMessage(String cmd, String cmdarg, String target, 
			String targetType, Properties optionalProps) 
				throws BrokerAdminException {
	BrokerAdminException bae;

        if (debug) Globals.stdOutPrintln("***** sendDebugMessage *****");
        ObjectMessage mesg = null;

	checkIfBusy();

        try {
            mesg = session.createObjectMessage();
            mesg.setJMSReplyTo(replyQueue);
            mesg.setIntProperty
		(MessageType.JMQ_MESSAGE_TYPE, MessageType.DEBUG);

	    if (cmd != null)
	        mesg.setStringProperty(MessageType.JMQ_CMD, cmd);
	    if (cmdarg != null)
	        mesg.setStringProperty(MessageType.JMQ_CMDARG, cmdarg);
	    if (target != null)  {
	        mesg.setStringProperty(MessageType.JMQ_TARGET, target);
	    }
	    if (targetType != null)  {
	        mesg.setStringProperty(MessageType.JMQ_TARGET_TYPE, targetType);
	    }
	    if (optionalProps != null)
	        mesg.setObject(optionalProps);

	    statusEvent = createStatusEvent(BrokerCmdStatusEvent.DEBUG,
						MessageType.DEBUG_REPLY,
						"DEBUG_REPLY");

	    if (debug)  {
		printMsgType(MessageType.DEBUG, "DEBUG");
	    }
            sender.send(mesg);

        } catch (Exception e) {
	    handleSendExceptions(e);
        }
    }

    /*
     * Receive DEBUG_REPLY message from broker.
     * A Hashtable is returned, containing name=value pairs.
     */
    public Hashtable receiveDebugReplyMessage() throws BrokerAdminException {
	return receiveDebugReplyMessage(true);
    }

    public Hashtable receiveDebugReplyMessage(boolean waitForResponse) 
	throws BrokerAdminException {

        if (debug) 
	Globals.stdOutPrintln("***** receiveDebugReplyMessage *****");
        ObjectMessage mesg = null;

        try {
            mesg = (ObjectMessage)receiveCheckMessageTimeout(false, waitForResponse);
            mesg.acknowledge();
	    clearStatusEvent();

	    checkReplyTypeStatus(mesg, MessageType.DEBUG_REPLY,
				"DEBUG_REPLY");

            Object obj;

            if ((obj = mesg.getObject()) != null) {
                if (obj instanceof Hashtable)
                    return (Hashtable)obj;
            }

        } catch (Exception e) {
	    handleReceiveExceptions(e);
        }

        return null;
    }

    /*
     * DEBUG END
     */


    /*
     * Sets a string that will be used to uniquely identify this 
     * broker instance.
     */
    public void setKey(String key)  {
	this.key = key;
    }

    /**
     * Returns a unique key of this broker instance.
     * If the key field is not set, host:port is returned
     */
    public String getKey() {
	if (key != null)  {
	    return (key);
	}

        return (getBrokerHost() + ":" + getBrokerPort());
    }

    /**
     * Returns true if this instance is connected to the broker.
     * Returns false otherwise.
     */
    public boolean isConnected() {
	return (isConnected);
    }

    /**
     * Sets the value of isConnected value.
     */
    public void setIsConnected(boolean b) {
        this.isConnected = b;
    }

    private String getErrorMessage(Message mesg) {
	String error = null;
	
	try {
	    error = mesg.getStringProperty(MessageType.JMQ_ERROR_STRING);
	} catch (JMSException jmse) {
	    if (debug)  {
	        Globals.stdErrPrint("Failed to retrieve the error message: ");
                Globals.stdErrPrintln(jmse.getMessage());
	        jmse.printStackTrace();
	    }
        } catch (Exception e) {
	    if (debug)  {
                Globals.stdErrPrintln("Exception caught: " + e.getMessage());
                e.printStackTrace();
	    }
        }

	return error;
    }

    /**
     * This method is used to force the connection close when shutdown is
     * performed on the broker.  This way the connection is closed immediately
     * and we rely on the client to do all the cleanup for us.  This is
     * called immediately as to avoid any unnecessary reconnect retries.
     */
    public void forceClose() {
	try {
	    connection.close();
	    isConnected = false;

	} catch (JMSException jmse) {
	    Globals.stdErrPrintln("JMSException caught: " + jmse.getMessage());
	    jmse.printStackTrace();
	}
    }

    // REVISIT: should this be synchronized to make sure isConnected is properly set?
    public void close() {
	if (!isConnected())  {
	    return;
	}

	try {
	    if (msgAckThread != null)  {
		msgAckThread.stop();
		msgAckThread = null;
	        if (debug) 
		    Globals.stdOutPrintln("***** Stopped msgAckThread thread...");
	    }

	    if (debug) Globals.stdOutPrintln("***** Closing sender and receiver...");
	    sender.close();
	    if (debug) Globals.stdOutPrintln("***** Closed sender.");
	    receiver.close();
	    if (debug) Globals.stdOutPrintln("***** Closed receiver.");

      	    if (debug)
	    Globals.stdOutPrintln("***** Closing queue session and queue connection...");
	    session.close();
	    if (debug) Globals.stdOutPrintln("***** Closed session.");
        } catch (JMSException jmse) {
            Globals.stdErrPrintln("JMSException caught: " + jmse.getMessage());
            jmse.printStackTrace();

        } catch (Exception e) {
            Globals.stdErrPrintln("Exception caught: " + e.getMessage());
            e.printStackTrace();
        }

	try {
	    connection.close();
	    if (debug) Globals.stdOutPrintln("***** Closed connection.");

	    isConnected = false;
	    setBusy(false);

        } catch (JMSException jmse) {
	    if (sslTransportUsed)  {
	        isConnected = false;
	        setBusy(false);
	    } else  {
                Globals.stdErrPrintln("JMSException caught: " + jmse.getMessage());
                jmse.printStackTrace();
	    }
        } catch (Exception e) {
	    if (sslTransportUsed)  {
	        isConnected = false;
	        setBusy(false);
	    } else  {
                Globals.stdErrPrintln("Exception caught: " + e.getMessage());
                e.printStackTrace();
	    }
        }
    }

    private void setFactoryAttr(String propName, String propVal) 
				throws JMSException  {
	if (qcf == null)  {
	    return;
	}

	qcf.setProperty(propName, propVal);
    }

    private String getFactoryAttr(String propName)  {
	if (qcf == null)  {
	    return (null);
	}

	String s;

	try  {
	    s = qcf.getProperty(propName);
	} catch (JMSException jmse)  {
	    s = null;
	}

	return (s);
    }

    private void createFactory(Properties brokerAttrs) throws BrokerAdminException  {
	BrokerAdminException	bae;

	qcf = new QueueConnectionFactory();

        try {
	    qcf.setConnectionType(ClientConstants.CONNECTIONTYPE_ADMIN);

	    for (Enumeration e = brokerAttrs.propertyNames(); 
				e.hasMoreElements(); ) {
	        String propName = (String)e.nextElement();
	        String value  = brokerAttrs.getProperty(propName);

	        if (value != null)  {
		    qcf.setProperty(propName, value);
	        }
	    }
	} catch (JMSException jmse) {
	    bae = new BrokerAdminException(BrokerAdminException.CONNECT_ERROR);
	    bae.setLinkedException(jmse);
	    bae.setBrokerHost(getBrokerHost());
	    bae.setBrokerPort(getBrokerPort());
            throw bae;
	    
	} catch (Exception e) {
	    bae = new BrokerAdminException(BrokerAdminException.CONNECT_ERROR);
	    bae.setLinkedException(e);
	    bae.setBrokerHost(getBrokerHost());
	    bae.setBrokerPort(getBrokerPort());
            throw bae;
	}

	if (debug)  {
	    Globals.stdOutPrintln("***** BrokerAdmin instance: " + getKey());
	    Globals.stdOutPrintln("BrokerAdmin created with timeout set to: "
			+ (this.timeout/1000)
			+ " seconds");
	    Globals.stdOutPrintln("BrokerAdmin created with num retries set to: " + this.numRetries);
	}

    }

    private void checkIfBusy() throws BrokerAdminException  { 
	if (isBusy())  {
            BrokerAdminException	bae;

            bae = new BrokerAdminException(
			BrokerAdminException.BUSY_WAIT_FOR_REPLY);
	    bae.setBrokerAdmin(this);
            throw bae;
	}
    }

    /*
     * Not used
    private void checkReplyTypeStatus(Message mesg, int msgType) 
					throws BrokerAdminException  {
        checkReplyTypeStatus(mesg, msgType, "");
    }
    */

    private Message receiveCheckMessageTimeout(boolean isShutdownReply) 
				throws BrokerAdminException  {
        return (receiveCheckMessageTimeout(isShutdownReply, true));
    }

    /*
     * Convenience method used to check the received mesg
     * if it is null.
     *
     * If it is null, this means that the receive() method timed out.
     * An appropriate number of retries is attempted after which
     * a BrokerAdminException is thrown. Each receive retry is made 
     * with a timeout that is a multiple of the original timeout.
     */
    private Message receiveCheckMessageTimeout(boolean isShutdownReply, 
			boolean waitForResponse) 
				throws BrokerAdminException  {
	Message mesg = null;
        BrokerAdminException	bae;
	long incrTimeout = timeout;
	int localNumRetries = 0;

	try  {
	    while (localNumRetries <= numRetries)  {
                mesg = (ObjectMessage)receiver.receive(incrTimeout);

                /* REVISIT: There is a timing problem in the protocol.  
                   The GOODBYE message could be processed before the SHUTDOWN_REPLY
                   message and therefore could be sending null as a value for 'mesg'
                   when receive() returns.  We will assume that the SHUTDOWN operation
                   was successful when we receive status == 200 or mesg == null.
                 */
	        if (mesg != null)  {
		    break;
	        } else  {
	            if (isShutdownReply) {
                        isConnected = false;
			break;
	            } else {
			localNumRetries++;
			incrTimeout += timeout;

			if (localNumRetries <= numRetries)  {
			    /*
			     * Send a status event so that something like
			     * the following can be printed:
			     *
			     *  Broker not responding, 
			     *	  retrying [1 of 2 attempts, timeout=20 seconds]
			     */
	                    BrokerCmdStatusEvent cse = 
				new BrokerCmdStatusEvent(this, this, 
					BrokerCmdStatusEvent.BROKER_BUSY);
			    cse.setNumRetriesAttempted(localNumRetries);
			    cse.setMaxNumRetries(numRetries);
			    cse.setRetryTimeount((incrTimeout/1000));
			    fireAdminEventDispatched(cse);
			} else {
		            /*
		             * It looks like we timed out waiting for a reply
			     * even after several retries.
		             * We should:
		             *	- fire off a thread to continue waiting for the reply.
		             *	   (this will mark this BrokerAdmin as 'busy').
		             *	- throw an exception saying the reply was not received.
		             */
    		            msgAckThread = new MessageAckThread(this);
    		            msgAckThread.start();

		            if (waitForResponse)
                                bae = new BrokerAdminException(
			            BrokerAdminException.REPLY_NOT_RECEIVED);
	 	            else
                                bae = new BrokerAdminException(
			            BrokerAdminException.IGNORE_REPLY_IF_RCVD);
                            throw bae;
	                }
	            }
	        }
	    }
        } catch (Exception e) {
	    handleReceiveExceptions(e);
        }

	return (mesg);
    }

    private void checkReplyTypeStatus(Message mesg, int msgType, String msgTypeString) 
					throws BrokerAdminException  {
        BrokerAdminException	bae;
	int			actualMsgType,
				actualReplyStatus;

        /* REVISIT: There is a timing problem in the protocol.  
           The GOODBYE message could be processed before the SHUTDOWN_REPLY
           message and therefore could be sending null as a value for 'mesg'
           when receive() returns.  We will assume that the SHUTDOWN operation
           was successful when we receive status == 200 or mesg == null.
         */
	if (mesg == null)  {
	    if (msgType == MessageType.SHUTDOWN_REPLY) {
                isConnected = false;
                return;
	    }
	}

	/*
	 * Fetch reply message type
	 */
	try  {
            actualMsgType = mesg.getIntProperty(MessageType.JMQ_MESSAGE_TYPE);
	} catch (JMSException jmse)  {
            bae = new BrokerAdminException(
			BrokerAdminException.PROB_GETTING_MSG_TYPE);
	    bae.setLinkedException(jmse);
            throw bae;
	}
        
	/*
	 * Fetch reply status code
	 */
	try  {
            actualReplyStatus = mesg.getIntProperty(MessageType.JMQ_STATUS);
	} catch (JMSException jmse)  {
            bae = new BrokerAdminException(
			BrokerAdminException.PROB_GETTING_STATUS);
	    bae.setLinkedException(jmse);
            throw bae;
	}

	if (debug)  {
	    Globals.stdOutPrintln("\tReplyMsgType="
			+ actualMsgType
			+ "(expecting "
			+ msgType
			+ "["
			+ msgTypeString
			+ "]), ReplyStatus="
			+ actualReplyStatus);
	}

	/*
	 * Both values must be correct
	 */
	if ((msgType == actualMsgType) && (actualReplyStatus == MessageType.OK))  {
            if (msgType == MessageType.SHUTDOWN_REPLY)
                isConnected = false;

	    return;
	}

	/*
	 * Otherwise, report an error
	 */
	String	errorStr = getErrorMessage(mesg);

	if (debug)  {
	    Globals.stdOutPrintln("\tJMQ_ERROR_STRING=" + errorStr);
	}

        bae = new BrokerAdminException(BrokerAdminException.MSG_REPLY_ERROR);
	bae.setBrokerErrorStr(errorStr);
	bae.setReplyMsgType(actualMsgType);
	bae.setReplyStatus(actualReplyStatus);
        throw bae;
    }

    private void printMsgType(int msgType, String msgTypeString)  {
        Globals.stdOutPrintln("\t"
		+ MessageType.JMQ_MESSAGE_TYPE
		+ "="
		+ msgType
		+ "("
		+ msgTypeString
		+ ")");
    }

    private void printDestinationInfoList(Vector v)  {
	Enumeration e = v.elements();

	Globals.stdOutPrintln("\t************************");
	while (e.hasMoreElements()) {
	    Object o = e.nextElement();

	    if (!(o instanceof DestinationInfo))  {
	        Globals.stdOutPrintln("\tprintDestinationInfoList: Vector contained object of type: "
				+ o.getClass().getName());
	        Globals.stdOutPrintln("\t************************");
		return;
	    }
	    DestinationInfo dInfo = (DestinationInfo)o;

	    printDestinationInfo(dInfo);

	    if (e.hasMoreElements())
	        Globals.stdOutPrintln("");
	}
	Globals.stdOutPrintln("\t************************");
    }

    private void printDestinationInfo(DestinationInfo dstInfo)  {
        Globals.stdOutPrintln("\tDestinationInfo:");
        Globals.stdOutPrintln("\t  name=" + dstInfo.name);
        Globals.stdOutPrintln("\t  type=" + dstInfo.type);
        Globals.stdOutPrintln("\t  nMessages=" + dstInfo.nMessages);
        Globals.stdOutPrintln("\t  nMessageBytes=" + dstInfo.nMessageBytes);
        Globals.stdOutPrintln("\t  nConsumers=" + dstInfo.nConsumers);
        Globals.stdOutPrintln("\t  maxMessages=" + dstInfo.maxMessages);
        Globals.stdOutPrintln("\t  maxMessageBytes=" + dstInfo.maxMessageBytes);
        Globals.stdOutPrintln("\t  maxMessageSize=" + dstInfo.maxMessageSize);
        Globals.stdOutPrintln("\t  maxFailoverConsumers=" + dstInfo.maxFailoverConsumers);
        Globals.stdOutPrintln("\t  maxActiveConsumers=" + dstInfo.maxActiveConsumers);
        Globals.stdOutPrintln("\t  destScope=" + dstInfo.destScope);
        Globals.stdOutPrintln("\t  destLimitBehavior=" + dstInfo.destLimitBehavior);
        Globals.stdOutPrintln("\t  destCDP=" + dstInfo.destCDP);
        Globals.stdOutPrintln("\t  maxPrefetch=" + dstInfo.maxPrefetch);
        Globals.stdOutPrintln("\t  maxProducers=" + dstInfo.maxProducers);
        Globals.stdOutPrintln("\t  autocreated=" + dstInfo.autocreated);
        Globals.stdOutPrintln("\t  naConsumers=" + dstInfo.naConsumers);
        Globals.stdOutPrintln("\t  nfConsumers=" + dstInfo.nfConsumers);
        Globals.stdOutPrintln("\t  destState=" + dstInfo.destState);
    }

    private void printServiceInfoList(Vector v)  {
	Enumeration e = v.elements();

	Globals.stdOutPrintln("\t************************");
	while (e.hasMoreElements()) {
	    Object o = e.nextElement();

	    if (!(o instanceof ServiceInfo))  {
	        Globals.stdOutPrintln("\tprintServiceInfoList: Vector contained object of type: "
				+ o.getClass().getName());
	        Globals.stdOutPrintln("\t************************");
		return;
	    }
	    ServiceInfo svcInfo = (ServiceInfo)o;

	    printServiceInfo(svcInfo);

	    if (e.hasMoreElements())
	        Globals.stdOutPrintln("");
	}
	Globals.stdOutPrintln("\t************************");
    }

    private void printServiceInfo(ServiceInfo svcInfo)  {
        Globals.stdOutPrintln("\tServiceInfo:");
        Globals.stdOutPrintln("\t  name=" + svcInfo.name);
        Globals.stdOutPrintln("\t  protocol=" + svcInfo.protocol);
        Globals.stdOutPrintln("\t  type=" + svcInfo.type);
        Globals.stdOutPrintln("\t  state=" + svcInfo.state);
        Globals.stdOutPrintln("\t  nConnections=" + svcInfo.nConnections);
        Globals.stdOutPrintln("\t  currentThreads=" + svcInfo.currentThreads);
        Globals.stdOutPrintln("\t  dynamicPort=" + svcInfo.dynamicPort);
        Globals.stdOutPrintln("\t  metrics=" + svcInfo.metrics);
        Globals.stdOutPrintln("\t  port=" + svcInfo.port);
        Globals.stdOutPrintln("\t  minThreads=" + svcInfo.minThreads);
        Globals.stdOutPrintln("\t  maxThreads=" + svcInfo.maxThreads);
    }

    private void printConnectionInfoList(Vector v)  {
	Enumeration e = v.elements();

	Globals.stdOutPrintln("\t************************");
	while (e.hasMoreElements()) {
	    Object o = e.nextElement();

	    if (!(o instanceof Hashtable))  {
	        Globals.stdOutPrintln("\tprintConnectionInfoList: Vector contained object of type: "
				+ o.getClass().getName()
				+ "(expected java.util.Hashtable)");
	        Globals.stdOutPrintln("\t************************");
		return;
	    }
	    Hashtable cxnInfo = (Hashtable)o;

	    printConnectionInfo(cxnInfo);

	    if (e.hasMoreElements())
	        Globals.stdOutPrintln("");
	}
	Globals.stdOutPrintln("\t************************");
    }

    private void printConnectionInfo(Hashtable cxnInfo)  {
        Globals.stdOutPrintln("\tConnection Info:");

	for (Enumeration e = cxnInfo.keys() ; e.hasMoreElements() ;) {
	    String curPropName = (String)e.nextElement();
	    String curValue;
	    Object tmpObj;

	    tmpObj = cxnInfo.get(curPropName);
	    curValue = tmpObj.toString();
    
            Globals.stdOutPrintln("\t  "
			+ curPropName
			+ "="
			+ curValue);
	}
    }

    private void printDurableInfoList(Vector v)  {
	Enumeration e = v.elements();

	Globals.stdOutPrintln("\t************************");
	while (e.hasMoreElements()) {
	    Object o = e.nextElement();

	    if (!(o instanceof DurableInfo))  {
	        Globals.stdOutPrintln("\tprintDurableInfoList: Vector contained object of type: "
				+ o.getClass().getName());
	        Globals.stdOutPrintln("\t************************");
		return;
	    }
	    DurableInfo durInfo = (DurableInfo)o;

	    printDurableInfo(durInfo);

	    if (e.hasMoreElements())
	        Globals.stdOutPrintln("");
	}
	Globals.stdOutPrintln("\t************************");
    }

    private void printDurableInfo(DurableInfo durInfo)  {
        Globals.stdOutPrintln("\tDurableInfo:");
        Globals.stdOutPrintln("\t  name=" + durInfo.name);
        Globals.stdOutPrintln("\t  clientID=" + durInfo.clientID);
        Globals.stdOutPrintln("\t  nMessages=" + durInfo.nMessages);
        Globals.stdOutPrintln("\t  isActive=" + durInfo.isActive);
        Globals.stdOutPrintln("\t  ConsumerInfo=" + durInfo.consumer);
    }

    private void printTxnInfoList(Vector v)  {
	Enumeration e = v.elements();

	Globals.stdOutPrintln("\t************************");
	while (e.hasMoreElements()) {
	    Object o = e.nextElement();

	    if (!(o instanceof Hashtable))  {
	        Globals.stdOutPrintln("\tprintTxnInfoList: Vector contained object of type: "
				+ o.getClass().getName()
				+ "(expected java.util.Hashtable)");
	        Globals.stdOutPrintln("\t************************");
		return;
	    }
	    Hashtable txnInfo = (Hashtable)o;

	    printTxnInfo(txnInfo);

	    if (e.hasMoreElements())
	        Globals.stdOutPrintln("");
	}
	Globals.stdOutPrintln("\t************************");
    }

    private void printTxnInfo(Hashtable txnInfo)  {
        Globals.stdOutPrintln("\tTransaction Info:");

	for (Enumeration e = txnInfo.keys() ; e.hasMoreElements() ;) {
	    String curPropName = (String)e.nextElement();
	    String curValue;
	    Object tmpObj;

	    tmpObj = txnInfo.get(curPropName);
	    curValue = tmpObj.toString();
    
            Globals.stdOutPrintln("\t  "
			+ curPropName
			+ "="
			+ curValue);
	}
    }



    private void printClusterList(Vector v)  {
	Enumeration e = v.elements();

	Globals.stdOutPrintln("\t************************");
	while (e.hasMoreElements()) {
	    Object o = e.nextElement();

	    if (!(o instanceof BrokerClusterInfo))  {
	        Globals.stdOutPrintln("\tprintClusterList: Vector contained object of type: "
				+ o.getClass().getName()
				+ "(expected BrokerClusterInfo)");
	        Globals.stdOutPrintln("\t************************");
		return;
	    }
	    BrokerClusterInfo bkrClsInfo = (BrokerClusterInfo)o;

	    printBkrClsInfo(bkrClsInfo);

	    if (e.hasMoreElements())
	        Globals.stdOutPrintln("");
	}
	Globals.stdOutPrintln("\t************************");
    }

    private void printBkrClsInfo(BrokerClusterInfo bkrClsInfo)  {
        Globals.stdOutPrintln("\tBroker Cluster Info:");
    }


    private void printJMXList(Vector v)  {
	Enumeration e = v.elements();

	Globals.stdOutPrintln("\t************************");
	while (e.hasMoreElements()) {
	    Object o = e.nextElement();

	    if (!(o instanceof Hashtable))  {
	        Globals.stdOutPrintln("\tprintJMXList: Vector contained object of type: "
				+ o.getClass().getName()
				+ "(expected java.util.Hashtable)");
	        Globals.stdOutPrintln("\t************************");
		return;
	    }
	    Hashtable jmxInfo = (Hashtable)o;

	    printJMXInfo(jmxInfo);

	    if (e.hasMoreElements())
	        Globals.stdOutPrintln("");
	}
	Globals.stdOutPrintln("\t************************");
    }

    private void printJMXInfo(Hashtable jmxInfo)  {
        Globals.stdOutPrintln("\tJMX Connector Info:");

	for (Enumeration e = jmxInfo.keys() ; e.hasMoreElements() ;) {
	    String curPropName = (String)e.nextElement();
	    String curValue;
	    Object tmpObj;

	    tmpObj = jmxInfo.get(curPropName);
	    curValue = tmpObj.toString();
    
            Globals.stdOutPrintln("\t  "
			+ curPropName
			+ "="
			+ curValue);
	}
    }




    private void handleSendExceptions(Exception e) 
				throws BrokerAdminException  {
	BrokerAdminException bae;

	if (e instanceof BrokerAdminException)  {
	    throw ((BrokerAdminException)e);
	} else if (e instanceof JMSException)  {
	    /*
	     * Handled separately from regular Exceptions in case we know
	     * enough to report them in a more useful way.
	     */
	    bae = new BrokerAdminException(BrokerAdminException.MSG_SEND_ERROR);
	    bae.setLinkedException(e);
            throw bae;
	} else {
	    bae = new BrokerAdminException(BrokerAdminException.MSG_SEND_ERROR);
	    bae.setLinkedException(e);
            throw bae;
	}
    }

    private void handleReceiveExceptions(Exception e) 
				throws BrokerAdminException  {
	BrokerAdminException bae;

	if (e instanceof BrokerAdminException)  {
	    /*
	     * Could be thrown by checkReplyTypeStatus()
	     */
	    throw ((BrokerAdminException)e);
	} else if (e instanceof JMSException)  {
	    /*
	     * Handled separately from regular Exceptions in case we know
	     * enough to report them in a more useful way.
	     */
	    bae = new BrokerAdminException(BrokerAdminException.MSG_REPLY_ERROR);
	    bae.setLinkedException(e);
            throw bae;
	} else {
	    bae = new BrokerAdminException(BrokerAdminException.MSG_REPLY_ERROR);
	    bae.setLinkedException(e);
            throw bae;
	}
    }

    /**
     * Add an admin event listener.
     * @param l admin event listener to add.
     */
    public void addAdminEventListener(AdminEventListener l)  {
        eListeners.addElement(l);
    }

    /**
     * Remove an admin event listener.
     * @param l admin event listener to remove.
     */
    public void removeAdminEventListener(AdminEventListener l)  {
        eListeners.removeElement(l);
    }

    /**
     * Remove all admin event listeners.
     */
    public void removeAllAdminEventListeners()  {
	eListeners.removeAllElements();
    }

    /**
     * Fire off/dispatch an admin event to all the listeners.
     * @param ae AdminEvent to dispatch to event listeners.
     */
    public void fireAdminEventDispatched(AdminEvent ae)  {
        for (int i = 0; i < eListeners.size(); i++) {
            ((AdminEventListener)eListeners.elementAt(i)).
		adminEventDispatched(ae);
        }
    }

    protected BrokerCmdStatusEvent createStatusEvent(int type, int replyType, 
					String replyTypeString)  {
	BrokerCmdStatusEvent cse = new BrokerCmdStatusEvent(this, this, type);
	cse.setReplyType(replyType);
	cse.setReplyTypeString(replyTypeString);

	return (cse);
    }

    protected void clearStatusEvent()  {
        statusEvent = null;
    }

    protected void sendStatusEvent(Message mesg, Exception ex)  {
	if (statusEvent != null)  {
	    /*
	     * Set success flag to true/false before
	     * firing off status event.
	     */

	    if (mesg == null)  {
		/*
		 * If mesg is null, ex should hold an exception
		 * that caused the failure
		 */
		statusEvent.setSuccess(false);
		statusEvent.setLinkedException(ex);
	    } else  {
		/*
		 * If mesg is not null, check for reply/status code
		 */
	        try  {
                    checkReplyTypeStatus(mesg, statusEvent.getReplyType(), 
					statusEvent.getReplyTypeString());

		    if (mesg instanceof ObjectMessage) {
			try {
		            statusEvent.setReturnedObject(
				((ObjectMessage)mesg).getObject());
			} catch (JMSException jmse) {
			    /*
			     * Problems retrieving the associated object.
			     * Report an error.
			     */
                	    statusEvent.setSuccess(false);
                	    statusEvent.setLinkedException(jmse);
			}
		    }
		    statusEvent.setSuccess(true);

	        } catch (BrokerAdminException bae)  {
		    statusEvent.setSuccess(false);
		    statusEvent.setLinkedException(bae);
	        }
	    }

            fireAdminEventDispatched(statusEvent);
            clearStatusEvent();
	}
    }

    /*
     * BEGIN INTERFACE ExceptionListener
     */
    public void onException(JMSException jmse) {
	BrokerErrorEvent bee = null;
        isConnected = false;

	/*
	 * Broker initiated connection close.
	 */
        if (ClientResources.X_BROKER_GOODBYE == jmse.getErrorCode()) {
	    /*
	     * Check to see if this BrokerAdmin has initiated the
	     * shutdown.  Only propagate the error message if it is
	     * NOT initiated by this BrokerAdmin.
	     */
	    if (!isInitiator()) {
                bee = new BrokerErrorEvent(this, BrokerErrorEvent.ALT_SHUTDOWN);
                bee.setBrokerHost(getBrokerHost());
                bee.setBrokerPort(getBrokerPort());
                bee.setBrokerName(getKey());
	    }
	/*
	 * Broker unexpectedly shutdown.
	 */
        } else if (jmse.getLinkedException() instanceof EOFException) {
            bee = new BrokerErrorEvent(this, BrokerErrorEvent.UNEXPECTED_SHUTDOWN);
	    bee.setBrokerHost(getBrokerHost());
	    bee.setBrokerPort(getBrokerPort());
	    bee.setBrokerName(getKey());
	/*
	 * Other misc. connection problems.
	 */
        } else {
            bee = new BrokerErrorEvent(this, BrokerErrorEvent.CONNECTION_ERROR);
            bee.setBrokerHost(getBrokerHost());
            bee.setBrokerPort(getBrokerPort());
            bee.setBrokerName(getKey());
	}
	if (bee != null)
            fireAdminEventDispatched(bee);
	removeAllAdminEventListeners();
    }
    /*
     * END INTERFACE ExceptionListener
     */
    
    public static void setDebug(boolean b)  {
	debug = b;
    }

    public static boolean getDebug()  {
	return (debug);
    }

    public void setAssociatedObj(Object obj)  {
	this.aObj = obj;
    }

    public Object getAssociatedObj()  {
	return (aObj);
    }
}

class MessageAckThread implements Runnable  {
    private Thread		ackThread = null;
    private BrokerAdmin		ba;
    private boolean		msgReceived = false;
    private long		timeout = 30000;
    private boolean		debug = false,
				stopRequested = false;

    public MessageAckThread(BrokerAdmin ba)  {
	debug = BrokerAdmin.getDebug();

	if (debug)  {
            Globals.stdOutPrintln("***** Created MessageAckThread");
	}
	this.ba = ba;
    }

    public synchronized void start()  {
	if (ackThread == null)  {
	    ackThread = new Thread(this, "JMQ Administration MessageAckThread");
	    ackThread.start();
	    if (debug)  {
                Globals.stdOutPrintln("***** Started MessageAckThread");
	    }
	}
    }

    public synchronized void stop()  {
	stopRequested = true;
    }

    public void run()  {
        Message mesg = null;

	ba.setBusy(true);

	while (!msgReceived)  {
	    try  {
	        mesg = ba.receiver.receive(timeout);

		if (mesg != null)  {
		    if (debug)  {
		        Globals.stdOutPrintln("***** MessageAckThread: received reply message !");
		    }
		    msgReceived = true;
		    if (debug)  {
		        Globals.stdOutPrintln("***** MessageAckThread: acknowledging reply message.");
		    }
                    mesg.acknowledge();
		    synchronized (this) {
		        ba.setBusy(false);
		        ba.sendStatusEvent(mesg, null);
		    }
		} else  {
		    synchronized (this)  {
			if (stopRequested)  {
		            if (debug)  {
		                Globals.stdOutPrintln("***** MessageAckThread: receive() timed out. Not retrying (stop requested).");
		            }
			    stopRequested = false;
			    ackThread = null;
			    return;
			}
		    }

		    if (debug)  {
		        Globals.stdOutPrintln("***** MessageAckThread: receive() timed out. Retrying...");
		    }
		}
	    } catch (Exception e)  {
		/*
	         * Caught exception while waiting for reply message.
		 * Report error in status event.
		 */
		synchronized (this) {
		    ba.setBusy(false);
		    ba.sendStatusEvent(null, e);
		}
	    }
	}

	ba.setBusy(false);
    }
}
