/*************************************************************************
* Bulgarian-English Dictionary
* Copyright (C) 2000, 2001  Radostin Radnev <radnev@yahoo.com>
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; if not, write to the Free Software
* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*************************************************************************/

#include <string.h>

#include "history.h"



//=== Class History ======================================================
// Written by Radostin Radnev - radnev@yahoo.com
// $Id: history.cpp,v 1.2 2001/03/11 05:21:32 radnev Exp $
//
// This class represents History object.
// It holds entered words and serves navigation
//========================================================================



//=== Constructor ========================================================
// Set defaults variables
// Invoke setSize to allocate memory
//========================================================================
History::History(const int items, const int length) {
	buf = NULL;
	len = length;
	setMaxSize(items);
} // End of Constructor



//=== Destructor =========================================================
// Free memory
//========================================================================
History::~History() {
	delete [] buf;
} // End of Destructor



//=== Set Max Size =======================================================
// (Re)Allocate memory and copy data if necessary
//========================================================================
void History::setMaxSize(const int items) {
	if (buf == NULL) {
		buf = new char[items*(len + 1)];
		currentSize = 0;
		current = 0;
		size = items;
	}
	else {
		char *b;
		int i;
		i = (size < items ? size : items);
		b = new char[items*(len + 1)];
		memcpy(b, buf, i*(len + 1));
		delete [] buf;
		buf = b;
		currentSize = i;
		if (current > currentSize - 1) {
			current = currentSize - 1;
		}
		size = items;
	}
} // End of setMaxSize



//=== Get Max Size =======================================================
// Return max number of items that can be stored
//========================================================================
int History::getMaxSize() {
	return size;
} // End of getMaxSize



//=== Get Current Size ===================================================
// Return number of current stored items
//========================================================================
int History::getCurrentSize() {
	return currentSize;
} // End of getCurrentSize



//=== Add Item ===========================================================
// Add item to the buffer
//========================================================================
void History::addItem(const char *item) {
	if (((currentSize > 0) && (strcmp(buf, item) != 0)) || (currentSize == 0)) {
		memmove(buf + len + 1, buf, (size - 1)*(len + 1));
		if ((signed)strlen(item) <= len) {
			strcpy(buf, item);
		}
		else {
			strncpy(buf, item, len);
			buf[len] = '\0';
		}
		if (currentSize < size) {
			currentSize++;
		}
		current = -1;
	}
} // End of addItem



//=== Get Current Item ===================================================
// Return current item
//========================================================================
char *History::getCurrentItem() {
	return (buf + current*(len + 1));
} // End of getCurrentItem



//=== Prev ===============================================================
// Go to previous item
//========================================================================
void History::prev() {
	if (current < currentSize - 1) {
		current++;
	}
} // End of prev



//=== Next ===============================================================
// Go to next item
//========================================================================
void History::next() {
	if (current > 0) {
		current--;
	}
} // End of next



//=== Is First ===========================================================
// Return true if current item is first item
//========================================================================
bool History::isFirst() {
	return (current == 0);
} // End of isFirst



//=== Is Last ============================================================
// Return true if current item is last item
//========================================================================
bool History::isLast() {
	return (current == (currentSize - 1));
} // End of isLast



//=== Is New =============================================================
// Return true if last operation was addItem
//========================================================================
bool History::isNew() {
	return (current == -1);
} // End of isNew
