/* -*- Mode: C; indent-tabs-mode: t; c-basic-offset: 8; tab-width: 8 -*- */
/*
 * GXMame
 *
 * Copyright 2002-2005 Stephane Pontier <shadow_walker@users.sourceforge.net>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#include <qstring.h>

#include "game_options.h"
#include "io.h"

extern "C" {
#include <string.h>
#include <stdlib.h>
}

GameOptions default_options;


void
game_options_free (GameOptions *opts)
{
	if (!opts || opts == &default_options)
		return;

	g_free (opts->additional_options);
	g_free (opts);
}

/** Loads the options for the given rom.
 * If rom is NULL then it loads the default options.
 * The options are loaded from: ~/gxmame/options/romname
 */
GameOptions *
load_options (RomEntry *rom)
{
	FILE *options_file;
	gchar *filename;
	gchar line[2000];
	gchar **option_value = NULL;
	GameOptions *target;
	GList *ctrlr_list = NULL;
	GList *my_list = NULL;

	if (rom == NULL) {
		target = &default_options;
		g_free (target->additional_options);
		
		filename = g_build_filename (g_get_home_dir (), DOT_GXMAME, "options", "default", NULL);
	} else {
		filename = g_build_filename (g_get_home_dir (), DOT_GXMAME, "options", rom->romname, NULL);
		/* if there is no file with that name, exit, will use default options */
		if (!g_file_test (filename, G_FILE_TEST_EXISTS)) {
			GXMAME_DEBUG ("option file for %s not found",rom->romname);
			g_free (filename);
			return NULL;
		}
		/* otherwise, create a xmame_game_options struct for this game */
		target = (GameOptions *)g_malloc0 (sizeof (GameOptions));
	}
	
	/* default options */

	/*** Video Related ***/
	target->fullscreen = TRUE;
	target->bpp = 0;
	target->arbheight = 0;
	target->heightscale = 1;
	target->widthscale = 1;
	/*	target->scale = 0.00000;*/
	target->effect = 0;
	target->autodouble = TRUE;		
	target->dirty = TRUE;
	target->scanlines = FALSE;
	target->artwork = TRUE;
	target->use_backdrops = TRUE;
	target->use_overlays = TRUE;
	target->use_bezels = TRUE;
	target->artwork_crop = FALSE;
	target->artwork_resolution = 0;
	target->frameskipper = 0;
	target->throttle = TRUE;
	target->sleepidle = FALSE;
	target->autoframeskip = TRUE;
	target->maxautoframeskip = 8;
	target->frameskip = 0;
	target->brightness = 1.00000;
	target->gamma_correction = 1.00000;
	target->norotate = FALSE;
	target->ror = FALSE;
	target->rol = FALSE;
	target->flipx = FALSE;
	target->flipy = FALSE;

	/*** Vector Games Related ***/
	target->vectorres_flag = FALSE;
	strcpy (target->vectorres, "640x480");
	target->beam = 1.000000;
	target->flicker = 0.000000;
	target->antialias = TRUE;
	target->translucency = TRUE;
	target->intensity = 1.5;

	/*display_opts (depend target)*/
	/*** X11 Related ***/
	target->x11_mode = 0;
	target->cursor = TRUE;
	target->mitshm = TRUE;
	target->xvext = TRUE;
	target->vidix = FALSE;
	target->force_yuv = 0;
	target->xsync = FALSE;
	target->privatecmap = FALSE;
	target->xil = TRUE;
	target->mtxil = FALSE;
	target->run_in_root_window = FALSE;
	target->root_window_id = 0;
	target->xvgeom_flag = FALSE;
	strcpy (target->geometry,"640x480");

	/*** X11-input related ***/
	/* (X11 & GL Glide) */
	target->grabmouse = FALSE;
	target->grabkeyboard = FALSE;
	target->winkeys = FALSE;
	strcpy (target->mapkey, " ");


	/*** OpenGL Related ***/
	target->gldblbuffer = TRUE;
	target->gltexture_size = FALSE;
	target->glforceblitmode = FALSE;
	target->glext78 = TRUE;
	target->glbilinear = TRUE;
	target->gldrawbitmap = TRUE;
	target->gldrawbitmapvec = TRUE;
	target->glcolormod = TRUE;
	target->glbeam = 1.000000;
	target->glalphablending = TRUE;
	target->glantialias = TRUE;
	target->glantialiasvec = TRUE;
	strcpy (target->gllibname, "libGL.so.1");
	strcpy (target->glulibname, "libGLU.so.1");
	target->cabview = FALSE;
	strcpy (target->cabinet, "glmamejau");
	target->glres_flag = FALSE;
	strcpy (target->glres, " ");
				

	/*** SDL Related ***/
	target->sdl_doublebuf = FALSE;
	target->sdl_auto_mode = TRUE;
	target->modenumber = 0;
	strcpy (target->sdlmapkey, " ");

	/*** GGI Related ***/
	target->ggilinear = FALSE;
	target->ggi_force_resolution = FALSE;
	target->xres = 640;
	target->yres = 480;

	/*** Svgalib Related ***/
	target->tweak = FALSE;
	target->planar = TRUE;
	target->linear = FALSE;
	target->centerx = 0;
	target->centery = 0;

	/*** FX (Glide) Related ***/
	strcpy (target->resolution, "640x480");
	target->fxgkeepaspect = TRUE;

	/*** Photon Related ***/
	target->render_mode = 0;

	/*** Photon-window Related ***/
	target->phcursor = TRUE;
                 
	/*** Photon-input Related ***/
	/*	target->phgrabmouse = FALSE;
		target->phwinkeys = FALSE;
		strcpy (target->phmapkey, " ");
	*/

	/*** Video Mode Selection Related ***/
	/*** (SVGA, xf86_dga_opts, GGI) ***/
	target->keepaspect = 1;
	target->displayaspectratio = 1.330000;
	strcpy (target->disablemode, " ");

	/*** Sound Related ***/
	target->sound = TRUE;
	target->samples = TRUE;
	target->fakesound = FALSE;
	target->samplefre = 22050;
	target->bufsize = 3.000000;
	target->volume = -3;
	strcpy (target->audiodevice, " ");
	strcpy (target->mixerdevice, " ");
	strcpy (target->soundfile, "xmameout.wav");

	/*** Input Related ***/
	target->joytype = 0;
	target->analogstick = FALSE;
	target->mouse = FALSE;
	target->hotrod = FALSE;
	target->hotrodse = FALSE;
	target->usbpspad = FALSE;
	target->rapidfire = FALSE;
	target->ugcicoin = FALSE;
	
	strcpy (target->keymap, " ");
	strcpy (target->ctrlr, "None");
#ifndef KXMAME
	strcpy (target->joydevname, get_joy_dev ());
#else
	strcpy (target->joydevname, "/dev/js0");
#endif
	target->joydevname[strlen (target->joydevname) - 1] = '\0';

	/***joy_pad_opts***  *FM townpad ***/
	strcpy (target->paddevname, "/dev/pad00");
	/***joy_x11_opts***/
	strcpy (target->x11joyname, " ");
	/***Xinput***/
	strcpy (target->XInput_trackball1, " ");
	strcpy (target->XInput_trackball2, " ");
	strcpy (target->XInput_trackball3, " ");
	strcpy (target->XInput_trackball4, " ");
	/* 	strcpy (target->XInput_joystick1, " "); */
	/* 	strcpy (target->XInput_joystick2, " "); */
	/* 	strcpy (target->XInput_joystick3, " "); */
	/* 	strcpy (target->XInput_joystick4, " "); */

	/*** Digital sound related ***/
	strcpy (target->dsp_plugin, " ");
	/*list-dsp-plugins  */
	target->timer = FALSE;
				

	/*** Sound mixer related ***/
	strcpy (target->sound_mixer_plugin, " ");
	/* list-mixer-plugins */

	/***arts***/
	target->artsBufferTime = 10;
		
	/*** Alsa Sound System 0.5***/
	/* list-alsa-cards */
	target->alsacard = 0;
	target->alsadevice = 0;
        	       
        /*** Alsa Sound System 0.9 ***/
	/* list-alsa-cards
	   list-alsa-pcm	*/
	strcpy (target->alsa_pcm, "default");
	target->alsa_buffer = 250000;
        	       
        /*** QNX Audio related ***/
	target->audio_preferred = TRUE;
				

	/*** Display  Related ***/
	/*	strcpy (target->display_plugin, " ");*/
	/* list-display-plugins	*/
	/*	target->dwidthscale = 1;
		target->dheightscale = 1;
		target->dscanlines = FALSE;
		target->display_aspect_ratio = 1.33;
		target->keep_aspect = TRUE;
	*/
	/* Misc */
	target->cheat = 1;
	target->keyboard_leds = 1;
	target->debug = 0;
	strcpy (target->debug_size, "640x480");
	target->log_flag = FALSE;
	strcpy (target->log, "xmame.log");
	target->cfgname_flag = FALSE;
	strcpy (target->cfgname, "");
	target->use_additional_options = FALSE;
	target->additional_options = g_strdup ("");
	
	target->skip_disclaimer = FALSE;
	target->skip_gameinfo = FALSE;
	target->bios = 0;
	
	
	options_file = fopen (filename, "r");
	g_free (filename);
	if (!options_file) {
		GXMAME_DEBUG ("options file not found");
		return (FALSE);
	}
	
	while (fgets (line, 500, options_file)) {
		if (!strncmp (line, "[Video]", 7)) {
			while (fgets (line, 500, options_file)) {
				if (*line == '\r' || *line == '\n')
					break;
				option_value = g_strsplit (g_strstrip (g_strdelimit (line, "[]\r\n", ' ')), "=", 0);
				if (option_value[1] != NULL) {
					if (!strcmp (option_value[0], "fullscreen")) {
						target->fullscreen = atoi ( option_value[1]);
					} else if (!strcmp (option_value[0], "bpp")) {
						target->bpp = atoi ( option_value[1]);
					} else if (!strcmp (option_value[0], "arbheight")) {
						target->arbheight = atoi ( option_value[1]);
					} else if (!strcmp (option_value[0], "heightscale")) {
						target->heightscale = atoi ( option_value[1]);
					} else if (!strcmp (option_value[0], "widthscale")) {
						target->widthscale = atoi ( option_value[1]);
						/*					} else if (!strcmp (option_value[0], "scale")) {
											target->scale = g_ascii_strtod ( option_value[1]);
						*/					} else if (!strcmp (option_value[0], "effect")) {
							target->effect = atoi ( option_value[1]);
						} else if (!strcmp (option_value[0], "autodouble")) {
							target->autodouble = atoi ( option_value[1]);
						} else if (!strcmp (option_value[0], "dirty")) {
							target->dirty = atoi ( option_value[1]);
						} else if (!strcmp (option_value[0], "scanlines")) {
							target->scanlines = atoi ( option_value[1]);
						} else if (!strcmp (option_value[0], "artwork")) {
							target->artwork = atoi ( option_value[1]);
						} else if (!strcmp (option_value[0], "use_backdrops")) {
							target->use_backdrops = atoi ( option_value[1]);
						} else if (!strcmp (option_value[0], "use_overlays")) {
							target->use_overlays = atoi ( option_value[1]);
						} else if (!strcmp (option_value[0], "use_bezels")) {
							target->use_bezels = atoi ( option_value[1]);
						} else if (!strcmp (option_value[0], "artwork_crop")) {
							target->artwork_crop = atoi ( option_value[1]);
						} else if (!strcmp (option_value[0], "artwork_resolution")) {
							target->artwork_resolution = atoi ( option_value[1]);
						} else if (!strcmp (option_value[0], "frameskipper")) {
							target->frameskipper = atoi ( option_value[1]);
						} else if (!strcmp (option_value[0], "throttle")) {
							target->throttle = atoi ( option_value[1]);
						} else if (!strcmp (option_value[0], "sleepidle")) {
							target->sleepidle = atoi ( option_value[1]);
						} else if (!strcmp (option_value[0], "autoframeskip")) {
							target->autoframeskip = atoi ( option_value[1]);
						} else if (!strcmp (option_value[0], "maxautoframeskip")) {
							target->maxautoframeskip = atoi ( option_value[1]);
						} else if (!strcmp (option_value[0], "frameskip")) {
							target->frameskip = atoi ( option_value[1]);
						} else if (!strcmp (option_value[0], "brightness")) {
							target->brightness = g_ascii_strtod (option_value[1], NULL);
							/*  converted old percentage to dosmame brightness complient 0.5-2.0 range*/
							if ( (target->brightness > 2) || (target->brightness < 0.5))
								target->brightness = (target->brightness) * (1.5 / 100) + 0.5;
						} else if (!strcmp (option_value[0], "gamma_correction")) {
							target->gamma_correction = g_ascii_strtod (option_value[1], NULL);
						} else if (!strcmp (option_value[0], "norotate")) {
							target->norotate = atoi ( option_value[1]);
						} else if (!strcmp (option_value[0], "ror")) {
							target->ror = atoi ( option_value[1]);
						} else if (!strcmp (option_value[0], "rol")) {
							target->rol = atoi ( option_value[1]);
						}  else if (!strcmp (option_value[0], "flipx")) {
							target->flipx = atoi ( option_value[1]);
						} else if (!strcmp (option_value[0], "flipy")) {
							target->flipy = atoi ( option_value[1]);
						}
					g_strfreev (option_value);
				}
			}
		}

		if (!strncmp (line, "[Vector]", 8)) {
			while (fgets (line, 500, options_file)) {
				if (*line == '\r' || *line == '\n')
					break;
				option_value = g_strsplit (g_strstrip (g_strdelimit (line, "[]\r\n", ' ')), "=", 0);
				if (option_value[1] != NULL) {
					if (!strcmp (option_value[0], "vectorres_flag")) {
						target->vectorres_flag = atoi (option_value[1]);
					} else if (!strcmp (option_value[0], "vectorres")) {
						g_snprintf (target->vectorres, 10, "%s", option_value[1]);
					} else if (!strcmp (option_value[0], "beam")) {
						target->beam = g_ascii_strtod (option_value[1], NULL);
					} else if (!strcmp (option_value[0], "flicker")) {
						target->flicker = g_ascii_strtod (option_value[1], NULL);
					} else if (!strcmp (option_value[0], "antialias")) {
						target->antialias = atoi ( option_value[1]);
					} else if (!strcmp (option_value[0], "translucency")) {
						target->translucency = atoi ( option_value[1]);
					} else if (!strcmp (option_value[0], "intensity")) {
						target->intensity = g_ascii_strtod ( option_value[1], NULL);
					}
					g_strfreev (option_value);
				}
			}
		}

		if (!strncmp (line, "[X11]", 5)) {	
			while (fgets (line, 500, options_file)) {
				if (*line == '\r' || *line == '\n')
					break;
				option_value = g_strsplit (g_strstrip (g_strdelimit (line, "[]\r\n", ' ')), "=", 0);
				if (option_value[1] != NULL) {
					if (!strcmp (option_value[0], "x11_mode")) {
						//target->x11_mode = g_ascii_strtod (option_value[1], NULL);
						target->x11_mode = QString(option_value[1]).toUInt();
					}
					g_strfreev (option_value);
				}
			}
		}

		if (!strncmp (line, "[X11-window]", 12)) {	
			while (fgets (line, 500, options_file)) {
				if (*line == '\r' || *line == '\n')
					break;
				option_value = g_strsplit (g_strstrip (g_strdelimit (line, "[]\r\n", ' ')), "=", 0);
				if (option_value[1] != NULL) {
					if (!strcmp (option_value[0], "cursor")) {
						target->cursor = atoi ( option_value[1]);
					} else if (!strcmp (option_value[0], "mitshm")) {
						target->mitshm = atoi ( option_value[1]);
					} else if (!strcmp (option_value[0], "xvext")) {
						target->xvext = atoi ( option_value[1]);
					} else if (!strcmp (option_value[0], "vidix")) {
						target->vidix = atoi ( option_value[1]);
					} else if (!strcmp (option_value[0], "force_yuv")) {
						target->force_yuv = atoi ( option_value[1]);
					} else if (!strcmp (option_value[0], "xsync")) {
						target->xsync = atoi ( option_value[1]);
					} else if (!strcmp (option_value[0], "privatecmap")) {
						target->privatecmap = atoi ( option_value[1]);
					} else if (!strcmp (option_value[0], "xil")) {
						target->xil = atoi ( option_value[1]);
					} else if (!strcmp (option_value[0], "mtxil")) {
						target->mtxil = atoi ( option_value[1]);
					} else if (!strcmp (option_value[0], "run_in_root_window")) {
						target->run_in_root_window = atoi ( option_value[1]);
					} else if (!strcmp (option_value[0], "root_window_id")) {
						target->root_window_id = atoi ( option_value[1]);
					} else if (!strcmp (option_value[0], "xvgeom_flag")) {
						target->xvgeom_flag = atoi ( option_value[1]);
					} else if (!strcmp (option_value[0], "geometry")) {
						g_snprintf (target->geometry, 10, "%s", option_value[1]);
					}
					g_strfreev (option_value);
				}
			}
		}

		if (!strncmp (line, "[X11-input]", 11)) {	
			while (fgets (line, 500, options_file)) {
				if (*line == '\r' || *line == '\n')
					break;
				option_value = g_strsplit (g_strstrip (g_strdelimit (line, "[]\r\n", ' ')), "=", 0);
				if (option_value[1] != NULL) {
					if (!strcmp (option_value[0], "grabmouse")) {
						target->grabmouse = atoi ( option_value[1]);
					} else if (!strcmp (option_value[0], "grabkeyboard")) {
						target->grabkeyboard = atoi ( option_value[1]);
					} else if (!strcmp (option_value[0], "winkeys")) {
						target->winkeys = atoi ( option_value[1]);
					} else if (!strcmp (option_value[0], "mapkey")) {
						g_snprintf (target->mapkey, 14, "%s", option_value[1]);
					}
					g_strfreev (option_value);
				}
			}
		}

		if (!strncmp (line, "[OpenGL]", 8)) {	
			while (fgets (line, 500, options_file)) {
				if (*line == '\r' || *line == '\n')
					break;
				option_value = g_strsplit (g_strstrip (g_strdelimit (line, "[]\r\n", ' ')), "=", 0);
				if (option_value[1] != NULL) {
					if (!strcmp (option_value[0], "gldblbuffer")) {
						target->gldblbuffer = atoi ( option_value[1]);
					} else if (!strcmp (option_value[0], "gltexture_size")) {
						target->gltexture_size = atoi ( option_value[1]);
					} else if (!strcmp (option_value[0], "glforceblitmode")) {
						target->glforceblitmode = atoi ( option_value[1]);
					} else if (!strcmp (option_value[0], "glext78")) {
						target->glext78 = atoi ( option_value[1]);
					} else if (!strcmp (option_value[0], "glbilinear")) {
						target->glbilinear = atoi ( option_value[1]);
					} else if (!strcmp (option_value[0], "gldrawbitmap")) {
						target->gldrawbitmap = atoi ( option_value[1]);
					} else if (!strcmp (option_value[0], "gldrawbitmapvec")) {
						target->gldrawbitmapvec = atoi ( option_value[1]);
					} else if (!strcmp (option_value[0], "glcolormod")) {
						target->glcolormod = atoi ( option_value[1]);
					} else if (!strcmp (option_value[0], "glbeam")) {
						target->glbeam = g_ascii_strtod (option_value[1], NULL);
					} else if (!strcmp (option_value[0], "glalphablending")) {
						target->glalphablending = atoi ( option_value[1]);
					} else if (!strcmp (option_value[0], "glantialias")) {
						target->glantialias = atoi ( option_value[1]);
					} else if (!strcmp (option_value[0], "glantialiasvec")) {
						target->glantialiasvec = atoi ( option_value[1]);
					} else if (!strcmp (option_value[0], "gllibname")) {
						g_snprintf (target->gllibname, 20, "%s", option_value[1]);
					} else if (!strcmp (option_value[0], "glulibname")) {
						g_snprintf (target->glulibname, 20, "%s", option_value[1]);
					} else if (!strcmp (option_value[0], "cabview")) {
						target->cabview = atoi ( option_value[1]);
					} else if (!strcmp (option_value[0], "cabinet")) {
						g_snprintf (target->cabinet, 20, "%s", option_value[1]);
					} else if (!strcmp (option_value[0], "glres_flag")) {
						target->glres_flag = atoi ( option_value[1]);
					} else if (!strcmp (option_value[0], "glres")) {
						g_snprintf (target->glres, 10, "%s", option_value[1]);
					}
					g_strfreev (option_value);
				}
			}
		}

		if (!strncmp (line, "[SDL]", 5)) {	
			while (fgets (line, 500, options_file)) {
				if (*line == '\r' || *line == '\n')
					break;
				option_value = g_strsplit (g_strstrip (g_strdelimit (line, "[]\r\n", ' ')), "=", 0);
				if (option_value[1] != NULL) {	
					if (!strcmp (option_value[0], "doublebuf")) {
						target->sdl_doublebuf = atoi ( option_value[1]);
					} else if (!strcmp (option_value[0], "sdl_auto_mode")) {
						target->sdl_auto_mode = atoi ( option_value[1]);
					} else if (!strcmp (option_value[0], "modenumber")) {
						target->modenumber = atoi ( option_value[1]);
					} else if (!strcmp (option_value[0], "sdlmapkey")) {
						g_snprintf (target->sdlmapkey, 14, "%s", option_value[1]);
					}
					g_strfreev (option_value);
				}
			}
		}
		if (!strncmp (line, "[GGI]", 5)) {	
			while (fgets (line, 500, options_file)) {
				if (*line == '\r' || *line == '\n')
					break;
				option_value = g_strsplit (g_strstrip (g_strdelimit (line, "[]\r\n", ' ')), "=", 0);
				if (option_value[1] != NULL) {
					if (!strcmp (option_value[0], "ggilinear")) {
						target->ggilinear = atoi ( option_value[1]);
					} else if (!strcmp (option_value[0], "ggi_force_resolution")) {
						target->ggi_force_resolution = atoi ( option_value[1]);
					} else if (!strcmp (option_value[0], "xres")) {
						target->xres = atoi ( option_value[1]);
					} else if (!strcmp (option_value[0], "yres")) {
						target->yres = atoi ( option_value[1]);
					}
					g_strfreev (option_value);
				}
			}
		}

		if (!strncmp (line, "[Svgalib]", 9)) {	
			while (fgets (line, 500, options_file)) {
				if (*line == '\r' || *line == '\n')
					break;
				option_value = g_strsplit (g_strstrip (g_strdelimit (line, "[]\r\n", ' ')), "=", 0);
				if (option_value[1] != NULL) {
					if (!strcmp (option_value[0], "tweak")) {
						target->tweak = atoi ( option_value[1]);
					} else if (!strcmp (option_value[0], "planar")) {
						target->planar = atoi ( option_value[1]);
					} else if (!strcmp (option_value[0], "linear")) {
						target->linear = atoi ( option_value[1]);
					} else if (!strcmp (option_value[0], "centerx")) {
						target->centerx = atoi ( option_value[1]);
					} else if (!strcmp (option_value[0], "centery")) {
						target->centery = atoi ( option_value[1]);
					}
					g_strfreev (option_value);
				}
			}
		}

		if (!strncmp (line, "[FX (Glide)]", 12)) {	
			while (fgets (line, 500, options_file)) {
				if (*line == '\r' || *line == '\n')
					break;
				option_value = g_strsplit (g_strstrip (g_strdelimit (line, "[]\r\n", ' ')), "=", 0);
				if (option_value[1] != NULL) {
					if (!strcmp (option_value[0], "resolution")) {
						g_snprintf (target->resolution, 10, "%s",option_value[1]);
					} else if (!strcmp (option_value[0], "fxgkeepaspect")) {
						target->fxgkeepaspect = atoi ( option_value[1]);
					}
					g_strfreev (option_value);
				}
			}
		}

		if (!strncmp (line, "[Photon]", 8)) {	
			while (fgets (line, 500, options_file)) {
				if (*line == '\r' || *line == '\n')
					break;
				option_value = g_strsplit (g_strstrip (g_strdelimit (line, "[]\r\n", ' ')), "=", 0);
				if (option_value[1] != NULL) {
					if (!strcmp (option_value[0], "render_mode")) {
						target->render_mode = atoi ( option_value[1]);
					}
					g_strfreev (option_value);
				}
			}
		}

		if (!strncmp (line, "[Photon-window]", 15)) {	
			while (fgets (line, 500, options_file)) {
				if (*line == '\r' || *line == '\n')
					break;
				option_value = g_strsplit (g_strstrip (g_strdelimit (line, "[]\r\n", ' ')), "=", 0);
				if (option_value[1] != NULL) {
					if (!strcmp (option_value[0], "phcursor")) {
						target->phcursor = atoi ( option_value[1]);
					}
					g_strfreev (option_value);
				}
			}
		}

		/*		if (!strncmp (line, "[Photon-input]", 14)) {	
				while (fgets (line, 500, options_file)) {
				if (*line == '\r' || *line == '\n')
				break;
				option_value = g_strsplit (g_strstrip (g_strdelimit (line, "[]\r\n", ' ')), "=", 0);
				if (option_value[1] != NULL) {
				if (!strcmp (option_value[0], "phgrabmouse")) {
				target->phgrabmouse = atoi ( option_value[1]);
				} else if (!strcmp (option_value[0], "phwinkeys")) {
				target->phwinkeys = atoi ( option_value[1]);
				} else if (!strcmp (option_value[0], "phmapkey")) {
				g_snprintf (target->phmapkey, 14, option_value[1]);
				} 
				g_strfreev (option_value);
				}
				}
				}*/

		if (!strncmp (line, "[Video Mode]", 12)) {	
			while (fgets (line, 500, options_file)) {
				if (*line == '\r' || *line == '\n')
					break;
				option_value = g_strsplit (g_strstrip (g_strdelimit (line, "[]\r\n", ' ')), "=", 0);
				if (option_value[1] != NULL) {
					if (!strcmp (option_value[0], "keepaspect")) {
						target->keepaspect = atoi ( option_value[1]);
					} else if (!strcmp (option_value[0], "displayaspectratio")) {
						target->displayaspectratio = g_ascii_strtod (option_value[1], NULL);
					} else if (!strcmp (option_value[0], "disablemode")) {
						g_snprintf (target->disablemode, 15, "%s", option_value[1]);
					} 
					g_strfreev (option_value);
				}
			}
		}

		if (!strncmp (line, "[Sound]", 7)) {	
			while (fgets (line, 500, options_file)) {
				if (*line == '\r' || *line == '\n')
					break;
				option_value = g_strsplit (g_strstrip (g_strdelimit (line, "[]\r\n", ' ')), "=", 0);
				if (option_value[1] != NULL) {
					if (!strcmp (option_value[0], "sound")) {
						target->sound = atoi ( option_value[1]);
					} else if (!strcmp (option_value[0], "samples")) {
						target->samples = atoi ( option_value[1]);
					} else if (!strcmp (option_value[0], "fakesound")) {
						target->fakesound = atoi ( option_value[1]);
					} else if (!strcmp (option_value[0], "samplefre")) {
						//target->samplefre = g_ascii_strtod ( option_value[1], NULL);
						target->samplefre = QString ( option_value[1] ).toInt();
					} else if (!strcmp (option_value[0], "bufsize")) {
						target->bufsize = g_ascii_strtod ( option_value[1], NULL);
					} else if (!strcmp (option_value[0], "volume")) {
						target->volume = atoi ( option_value[1]);
					} else if (!strcmp (option_value[0], "audiodevice")) {
						g_snprintf (target->audiodevice, 20, "%s", option_value[1]);
					} else if (!strcmp (option_value[0], "mixerdevice")) {
						g_snprintf (target->mixerdevice, 20, "%s", option_value[1]);
					} else if (!strcmp (option_value[0], "soundfile")) {
						g_snprintf (target->soundfile, 20, "%s", option_value[1]);
					}
					g_strfreev (option_value);
				}
			}
		}

		if (!strncmp (line, "[Input]", 7)) {
			while (fgets (line, 500, options_file)) {
				if (*line == '\r' || *line == '\n')
					break;
				option_value = g_strsplit (g_strstrip (g_strdelimit (line, "[]\r\n", ' ')), "=", 0);
				if (option_value[1] != NULL) {
					if (!strcmp (option_value[0], "joytype")) {
						target->joytype = atoi ( option_value[1]);
					} else if (!strcmp (option_value[0], "analogstick")) {
						target->analogstick = atoi ( option_value[1]);
					} else if (!strcmp (option_value[0], "mouse")) {
						target->mouse = atoi ( option_value[1]);
					} else if (!strcmp (option_value[0], "hotrod")) {
						//target->hotrod = g_ascii_strtod (option_value[1], NULL);
						target->hotrod = (QString (option_value[1] ).toInt() != 0);
					} else if (!strcmp (option_value[0], "hotrodse")) {
						//target->hotrodse = g_ascii_strtod (option_value[1], NULL);
						target->hotrodse = (QString (option_value[1] ).toInt() != 0);
					} else if (!strcmp (option_value[0], "usbpspad")) {
						target->usbpspad = atoi ( option_value[1]);
					} else if (!strcmp (option_value[0], "rapidfire")) {
						target->rapidfire = atoi ( option_value[1]);
					} else if (!strcmp (option_value[0], "keymap")) {
						g_snprintf (target->keymap, 4, "%s", option_value[1]);
					} else if (!strcmp (option_value[0], "ugcicoin")) {
						target->ugcicoin = atoi ( option_value[1]);
					} else if (!strcmp (option_value[0], "ctrlr")) {
						g_snprintf (target->ctrlr, 20, "%s", option_value[1]);
					}

					g_strfreev (option_value);
				}
			}
		}

		if (!strncmp (line, "[i386]", 6)) {
			while (fgets (line, 500, options_file)) {
				if (*line == '\r' || *line == '\n')
					break;
				option_value = g_strsplit (g_strstrip (g_strdelimit (line, "[]\r\n", ' ')), "=", 0);
				if (option_value[1] != NULL) {
					if (!strcmp (option_value[0], "joydevname")) {
						g_snprintf (target->joydevname, 20, "%s", option_value[1]);
					}
					g_strfreev (option_value);
				}
			}
		}

		if (!strncmp (line, "[joy_pad_opts]", 14)) {	
			while (fgets (line, 500, options_file)) {
				if (*line == '\r' || *line == '\n')
					break;
				option_value = g_strsplit (g_strstrip (g_strdelimit (line, "[]\r\n", ' ')), "=", 0);
				if (option_value[1] != NULL) {
					if (!strcmp (option_value[0], "paddevname")) {
						g_snprintf (target->paddevname, 20, "%s", option_value[1]);
					}
					g_strfreev (option_value);
				}
			}
		}

		if (!strncmp (line, "[joy_x11_opts]", 14)) {	
			while (fgets (line, 500, options_file)) {
				if (*line == '\r' || *line == '\n')
					break;
				option_value = g_strsplit (g_strstrip (g_strdelimit (line, "[]\r\n", ' ')), "=", 0);
				if (option_value[1] != NULL) {
					if (!strcmp (option_value[0], "x11joyname")) {
						g_snprintf (target->x11joyname, 20, "%s", option_value[1]);
					}
					g_strfreev (option_value);
				}
			}
		}

		if (!strncmp (line, "[Xinput]", 8)) {	
			while (fgets (line, 500, options_file)) {
				if (*line == '\r' || *line == '\n')
					break;
				option_value = g_strsplit (g_strstrip (g_strdelimit (line, "[]\r\n", ' ')), "=", 0);
				if (option_value[1] != NULL) {
					if (!strcmp (option_value[0], "XInput_trackball1")) {
						g_snprintf (target->XInput_trackball1, 20, "%s", option_value[1]);
					} else if (!strcmp (option_value[0], "XInput_trackball2")) {
						g_snprintf (target->XInput_trackball2, 20, "%s", option_value[1]);
					} else if (!strcmp (option_value[0], "XInput_trackball3")) {
						g_snprintf (target->XInput_trackball3, 20, "%s", option_value[1]);
					} else if (!strcmp (option_value[0], "XInput_trackball4")) {
						g_snprintf (target->XInput_trackball4, 20, "%s", option_value[1]);
						/*					} else if (!strcmp (option_value[0], "XInput_joystick1")) {
											g_snprintf (target->XInput_joystick1, 20, option_value[1]);
											} else if (!strcmp (option_value[0], "XInput_joystick2")) {
											g_snprintf (target->XInput_joystick2, 20, option_value[1]);
											} else if (!strcmp (option_value[0], "XInput_joystick3")) {
											g_snprintf (target->XInput_joystick3, 20, option_value[1]);
											} else if (!strcmp (option_value[0], "XInput_joystick4")) {
											g_snprintf (target->XInput_joystick4, 20, option_value[1]);
						*/					}
					g_strfreev (option_value);
				}
			}
		}

		if (!strncmp (line, "[Digital sound]", 15)) {
			while (fgets (line, 500, options_file)) {
				if (*line == '\r' || *line == '\n')
					break;
				option_value = g_strsplit (g_strstrip (g_strdelimit (line, "[]\r\n", ' ')), "=", 0);
				if (option_value[1] != NULL) {
					if (!strcmp (option_value[0], "dsp_plugin")) {
						g_snprintf (target->dsp_plugin, 20, "%s", option_value[1]);
					} else if (!strcmp (option_value[0], "timer")) {
						target->timer = atoi ( option_value[1]);
					} 
					g_strfreev (option_value);
				}
			}
		}

		if (!strncmp (line, "[Sound mixer]", 13)) {
			while (fgets (line, 500, options_file)) {
				if (*line == '\r' || *line == '\n')
					break;
				option_value = g_strsplit (g_strstrip (g_strdelimit (line, "[]\r\n", ' ')), "=", 0);
				if (option_value[1] != NULL)
					{
						if (!strcmp (option_value[0], "sound_mixer_plugin")) {
							g_snprintf (target->sound_mixer_plugin, 20, "%s", option_value[1]);
						} 
						g_strfreev (option_value);
					}
			}
		}

		if (!strncmp (line, "[arts]", 6)) {
			while (fgets (line, 500, options_file)) {
				if (*line == '\r' || *line == '\n')
					break;
				option_value = g_strsplit (g_strstrip (g_strdelimit (line, "[]\r\n", ' ')), "=", 0);
				if (option_value[1] != NULL) {
					if (!strcmp (option_value[0], "artsBufferTime")) {
						target->artsBufferTime = atoi ( option_value[1]);
					} 
					g_strfreev (option_value);
				}
			}
		}

		if (!strncmp (line, "[Alsa Sound System 0.5]", 23)) {
			while (fgets (line, 500, options_file)) {
				if (*line == '\r' || *line == '\n')
					break;
				option_value = g_strsplit (g_strstrip (g_strdelimit (line, "[]\r\n", ' ')), "=", 0);
				if (option_value[1] != NULL) {
					if (!strcmp (option_value[0], "alsacard")) {
						target->alsacard = atoi ( option_value[1]);
					} else if (!strcmp (option_value[0], "alsadevice")) {
						target->alsadevice = atoi ( option_value[1]);
					}
					g_strfreev (option_value);
				}
			}
		}

		if (!strncmp (line, "[Alsa Sound System 0.9]", 23)) {	
			while (fgets (line, 500, options_file)) {
				if (*line == '\r' || *line == '\n')
					break;
				option_value = g_strsplit (g_strstrip (g_strdelimit (line, "[]\r\n", ' ')), "=", 0);
				if (option_value[1] != NULL) {
					if (!strcmp (option_value[0], "alsa_pcm")) {
						g_snprintf (target->alsa_pcm, 20, "%s", option_value[1]);
					} else if (!strcmp (option_value[0], "alsa_buffer")) {
						target->alsa_buffer = atoi ( option_value[1]);
					}
					g_strfreev (option_value);
				}
			}
		}

		if (!strncmp (line, "[QNX Audio]", 11)) {	
			while (fgets (line, 500, options_file)) {
				if (*line == '\r' || *line == '\n')
					break;
				option_value = g_strsplit (g_strstrip (g_strdelimit (line, "[]\r\n", ' ')), "=", 0);
				if (option_value[1] != NULL) {
					if (!strcmp (option_value[0], "audio_preferred")) {
						target->audio_preferred = atoi ( option_value[1]);
					}
					g_strfreev (option_value);
				}
			}
		}

		/*		if (!strncmp (line, "[Display]", 9)) {	
				while (fgets (line, 500, options_file)) {
				if (*line == '\r' || *line == '\n')
				break;
				option_value = g_strsplit (g_strstrip (g_strdelimit (line, "[]\r\n", ' ')), "=", 0);
				if (option_value[1] != NULL) {
				if (!strcmp (option_value[0], "display_plugin")) {
				g_snprintf (target->display_plugin, 20, option_value[1]);
				} else if (!strcmp (option_value[0], "dwidthscale")) {
				target->dwidthscale = atoi ( option_value[1]);
				} else if (!strcmp (option_value[0], "dheightscale")) {
				target->dheightscale = atoi ( option_value[1]);
				} else if (!strcmp (option_value[0], "dscanlines")) {
				target->dscanlines = atoi ( option_value[1]);
				} else if (!strcmp (option_value[0], "display_aspect_ratio")) {
				target->display_aspect_ratio = g_ascii_strtod (option_value[1], NULL);
				} else if (!strcmp (option_value[0], "keep_aspect")) {
				target->keep_aspect = atoi ( option_value[1]);
				}
				g_strfreev (option_value);
				}
				}
				}*/

		if (!strncmp (line, "[Misc]", 6)) {	
			while (fgets (line, 500, options_file)) {
				if (*line == '\r' || *line == '\n')
					break;
				option_value = g_strsplit (g_strstrip (g_strdelimit (line, "[]\r\n", ' ')), "=", 0);
				if (option_value[1] != NULL) {
					if (!strcmp (option_value[0], "cheat")) {
						target->cheat = atoi ( option_value[1]);
					} else if (!strcmp (option_value[0], "keyboard_leds")) {
						target->keyboard_leds = atoi ( option_value[1]);
					} else if (!strcmp (option_value[0], "debug")) {
						target->debug = atoi ( option_value[1]);
					} else if (!strcmp (option_value[0], "debug_size")) {
						g_snprintf (target->debug_size, 10, "%s", option_value[1]);
					} else if (!strcmp (option_value[0], "log_flag")) {
						target->log_flag = atoi ( option_value[1]);
					} else if (!strcmp (option_value[0], "log")) {
						g_snprintf (target->log, 20, "%s", option_value[1]);
					} else if (!strcmp (option_value[0], "cfgname_flag")) {
						target->cfgname_flag = atoi ( option_value[1]);
					} else if (!strcmp (option_value[0], "cfgname")) {
						g_snprintf (target->cfgname, 20, "%s", option_value[1]);
					} else if (!strcmp (option_value[0], "use_additional_options")) {
						target->use_additional_options = atoi ( option_value[1]);
					} else if (!strcmp (option_value[0], "additional_options")) {
						g_free (target->additional_options);
						target->additional_options = g_strdup (option_value[1]);
					} else if (!strcmp (option_value[0], "skip_disclaimer")) {
						target->skip_disclaimer = atoi ( option_value[1]);
					} else if (!strcmp (option_value[0], "skip_gameinfo")) {
						target->skip_gameinfo = atoi ( option_value[1]);
					} else if (!strcmp (option_value[0], "bios")) {
						target->bios = atoi ( option_value[1]);
					}
					g_strfreev (option_value);
				}
			}
		}
	}

	fclose (options_file);
	
	/* we should not allow xv vidix and DGA at the same time, priority to older (DGA) */
	if (target->x11_mode == 1) {
		target->xvext = FALSE;
		target->vidix = FALSE;
	} else if (target->xvext == 1)
		target->vidix = FALSE;
	
	/* if ctrlr is not available in the list then -> None */
	ctrlr_list = get_ctrlr_list ();
	for (my_list = g_list_first (ctrlr_list); (my_list != NULL); my_list = g_list_next (my_list)) {
		if (strncmp ((char *)my_list->data, target->ctrlr, 20) == 0)
			break;
	}

	if (my_list == NULL)
		strcpy (target->ctrlr, "None");
	g_list_free (ctrlr_list);

	return target;
}

/**
 * Saves the options for the given rom.
 * The filename is ~/.gxmame/options/romname
 *
 * If the rom is NULL then it saves the default options
 * and the opts parameter is not used.
 */
gboolean
save_options (RomEntry    *rom,
	      GameOptions *target)
{
	FILE *options_file = NULL;
	gchar *filename;
	gchar double_buffer[FLOAT_BUF_SIZE];
	
	if (rom == NULL) {
		target = &default_options;
		filename = g_build_filename (g_get_home_dir (), DOT_GXMAME, "options", "default", NULL);	
	} else {
		/* no game preferences to save */
		if (!target)
			return TRUE;

		filename = g_build_filename (g_get_home_dir (), DOT_GXMAME, "options", rom->romname, NULL);	
	}
	
	options_file = fopen (filename, "w");	
	g_free (filename);
	if (!options_file) {
		GXMAME_DEBUG ("unable to write %s option", (rom)? (rom->romname):"default");
		return FALSE;
	}

	fputs ("[Video]\n", options_file);
	fprintf (options_file, "fullscreen=%i\n", target->fullscreen);
	fprintf (options_file, "bpp=%i\n", target->bpp);
	fprintf (options_file, "arbheight=%i\n", target->arbheight);
	fprintf (options_file, "heightscale=%i\n", target->heightscale);
	fprintf (options_file, "widthscale=%i\n", target->widthscale);
	/*	fprintf (options_file, "scale=%s\n", my_dtostr (double_buffer, target->scale)); */
	fprintf (options_file, "effect=%i\n", target->effect);
	fprintf (options_file, "autodouble=%i\n", target->autodouble);
	fprintf (options_file, "dirty=%i\n", target->dirty);
	fprintf (options_file, "scanlines=%i\n", target->scanlines);
	fprintf (options_file, "artwork=%i\n", target->artwork);
	fprintf (options_file, "use_backdrops=%i\n", target->use_backdrops);
	fprintf (options_file, "use_overlays=%i\n", target->use_overlays);
	fprintf (options_file, "use_bezels=%i\n", target->use_bezels);
	fprintf (options_file, "artwork_crop=%i\n", target->artwork_crop);
	fprintf (options_file, "artwork_resolution=%i\n", target->artwork_resolution);
	fprintf (options_file, "frameskipper=%i\n", target->frameskipper);
	fprintf (options_file, "throttle=%i\n", target->throttle);
	fprintf (options_file, "sleepidle=%i\n", target->sleepidle);
	fprintf (options_file, "autoframeskip=%i\n", target->autoframeskip);
	fprintf (options_file, "maxautoframeskip=%i\n", target->maxautoframeskip);
	fprintf (options_file, "frameskip=%i\n", target->frameskip);
	fprintf (options_file, "brightness=%s\n", my_dtostr (double_buffer, target->brightness));
	fprintf (options_file, "gamma_correction=%s\n", my_dtostr (double_buffer, target->gamma_correction));
	fprintf (options_file, "norotate=%i\n", target->norotate);
	fprintf (options_file, "ror=%i\n", target->ror);
	fprintf (options_file, "rol=%i\n", target->rol);
	fprintf (options_file, "flipx=%i\n", target->flipx);
	fprintf (options_file, "flipy=%i\n", target->flipy);

	fputs ("\n[Vector]\n", options_file);
	fprintf (options_file, "vectorres_flag=%i\n", target->vectorres_flag);
	fprintf (options_file, "vectorres=%s\n", target->vectorres);
	fprintf (options_file, "beam=%s\n", my_dtostr (double_buffer, target->beam));
	fprintf (options_file, "flicker=%s\n", my_dtostr (double_buffer, target->flicker));
	fprintf (options_file, "antialias=%i\n", target->antialias);
	fprintf (options_file, "translucency=%i\n", target->translucency);
	fprintf (options_file, "intensity=%s\n", my_dtostr (double_buffer, target->intensity));
	
	fputs ("\n[X11]\n", options_file);
	fprintf (options_file, "x11_mode=%i\n", target->x11_mode);
		
	fputs ("\n[X11-window]\n", options_file);
	fprintf (options_file, "cursor=%i\n", target->cursor);
	
	fprintf (options_file, "mitshm=%i\n", target->mitshm);
	fprintf (options_file, "xvext=%i\n", target->xvext);
	fprintf (options_file, "vidix=%i\n", target->vidix);
	fprintf (options_file, "force_yuv=%i\n", target->force_yuv);
	fprintf (options_file, "xsync=%i\n", target->xsync);
	fprintf (options_file, "privatecmap=%i\n", target->privatecmap);
	fprintf (options_file, "xil=%i\n", target->xil);
	fprintf (options_file, "mtxil=%i\n", target->mtxil);
	fprintf (options_file, "run_in_root_window=%i\n", target->run_in_root_window);
	fprintf (options_file, "root_window_id=%i\n", target->root_window_id);
	fprintf (options_file, "xvgeom_flag=%i\n", target->xvgeom_flag);
	fprintf (options_file, "geometry=%s\n", target->geometry);
	
	fputs ("\n[X11-input]\n", options_file);
	fprintf (options_file, "grabmouse=%i\n", target->grabmouse);
	fprintf (options_file, "grabkeyboard=%i\n", target->grabkeyboard);
	fprintf (options_file, "winkeys=%i\n", target->winkeys);
	fprintf (options_file, "mapkey=%s\n", target->mapkey);

	fputs ("\n[OpenGL]\n", options_file);
	fprintf (options_file, "gldblbuffer=%i\n", target->gldblbuffer);
	fprintf (options_file, "gltexture_size=%i\n", target->gltexture_size);
	fprintf (options_file, "glforceblitmode=%i\n", target->glforceblitmode);
	fprintf (options_file, "glext78=%i\n", target->glext78);
	fprintf (options_file, "glbilinear=%i\n", target->glbilinear);
	fprintf (options_file, "gldrawbitmap=%i\n", target->gldrawbitmap);
	fprintf (options_file, "gldrawbitmapvec=%i\n", target->gldrawbitmapvec);
	fprintf (options_file, "glcolormod=%i\n", target->glcolormod);
	fprintf (options_file, "glalphablending=%i\n", target->glalphablending);
	fprintf (options_file, "glantialias=%i\n", target->glantialias);
	fprintf (options_file, "glantialiasvec=%i\n", target->glantialiasvec);
	fprintf (options_file, "gllibname=%s\n", target->gllibname);
	fprintf (options_file, "glulibname=%s\n", target->glulibname);
	fprintf (options_file, "cabview=%i\n", target->cabview);
	fprintf (options_file, "cabinet=%s\n", target->cabinet);
	fprintf (options_file, "glres_flag=%i\n", target->glres_flag);
	fprintf (options_file, "glres=%s\n", target->glres);

	fputs ("\n[SDL]\n", options_file);
	fprintf (options_file, "doublebuf=%i\n", target->sdl_doublebuf);
	fprintf (options_file, "sdl_auto_mode=%i\n", target->sdl_auto_mode);
	fprintf (options_file, "modenumber=%i\n", target->modenumber);
	fprintf (options_file, "sdlmapkey=%s\n", target->sdlmapkey);
	
	fputs ("\n[GGI]\n", options_file);
	fprintf (options_file, "ggilinear=%i\n", target->ggilinear);
	fprintf (options_file, "ggi_force_resolution=%i\n", target->ggi_force_resolution);
	fprintf (options_file, "xres=%i\n", target->xres);
	fprintf (options_file, "yres=%i\n", target->yres);
	
	fputs ("\n[Svgalib]\n", options_file);
	fprintf (options_file, "tweak=%i\n", target->tweak);
	fprintf (options_file, "planar=%i\n", target->planar);
	fprintf (options_file, "linear=%i\n", target->linear);
	fprintf (options_file, "centerx=%i\n", target->centerx);
	fprintf (options_file, "centery=%i\n", target->centery);

	fputs ("\n[FX (Glide)]\n", options_file);
	fprintf (options_file, "resolution=%s\n", target->resolution);
	fprintf (options_file, "fxgkeepaspect=%i\n", target->fxgkeepaspect);
	
	fputs ("\n[Photon]\n", options_file);
	fprintf (options_file, "render_mode=%i\n", target->render_mode);
	
	fputs ("\n[Photon-window]\n", options_file);
	fprintf (options_file, "phcursor=%i\n", target->phcursor);
	
	/*	fputs ("\n[Photon-input]\n", options_file);
		fprintf (options_file, "phgrabmouse=%i\n", target->phgrabmouse);
		fprintf (options_file, "phwinkeys=%i\n", target->phwinkeys);
		fprintf (options_file, "phmapkey=%s\n", target->phmapkey);
	*/	
	fputs ("\n[Video Mode]\n", options_file);
	fprintf (options_file, "keepaspect=%i\n", target->keepaspect);
	fprintf (options_file, "displayaspectratio=%s\n", 
		 my_dtostr (double_buffer, target->displayaspectratio));
	fprintf (options_file, "disablemode=%s\n", target->disablemode);
	
	fputs ("\n[Sound]\n", options_file);
	fprintf (options_file, "sound=%i\n", target->sound);
	fprintf (options_file, "samples=%i\n", target->samples);
	fprintf (options_file, "fakesound=%i\n", target->fakesound);
	fprintf (options_file, "samplefre=%i\n", target->samplefre);
	fprintf (options_file, "bufsize=%s\n", my_dtostr (double_buffer, target->bufsize));
	fprintf (options_file, "volume=%i\n", target->volume);
	fprintf (options_file, "audiodevice=%s\n", target->audiodevice);
	fprintf (options_file, "mixerdevice=%s\n", target->mixerdevice);
	fprintf (options_file, "soundfile=%s\n", target->soundfile);
	
	fputs ("\n[Input]\n", options_file);
	fprintf (options_file, "joytype=%i\n", target->joytype);
	fprintf (options_file, "analogstick=%i\n", target->analogstick);
	fprintf (options_file, "mouse=%i\n", target->mouse);
	fprintf (options_file, "hotrod=%i\n", target->hotrod);
	fprintf (options_file, "hotrodse=%i\n", target->hotrodse);
	fprintf (options_file, "usbpspad=%i\n", target->usbpspad);
	fprintf (options_file, "rapidfire=%i\n", target->rapidfire);
	fprintf (options_file, "ugcicoin=%i\n", target->ugcicoin);
	fprintf (options_file, "keymap=%s\n", target->keymap);
	fprintf (options_file, "ctrlr=%s\n", target->ctrlr);

	fputs ("\n[i386]\n", options_file);
	fprintf (options_file, "joydevname=%s\n", target->joydevname);
	
	fputs ("\n[joy_pad_opts]\n", options_file);
	fprintf (options_file, "paddevname=%s\n", target->paddevname);
	
	fputs ("\n[joy_x11_opts]\n", options_file);
	fprintf (options_file, "x11joyname=%s\n", target->x11joyname);
	
	fputs ("\n[Xinput]\n", options_file);
	fprintf (options_file, "XInput_trackball1=%s\n", target->XInput_trackball1);
	fprintf (options_file, "XInput_trackball2=%s\n", target->XInput_trackball2);
	fprintf (options_file, "XInput_trackball3=%s\n", target->XInput_trackball3);
	fprintf (options_file, "XInput_trackball4=%s\n", target->XInput_trackball4);
	/*	fprintf (options_file, "XInput_joystick1=%s\n", target->XInput_joystick1);
		fprintf (options_file, "XInput_joystick2=%s\n", target->XInput_joystick2);
		fprintf (options_file, "XInput_joystick3=%s\n", target->XInput_joystick3);
		fprintf (options_file, "XInput_joystick4=%s\n", target->XInput_joystick4);
	*/
	
	fputs ("\n[Digital sound]\n", options_file);
	fprintf (options_file, "dsp_plugin=%s\n", target->dsp_plugin);
	fprintf (options_file, "timer=%i\n", target->timer);

	fputs ("\n[Sound mixer]\n", options_file);
	fprintf (options_file, "sound_mixer_plugin=%s\n", target->sound_mixer_plugin);

	fputs ("\n[arts]\n", options_file);
	fprintf (options_file, "artsBufferTime=%i\n", target->artsBufferTime);
	
	fputs ("\n[Alsa Sound System 0.5]\n", options_file);
	fprintf (options_file, "alsacard=%i\n", target->alsacard);
	fprintf (options_file, "alsadevice=%i\n", target->alsadevice);
	
	fputs ("\n[Alsa Sound System 0.9]\n", options_file);
	fprintf (options_file, "alsa_pcm=%s\n", target->alsa_pcm);
	fprintf (options_file, "alsa_buffer=%i\n", target->alsa_buffer);
	
	fputs ("\n[QNX Audio]\n", options_file);
	fprintf (options_file, "audio_preferred=%i\n", target->audio_preferred);
	
	/*	fputs ("\n[Display]\n", options_file);
		fprintf (options_file, "display_plugin=%s\n", target->display_plugin);
		fprintf (options_file, "dwidthscale=%i\n", target->dwidthscale);
		fprintf (options_file, "dheightscale=%i\n", target->dheightscale);
		fprintf (options_file, "dscanlines=%i\n", target->dscanlines);
		fprintf (options_file, "display_aspect_ratio=%s\n", my_dtostr (double_buffer, target->aspect_ratio));
		fprintf (options_file, "keep_aspect=%i\n", target->keep_aspect);
	*/	
	
	fputs ("\n[Misc]\n", options_file);
	fprintf (options_file, "cheat=%i\n", target->cheat);
	fprintf (options_file, "keyboard_leds=%i\n", target->keyboard_leds);
	fprintf (options_file, "debug=%i\n", target->debug);
	fprintf (options_file, "debug_size=%s\n", target->debug_size);
	fprintf (options_file, "log_flag=%i\n", target->log_flag);
	fprintf (options_file, "log=%s\n", target->log);
	fprintf (options_file, "cfgname_flag=%i\n", target->cfgname_flag);
	fprintf (options_file, "cfgname=%s\n", target->cfgname);
	fprintf (options_file, "use_additional_options=%i\n", target->use_additional_options);
	fprintf (options_file, "additional_options=%s\n", target->additional_options);
	fprintf (options_file, "skip_disclaimer=%i\n", target->skip_disclaimer);
	fprintf (options_file, "skip_gameinfo=%i\n", target->skip_gameinfo);
	fprintf (options_file, "bios=%i\n", target->bios);
	
	fclose (options_file);
	return TRUE;
}
