/*
    libfame - Fast Assembly MPEG Encoder Library
    Copyright (C) 2000-2001 Damien Vincent

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include <stdio.h>
#include <stdlib.h>
#include "fame.h"
#include "fame_rate.h"
#include "fame_monitor.h"

static void rate_init(fame_rate_t *rate,
		      int mb_width,
		      int mb_height,
		      int bitrate,
		      char *coding,
		      fame_frame_statistics_t *stats_list,
		      fame_global_statistics_t *global_stats,
		      unsigned int flags);
static void rate_close(fame_rate_t *rate);
static void rate_enter(fame_rate_t *rate,
		       fame_yuv_t **ref,
		       fame_yuv_t *current,
		       unsigned char *shape,
		       char coding,
		       fame_frame_statistics_t *frame_stats);
static int rate_global_estimation(fame_rate_t *rate);
static int rate_local_estimation(fame_rate_t *rate,
				 int mb_x, int mb_y,
				 short *blocks[6],
                                 fame_motion_coding_t motion_coding);
static void rate_leave(fame_rate_t *rate,
		       int spent);

FAME_CONSTRUCTOR(fame_rate_t)
{
  FAME_OBJECT(this)->name = "rate estimation";
  FAME_RATE(this)->init = rate_init;
  FAME_RATE(this)->close = rate_close;
  FAME_RATE(this)->enter = rate_enter;
  FAME_RATE(this)->global_estimation = rate_global_estimation;
  FAME_RATE(this)->local_estimation = rate_local_estimation;
  FAME_RATE(this)->leave = rate_leave;
  FAME_RATE(this)->flags = 0xffffffff;
  return(this);
}

/*  rate_init                                                                */
/*                                                                           */
/*  Description:                                                             */
/*    Initialise rate estimation.                                            */
/*                                                                           */
/*  Arguments:                                                               */
/*    fame_rate_t *rate: the rate estimation                                 */
/*    int mb_width: width in macroblocks                                     */
/*    int mb_height: height in macroblocks                                   */
/*                                                                           */
/*  Return value:                                                            */
/*    Rate.                                                                  */

static void rate_init(fame_rate_t *rate,
		      int mb_width,
		      int mb_height,
		      int bitrate,
		      char *coding,
		      fame_frame_statistics_t *stats_list,
		      fame_global_statistics_t *global_stats,
		      unsigned int flags)
{
  rate->mb_width = mb_width;
  rate->mb_height = mb_height;
  rate->bitrate = bitrate;
  rate->available = 0;
  rate->spent = 0;
  rate->global_scale = 8;
  rate->flags &= flags;
  if (stats_list) 
    rate->flags |= FAME_RATE_SECOND_PASS;
  else
    rate->flags &= ~FAME_RATE_SECOND_PASS;
}

/*  rate_close                                                               */
/*                                                                           */
/*  Description:                                                             */
/*    Release rate estimation.                                               */
/*                                                                           */
/*  Arguments:                                                               */
/*    fame_rate_t *rate: the rate estimation                                 */
/*                                                                           */
/*  Return value:                                                            */
/*    Rate.                                                                  */

static void rate_close(fame_rate_t *rate)
{
}

/*  rate_enter                                                               */
/*                                                                           */
/*  Description:                                                             */
/*    Prepare for a new frame.                                               */
/*                                                                           */
/*  Arguments:                                                               */
/*    fame_rate_t *rate: the rate estimation                                 */
/*    fame_yuv_t **ref: the reference frames (half-pel)                      */
/*    fame_yuv_t *current: the current frame                                 */
/*    unsigned char *shape: the current shape                                */
/*                                                                           */
/*  Return value:                                                            */
/*    Rate.                                                                  */

static void rate_enter(struct _fame_rate_t_ *rate,
		       fame_yuv_t **ref,
		       fame_yuv_t *current,
		       unsigned char *shape,
		       char coding,
		       fame_frame_statistics_t *frame_stats)
{
  rate->ref = ref;
  rate->current = current;
  rate->shape = shape;
  rate->coding = coding;

  if (frame_stats) 
  {
    frame_stats->target_bits = rate->available;      
  }
}

/*  rate_leave                                                               */
/*                                                                           */
/*  Description:                                                             */
/*    Finish estimating a frame.                                             */
/*                                                                           */
/*  Arguments:                                                               */
/*    fame_rate_t *rate: the rate estimation                                 */
/*                                                                           */
/*  Return value:                                                            */
/*    Rate.                                                                  */

static void rate_leave(fame_rate_t *rate, int spent)
{  
  rate->spent = spent;
  rate->available -= spent;
}

static int rate_global_estimation(fame_rate_t *rate)
{
  return(rate->global_scale);
}

static int rate_local_estimation(fame_rate_t *rate,
				 int mb_x, int mb_y,
				 short *blocks[6],
                                 fame_motion_coding_t motion_coding)
{
  int dquant = 0;

  if(motion_coding != motion_inter4v)
  {
    /* TODO */
  }

  return(dquant);
}












