/* ============================================================
 * File  : streambrowser.h
 * Author: Eric Giesselbach <ericgies@kabelfoon.nl>
 * Date  : 2003-12-22
 * Description : av stream repository handling
 *
 * Copyright 2003 by Eric Giesselbach

 * This program is free software; you can redistribute it
 * and/or modify it under the terms of the GNU General
 * Public License as published bythe Free Software Foundation;
 * either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * ============================================================ */

// development snapshot, many todo's

#ifndef STORAGEHANDLERS_H
#define STORAGEHANDLERS_H

#include <iostream>
#include <qwidget.h>
#include <qdatetime.h>
#include <qprocess.h>
#include <qhttp.h>
#include <qvaluevector.h>


typedef QValueVector<QString> ValueList;


class Record
{
  public:
    Record(int off, int len);

    int offset;
    int length;
    ValueList values;
};

class ChangedRecord
{
  public:
    bool    error;
    int     ident; // source identifier

    ValueList oldValues;  // on update: existing item data
    ValueList values;     // item (on update: new item data)

    void resetState();
};

class RecordList : public QPtrList<Record>
{
  public:
    RecordList() { urlCompare = false; }
    virtual int compareItems( QPtrCollection::Item i1, QPtrCollection::Item i2 );
    // if record is not valid edit record or return false to prevent insertion in list:
    virtual bool validateItem( Record *record );
    bool urlCompare;
    // key fields
    int key_pri;
    int key_mul1;
    int key_mul2;
};


/*
  Absolutely NOT thread safe

  Storage constraints:
    1) url is unique
    2) (folder, name) is unique

    --> on insertRecord of unique url with non-unique (folder,name),
        name is changed to url (variable name is passed by reference).

    bool loadList()
      clears memory list, reloads records from resource

    bool storeList()
      clear current selected storage and insert records from memory list
      -> selectDbStorage + loadList + selectFileStorage + storeList copies db records to file.
      -> In case of file storage: selectFileStorage + removeRecord + storeList compacts file
*/

class GenStorage : public QObject
{
    Q_OBJECT

  public:

    enum RequestType
    {
      idle = 0,
      sethost,
      handshake,
      getlist,
      savelist,
      insert,
      update,
      remove
    };

    // copied in Storage class
    enum EventType
    {
      selected = 0,
      loaded,
      saved
    };

    GenStorage(QString name, int accessType, int key_pri, int key_mul1, int key_mul2);
    ~GenStorage();

    virtual QString getStorageDescription() = 0;  // string identifying resource

    int        getAccessType()    { return accessType;      }
    QString    getStorageName()   { return accessName;      }
    bool       getSynchronized()  { return accessInSync;    }
    bool       storageReadOnly()  { return accessReadOnly;  }
    QString&   getLastError()     { return accessLastError; }

    // list manipulation
    virtual bool loadList (int ident, QString& error);
    virtual bool storeList(int ident, RecordList& list, QString& error);

    RecordList& getListHandler() { return recordList; }

    // record manipulation
    virtual bool insertRecord(int ident, ValueList& values, QString& error);
    virtual bool updateRecord(int ident, ValueList& oldValues, ValueList& values, QString& error);
    virtual bool removeRecord(int ident, ValueList values, QString& error);

    // list walk
    void resetRecordList() { listreset = true; }
    bool getNextRecord(RecordList& list, ValueList& values);

  signals:
    void storageEvent(int ident, int eventType, bool error); // ident=0: internal use
    void recordInserted(ChangedRecord* rec);
    void recordUpdated(ChangedRecord* rec);
    void recordRemoved(ChangedRecord* rec);

  protected:
    RecordList   recordList;
    bool         listreset;
    ChangedRecord *changedRecord; // last changed/removed/inserted record

    virtual void closeStorage() = 0;

    int findItemKeyIndex(ValueList& value);       // find item by key (see Record class)

    int findItemResourceIndex(ValueList& values); // find item by resource (see Record class)
    ValueList getItemValues(int index);

  // access properties
    QString       accessResource;      // table or file name
    bool          accessReadOnly;      // storage is readonly
    QString       accessName;          // storage name
  // access states
    bool          accessInSync;        // list in memory and storage are synchronized
    bool          accessNeedsCompact;  // storage wants cleanup before app close.
    ValueList     accessDbFields;      //
  // access processing
    RequestType   accessRequestType;   // current request
    QString       accessLastError;     // last error
    int           accessSourceIdent;   // caller identifier
    int           accessType;

    void resetState();
};


class FileStorage : public GenStorage
{
   Q_OBJECT

   public:
     FileStorage(QString name, int accessType, int key_pri, int key_mul1, int key_mul2);
     ~FileStorage();

     bool    openFileStorage(int ident, QString fileName);
     QString getStorageDescription();

     bool    loadList(int ident, QString& error);
     bool    storeList(int ident, RecordList& list, QString& error); // overwrite
     bool    insertRecord(int ident, ValueList& values, QString& error);
     bool    updateRecord(int ident, ValueList& oldValues, ValueList& values, QString& error);
     bool    removeRecord(int ident, ValueList values, QString& error);

   protected:
     void    closeStorage();

   private:
     bool    appendFileRecord();
     bool    blankFileRecord();
     bool    loadListFromFile();
     bool    saveListToFile(RecordList& list);

     QFile   myFile;
};

class DatabaseStorage : public GenStorage
{

  Q_OBJECT

  public:
    DatabaseStorage(QString table, QString name,
                                     int accessType, int key_pri, int key_mul1, int key_mul2);
    ~DatabaseStorage();

    QString getStorageDescription();

    bool openDatabaseStorage(int ident, QString dbDriver, QString hostName,
                                              unsigned int port, QString dbName, QString table,
                                              QString login, QString password);
    bool openDefaultDb(int ident);
    bool loadList(int ident, QString& error);
    bool storeList(int ident, RecordList& list, QString& error);
    bool insertRecord(int ident, ValueList& values, QString& error);
    bool updateRecord(int ident, ValueList& oldValues, ValueList& values, QString& error);
    bool removeRecord(int ident, ValueList values, QString& error);

  protected:
    void closeStorage();

  private:
    ValueList accessDbFields;
    QString defaultTable;    //
    QString defaultName;     //

    bool loadDbFields();
    bool loadListFromDb();
    bool saveListToDb(RecordList& list);
    bool insertDbRecord();
    bool updateDbRecord();
    bool removeDbRecord();

};


//----------------------- Web Storage ---------------------------

class WebStorage : public GenStorage
{
  Q_OBJECT

  public:
    WebStorage(QString name, int accessType, int key_pri, int key_mul1, int key_mul2);
    ~WebStorage();

    QString getStorageDescription();

    void openWebStorage(int ident, QString url, QString login, QString password);
    bool loadList(int ident, QString& error);
    bool storeList(int ident, RecordList& list, QString& error); // no individual errors
    bool insertRecord(int ident, ValueList& values, QString& error);
    bool updateRecord(int ident, ValueList& oldValues, ValueList& values, QString& error);
    bool removeRecord(int ident, ValueList values, QString& error);

  protected:
    void closeStorage();

  private slots:
    void slotRequestFinished(int id, bool error);

  private:
    QHttp *http;
    QString host;

    QString accessLogin, accessPassword, authQuery;

    void processWebResponse(bool error);
    bool getWebResponse();
    void parseWebList(QString& data);
    void postToWeb(QString data, bool usePost = true);
    bool removeWebRecord();
    bool updateWebRecord();
    bool insertWebRecord();
    bool saveListToWeb(RecordList& list);
};

#endif /* STORAGEHANDLERS_H */



