/*
**	parse_image.c
**
**	This file is subject to the terms and conditions of the GNU General Public
**	License.  See the file COPYING in the main directory of this archive
**	for more details.
**
**	This file parses the kernel image and the ramdisk image and can load them
**	into memory.  Pretty specialized.
**	See parse_image.h for structure declarations.
**
**/

#include <Aliases.h>

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <stddef.h>
#include <string.h>
#include <ctype.h>
/*#include <unix.h>*/

/* linux specific include files */
#include "linux_a.out.h"
#include "linux_elf.h"
#include "asm_page.h"
#include "asm_setup.h"

#include <ShutDown.h>
#include <Gestalt.h> // JDJ
#include <Endian.h>
#include <ToolUtils.h>
//#include "penguin.h"
#include "bootstrap_prototypes.h"
#include "appkill_prototypes.h"
#include "penguin_prototypes.h"
#include "penguin_utils.h"
#include "prefs_prototypes.h"
#include "asm_prototypes.h"
#include "videocard_prototypes.h"

#include "MMU.h"

#include "stream.h"

#define	GZIP_MAGIC     '\037\213'	/* Magic header for gzip files, 1F 8B */
#define	OLD_GZIP_MAGIC '\037\236'	/* Magic header for gzip 0.5 = freeze 1.x */
#define BZ2_MAGIC      '\102\132'   /* Magic header for bz2, 42 5A */

/* Prototypes */
static long		get_image_size(ConstFSSpecPtr pFSS);
static long		get_unzip_size(ConstFSSpecPtr pFSS);
static long		read_image(ConstFSSpecPtr pFSS, char *bptr, long nbytes);
static long		chunk_sread(char *dst, long size);

/*
 *	parse_kernel_image
 *
 */
unsigned long parse_kernel_image (struct Kernel_Image_Info *kernel_info)
{
//	short						kfd;
	long						how_much;
	unsigned long				kernel_size, min_addr, max_addr;
	Boolean						junk;
	OSErr						err;
	FSSpec						kernel_spec;
	int							i;
	struct Kernel_Image_Info	temp_info;
	char						fname[256];
	unsigned long				magic;

	/* open kernel executable and read exec header */
	if ((err = ResolveAlias(&app_spec, config.kernel_alias, &kernel_spec, &junk)) != noErr)  {
		ErrorNum ("Unable to open kernel file (couldn't resolve alias, error %d)\n", err);
	}

	/* get filename from FSSpec */
	if (path_from_FSSpec(&kernel_spec, (unsigned char *)fname) != noErr) {
		ErrorNum ("Unable to open kernel file (path_from_FSSpec error %d)\n", err);
	}
	fname[fname[0]+1] = 0;
	BlockMoveData(&fname[1], &fname[0], fname[0]+1);

	/* initialize streams */
	stream_init();
	stream_push(&file_mod);
	if (sopen(fname) < 0) {
		sclose();
		Error ("Unable to determine kernel file format (sopen failed)\n");
	}
	
	how_much = chunk_sread((char *)&magic, sizeof(magic));
	if (how_much != sizeof(magic)) {
		sclose();
		Error ("Unable to determine kernel file format (sread error)\n");
	}
	
	magic = HiWord(magic); /* get the first 16 bits */
	
//	cprintf("Magic Number: %X\n", magic);
	
	sclose();
	/* initialize streams, with decompression */
	stream_init();
	stream_push(&file_mod);
	
	if ((magic == GZIP_MAGIC) || (magic == OLD_GZIP_MAGIC)) {
		stream_push(&gunzip_mod);
	} else if (magic == BZ2_MAGIC) {
		stream_push(&bunzip2_mod);
	}
	
	/* open kernel image */
	if (sopen(fname) < 0) {
		sclose();
		Error ("Unable to open kernel file (sopen failed)\n");
	}

	/* read kernel exec header */
	how_much = chunk_sread((char *)&temp_info.exec, sizeof(temp_info.exec));
	if (how_much != sizeof(temp_info.exec))  {
		sclose();
		Error ("Unable to read exec header from kernel file (sread error)\n");
	}

	/* Check kernel format */
	switch (N_MAGIC(temp_info.exec))  {
		case ZMAGIC:
			temp_info.type = KERNEL_AOUT;
			temp_info.aout_text_offset = N_TXTOFF(temp_info.exec);
			cprintf("Kernel format: a.out (ZMAGIC)\n");
			break;
		case QMAGIC:
			temp_info.type = KERNEL_AOUT;
			temp_info.aout_text_offset = sizeof(temp_info.exec);
			/* the text size includes the exec header; remove this */
			temp_info.exec.a_text -= sizeof(temp_info.exec);
			cprintf("Kernel format: a.out (QMAGIC)\n");
			break;
		default:
			/* Try to parse it as an ELF header */
			if (sseek(0, SEEK_SET) == -1) {
				sclose();
				Error("Failed to seek to start\n");
			}

			how_much = chunk_sread((char *)&temp_info.exec_elf, sizeof (temp_info.exec_elf));
			if (	(how_much != sizeof(temp_info.exec_elf)) 
					|| (memcmp (&temp_info.exec_elf.e_ident[EI_MAG0], ELFMAG, SELFMAG) != 0) ) {
				sclose();
				ErrorNum("Wrong magic number %lo in kernel header\n", N_MAGIC(temp_info.exec));
			}
			
			temp_info.type = KERNEL_ELF;
			
			/* A few plausibility checks */
			if (	(temp_info.exec_elf.e_type != ET_EXEC)
					|| (temp_info.exec_elf.e_machine != EM_68K)
					|| (temp_info.exec_elf.e_version != EV_CURRENT) ) {
				sclose();
				Error ("Invalid ELF header contents in kernel\n");
			}

			/* Load the program headers */
			temp_info.phdrs_elf = (Elf32_Phdr *) NewPtrClear (temp_info.exec_elf.e_phnum * sizeof (Elf32_Phdr));
			if (temp_info.phdrs_elf == NULL) {
				sclose();
				ErrorNum ("Unable to allocate memory for kernel headers (%d bytes)", temp_info.exec_elf.e_phnum * sizeof (Elf32_Phdr));
			}

			if (sseek(temp_info.exec_elf.e_phoff, SEEK_SET) == -1) {
				sclose();
				Error("Failed to seek to program headers\n");
			}

			how_much = chunk_sread((char *)temp_info.phdrs_elf, temp_info.exec_elf.e_phnum * sizeof (*temp_info.phdrs_elf));
			if ( (how_much < (temp_info.exec_elf.e_phnum * sizeof (*temp_info.phdrs_elf))) ) {
				sclose();
				ErrorNum ("Unable to read program headers from kernel file (error %d)\n", err);
			}
			cprintf("Kernel format: ELF\n");
			break;
			
	}
	
	sclose();

	switch (temp_info.type)
	{
		case KERNEL_AOUT:
			/* Align bss size to multiple of four */
			temp_info.exec.a_bss = (temp_info.exec.a_bss + 3) & ~3;
			kernel_size = temp_info.exec.a_text + temp_info.exec.a_data + temp_info.exec.a_bss;
			break;
			
		case KERNEL_ELF:
			/* calculate the total required amount of memory */
			min_addr = 0xffffffff;
			max_addr = 0;
			for (i = 0; i < temp_info.exec_elf.e_phnum; i++) 
			{
				if (min_addr > temp_info.phdrs_elf[i].p_vaddr)
					min_addr = temp_info.phdrs_elf[i].p_vaddr;
				if (max_addr < temp_info.phdrs_elf[i].p_vaddr + temp_info.phdrs_elf[i].p_memsz)
					max_addr = temp_info.phdrs_elf[i].p_vaddr + temp_info.phdrs_elf[i].p_memsz;
			}
			/* This is needed for newer linkers that include the header in
			   the first segment.  */
			if (min_addr == 0) 
			{
				min_addr = PAGE_SIZE;
				temp_info.phdrs_elf[0].p_vaddr += PAGE_SIZE;
				temp_info.phdrs_elf[0].p_offset += PAGE_SIZE;
				temp_info.phdrs_elf[0].p_filesz -= PAGE_SIZE;
				temp_info.phdrs_elf[0].p_memsz -= PAGE_SIZE;
			}
			kernel_size = max_addr - min_addr;
			break;
			
		default:
			ErrorNum ("Kernel type incorrectly determined: %d", temp_info.type);
	}
	
	/*
	 *	Copy information to the caller
	 */
	*kernel_info = temp_info;
	
	return kernel_size;
}

/*
 *	parse_read_kernel
 *
 */
void parse_read_kernel (struct Kernel_Image_Info *kernel_info, unsigned long where)
{
//	short		kfd;
	long		how_much;
	Boolean		junk;
	OSErr		err;
	FSSpec		kernel_spec;
	int			i;
	char		fname[256];
	unsigned long magic;

	/* open kernel executable and read exec header */
	if ((err = ResolveAlias(&app_spec, config.kernel_alias, &kernel_spec, &junk)) != noErr) 
		ErrorNum ("Unable to open kernel file (couldn't resolve alias, error %d)\n", err);

	/* get filename from FSSpec */
	if (path_from_FSSpec(&kernel_spec, (unsigned char *)fname) != noErr) {
		ErrorNum ("Unable to open kernel file (path_from_FSSpec error %d)\n", err);
	}
	fname[fname[0]+1] = 0;
	BlockMoveData(&fname[1], &fname[0], fname[0]+1);


	/* initialize streams */
	stream_init();
	stream_push(&file_mod);
	if (sopen(fname) < 0) {
		sclose();
		Error ("Unable to determine kernel file format (sopen failed)\n");
	}
	
	how_much = chunk_sread((char *)&magic, sizeof(magic));
	if (how_much != sizeof(magic)) {
		sclose();
		Error ("Unable to determine kernel file format (sread error)\n");
	}
	
	magic = HiWord(magic); /* get the first 16 bits */
	
//	cprintf("Magic Number: %X\n", magic);
	
	sclose();
	/* initialize streams, with decompression */
	stream_init();
	stream_push(&file_mod);
	
	if ((magic == GZIP_MAGIC) || (magic == OLD_GZIP_MAGIC)) {
		stream_push(&gunzip_mod);
	} else if (magic == BZ2_MAGIC) {
		stream_push(&bunzip2_mod);
	}
	
	/* open kernel image */
	if (sopen(fname) < 0) {
		sclose();
		Error ("Unable to open kernel file (sopen failed)\n");
	}

	/* read the text and data segments from the kernel image */
	switch (kernel_info->type) {
		case KERNEL_ELF:
			for (i = 0; i < kernel_info->exec_elf.e_phnum; i++) {

				if (sseek(kernel_info->phdrs_elf[i].p_offset, SEEK_SET) == -1) {
					sclose();
					ErrorNum ("Failed to seek to segment %d", i);
				}

				how_much = chunk_sread((char *)(where + kernel_info->phdrs_elf[i].p_vaddr - PAGE_SIZE),
							kernel_info->phdrs_elf[i].p_filesz);
				if (how_much != kernel_info->phdrs_elf[i].p_filesz) {
					sclose();
					ErrorNum ("Failed to read segment %d", i);
				}

				if (config.debug_segment_info)
					cprintf("Read %ld bytes for segment %d, requested %ld\n",
							how_much, i, kernel_info->phdrs_elf[i].p_filesz);

			}
			break;
		
		case KERNEL_AOUT:
			if (sseek(kernel_info->aout_text_offset, SEEK_SET) == -1) {
				sclose();
				Error( "Failed to seek to text segment\n" );
			}

			if (chunk_sread((char *)where, kernel_info->exec.a_text) != kernel_info->exec.a_text) {
				sclose();
				Error ("Failed to read text");
			}

			if (chunk_sread((char *)(where + kernel_info->exec.a_text), kernel_info->exec.a_data) != kernel_info->exec.a_data) {
				sclose();
				Error ("Failed to read data");
			}

		default:
			sclose();
			Error ("Unknown kernel_info->type\n");
			break;
	}

	sclose();
}


// ============= ============= ============= ============= ============= =============
// ============= ============= ============= ============= ============= =============
// ============= ============= ============= ============= ============= =============
#if 0

/*
 *	parse_kernel_image
 *
 */
unsigned long parse_kernel_image (struct Kernel_Image_Info *kernel_info)
{
	short kfd;
	long how_much;
	unsigned long kernel_size, min_addr, max_addr;
	Boolean junk;
	OSErr err;
	FSSpec kernel_spec;
	int i;
	
	struct Kernel_Image_Info		temp_info;

	/* open kernel executable and read exec header */
	if ((err = ResolveAlias(&app_spec, config.kernel_alias, &kernel_spec, &junk)) != noErr) 
	{
		ErrorNum ("Unable to open kernel file (couldn't resolve alias, error %d)\n", err);
	}
	if ((err = FSpOpenDF(&kernel_spec, fsRdPerm, &kfd)) != noErr) 
	{
		ErrorNum ("Unable to open kernel file (OpenDF failed, error %d)\n", err);
	}

	how_much = sizeof(temp_info.exec);
	err = FSRead (kfd, &how_much, (void *)&temp_info.exec);
	if ((how_much != sizeof(temp_info.exec)) || (err != noErr)) 
	{
		ErrorNum ("Unable to read exec header from kernel file (error &d)\n", err);
	}

	switch (N_MAGIC(temp_info.exec)) 
	{
	case ZMAGIC:
		temp_info.type = KERNEL_AOUT;
		temp_info.aout_text_offset = N_TXTOFF(temp_info.exec);
		cprintf("Kernel format: a.out (ZMAGIC)\n");
		break;
	case QMAGIC:
		temp_info.type = KERNEL_AOUT;
		temp_info.aout_text_offset = sizeof(temp_info.exec);
		/* the text size includes the exec header; remove this */
		temp_info.exec.a_text -= sizeof(temp_info.exec);
		cprintf("Kernel format: a.out (QMAGIC)\n");
		break;
	default:
		/* Try to parse it as an ELF header */
		SetFPos(kfd, fsFromStart, 0);
		how_much = sizeof (temp_info.exec_elf);
		err = FSRead (kfd, &how_much, (void *)&temp_info.exec_elf);
		if (   (err != noErr)
			|| (how_much != sizeof(temp_info.exec_elf)) 
			|| memcmp (&temp_info.exec_elf.e_ident[EI_MAG0], ELFMAG, SELFMAG) != 0)
		{
			ErrorNum("Wrong magic number %lo in kernel header\n", N_MAGIC(temp_info.exec));
		}
		
		temp_info.type = KERNEL_ELF;
		
		/* A few plausibility checks */
		if (   temp_info.exec_elf.e_type != ET_EXEC 
			|| temp_info.exec_elf.e_machine != EM_68K
			|| temp_info.exec_elf.e_version != EV_CURRENT) 
		{
			Error ("Invalid ELF header contents in kernel\n");
		}
		/* Load the program headers */
		temp_info.phdrs_elf = (Elf32_Phdr *) NewPtrClear (temp_info.exec_elf.e_phnum * sizeof (Elf32_Phdr));
		
		if (temp_info.phdrs_elf == NULL) 
		{
			ErrorNum ("Unable to allocate memory for kernel headers (%d bytes)", temp_info.exec_elf.e_phnum * sizeof (Elf32_Phdr));
		}
		
		SetFPos(kfd, fsFromStart, temp_info.exec_elf.e_phoff);
		how_much = temp_info.exec_elf.e_phnum * sizeof (*temp_info.phdrs_elf);
		err = FSRead (kfd, &how_much, (void *)temp_info.phdrs_elf);
		if ((how_much < temp_info.exec_elf.e_phnum * sizeof (*temp_info.phdrs_elf)) || (err != noErr)) 
		{
			ErrorNum ("Unable to read program headers from kernel file (error %d)\n", err);
		}
		cprintf("Kernel format: ELF\n");
		break;
	}
	
	FSClose (kfd);

	switch (temp_info.type)
	{
		case KERNEL_AOUT:
			/* Align bss size to multiple of four */
			temp_info.exec.a_bss = (temp_info.exec.a_bss + 3) & ~3;
			kernel_size = temp_info.exec.a_text + temp_info.exec.a_data + temp_info.exec.a_bss;
			break;
			
		case KERNEL_ELF:
			/* calculate the total required amount of memory */
			min_addr = 0xffffffff;
			max_addr = 0;
			for (i = 0; i < temp_info.exec_elf.e_phnum; i++) 
			{
				if (min_addr > temp_info.phdrs_elf[i].p_vaddr)
					min_addr = temp_info.phdrs_elf[i].p_vaddr;
				if (max_addr < temp_info.phdrs_elf[i].p_vaddr + temp_info.phdrs_elf[i].p_memsz)
					max_addr = temp_info.phdrs_elf[i].p_vaddr + temp_info.phdrs_elf[i].p_memsz;
			}
			/* This is needed for newer linkers that include the header in
			   the first segment.  */
			if (min_addr == 0) 
			{
				min_addr = PAGE_SIZE;
				temp_info.phdrs_elf[0].p_vaddr += PAGE_SIZE;
				temp_info.phdrs_elf[0].p_offset += PAGE_SIZE;
				temp_info.phdrs_elf[0].p_filesz -= PAGE_SIZE;
				temp_info.phdrs_elf[0].p_memsz -= PAGE_SIZE;
			}
			kernel_size = max_addr - min_addr;
			break;
			
		default:
			ErrorNum ("Kernel type incorrectly determined: %d", temp_info.type);
	}
	
	/*
	 *	Copy information to the caller
	 */
	*kernel_info = temp_info;
	
	return kernel_size;
}

/*
 *	parse_read_kernel
 *
 */
void parse_read_kernel (struct Kernel_Image_Info *kernel_info, unsigned long where)
{
	short kfd;
	long how_much;
	Boolean junk;
	OSErr err;
	FSSpec kernel_spec;
	int i;

	/* open kernel executable and read exec header */
	if ((err = ResolveAlias(&app_spec, config.kernel_alias, &kernel_spec, &junk)) != noErr) 
		ErrorNum ("Unable to open kernel file (couldn't resolve alias, error %d)\n", err);

	if ((err = FSpOpenDF(&kernel_spec, fsRdPerm, &kfd)) != noErr) 
		ErrorNum ("Unable to open kernel file (OpenDF failed, error %d)\n", err);

	/* read the text and data segments from the kernel image */
	switch (kernel_info->type)
	{
		case KERNEL_ELF:
			for (i = 0; i < kernel_info->exec_elf.e_phnum; i++)
			{
				if (SetFPos (kfd, fsFromStart, kernel_info->phdrs_elf[i].p_offset) != noErr)
					ErrorNum ("Failed to seek to segment %d", i);
	
				how_much = kernel_info->phdrs_elf[i].p_filesz;
				err = FSRead (kfd, &how_much, (void *)(where + kernel_info->phdrs_elf[i].p_vaddr - PAGE_SIZE));
				if ((how_much != kernel_info->phdrs_elf[i].p_filesz) || (err != noErr))
					ErrorNum ("Failed to read segment %d", i);
			}
			break;
		
		case KERNEL_AOUT:
			if (SetFPos (kfd, fsFromStart, kernel_info->aout_text_offset) != noErr) 
				Error ("Failed to seek to text");
	
			how_much = kernel_info->exec.a_text;
			err = FSRead (kfd, &how_much, (void *)where);
			if ((how_much != kernel_info->exec.a_text) || (err != noErr))
				Error ("Failed to read text");
	
			/* data follows immediately after text */
			how_much = kernel_info->exec.a_data;
			err = FSRead (kfd, &how_much, (void *) (where + kernel_info->exec.a_text));
			if ((how_much != kernel_info->exec.a_data) || (err != noErr))
				Error ("Failed to read data");
			break;
	}
	FSClose (kfd);
}

#endif
// ============= ============= ============= ============= ============= =============
// ============= ============= ============= ============= ============= =============
// ============= ============= ============= ============= ============= =============

/*
 *	parse_ramdisk
 *
 */
unsigned long parse_ramdisk (void)
{
	OSErr err;
	FSSpec ramdisk_spec;
	Boolean junk;
	short	rfd;
	long how_much;
	unsigned long rd_size;

	/* init ramdisk */
	if (config.do_ramdisk) 
	{
		if ((err = ResolveAlias(&app_spec, config.ramdisk_alias, &ramdisk_spec, &junk)) != noErr) 
		{
			ErrorNum ("Unable to open ramdisk file (couldn't resolve alias, error %d)\n", err);
		}

		if (FSpOpenDF(&ramdisk_spec, fsRdPerm, &rfd) != noErr) 
		{
			ErrorNum ("Unable to open ramdisk file (OpenDF failed, error %d)\n", err);
		}
		how_much = 0;
		GetEOF(rfd, &how_much);
		FSClose (rfd);

		rd_size = (how_much + 1023) & ~1023;
	} else
		rd_size = 0;

	bi.ramdisk_size = rd_size / 1024;

	return rd_size;
}


/*
 *	parse_read_ramdisk
 *
 */
void parse_read_ramdisk (unsigned long size, unsigned long where)
{
	OSErr		err;
	FSSpec		ramdisk_spec;
	Boolean		junk;
	short		rfd;
	long		how_much, pad_bytes;
	char		*pad_ptr;

	how_much = 0;

	/* init ramdisk */
	if (config.do_ramdisk && size > 0) 
	{
		if ((err = ResolveAlias(&app_spec, config.ramdisk_alias, &ramdisk_spec, &junk)) != noErr) 
		{
			ErrorNum ("Unable to open ramdisk file (couldn't resolve alias, error %d)\n", err);
		}

		if (FSpOpenDF(&ramdisk_spec, fsRdPerm, &rfd) != noErr) 
		{
			ErrorNum ("Unable to open ramdisk file (OpenDF failed, error %d)\n", err);
		}
		if (SetFPos (rfd, fsFromStart, 0) != noErr) 
		{
			Error ("Failed to seek to beginning of ramdisk file");
		}

		how_much = size;
		err = FSRead (rfd, &how_much, (void *)where);
		if ( err != noErr )
		{
			if (err != eofErr)
				ErrorNum ("Failed to read ramdisk file (error %d)", err);
			else {
				/* size was probably aligned on 1K multiple, verify that and pad with zeros */
				pad_bytes = size - how_much;
				if (pad_bytes < 1024)
				{
					/* Image was most certainly aligned - do the padding */
					pad_ptr = (char *)(((char *)where) + how_much);
					while (--pad_bytes >= 0)
						*pad_ptr++ = 0;
				}
				else
					ErrorNum ("Failed to read ramdisk file (error %d)", err);
			}
		}
		FSClose (rfd);

	}
}

/*
 *	get_image_size
 *
 *	Get size of file, checks for gzipped files
 *
 *	Return expanded size if gzipped
 *	Return file size if not gzipped
 *	Return -1 on error
 */
long get_image_size(ConstFSSpecPtr pFSS)
{
	short		refNum, magic;
	long		result, count, gzsize;

	result = -1;

	if (FSpOpenDF(pFSS, fsRdPerm, &refNum) == noErr) {
		
		count = sizeof(magic);
		if (FSRead(refNum, &count, &magic) == noErr) {

			if ( (count == sizeof(magic)) && ((magic == GZIP_MAGIC) || (magic == OLD_GZIP_MAGIC)) ) {

				if (SetFPos(refNum, fsFromLEOF, -4) == noErr) {
			
					count = sizeof(gzsize);
					if (FSRead(refNum, &count, &gzsize) == noErr) {

						/* if size != 0 then we got the uncompressed size
						 * if size == 0 then the file is probably padded
						 * to an even multiple of the device read size (512 bytes)
						 * and we have read size as zero
						 * In this case we'll have to <gasp> unzip the file to
						 * get it's real size...
						 */ 
						if (gzsize != 0) {
							/* Great! Just return the size
							 */
							result = EndianS32_LtoB(gzsize);
							cprintf("<GZIP file, size by field (%ldK>\n", (long)(result / 1024L));
						}
						else {
							/* S   happens, we'll have to unzip the
							 * whole thing...
							 */
							cprintf("<GZIP file, size by unzip - please wait...>\n");
							FSClose(refNum);
							refNum = -1;
							result = get_unzip_size(pFSS);
							if (result == -1)
								cprintf("<GZIP size unpack failure>\n");
						}
					}
				}
			}
			else {
				if (GetEOF(refNum, &count) == noErr)
					result = count;
			}
		}

		if (refNum != -1)
			FSClose(refNum);
	}

	return result;
}

/*
 *	get_unzip_size
 *
 *	Unzip the file to get it's size
 *
 *	Return expanded size
 *	Return -1 on error
 */
static long
get_unzip_size(ConstFSSpecPtr pFSS)
{
	Str255		path_str;
	long		result = -1;
	long		total_read, nread;

	/* Convert FSSpec to full pathname, stream uses pathnames
	 * Chicken out if failed conversion
	 */
	if (path_from_FSSpec(pFSS, path_str) != noErr)
		return -1;
	path_str[path_str[0]+1] = 0;

	/* Open and push stream modules */
	stream_init();
	stream_push(&file_mod);
	stream_push(&gunzip_mod);

	/* Open compressed file */
	if (sopen((const char *)&path_str[1]) >= 0) {

		total_read = 0;

		/* Sigh... Read and unzip the file */
		do {
			nread = sread(gzip_rdbuf, GZIP_RDBUF_SIZE);
			if (nread >= 0) {
				total_read += nread;
			}
			else if (nread < 0) {
				nread = -1;
				total_read = -1;
			}
		} while ( nread == GZIP_RDBUF_SIZE );
		
		result = total_read;
	}

	sclose();

	return result;
}

/*
 *	read_image
 *
 *	Read/load the file
 *	Handles compress/decompressed file images
 *	Does the reading in one pass
 *	Read nbytes into bptr
 *
 *	Return expanded size
 *	Return -1 on error
 */
static long
read_image(ConstFSSpecPtr pFSS, char *bptr, long nbytes)
{
	Str255		path_str;
	long		result = -1;

	nbytes = nbytes;

	/* Convert FSSpec to full pathname, stream uses pathnames
	 * Chicken out if failed conversion
	 */
	if (path_from_FSSpec(pFSS, path_str) != noErr)
		return -1;
	path_str[path_str[0]+1] = 0;

	/* Open and push stream modules */
	stream_init();
	stream_push(&file_mod);
	stream_push(&gunzip_mod);

	/* Read/load image */
	if (sopen((const char *)&path_str[1]) >= 0) {

#if 1
		result = sread(bptr, nbytes+1);
		if (result != nbytes)
			result = -1;
#else	
		{
			long		nread, total_read;

			total_read = 0;

			cprintf("mem allocated for unzip: %ld", nbytes);
			do {
				nread = sread(gzip_rdbuf, (GZIP_RDBUF_SIZE) );
				if (nread >= 0) {
					total_read += nread;
					cprintf(" |%ld| ", total_read);
					memcpy(bptr, gzip_rdbuf, nread);
					bptr += nread;
				}
				else if (nread < 0) {
cprintf("sread error, after %ld bytes\n", total_read);
					nread = -1;
					total_read = -1;
				}
			} while ( nread == (GZIP_RDBUF_SIZE) );
			
			result = total_read;

		}

#endif

	}

	sclose();

	return result;
}

/*
 *	chunk_sread
 *
 *	Split > 32K sread()'s into several 32K reads
 *
 *	Return size read
 *	Return -1 on error
 */

#if GZIP_RDBUF_SIZE != 32768
#error // Buffer size must be 32768
#endif

static long
chunk_sread(char *dst, long size)
{
	int		i;
	long	nread, count, num_chunks, last_count;

	if (size <= 0)
		return 0;

	nread = 0;
	num_chunks = size / 32768;
	last_count = size & 32767;

	// read n chunks of size 32K
	for(i = 0; i < num_chunks; i++) {
		count = sread(gzip_rdbuf, 32768);
		if (count != 32768) {
			// error
			nread = -1;
			break;
		} else {
			BlockMoveData(gzip_rdbuf, dst+nread, count);
			nread += count;
		}
	}

	// read last n bytes
	if ((nread >= 0) && (last_count > 0)) {
		count = sread(gzip_rdbuf, last_count);
		if (count != last_count) {
			// error
			nread = -1;
		} else {
			BlockMoveData(gzip_rdbuf, dst+nread, count);
			nread += count;
		}
	}

	return nread;
}

