/*
 * rtmain.h
 * 
 * Copyright (c) 2000-2005 by Florian Fischer (florianfischer@gmx.de)
 * and Martin Trautmann (martintrautmann@gmx.de) 
 * 
 * This file may be distributed and/or modified under the terms of the 
 * GNU General Public License version 2 as published by the Free Software 
 * Foundation. 
 * 
 * This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
 * WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 * 
 */

/** \file
  * Implements main functions for all supported systems.
  * This include file should be included by, and only by, your application's main cpp file. 
  * Unlike every other libRT header file, <tt>main.h</tt> includes system include files.
  * That means that you should not use the <tt>using namespace lrt;</tt> directive in your
  * application's main cpp file, as you can never know what will be in the top namespace
  * for other systems.
  */


#include "rtsystem.h"
#include "rtstring.h"
#include "rtcollect.h"
#include "rtfile.h"

// fwd declare: user main function
/** The main function for libRT users. Every application using libRT should implement
  * this function, rather than the system-dependent one.
  * @param args The application's command line arguments. Note that they do <b>not</b> 
  *             include the executable name.
  * @return The application's return code.
  * @see lrt::File::getExecutableFile()
  */
int rtMain(const lrt::Array<lrt::String> &args);


#ifdef __SYMBIAN32__
#include <e32base.h>
#include <e32std.h>

#include "rtetools.h"

void ExceptionHandler(TExcType exc)
{
	const TText* name; 
	switch(exc) {
		case EExcGeneral:             	name = _S("Exc-General"); break; 
		case EExcIntegerDivideByZero: 	name = _S("Exc-IntDivBy0"); break;
		case EExcSingleStep:          	name = _S("Exc-SingleStep"); break; 
		case EExcBreakPoint:          	name = _S("Exc-BreakPoint"); break;
		case EExcIntegerOverflow:     	name = _S("Exc-IntOverflow"); break; 
		case EExcBoundsCheck: 			name = _S("Exc-BoundsCk"); break; 
		case EExcInvalidOpCode: 		name = _S("Exc-InvOpCode"); break; 
		case EExcDoubleFault: 			name = _S("Exc-DoubleFault"); break; 
		case EExcStackFault: 			name = _S("Exc-StackFault"); break; 
		case EExcAccessViolation: 		name = _S("Exc-AccViol"); break; 
		case EExcPrivInstruction: 		name = _S("Exc-PrivInst"); break; 
		case EExcAlignment: 			name = _S("Exc-Alignment"); break; 
		case EExcPageFault: 			name = _S("Exc-PageFault"); break; 
		case EExcFloatDenormal: 		name = _S("Exc-FDenorm"); break; 
		case EExcFloatDivideByZero: 	name = _S("Exc-FDivBy0"); break; 
		case EExcFloatInexactResult: 	name = _S("Exc-FInexRes"); break; 
		case EExcFloatInvalidOperation: name = _S("Exc-FInvOp"); break; 
		case EExcFloatOverflow: 		name = _S("Exc-FOverflow"); break; 
		case EExcFloatStackCheck: 		name = _S("Exc-FStackCk"); break; 
		case EExcFloatUnderflow: 		name = _S("Exc-FUnderflow"); break; 
		case EExcAbort: 				name = _S("Exc-Abort"); break; 
		case EExcKill: 					name = _S("Exc-Kill"); break; 
		case EExcUserInterrupt: 		name = _S("Exc-UserIntr"); break; 
		default: name = _S("Exc-Unknown"); break; 
	};
	TBuf<16> buf = name;	
	
	User::Panic(buf, 0); 
}

GLDEF_C TInt E32Main() // main function called by E32
{
	RThread myThread;
	myThread.SetExceptionHandler(&ExceptionHandler, 
		KExceptionAbort | KExceptionKill | KExceptionUserInterrupt 
		| KExceptionFpe | KExceptionDebug | KExceptionFault 
		| KExceptionInteger); 
	
	
	lrt::System::cleanup=CTrapCleanup::New(); // get clean-up stack

	RProcess myProcess;
	TFileName fn = myProcess.FileName();
#ifndef _UNICODE    // actually: #if Epoc Version 5, but that's impossible
	TCommand cmd = myProcess.CommandLine();
#else
	TBuf<0x100> cmd;
	myProcess.CommandLine(cmd);
#endif
	lrt::String cmdLine = MYSTRING(cmd);
	cmdLine = cmdLine.trim();
	lrt::String file = MYSTRING(fn);
	file = file.trim();
	if(!file.compareIgnoreCase(cmdLine)) // started (tapped) from System 
	{
		cmdLine = lrt::System::readLine("Arguments: ");
	}
	lrt::File::initialize(file);
	lrt::File::setCurrentFolder(lrt::File(file));
	lrt::Array<lrt::String> args = cmdLine.split(" ");

	int retval = 0;
	TRAPD(error,retval = rtMain(args));
	__ASSERT_ALWAYS(!error,User::Panic(_L("LibRT Error"),error));

	if(lrt::System::isInteractive()) {
		lrt::System::println("--- Press any key to exit ---");
		lrt::System::read();
	}
	lrt::System::exit(retval);

	return 0; // unreachable
}

#else

/** LibRT's implementation of the stdlib main function. 
  * This method basically just calls the user main function <tt>rtMain()</tt>,
  * which should be implemented instead by library users.
  */
int main(int argn, char **argv) // main function called by Windows et al.
{ 
	lrt::File::initialize(argv[0]);
	lrt::Array<lrt::String>* arr 
	    = new lrt::Array<lrt::String>(lrt::Math::max(argn-1,0));
	for(int i = 1; i < argn; i++)
		(*arr)[i-1] = lrt::String(argv[i]);

	int ret = rtMain(*arr);
	delete arr;
	lrt::System::exit(ret);
	return 0; // unreachable
}
#endif
