/*
 * rtstreams.epoc.cpp
 * 
 * Copyright (c) 2000-2005 by Florian Fischer (florianfischer@gmx.de)
 * and Martin Trautmann (martintrautmann@gmx.de) 
 * 
 * This file may be distributed and/or modified under the terms of the 
 * GNU General Public License version 2 as published by the Free Software 
 * Foundation. 
 * 
 * This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
 * WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 * 
 */

////////////// File Streams using EPOC ////////////////////

#ifndef __LRT_STREAMS_EPOC__
#define __LRT_STREAMS_EPOC__

#include "rtstreams.h"

#include "rtstring.h"
#include "rtfile.h"

#include "rtetools.h"

#include <f32file.h>

namespace lrt {

FileInputStream::FileInputStream(const String &filename, bool textMode) : 
		markPos(0), open(false), session(0), file(0), buf(0), pos(0)
{
	String fname = File(filename).getName();
	init(fname, textMode);
}

FileInputStream::FileInputStream(const File &file, bool textMode) : 
		markPos(0), open(false), session(0), file(0), buf(0), pos(0)
{
	String fname = file.getName();
	init(fname, textMode);
}

void FileInputStream::init(const String &fname, bool textMode) 
{
	buf = new TBuf8<80>;
	session = new RFs();
	if(session->Connect() < 0) { _fail = true; return; }
	file = new RFile();
	TFileMode mode = EFileShareReadersOnly;
	if(textMode) mode = (TFileMode)(mode | EFileStreamText);
	if(file->Open(*session, ESTRING(fname), mode) < 0) 
		_fail = true;
	else open = true;
}
  
bool FileInputStream::markSupported()
{
	return true;
}

void FileInputStream::mark()
{
	if(!open) return;
	markPos = 0;
	file->Seek(ESeekCurrent, markPos);
	// undo read-ahead
	markPos -= buf->Length();
	markPos += pos;
}

void FileInputStream::reset()
{
	if(!open) return;
	file->Seek(ESeekStart, markPos);
	buf->SetLength(0);
	pos = 0;
}

int FileInputStream::read()
{
	if(!open) return -1;
	if(pos >= buf->Length()) {
		int rval = file->Read(*buf, 79);
		if(rval < 0) _fail = true;
		if((rval < 0) || (buf->Length() == 0))
			return -1;
		pos = 0;
	}
	return (*buf)[pos++];
}

bool FileInputStream::eos()
{
	if(!open) return true;
	
	// attempt to fetch new data if the buffer is empty
	if(pos >= buf->Length()) {
		int rval = file->Read(*buf, 79);
		if(rval < 0) _fail = true;
		if((rval < 0) || (buf->Length() == 0))
			return true;
		pos = 0;
	}

	// do we have more data in our buffer?
	if(pos < buf->Length())
		return false;
	else return true;
}

void FileInputStream::close()
{
	if(open) {
		file->Close();
		session->Close();
		open = false;
	}
}

FileInputStream::~FileInputStream()
{
	close();
	delete file;
	delete session;
	delete buf;
}



FileOutputStream::FileOutputStream(const String &fileName, bool append) : open(false)
{
	String fname = File(fileName).getName();
	init(fname, append);
}

FileOutputStream::FileOutputStream(const File &file, bool append) : open(false)
{
	String fname = file.getName();
	init(fname, append);
}

void FileOutputStream::init(const String &fname, bool append)
{
	session = new RFs();
	if(session->Connect() < 0) { _fail = true; return; }
	file = new RFile();
	TFileMode mode = EFileShareExclusive;
	int ret;
	if(!append)
		ret = file->Replace(*session, ESTRING(fname), mode);
	else
	{
		mode = (TFileMode)(mode | EFileWrite);
		ret = file->Open(*session, ESTRING(fname), mode);
		int seek = 0;
		file->Seek(ESeekEnd, seek);
	}
	if(ret < 0) 
		_fail = true;
	else
		open = true;
}

bool FileOutputStream::write(int b)
{
	if(!open) return false;
	TBuf8<1> buf(1);
	buf[0] = (unsigned char)b;
	int rval = file->Write(buf, 1);
	if(rval < 0)
	{
		_fail = true;
		return false;
	}
	else return true;
}

bool FileOutputStream::write(const Array<char> &b, int off, int len)
{
	if(!open) return false;
	const char *ptr = b.getData();
	int rval = file->Write(_L8(ptr + off), len);
	if(rval < 0)
	{
		_fail = true;
		return false;
	}
	else return true;
}

void FileOutputStream::flush()
{
	if(!open) return;
	file->Flush();
}

void FileOutputStream::close()
{
	if(open) {
		file->Close();
		session->Close();
		open = false;
	}
}

FileOutputStream::~FileOutputStream()
{
	close();
	delete file;
	delete session;
}

} // namespace

#endif

