/*
 * rtsystem.win32.cpp
 * 
 * Copyright (c) 2000-2005 by Florian Fischer (florianfischer@gmx.de)
 * and Martin Trautmann (martintrautmann@gmx.de) 
 * 
 * This file may be distributed and/or modified under the terms of the 
 * GNU General Public License version 2 as published by the Free Software 
 * Foundation. 
 * 
 * This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
 * WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 * 
 */

///// Win32 implementation of System functions

#ifndef __LRT_SYSTEM_WIN32__
#define __LRT_SYSTEM_WIN32__

#include <windows.h>
#include <stdlib.h>
#include <sys/timeb.h>
#include <time.h>

#include "rtsystem.h"
#include "rtfile.h"

namespace lrt {

Time::Time(const String& str) : sec(0), msec(0)
{
	Array<String> parts = str.split(" :", "");
	if(parts.length() != 7) // there should be 7 parts
		return;
	struct tm myTm;
	myTm.tm_isdst = -1;

	myTm.tm_mon = getMonthNum(parts[1]);
	myTm.tm_mday = parts[2].intValue(1);
	myTm.tm_hour = parts[3].intValue(0);
	myTm.tm_min = parts[4].intValue(0);
	myTm.tm_sec = parts[5].intValue(0);
	myTm.tm_year = parts[6].intValue(1970) - 1900;

	time_t myTime = mktime(&myTm);
	sec = (int)myTime;
}

Time Time::getCurrentTime()
{
	Time ret;
	struct timeb tp;
	ftime( &tp );
	ret.sec = tp.time;
	ret.msec = tp.millitm;
	return ret;
}

String Time::toString()
{
	time_t tm = (long)sec;
	return String(ctime(&tm));
}

HANDLE SYSstdin, SYSstdout;
int startY = 0;

void System::message(const String& str)
{
	if(interactive) {
		String title(*appName);
		title += " Message";
		MessageBox(0, str.cStr(), title.cStr(), MB_OK);
	}
	else
		println(str);
}

void System::exit(int errnum)
{
	finalize();
	FlushFileBuffers(SYSstdout);
	ExitProcess(errnum);
}


void LrtSysWriteToConsole(const char* str, int len)
{
	// convert '\n' to '\r\n'
	char* buf = new char[2 * len + 1]; 

	int addlen = 0; 
	char lastCh = 0; 
	for(int i = 0; i < len; i++) {
		char ch = str[i]; 
		if(ch == '\n' && lastCh != '\r') {
			buf[i + addlen] = '\r'; 
			addlen++;
		}
		buf[i + addlen] = ch; 
		lastCh = ch; 
	}
	len += addlen; 
	buf[len] = 0; 

	// ... and, if it's a console, also convert ANSI to OEM charset. 
	if(GetFileType(SYSstdout) == FILE_TYPE_CHAR) {
		// convert str to OEM charset if printing on console
		CharToOem(buf, buf); // according to MSDN docs, this is safe in ANSI BUILD ONLY. 
	}

	unsigned long numb; 
	WriteFile(SYSstdout, buf, len, &numb, 0); 
	delete[] buf; 
}



void System::exit(int errnum, const String &errmsg)
{
	if(interactive) {
		String title(*appName);
		title += " Error";
		MessageBox(0, errmsg.cStr(), title.cStr(), MB_OK);
	}
	else {
		LrtSysWriteToConsole(errmsg.cStr(), errmsg.length()); 
	}
	FlushFileBuffers(SYSstdout);
	finalize();
	ExitProcess(errnum);
}

// my console control handler function, which calls the registered function
BOOL WINAPI MyCtrlHandlerFunction(DWORD type)
{
	if(!lrt::System::lastBreakFunction) // do nothing, no function registered (but shouldn't happen)
		return FALSE; 
	// ok, a function was registered, call it
	lrt::System::lastBreakFunction(lrt::System::CTRL_C);
	return TRUE;
}

void System::setOnBreak(System::BreakFunction func, bool)
{
	if(lastBreakFunction) { // remove it
		::SetConsoleCtrlHandler(MyCtrlHandlerFunction, false);
		lastBreakFunction = 0;
	}
	if(func) { // add the new one
		::SetConsoleCtrlHandler(MyCtrlHandlerFunction, true);
		lastBreakFunction = func;
	}
}

void System::initialize()
{
    SYSstdin = GetStdHandle(STD_INPUT_HANDLE);
    SYSstdout = GetStdHandle(STD_OUTPUT_HANDLE);
	CONSOLE_SCREEN_BUFFER_INFO info;
	GetConsoleScreenBufferInfo(SYSstdout, &info);
	startY = info.dwCursorPosition.Y;
	if(startY > info.dwSize.Y - 20) startY = info.dwSize.Y - 20;
	init = true;
}

void System::print(const char *cStr)
{
	if(!init) initialize();
	LrtSysWriteToConsole(cStr, lstrlen(cStr)); 
}

void System::print(const String &str)
{
	if(!init) initialize();
	LrtSysWriteToConsole(str.cStr(), str.length()); 
}

void System::println()
{
	if(!init) initialize();
	unsigned long numb;
	WriteFile(SYSstdout, "\r\n", 2, &numb, 0);
}

void System::println(const String &str)
{
	if(!init) initialize();
	unsigned long numb;
	LrtSysWriteToConsole(str.cStr(), str.length()); 
	WriteFile(SYSstdout, "\r\n", 2, &numb, 0);
}

char System::read()
{
	if(!init) initialize();
	unsigned long oldMode;
	if(!GetConsoleMode(SYSstdin, &oldMode)) return 0;
	if(!SetConsoleMode(SYSstdin, oldMode & ~(ENABLE_LINE_INPUT | ENABLE_ECHO_INPUT))) return 0;
	char buf[] = "    ";
	unsigned long numb;
	ReadFile(SYSstdin, buf, 1, &numb, 0);
	SetConsoleMode(SYSstdin, oldMode);
	return buf[0];	
}

String System::readLine(const String& query)
{
	if(!init) initialize();
	char buf[256];
	if(query.length() > 0) print(query);
	unsigned long numb;
	ReadFile(SYSstdin, buf, 255, &numb, 0);
	while((numb > 0) && ((buf[numb-1] == '\n') || (buf[numb-1] == '\r')))
		numb--;
	buf[numb] = 0;
	return String(buf);
}

void System::setPos(int x, int y)
{
	if(!init) initialize();
	COORD position;
	position.X = x;
	position.Y = y + startY;
	SetConsoleCursorPosition(SYSstdout, position);
}

void System::copyMemory(const void* src, void* dest, int length)
{
#ifndef __NOCHECK__
	if(length < 0)
		System::exit(-1, "length < 0 in System::copyMemory()");
#endif
	::memcpy(dest, src, length);
}

void System::fillMemory(void* ptr, int length, char data)
{
#ifndef __NOCHECK__
	if(length < 0)
		System::exit(-1, "length < 0 in System::fillMemory()");
#endif
	::memset(ptr, data, length);
}

void System::finalize()
{
	File::finalize();
	init = false;
}

} // namespace


#endif
