/*
 * insertview.cpp
 * 
 * Copyright (c) 2000-2005 by Florian Fischer (florianfischer@gmx.de)
 * and Martin Trautmann (martintrautmann@gmx.de) 
 * 
 * This file may be distributed and/or modified under the terms of the 
 * GNU General Public License version 2 as published by the Free Software 
 * Foundation. 
 * 
 * This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
 * WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 * 
 */

// ----------------------------------------------------------------------------
// headers
// ----------------------------------------------------------------------------

#include "insertview.h"

#include <wx/toolbar.h>
#include <wx/config.h>
#include <wx/txtstrm.h>

#include "robotop.h"
#include "tourview.h"
#include "dialogs.h"
#include "wxext.h"

#include <stdlib.h>

// Include the XPM resources
#include "bitmaps/start.xpm"

#if !defined(__WXMSW__) || !defined(__WXDLL__) // which has it
#ifndef min
  #define min(a,b) (((a)<(b))?a:b)
  #define max(a,b) (((a)>(b))?a:b)
#endif
#endif

BEGIN_EVENT_TABLE(InsertFrame, wxxMDIChildFrame)
  EVT_MENU(wxID_CLOSE,          InsertFrame::OnCloseCommand)
  EVT_MENU(InsertFrame_Start,     InsertFrame::OnStart)
  EVT_CLOSE(InsertFrame::OnCloseWindow)
  EVT_IDLE(InsertFrame::OnIdle)
  EVT_TIMER(InsertFrame_Timer, InsertFrame::OnTimer)
END_EVENT_TABLE()

InsertFrame::InsertFrame(TournamentFrame *parent)
  : wxxMDIChildFrame(parent->parent, wxT("[Temporary Title]"), wxT("InsertFrame")),
    chartsFrame(parent), rtOutputWin(0), running(false), curFiles(), newFiles(), process(0), status(),
	refreshTimer(0)
{
	UpdateTitle();

	wxBoxSizer *sizer = new wxBoxSizer( wxVERTICAL );
	wxPanel* panel = new wxPanel(this); 

	toolBar = new wxToolBar( panel, -1, wxDefaultPosition, wxDefaultSize,
				  wxNO_BORDER | wxTB_FLAT | wxTB_HORIZONTAL | wxTB_NODIVIDER);
	InitToolBar( toolBar );
	wxxAddCloseBar(panel, this, sizer, toolBar, (wxCommandEventFunction) &InsertFrame::OnCloseCommand);

	wxBoxSizer* progSizer = new wxBoxSizer(wxHORIZONTAL);
	progressBar = new wxGauge(panel, -1, 100);
	progSizer->Add(progressBar, 1, wxEXPAND | wxALL, 2);
	progressLabel = new wxStaticText(panel, -1, _("    0%"));
	progSizer->Add(progressLabel, 0, wxALIGN_CENTER_VERTICAL | wxALL, 2);
	sizer->Add(progSizer, 0, wxEXPAND | wxALL, 2);

	wxBoxSizer* statSizer = new wxBoxSizer(wxHORIZONTAL);
	statSizer->Add(new wxStaticText(panel, -1, _("Status:")), 0, wxEXPAND | wxALL, 2);
	statusLabel = new wxStaticText(panel, -1, _("No files waiting."), wxDefaultPosition, wxDefaultSize, wxST_NO_AUTORESIZE);
	statSizer->Add(statusLabel, 1, wxEXPAND | wxALL, 2);
	sizer->Add(statSizer, 0, wxEXPAND | wxALL, 2);

	rtOutputWin = new wxTextCtrl(panel, -1, wxT(""), wxDefaultPosition, wxDefaultSize, wxTE_MULTILINE);
	rtOutputWin->SetFont(wxFont(10, wxMODERN, wxNORMAL, wxNORMAL));

	sizer->Add( rtOutputWin,	1, wxEXPAND | wxALL, 2 );

	panel->SetSizer(sizer); 

	// size myself
	wxBoxSizer* mySizer = new wxBoxSizer(wxVERTICAL); 
	mySizer->Add(panel, 1, wxEXPAND | wxALL, 0); 
	SetSizer(mySizer);
	SetSizeHints(mySizer->GetMinSize()); // set size hints to honour mininum size
	Layout(); 

	SetMenuBar(CreateMenu());

	refreshTimer = new wxTimer(this, InsertFrame_Timer);
	refreshTimer->Start(1000); // every second only (1'000 msec)
}

bool InsertFrame::Destroy()
{
	// it seems better to actually destroy the timer
	delete refreshTimer;
	refreshTimer = 0;

	if(chartsFrame)
		chartsFrame->insertFrame = 0;
	if(process) {
		process->Break(); 
		process->parent = 0; 
		process = 0; 
	}
	/* don't delete the process! 
	 * It will do it itself when OnTerminate() is called. 
	 */
	//delete process;
	//process = 0; 

	return wxxMDIChildFrame::Destroy(); 
}

wxMenuBar* InsertFrame::CreateMenu()
{
    // create a menu bar
    wxMenu *menuFile = new wxMenu;
    menuFile->Append(RoboTop_New,  _("&New Competition...\tCtrl-N"), _("Create a new competition"));
    menuFile->Append(RoboTop_Open, _("&Open Competition...\tCtrl-O"), _("Open a competition"));
	menuFile->Append(wxID_CLOSE,   _("&Close View\tCtrl-F4"), _("Close this view"));
    menuFile->AppendSeparator();
    menuFile->Append(RoboTop_Quit, _("E&xit\tAlt-X"), _("Quit this program"));

	wxMenu* menuExtras = new wxMenu;
	menuExtras->Append(RoboTop_Settings, _("&Settings..."), _("Configure RoboTop settings"));
#ifdef __WXMSW__
	menuExtras->Append(RoboTop_Fullscreen, _("&Fullscreen\tF11"), _("Show RoboTop fullscreen"), true); // checkable
#endif

    // the "About" item should be in the help menu
    wxMenu *helpMenu = new wxMenu;
    helpMenu->Append(RoboTop_Help, _("Using RoboTop\tF1"), _("Show help for RoboTop"));
    helpMenu->Append(RoboTop_About, _("&About..."), _("Show about dialog"));

    // now append the freshly created menu to the menu bar...
    wxMenuBar *menuBar = new wxMenuBar();
    menuBar->Append(menuFile, _("&File"));
	menuBar->Append(menuExtras, _("&Extras"));
    menuBar->Append(helpMenu, _("&Help"));

	return menuBar;
}

bool InsertFrame::IsRunning()
{
	return running;
}

bool InsertFrame::CopyFiles(const wxArrayString& files)
{
	// NOTE: we don't need the IsRunning() check here 
	//  - it's always safe to copy new bots into the folder. 

	if(!chartsFrame)
		return false;
	
	// copy the files
	wxString copyToFolder = GetMainTourFolder() + chartsFrame->configData->tourFolder +
		wxFILE_SEP_PATH + wxT("newbots") + wxFILE_SEP_PATH;
	wxString resFolder = GetMainTourFolder() + chartsFrame->configData->tourFolder +
		wxFILE_SEP_PATH + wxT("results") + wxFILE_SEP_PATH;
	bool ret = true;
	for(size_t i = 0; i < files.GetCount(); i++)
	{
		wxString fromFile = files[i];
		wxString shortFile = fromFile.AfterLast(wxFILE_SEP_PATH);
		wxString toFile = copyToFolder + shortFile;
		wxString resFile = resFolder + shortFile;
		// check if we already have the same file in the result folder
		if(wxFileExists(resFile) && (wxxFileSize(resFile) == wxxFileSize(fromFile))) 
		{
			int res = wxMessageBox(wxString::Format(_("It seems that the file %s was already added to the competition\nin the same version. Would you like to resimulate it?"), shortFile.c_str()), 
				_("Question"), wxYES_NO | wxICON_QUESTION, this);
			if(res == wxNO) // don't resim => skip the file
				continue;
		}
		ret &= wxCopyFile(fromFile, toFile);
	}

	// added convenience for our users:
	// auto-scan for the new files, if RoboTour isnt running (check is done in ScanForFiles())
	ScanForFiles();

	return ret;
}

void InsertFrame::ScanForFiles()
{
	if(IsRunning())
		return;
	if(!chartsFrame)
		return;

	wxString myTourFolder = GetMainTourFolder() + chartsFrame->configData->tourFolder +
		wxFILE_SEP_PATH;

	// Go, wxDog! Find Tha Files! And I want them in one piece! ;-)
	wxDirCommand(newFiles, myTourFolder + wxT("newbots") + wxFILE_SEP_PATH + wxT("*.r*"), wxFILE);
	wxDirCommand(curFiles, myTourFolder + wxT("curbots") + wxFILE_SEP_PATH + wxT("*.r*"), wxFILE);

	if(newFiles.GetCount() + curFiles.GetCount()) // we've got some
	{
		toolBar->EnableTool(InsertFrame_Start, true);
		InitProgress();
	}
}

int InsertFrame::GetNumWaitingFiles()
{
	return newFiles.GetCount() + curFiles.GetCount(); 
}

void InsertFrame::InitToolBar(wxToolBar* toolBar)
{
	const int numBitmaps = 1;
    wxBitmap* bitmaps[numBitmaps];

    bitmaps[0] = new wxBitmap( start_xpm );

    toolBar->AddTool( InsertFrame_Start,    *(bitmaps[0]), _("Start Simulation"), _("Run RoboTour with the selected bots"));

    toolBar->Realize();
    for (int i = 0; i < numBitmaps; i++)
        delete bitmaps[i];
	
	toolBar->EnableTool(InsertFrame_Start, false); // disable tool initially

}

void InsertFrame::UpdateTitle()
{
	if(!chartsFrame)
		return;
	SetTitle(chartsFrame->configData->tourName + _(" - [Insert View]"));
}

// Progress system
void InsertFrame::InitProgress()
{
	if(IsRunning())
		return;
	int chartSize = 20;
	if(chartsFrame)
		chartSize = chartsFrame->configData->chartSize;
	int numFiles = newFiles.GetCount() + curFiles.GetCount();
	// init status
	wxString plurals[3];
	plurals[0] = _("No robots waiting.");
	plurals[1] = _("One robot waiting.");
	plurals[2] = _("%d robots waiting.");
	statusLabel->SetLabel(wxString::Format(plurals[min(2, numFiles)], numFiles));

	if(!numFiles) numFiles = 1; // we'll get problems if having 0 for the progBar
	progressBar->SetRange(numFiles * chartSize);
	SetProgress(0);
}

void InsertFrame::SetProgress(int numPlayed)
{
	int maxVal = progressBar->GetRange();
	progressBar->SetValue(min(numPlayed, maxVal));
	progressLabel->SetLabel(wxString::Format(wxT("%3d%%"), (numPlayed * 100) / maxVal) );
}


// event handlers (these functions should _not_ be virtual)
void InsertFrame::OnStart(wxCommandEvent& event)
{
	if(IsRunning())
		return;
	if(!chartsFrame) {
		wxLogError(_("Cannot start without charts frame open."));
		return;
	}
	// init status, then call NextFile();
	rtOutputWin->Clear();
	status.numFile = -1; // is inc'd before use
	status.totalFiles = curFiles.GetCount() + newFiles.GetCount();
	if(!status.totalFiles)
	{
		wxMessageBox(_("There are no robots waiting. "));
		return;
	}

	chartsFrame->SetNeedMakehtml(); 

	running = true; // now we won't return anymore
	toolBar->EnableTool(InsertFrame_Start, false);

	status.numRepeats = chartsFrame->configData->numRepeats;
	status.localFile = wxT("");
	wxConfigBase* globCfg = wxConfig::Get();
	status.rtExe = globCfg->Read(wxT("RoboTourExecutable"), wxT("robotour"));
	status.tourFolder = GetMainTourFolder() + chartsFrame->configData->tourFolder +
		wxFILE_SEP_PATH;
	status.optionSet = chartsFrame->configData->optionSet;
	status.chartSize = chartsFrame->configData->chartSize;
	status.numSim = 0; 

	NextFile();
}

void InsertFrame::NextFile()
{
	if(process)
		wxLogError(_("In InsertFrame::NextFile(): Process is not NULL?"));

	wxString newbotsFolder = status.tourFolder + wxT("newbots") + wxFILE_SEP_PATH;
	wxString curbotsFolder = status.tourFolder + wxT("curbots") + wxFILE_SEP_PATH;
	wxString resultsFolder = status.tourFolder + wxT("results") + wxFILE_SEP_PATH;

	// delete previous bot
	if(!status.localFile.IsEmpty()) 
	{
		if(!wxRemoveFile(curbotsFolder + status.localFile))
			wxLogError(_("Couldn't delete %s%s."), curbotsFolder.c_str(), status.localFile.c_str());
	}

	// setup status
	status.numFile++;
	SetProgress(status.numFile * status.chartSize);
	int stillToDo = curFiles.GetCount() + newFiles.GetCount();
	if(!stillToDo)
	{
		// finish up...
		statusLabel->SetLabel(_("Done."));
		ScanForFiles();
		running = false;
		if(chartsFrame) {
            // don't run MakeHtml here, we're still inside a process destruction
			chartsFrame->SetNeedMakehtml(); 
			chartsFrame->Redisplay(false);
			chartsFrame->InformListeners(EVT_TOUR_UPDATED);
		}
		return;
	}
	status.numSim = 0;
	status.isCur = (curFiles.GetCount() ? true : false);
	wxArrayString& source = (status.isCur ? curFiles : newFiles);
	int itemNr = source.GetCount() - 1;
	wxString botFile = source[itemNr];
	source.Remove(itemNr);
	status.localFile = botFile.AfterLast(wxFILE_SEP_PATH);

	// push the file around
	if(!status.isCur)
		if(!wxRenameFile(botFile, curbotsFolder + status.localFile)) {
			wxLogError(_("Couldn't rename %s to %s%s."), botFile.c_str(), curbotsFolder.c_str(),
				status.localFile.c_str());
			return;
		}
	if(!wxCopyFile(curbotsFolder + status.localFile, resultsFolder + status.localFile)) {
		wxLogError(_("Couldn't copy %s%s to %s%s."), curbotsFolder.c_str(), status.localFile.c_str(), 
			resultsFolder.c_str(), status.localFile.c_str());
		return;
	}

	// construct args line
	wxString argsResultFolder(resultsFolder);
#ifdef __WXMSW__ // avoids the Windows "feature" which interpretes \" in command lines as " instead of \"
	if(argsResultFolder[argsResultFolder.Len() - 1] == '\\')
		argsResultFolder += '\\';
#endif
	wxString args = wxString::Format(wxT("-cf \"%s\" -n %d -%c %d -o \"%s\" \"%s\""), 
		argsResultFolder.c_str(), status.numRepeats, (status.isCur? 'T' : 't'), status.chartSize, 
		status.optionSet.c_str(), status.localFile.c_str());
	// sound?
	bool useSound; wxConfig::Get()->Read(wxT("UseSound"), &useSound, false);
	if(useSound) args += wxT(" -sound");
	
	// run RoboTour
	process = new RoboTourProcess(this, args);

}

void InsertFrame::OnCloseCommand(wxCommandEvent& event)
{
	wxCloseEvent pseudoEvent;
	pseudoEvent.SetCanVeto(true);
	OnCloseWindow(pseudoEvent);
}

void InsertFrame::OnCloseWindow(wxCloseEvent& event)
{
	if(event.CanVeto() && IsRunning()) // still simulates! User probably doesn't want to close the tour. 
	{
		int res = wxMessageBox(_("RoboTour is still running!\nDo you really want to stop it?"), _("Confirm"), wxYES_NO, this);
		if(res == wxNO)
		{
			event.Veto();
			return;
		}
	}
	if(IsRunning()) {
		/*********** STOP ROBOTOUR HERE! *****************/
		if(process) {
			// we are going away
			process->parent = 0; 
			// stop the process too
			process->Break();
		}
	}
	if(chartsFrame)
		chartsFrame->insertFrame = 0;
	Destroy();
}

void InsertFrame::OnIdle(wxIdleEvent& event)
{
	if(process) {
		if(process->HasInput())
			event.RequestMore();
	}
}

void InsertFrame::OnTimer(wxTimerEvent& event)
{
	if(IsRunning()) {
		if(process)
			while(process->HasInput());
        wxGetApp().Yield(true /* only if needed */); // process events
	}
	else {
		ScanForFiles();
	}
}

/////////////////////////////// ////////////////////////////////////////////////
///////////// ///////////////////////////////////////////// ////////////////////
/////////////////// RoboTourProcess

RoboTourProcess::RoboTourProcess(InsertFrame* parent, const wxString& args) 
		: wxProcess(parent), parent(parent)		
{
	Redirect();
	// get robotour exe
	wxString rtExe = parent->status.rtExe;
	
	int flags = wxEXEC_ASYNC | wxEXEC_MAKE_GROUP_LEADER; 

	if(!(pid = wxExecute(wxT("\"") + rtExe + wxT("\" ") + args, flags, this)))
	{
		wxLogError(_("RoboTour (%s) execution failed."), rtExe.c_str());
		if(parent->process == this) {
			parent->process = 0;
		}
		delete this;
		return;
	}
	else
		parent->rtOutputWin->AppendText(wxT("[") + rtExe + wxT(" ") + args + wxT("]\n"));
}

RoboTourProcess::~RoboTourProcess()  // NOTE: do not assume that parent is still valid in there!
{
	// Just Kill RoboTour and done...
	Break(); 
}

void RoboTourProcess::OnTerminate(int pid, int status)  // NOTE: do not assume that parent is still valid in there!
{
	this->pid = 0;

	if(parent) {
		// show the rest of the output
		while ( HasInput() )
			;
		if(status)
			parent->rtOutputWin->AppendText(
				wxString::Format(_("*** WARNING! RoboTour exited with non-successful status: %d ***\n"), status));

		if(parent->process == this) {
			parent->process = 0;
			parent->NextFile(); // next bot...
		}
	}

	delete this;
}

bool RoboTourProcess::HasInput()
{
    bool hasInput = false;

    wxInputStream& is = *GetInputStream();
    if ( is.CanRead() )
    {
        wxTextInputStream tis(is, wxT("\t")
#if wxUSE_UNICODE
			, *wxConvCurrent
#endif
			);

		wxString line = tis.ReadLine();
		line += wxT("\n");
		if(parent) {
			if(line.StartsWith(wxT("***"))) // special msg
			{
				parent->status.numSim++;
				parent->SetProgress(parent->status.numFile * parent->status.chartSize + parent->status.numSim);
				parent->statusLabel->SetLabel(line);
			}
			parent->rtOutputWin->AppendText(line);
		}

        hasInput = true;
    }

    return hasInput;
}

void RoboTourProcess::Break() // cancel simulation 
{
	wxKillError err; 
	if(pid) {
		wxSignal sig = wxSIGINT; // like CTRL-C
		int flag = wxKILL_CHILDREN; 
#ifdef __WXMSW__
		sig = wxSIGKILL; // unfortunately, this doesn't work otherwise on MSW, because RoboTour has no windows open
#endif		
		
		if(wxKill(pid, sig, &err, flag) < 0 && err != wxKILL_NO_PROCESS) // error? 
			wxMessageBox(wxString::Format(_("Error %d while stopping RoboTour."), err), _("Stop error")); 
	}

	pid = 0;
	if(parent && parent->process == this) 
		parent->process = 0;

	// NO! OnTerminate() will be called, and we must react to it... 
	// delete this;
}

