/*
 * wxini.cpp
 * 
 * Copyright (c) 2000-2005 by Florian Fischer (florianfischer@gmx.de)
 * and Martin Trautmann (martintrautmann@gmx.de) 
 * 
 * This file may be distributed and/or modified under the terms of the 
 * GNU General Public License version 2 as published by the Free Software 
 * Foundation. 
 * 
 * This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
 * WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 * 
 */

#include <wx/wx.h>

#include "wxini.h"

#include <wx/txtstrm.h>
#include <wx/wfstream.h>

wxxIniFile::wxxIniFile() : hash(new wxHashTable(wxKEY_STRING, wxxINI_AVG_GROUPSIZE))
{
	hash->DeleteContents(true);
}

wxxIniFile::~wxxIniFile() 
{
	delete hash;
}

void wxxIniFile::AddSection(wxxIniSection* section)
{
	wxString lcName = section->GetName().Lower();
	hash->Put(lcName.c_str(), section);
}

bool wxxIniFile::DelSection(const wxString& name)
{
	wxString lcName = name.Lower();
	wxObject* sec = hash->Delete(lcName.c_str());
	if(sec) // it was actually there
	{
		delete sec;
		return true;
	}
	// there was no such section
	return false;
}

wxxIniSection* wxxIniFile::GetSection(const wxString& name)
{
	wxString lcName = name.Lower();
	return (wxxIniSection*) hash->Get(lcName.c_str());
}

wxxIniSection& wxxIniFile::operator[](const wxString& name)
{
	wxString lcName = name.Lower();
	wxxIniSection* sec = (wxxIniSection*) hash->Get(lcName.c_str());
	if(sec)
		return *sec;
	// doesn't exist
	sec = new wxxIniSection(name);
	hash->Put(lcName.c_str(), sec);
	return *sec;
}

bool wxxIniFile::Write(const wxString& file)
{
	wxFileOutputStream output(file);
	if(!output.IsOk())
		return false;
	// robot files are in ISO8859-1 char set
	wxTextOutputStream text(output, wxEOL_NATIVE
#if wxUSE_UNICODE
		, wxConvISO8859_1
#endif
		);

	wxxIniSection* curSection = NULL;
	First();
	while((curSection = Next()) != NULL)
	{
		text.WriteString(wxT("")); // empty line
		text.WriteString(wxT("[") + curSection->GetName() + wxT("]"));
		
		wxxIniOption* curOption = NULL;
		curSection->First();
		while((curOption = curSection->Next()) != NULL)
			text.WriteString(curOption->name + wxT("=") + curOption->value);
	}

	return output.IsOk();
}

bool wxxIniFile::Read(const wxString& file)
{
	if(!wxFileExists(file))
		return false;
	wxFileInputStream input(file);
	if(!input.IsOk())
		return false;
	// robot files are in ISO8859-1 char set
	wxTextInputStream text(input, wxT("\t")
#if wxUSE_UNICODE
		, wxConvISO8859_1
#endif
		);

	wxxIniSection* curSection = NULL;

	while(!input.Eof()) {
		wxString line = text.ReadLine();
		line.Trim(false); // from left
		if(line.IsEmpty())
			continue;
		if(line.GetChar(0) == ';') // INI comment
			continue;
		if(line.GetChar(0) == '[') //INI section
		{
			curSection = new wxxIniSection(line.Mid(1).BeforeFirst(']'));
			AddSection(curSection);
		}
		else // normal data
		{
			if(curSection)
			{
				int eqIndex = line.Find('=');
				curSection->Set(line.Left(eqIndex), line.Mid(eqIndex + 1));
			}
		}
	}

	// if it is a write error or such, return false
	return (input.GetLastError() == wxSTREAM_EOF);
}

void wxxIniFile::First()
{
	hash->BeginFind();
}

wxxIniSection* wxxIniFile::Next()
{
	wxNode* node = hash->Next();
	if(node)
		return (wxxIniSection*) node->Data();
	else
		return 0;
}

size_t wxxIniFile::GetCount()
{
	return hash->GetCount();
}


/////////// /////////////// /////////////
wxxIniSection::wxxIniSection(const wxString& name) : name(name), 
	hash(new wxHashTable(wxKEY_STRING, wxxINI_AVG_GROUPSIZE))
{
	hash->DeleteContents(true);
}

wxxIniSection::~wxxIniSection()
{
	delete hash;
}

const wxString& wxxIniSection::GetName() const
{
	return name;
}

wxString wxxIniSection::Get(const wxString& name) const
{
	wxString lcName = name.Lower();
	wxxIniOption* opt = (wxxIniOption*) hash->Get(lcName.c_str());
	if(opt)
		return opt->value;
	else
		return wxEmptyString;
}

wxString& wxxIniSection::operator[](const wxString& name)
{
	wxString lcName = name.Lower();
	wxxIniOption* opt = (wxxIniOption*) hash->Get(lcName.c_str());
	if(opt)
		return opt->value;
	else { // there's no such option
		opt = new wxxIniOption(name);
		hash->Put(lcName.c_str(), opt);
		return opt->value;
	}
}

long wxxIniSection::GetLong(const wxString& name, long def) const
{
	wxString lcName = name.Lower();
	wxxIniOption* opt = (wxxIniOption*) hash->Get(lcName.c_str());
	if(!opt) 
		return def;
	long ret;
	if(opt->value.ToLong(&ret))
		return ret;
	return def;
}

bool wxxIniSection::GetBool(const wxString& name, bool def) const
{
	wxString lcName = name.Lower();
	wxxIniOption* opt = (wxxIniOption*) hash->Get(lcName.c_str());
	if(!opt) 
		return def;
	bool ret = def;
	wxString lcVal = opt->value.Lower();
	if((lcVal == wxT("yes")) || (lcVal == wxT("true")) || (lcVal == wxT("on")))
		ret = true;
	else if((lcVal == wxT("no")) || (lcVal == wxT("false")) || (lcVal == wxT("off")))
		ret = false;
	else {
		long num;
		if(lcVal.ToLong(&num)) {
			if(num)
				ret = true;
			else
				ret = false;
		}
	}
	return ret;
}

void wxxIniSection::Set(const wxString& name, const wxString& value)
{
	wxString lcName = name.Lower();
	wxxIniOption* opt = (wxxIniOption*) hash->Get(lcName.c_str());
	if(opt)
		opt->value = value;
	else
		hash->Put(lcName.c_str(), new wxxIniOption(name, value));
}

void wxxIniSection::Set(const wxString& name, long value)
{
	wxString lcName = name.Lower();
	wxxIniOption* opt = (wxxIniOption*) hash->Get(lcName.c_str());
	if(opt)
		opt->value.Printf(wxT("%ld"), value);
	else
		hash->Put(lcName.c_str(), new wxxIniOption(name, wxString::Format(wxT("%ld"), value)));
}

void wxxIniSection::Set(const wxString& name, bool value)
{
	wxString lcName = name.Lower();
	wxxIniOption* opt = (wxxIniOption*) hash->Get(lcName.c_str());
	wxString val = ( value ? wxT("true") : wxT("false"));
	if(opt)
		opt->value = val;
	else
		hash->Put(lcName.c_str(), new wxxIniOption(name, val));
}

bool wxxIniSection::Del(const wxString& name)
{
	wxString lcName = name.Lower();
	wxObject* opt = hash->Delete(lcName.c_str());
	if(opt) // it was actually there
	{
		delete opt;
		return true;
	}
	// there was no such option
	return false;
}
	
void wxxIniSection::First()
{
	hash->BeginFind();
}

wxxIniOption* wxxIniSection::Next()
{
	wxNode* node = hash->Next();
	if(node)
		return (wxxIniOption*) node->Data();
	else
		return 0;
}

size_t wxxIniSection::GetCount()
{
	return hash->GetCount();
}


/////////////////////// /////////////////////////// ////////////////////////////
wxxIniOption::wxxIniOption() 
{
}

wxxIniOption::wxxIniOption(const wxString& name) : name(name)
{
}

wxxIniOption::wxxIniOption(const wxString& name, const wxString& value) : 
	name(name), value(value)
{
}

wxxIniOption& wxxIniOption::operator=(const wxString& value)
{
	this->value = value;
	return *this;
}

