/*
 * robmanage.h
 * 
 * Copyright (c) 2000-2005 by Florian Fischer (florianfischer@gmx.de)
 * and Martin Trautmann (martintrautmann@gmx.de) 
 * 
 * This file may be distributed and/or modified under the terms of the 
 * GNU General Public License version 2 as published by the Free Software 
 * Foundation. 
 * 
 * This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
 * WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 * 
 */

/** @file
  * Declares classes related to tournament control (SimManager)
  */

#ifndef __RT_MANAGE__
#define __RT_MANAGE__

#include <rtcollect.h>
#include <rtstring.h>

#include "robbase.h"
#include "robglob.h"

namespace rt {

/** Information about the tournament which is valid throughout the tournament, 
  * and therefore given only one (in TourDisplayer::init()) to the TourDisplayer.
  */
class TourInitInfo {
public:
	TourInitInfo() : numParticipants(0), numSimulations(0) {}
	TourInitInfo(int numParticipants, int numSimulations);
	int numParticipants;
	int numSimulations;
};

/** The tournament results (until now) for a certain Program. (It is identified by its
  * file name.) */
class TourResult {
public:
	static int compareByPoints(const TourResult& t1, const TourResult& t2);
	/// DON'T use!
	TourResult();
	/// Use this one instead. 
	TourResult(const lrt::StringMap<Program::HeaderField>& headers, const lrt::String& fileName);
	~TourResult() {}
	lrt::StringMap<Program::HeaderField> headers; ///< bot header information
	lrt::String fileName; ///< bot file name
	int wins, looses, ties;
	int rank;
	float points;
};

enum TourStatusType {
	tourAbort = -1,
	tourComplete = 0,
	tourContinue = tourComplete
};

/** Can display the tournament results. */
class TourDisplayer {

public:
	virtual void init(const TourInitInfo& info) = 0;
	virtual TourStatusType update(const lrt::Array<TourResult>& status) = 0;
	virtual void exit(TourStatusType exitType, const lrt::Array<TourResult>& result) = 0;

	virtual ~TourDisplayer() {}
protected:
	TourDisplayer() {}
};

/** Options for the simulations created by the SimManager. 
  * On destruction, SimOptions deletes all the contents of its pointer vectors etc. 
  * But not the errHandler! */
class SimOptions {
public:
	int numRepeats;
	lrt::Vector<lrt::String> files;
	lrt::Vector<SimSupervisor*> supervisors; // these are the output supervisors created by rt, 
	// the StdVictorySupervisor must be created by the Manager
	lrt::Vector<ExecSupervisor*> execSupervisors; 
	lrt::Vector<TourDisplayer*> tourDisps; // displays results
	const ErrorHandler* errHandler;
	Globals glob;

	SimOptions(const ErrorHandler* frontend);
	~SimOptions();

};

//////////// virtual base manager
/** Manages a tournament. Loads the programs and sets up the simulations.  */
class SimManager {

public:
	/// creates supervisors, results and programs. 
	virtual void setup(SimOptions* const options);
	virtual void run() = 0;
	/// destroys managerSupervisors and programs, but NOT options.
	virtual ~SimManager();

protected:
	SimOptions* options;
	Simulation* curSim;
	lrt::Vector<SimSupervisor*> managerSupervisors;
	lrt::Vector<Program*> programs;
	lrt::Vector<TourResult> results;
	/** creates a StdBotCreator and a StdVictorySupervisor. If you'd like
	  * different ones, just override this method. */
	virtual void createSupervisors();
	/// creates Programs and empty results for them
	void createPrograms();
	/// Do one sim between the given programs, using standard scoring.
	TourStatusType doStdSim(const lrt::Array<int>& programNums);
	/// Do all (i.e.\ options->numRepeats) sims between the given programs.
	TourStatusType doAllStdSims(const lrt::Array<int>& programNums);

	virtual TourInitInfo getTourInitInfo() = 0;

	SimManager();
};

/////////// derived managers
/** Plays a charts or single tournament. (Command line options -c and -s) */
class ChartsManager : public SimManager{

public:
	ChartsManager(bool playAll);
	virtual ~ChartsManager();

	virtual void run();

protected:
	virtual TourInitInfo getTourInitInfo();
	bool playAll;
};


/** Play a all-in-one tournament. (Command line option -i) */
class AllInOneManager : public SimManager{

public:
	AllInOneManager();
	virtual ~AllInOneManager();

	virtual void run();

protected:
	virtual TourInitInfo getTourInitInfo();
};



} // namespace

#endif

