package org.herac.tuxguitar.gui.editors.tab;

import org.herac.tuxguitar.gui.editors.tab.layout.ViewLayout;
import org.herac.tuxguitar.song.models.MeasureHeader;


public class MeasureHeaderGui {
    /**
     * Espacio por defecto del timeSignature
     */
    public static final int DEFAULT_TIME_SIGNATURE_SPAN = 40;
    /**
     * Espacio por defecto a la izquierda
     */
    public static final int DEFAULT_LEFT_SPAN = 15;
    /**
     * Espacio por defecto a la derecha
     */
    public static final int DEFAULT_RIGHT_SPAN = 15;    	
	
	private static final int PAINT_TEMPO = 0x01;
	
	private static final int PAINT_TRIPLET_FEEL = 0x02;
	
	private static final int PAINT_TIME_SIGNATURE = 0x04;
	
	
	private MeasureHeader header;
	
	private int paintFlags;
	
	private int maxQuarterSpan;
	
	private int maxClefSpan;
	
	private int maxKeySignatureSpan;		
	
	private int width;
	
	public MeasureHeaderGui(MeasureHeader header){				
		this.header = header;
	}

	public void reset() {
		this.paintFlags = 0;
		this.maxQuarterSpan = 0;
    	this.maxClefSpan = 0;
    	this.maxKeySignatureSpan = 0;
	}
	
	public void update(ViewLayout layout) {    
		reset();
    	calculateMeasureChanges(layout);
    	calculateWidth();
	}
	
    public void calculateMeasureChanges(ViewLayout layout) {    	    	
    	MeasureHeader previous = layout.getSongManager().getPrevMeasureHeader(header);
        if(previous == null){
        	this.paintFlags |= PAINT_TEMPO; 
        	this.paintFlags |= ((header.getTripletFeel() != MeasureHeader.TRIPLET_FEEL_NONE)?PAINT_TRIPLET_FEEL:0);
        	this.paintFlags |= PAINT_TIME_SIGNATURE;
        }else{
        	//Tempo
        	if(header.getTempo().getValue() != previous.getTempo().getValue()){
        		this.paintFlags |= PAINT_TEMPO; 
            }                 
        	//Triplet Feel
        	if(header.getTripletFeel() != previous.getTripletFeel()){
        		this.paintFlags |= PAINT_TRIPLET_FEEL;
        	}
        	//Time Signature
            int thisNumerator = header.getTimeSignature().getNumerator();
            int thisValue = header.getTimeSignature().getDenominator().getValue();
            int prevNumerator = previous.getTimeSignature().getNumerator();
            int prevValue = previous.getTimeSignature().getDenominator().getValue();            
            if(thisNumerator != prevNumerator || thisValue != prevValue){
            	this.paintFlags |= PAINT_TIME_SIGNATURE;
            }            
        }
    }	
	
    /**
     * Calcula el width del compas
     */
    public void calculateWidth() {
        double quartersInSignature = ((1.00 / (double)header.getTimeSignature().getDenominator().getValue()) * 4.00) * (double)header.getTimeSignature().getNumerator();        
        this.width = (int)((double)getMaxQuarterSpan() * quartersInSignature);
        this.width += getFirstNoteSpan();
        this.width += (header.getNumberOfRepetitions() > 0)?20:0;        
        this.width += DEFAULT_LEFT_SPAN;
        this.width += DEFAULT_RIGHT_SPAN;        
    }    
    
	public boolean shouldPaintTempo(){
		return ( (this.paintFlags & PAINT_TEMPO) != 0 );
	}	
	
	public boolean shouldPaintTripletFeel(){
		return ( (this.paintFlags & PAINT_TRIPLET_FEEL) != 0 );
	}	
		
	public boolean shouldPaintTimeSignature(){
		return ( (this.paintFlags & PAINT_TIME_SIGNATURE) != 0 );
	}

    public int getMaxQuarterSpan() {
        return maxQuarterSpan;
    }
    public void notifyQuarterSpan(int span) {
    	this.maxQuarterSpan = ((span > maxQuarterSpan)?span:maxQuarterSpan);
    }

    public int getTempoSpan(){
    	return (shouldPaintTempo()?50:0);
    }
    
    public int getTripletFeelSpan(){
    	return (shouldPaintTripletFeel()?50:0);
    }
    
    public int getTimeSignatureSpan(){
        return (shouldPaintTimeSignature()?DEFAULT_TIME_SIGNATURE_SPAN:0);
    }    
    
    public int getClefSpan(){
        return this.maxClefSpan;
    }    
    
    public int getKeySignatureSpan(){          
        return this.maxKeySignatureSpan;
    }      
    
    public int getFirstNoteSpan(){
    	int span = 0;
    	int topSpan = getTempoSpan() + getTripletFeelSpan();
    	int middleSpan = getClefSpan() + getKeySignatureSpan() + getTimeSignatureSpan();
    	    	
    	return Math.max(topSpan,middleSpan);
    }	
	
	public void notifyClefSpan(int span){
		this.maxClefSpan = ((span > maxClefSpan)?span:maxClefSpan);
	}    
    
	public void notifyKeySignatureSpan(int span){
		this.maxKeySignatureSpan = ((span > maxKeySignatureSpan)?span:maxKeySignatureSpan);
	}
	
	public int getWidth(){
		return this.width;
	}
	
	
	

}
