/*
Copyright (C) 1997-2001 Id Software, Inc.

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  

See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

*/
#include "q_shared.h"

//============================================================================

char	*SOUND_EXTENSIONS[] = { ".wav", ".ogg" };
size_t	NUM_SOUND_EXTENSIONS = 2;

char	*IMAGE_EXTENSIONS[] = { ".jpg", ".tga", ".pcx" };
size_t	NUM_IMAGE_EXTENSIONS = 3;

//============================================================================

/*
================
COM_UserFilename

Changes \ character to / characters in the string
Does NOT validate the string at all
Must be used before other functions are aplied to the string (or those functions might function improperly)
================
*/
void COM_UserFilename( char *filename )
{
	char *p;

	assert( filename );

	p = filename;
	while( *p && (p = strchr(p, '\\')) ) {
		*p = '/';
		p++;
	}
}

/*
================
COM_ValidateFilename
================
*/
qboolean COM_ValidateFilename( const char *filename )
{
	assert( filename );

	if( !filename || !filename[0] )
		return qfalse;

	// we don't allow \ in filenames, all user inputted \ characters are converted to /
	if( strchr(filename, '\\') )
		return qfalse;

	return qtrue;
}

/*
================
COM_ValidateRelativeFilename
================
*/
qboolean COM_ValidateRelativeFilename( const char *filename )
{
	if( !COM_ValidateFilename(filename) )
		return qfalse;

	if( strstr(filename, "..") || strstr(filename, "//") )
		return qfalse;

	if( *filename == '/'  || *filename == '.' )
		return qfalse;

	return qtrue;
}

//============
//COM_StripExtension
//============
void COM_StripExtension( char *filename )
{
	char	*src, *last = NULL;

	last = strrchr( filename, '/' );
	src = strrchr( last ? last : filename, '.' );
	if( src && *(src+1) )
		*src = 0;
}

//============
//COM_FileExtension
//============
const char *COM_FileExtension( const char *filename )
{
	const char	*src, *last;

	if ( !*filename )
		return filename;

	last = strrchr( filename, '/' );
	src = strrchr( last ? last : filename, '.' );
	if( src && *(src+1) )
		return src;

	return NULL;
}

//==================
//COM_DefaultExtension
//If path doesn't have extension, appends one to it
//If there is no room for it overwrites the end of the path
//==================
void COM_DefaultExtension( char *path, const char *extension, size_t size )
{
	const char	*src, *last;
	size_t	extlen;

	assert( extension && extension[0] && strlen(extension) < size );

	extlen = strlen(extension);

	// if path doesn't have a .EXT, append extension
	// (extension should include the .)
	last = strrchr( path, '/' );
	src = strrchr( last ? last : path, '.' );
	if( src && *(src+1) )
		return;						// it has an extension

	if( strlen(path)+extlen >= size )
		path[size-extlen-1] = 0;
	Q_strncatz( path, extension, size );
}

//==================
//COM_ReplaceExtension
//Replaces current extension, if there is none appends one
//If there is no room for it overwrites the end of the path
//==================
void COM_ReplaceExtension( char *path, const char *extension, size_t size )
{
	assert( path );
	assert( extension && extension[0] && strlen(extension) < size );

	COM_StripExtension( path );
	COM_DefaultExtension( path, extension, size );
}

//============
//COM_FileBase
//============
const char *COM_FileBase( const char *in )
{
	const char *s;

	s = strrchr( in, '/' );
	if( s )
		return s+1;

	return in;
}

//============
//COM_StripFilename
//
//Cuts the string of, at the last / or erases the whole string if not found
//============
void COM_StripFilename( char *filename )
{
	char	*p;

	p = strrchr( filename, '/' );
	if( !p )
		p = filename;

	*p = 0;
}

//============
//COM_FilePathLength
//
//Returns the length from start of string to the character before last /
//============
int COM_FilePathLength( const char *in )
{
	const char *s;

	s = strrchr( in, '/' );
	if( !s )
		s = in;

	return s - in;
}

//============================================================================
//
//					BYTE ORDER FUNCTIONS
//
//============================================================================

#if !defined(ENDIAN_LITTLE) && !defined(ENDIAN_BIG)
short   (*BigShort) (short l);
short   (*LittleShort) (short l);
int     (*BigLong) (int l);
int     (*LittleLong) (int l);
float   (*BigFloat) (float l);
float   (*LittleFloat) (float l);
#endif

short   ShortSwap (short l)
{
	qbyte    b1, b2;

	b1 = l&255;
	b2 = (l>>8)&255;

	return (b1<<8) + b2;
}

#if !defined(ENDIAN_LITTLE) && !defined(ENDIAN_BIG)
static short   ShortNoSwap (short l)
{
	return l;
}
#endif

int    LongSwap (int l)
{
	qbyte    b1, b2, b3, b4;

	b1 = l&255;
	b2 = (l>>8)&255;
	b3 = (l>>16)&255;
	b4 = (l>>24)&255;

	return ((int)b1<<24) + ((int)b2<<16) + ((int)b3<<8) + b4;
}

#if !defined(ENDIAN_LITTLE) && !defined(ENDIAN_BIG)
static int     LongNoSwap (int l)
{
	return l;
}
#endif

float FloatSwap (float f)
{
	union
	{
		float   f;
		qbyte   b[4];
	} dat1, dat2;


	dat1.f = f;
	dat2.b[0] = dat1.b[3];
	dat2.b[1] = dat1.b[2];
	dat2.b[2] = dat1.b[1];
	dat2.b[3] = dat1.b[0];
	return dat2.f;
}

#if !defined(ENDIAN_LITTLE) && !defined(ENDIAN_BIG)
static float FloatNoSwap (float f)
{
	return f;
}
#endif


//================
//Swap_Init
//================
void Swap_Init (void)
{
#if !defined(ENDIAN_LITTLE) && !defined(ENDIAN_BIG)
	qbyte    swaptest[2] = {1,0};

// set the byte swapping variables in a portable manner
	if ( *(short *)swaptest == 1)
	{
		BigShort = ShortSwap;
		LittleShort = ShortNoSwap;
		BigLong = LongSwap;
		LittleLong = LongNoSwap;
		BigFloat = FloatSwap;
		LittleFloat = FloatNoSwap;
	}
	else
	{
		BigShort = ShortNoSwap;
		LittleShort = ShortSwap;
		BigLong = LongNoSwap;
		LittleLong = LongSwap;
		BigFloat = FloatNoSwap;
		LittleFloat = FloatSwap;
	}
#endif
}


//=============
//TempVector
//
//This is just a convenience function
//for making temporary vectors for function calls
//=============
float	*tv (float x, float y, float z)
{
	static	int		index;
	static	vec3_t	vecs[8];
	float	*v;

	// use an array so that multiple tempvectors won't collide
	// for a while
	v = vecs[index];
	index = (index + 1)&7;

	v[0] = x;
	v[1] = y;
	v[2] = z;

	return v;
}

//=============
//VectorToString
//
//This is just a convenience function for printing vectors
//=============
char	*vtos (vec3_t v)
{
	static	int		index;
	static	char	str[8][32];
	char	*s;

	// use an array so that multiple vtos won't collide
	s = str[index];
	index = (index + 1)&7;

	Q_snprintfz (s, 32, "(%+6.3f %+6.3f %+6.3f)", v[0], v[1], v[2]);

	return s;
}

//============
//va
//
//does a varargs printf into a temp buffer, so I don't need to have
//varargs versions of all text functions.
//============
char *va( const char *format, ... )
{
	va_list		argptr;
	static int  str_index;
	static char	string[8][2048];

	str_index = (str_index+1) & 7;
	va_start( argptr, format );
	Q_vsnprintfz( string[str_index], sizeof(string[str_index]), format, argptr );
	va_end( argptr );

	return string[str_index];
}


char	com_token[MAX_TOKEN_CHARS];

//==============
//COM_ParseExt
//
//Parse a token out of a string
//==============
char *COM_ParseExt2( char **data_p, qboolean nl, qboolean sq )
{
	int		c;
	int		len;
	char	*data;
	qboolean newlines = qfalse;

	data = *data_p;
	len = 0;
	com_token[0] = 0;

	if (!data)
	{
		*data_p = NULL;
		return "";
	}

// skip whitespace
skipwhite:
	while ( (c = *data) <= ' ')
	{
		if (c == 0)
		{
			*data_p = NULL;
			return "";
		}
		if (c == '\n')
			newlines = qtrue;
		data++;
	}

	if ( newlines && !nl ) {
		*data_p = data;
		return com_token;
	}

// skip // comments
	if (c == '/' && data[1] == '/')
	{
		data += 2;

		while (*data && *data != '\n')
			data++;
		goto skipwhite;
	}

// skip /* */ comments
	if (c == '/' && data[1] == '*')
	{
		data += 2;

		while (1)
		{
			if (!*data)
				break;
			if (*data != '*' || *(data+1) != '/')
				data++;
			else
			{
				data += 2;
				break;
			}
		}
		goto skipwhite;
	}

// handle quoted strings specially
	if (c == '\"')
	{
		if (sq)
			data++;
		while (1)
		{
			c = *data++;
			if (c=='\"' || !c)
			{
				if( !c )
					data--;

				if ((len < MAX_TOKEN_CHARS) && (!sq))
				{
					com_token[len] = '\"';
					len++;
//					data++;
				}

				if (len == MAX_TOKEN_CHARS)
				{
//					Com_Printf ("Token exceeded %i chars, discarded.\n", MAX_TOKEN_CHARS);
					len = 0;
				}
				com_token[len] = 0;
				*data_p = data;
				return com_token;
			}
			if (len < MAX_TOKEN_CHARS)
			{
				com_token[len] = c;
				len++;
			}
		}
	}

// parse a regular word
	do
	{
		if (len < MAX_TOKEN_CHARS)
		{
			com_token[len] = c;
			len++;
		}
		data++;
		c = *data;
	} while (c>32);

	if (len == MAX_TOKEN_CHARS)
	{
//		Com_Printf ("Token exceeded %i chars, discarded.\n", MAX_TOKEN_CHARS);
		len = 0;
	}
	com_token[len] = 0;

	*data_p = data;
	return com_token;
}

//==============
//COM_RemoveColorTokens
//
//Remove color tokens from a string
//==============
char *COM_RemoveColorTokens( const char *in )
{
	static char cleanString[MAX_STRING_CHARS];
	char *out = cleanString;
	qboolean colorflag = qfalse;

	memset( cleanString, 0, MAX_STRING_CHARS);

	while( *in )
	{
		if( colorflag ) {
			if( *in == Q_COLOR_ESCAPE )
				*out++ = *in;	
			colorflag = qfalse;
		} else if( *in == Q_COLOR_ESCAPE )
			colorflag = qtrue;
		else
			*out++ = *in;
		++in;
	}

	return cleanString;
}

//==============
//COM_RemoveJunkChars
//
//Remove junk chars from a string (created for autoaction filenames)
//==============
char *COM_RemoveJunkChars( const char *in )
{
   static char cleanString2[MAX_STRING_CHARS];
   char *out = cleanString2;

   memset( cleanString2, 0, MAX_STRING_CHARS );
   
   while( *in )
   {
   
      if( isalpha(*in) || isdigit(*in) )
      {
         // keep it
         *out=*in;
         in++;
         out++;
      }
      else if ( *in == '<' || *in == '[' || *in == '{' ) 
      {
         *out = '(';
         in++;
         out++;
      }
      else if ( *in == '>' || *in == ']' || *in == '}' ) 
      {
         *out = ')';
         in++;
         out++;
      }
      else if ( *in == '.' ) 
      {
         *out = '_';
         in++;
         out++;
      }
	  else
	  {
		  // another char
		  // skip it
		  in++;
	  }
   }
   
   return cleanString2;
}

//==============
//COM_ReadColorRGBString
//==============
int COM_ReadColorRGBString( const char *in ) {
	static int playerColor[3];
	if( in && strlen(in) ) {
		if( sscanf(in, "%i %i %i", &playerColor[0], &playerColor[1], &playerColor[2]) == 3 ) {
			return COLOR_RGB( playerColor[0], playerColor[1], playerColor[2] );
		}
	}
	return -1;
}


//============================================================================
//
//					LIBRARY REPLACEMENT FUNCTIONS
//
//============================================================================

//==============
//Q_strncpyz
//==============
void Q_strncpyz( char *dest, const char *src, size_t size )
{
#ifdef HAVE_STRLCPY
	strlcpy( dest, src, size );
#else
	if( size ) {
		while( --size && (*dest++ = *src++) );
		*dest = '\0';
	}
#endif
}

//==============
//Q_strncatz
//==============
void Q_strncatz( char *dest, const char *src, size_t size )
{
#ifdef HAVE_STRLCAT
	strlcat( dest, src, size );
#else
	if( size ) {
		while( --size && *dest++ );
		if( size ) {
			dest--; size++;
			while( --size && (*dest++ = *src++) );
		}
		*dest = '\0';
	}
#endif
}

//==============
//Q_vsnprintfz
//==============
int Q_vsnprintfz( char *dest, size_t size, const char *format, va_list argptr )
{
	int len;

	assert( dest );
	assert( size );

	len = vsnprintf( dest, size, format, argptr );
	dest[size-1] = 0;

	return len;
}

//==============
//Q_snprintfz
//==============
void Q_snprintfz( char *dest, size_t size, const char *format, ... )
{
	va_list	argptr;

	va_start( argptr, format );
	Q_vsnprintfz( dest, size, format, argptr );
	va_end( argptr );
}

//==============
//Q_strupr
//==============
char *Q_strupr( char *s )
{
	char *p;

	if( s ) {
		for( p = s; *s; s++ )
			*s = toupper( *s );
		return p;
	}

	return NULL;
}

//==============
//Q_strlwr
//==============
char *Q_strlwr( char *s )
{
	char *p;

	if( s ) {
		for( p = s; *s; s++ )
			*s = tolower( *s );
		return p;
	}

	return NULL;
}

/*
==============
Q_isdigit
==============
*/
qboolean Q_isdigit( const char *str )
{
	if( str && *str ) {
		while( isdigit( *str ) ) str++;
		if( !*str )
			return qtrue;
	}
	return qfalse;
}

//============================================================================
//
//					WILDCARD COMPARES FUNCTIONS
//
//============================================================================

//==============
//Q_WildCmpAfterStar
//==============
static qboolean Q_WildCmpAfterStar( const char *pattern, const char *text )
{
	char c, c1;
	const char *p = pattern, *t = text;

	while( (c = *p++) == '?' || c == '*' ) {
		if( c == '?' && *t++ == '\0' )
			return qfalse;
	}

	if( c == '\0' )
		return qtrue;

	for( c1 = ((c == '\\') ? *p : c); ; ) {
		if( tolower( *t ) == c1 && Q_WildCmp( p - 1, t ) )
			return qtrue;
		if( *t++ == '\0' )
			return qfalse;
	}
}

//==============
//Q_WildCmp
//==============
qboolean Q_WildCmp( const char *pattern, const char *text )
{
	char c;

	while( (c = *pattern++) != '\0' ) {
		switch( c ) {
			case '?':
				if( *text++ == '\0' )
					return qfalse;
				break;
			case '\\':
				if( tolower( *pattern++ ) != tolower( *text++ ) )
					return qfalse;
				break;
			case '*':
				return Q_WildCmpAfterStar( pattern, text );
			default:
				if( tolower( c ) != tolower( *text++ ) )
					return qfalse;
		}
	}

	return (*text == '\0');
}


//=====================================================================
//
//  INFO STRINGS
//
//=====================================================================

//==================
//Info_ValidateValue
//==================
static qboolean Info_ValidateValue( const char *value )
{
	assert( value );

	if( !value )
		return qfalse;

	if( strlen(value) >= MAX_INFO_VALUE )
		return qfalse;

	if( strchr(value, '\\') )
		return qfalse;

	if( strchr(value, ';') )
		return qfalse;

	if( strchr(value, '"') )
		return qfalse;

	if( strchr(value, '\xFF') )
		return qfalse;

	return qtrue;
}

//==================
//Info_ValidateKey
//==================
static qboolean Info_ValidateKey( const char *key )
{
	assert( key );

	if( !key )
		return qfalse;

	if( !key[0] )
		return qfalse;

	if( strlen(key) >= MAX_INFO_KEY )
		return qfalse;

	if( strchr(key, '\\') )
		return qfalse;

	if( strchr(key, ';') )
		return qfalse;

	if( strchr(key, '"') )
		return qfalse;

	if( strchr(key, '\xFF') )
		return qfalse;

	return qtrue;
}

//==================
//Info_Validate
//
//Some characters are illegal in info strings because they
//can mess up the server's parsing
//==================
qboolean Info_Validate( const char *info )
{
	const char	*p, *start;

	assert( info );

	if( !info )
		return qfalse;

	if( strlen(info) >= MAX_INFO_STRING )
		return qfalse;

	if( strchr(info, '\"') )
		return qfalse;

	if( strchr(info, ';') )
		return qfalse;

	if( strchr(info, '"') )
		return qfalse;

	// wsw: r1q2: check for end-of-message-in-string exploit
	if( strchr(info, '\xFF') )
		return qfalse;

	p = info;

	while( p && *p ) {
		if( *p++ != '\\' )
			return qfalse;

		start = p;
		p = strchr( start, '\\' );
		if( !p ) // missing key
			return qfalse;
		if( p - start >= MAX_INFO_KEY ) // too long
			return qfalse;

		p++; // skip the \ char

		start = p;
		p = strchr( start, '\\' );
		if( (p && p - start >= MAX_INFO_KEY) || (!p && strlen(start) >= MAX_INFO_KEY) ) // too long
			return qfalse;
	}

	return qtrue;
}

//==================
//Info_FindKey
//
//Returns the pointer to the \ character if key is found
//Otherwise returns NULL
//==================
static char *Info_FindKey( const char *info, const char *key )
{
	const char	*p, *start;

	assert( Info_Validate(info) );
	assert( Info_ValidateKey(key) );

	if( !Info_Validate(info) || !Info_ValidateKey(key) )
		return NULL;

	p = info;

	while( p && *p )
	{
		start = p;

		p++; // skip the \ char
		if( !strncmp(key, p, strlen(key)) && p[strlen(key)] == '\\' )
			return (char *)start;

		p = strchr( p, '\\' );
		if( !p )
			return NULL;

		p++; // skip the \ char
		p = strchr( p, '\\' );
	}

	return NULL;
}

//===============
//Info_ValueForKey
//
//Searches the string for the given
//key and returns the associated value, or NULL
//===============
char *Info_ValueForKey( const char *info, const char *key )
{
	static char value[2][MAX_INFO_VALUE];	// use two buffers so compares work without stomping on each other
	static int	valueindex;
	const char	*p, *start;
	size_t		len;

	assert( info && Info_Validate(info) );
	assert( key && Info_ValidateKey(key) );

	if( !Info_Validate(info) || !Info_ValidateKey(key) )
		return NULL;

	valueindex ^= 1;

	p = Info_FindKey( info, key );
	if( !p )
		return NULL;

	p++; // skip the \ char
	p = strchr( p, '\\' );
	if( !p )
		return NULL;

	p++; // skip the \ char
	start = p;
	p = strchr( p, '\\' );
	if( !p ) {
		len = strlen(start);
	} else {
		len = p - start;
	}

	if( len >= MAX_INFO_VALUE ) {
		assert( qfalse );
		return NULL;
	}
	strncpy( value[valueindex], start, len );
	value[valueindex][len] = 0;

	return value[valueindex];
}

//==================
//Info_RemoveKey
//==================
void Info_RemoveKey( char *info, const char *key )
{
	char		*start, *p;
	size_t		len;

	assert( info && Info_Validate(info) );
	assert( key && Info_ValidateKey(key) );

	if( !Info_Validate(info) || !Info_ValidateKey(key) )
		return;

	p = Info_FindKey( info, key );
	if( !p )
		return;

	start = p;

	p++; // skip the \ char
	p = strchr( p, '\\' );
	if( p ) {
		p++; // skip the \ char
		p = strchr( p, '\\' );
	}

	if( !p ) {
		*start = 0;
	} else {
		// wsw : aiwa : fixed possible source and destination overlap with strcpy()
		len = strlen(p);
		memmove( start, p, len );
		start[len] = 0;
	}
}

//==================
//Info_SetValueForKey
//==================
qboolean Info_SetValueForKey( char *info, const char *key, const char *value )
{
	char	pair[MAX_INFO_KEY + MAX_INFO_VALUE + 1];

	assert( info && Info_Validate(info) );
	assert( key && Info_ValidateKey(key) );
	assert( value && Info_ValidateValue(value) );

	if( !Info_Validate(info) || !Info_ValidateKey(key) || !Info_ValidateValue(value) )
		return qfalse;

	Info_RemoveKey( info, key );

	Q_snprintfz( pair, sizeof(pair), "\\%s\\%s", key, value);

	if( strlen(pair) + strlen(info) > MAX_INFO_STRING )
		return qfalse;

	Q_strncatz( info, pair, MAX_INFO_STRING );

	return qtrue;
}
