// ---------------------------------------------------------------------------
// - InputOutput.hpp                                                         -
// - standard object library - input-output stream class definition          -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2007 amaury darsch                                   -
// ---------------------------------------------------------------------------


#ifndef  AFNIX_INPUTOUTPUT_HPP
#define  AFNIX_INPUTOUTPUT_HPP

#ifndef  AFNIX_INPUT_HPP
#include "Input.hpp"
#endif

#ifndef  AFNIX_OUTPUT_HPP
#include "Output.hpp"
#endif

namespace afnix {

  /// The InputOutput class implements an input-output stream with a buffer
  /// which holds character during the processing of transit between the 
  /// stream to the input stream. The theory of operation goes as follow.
  /// The internal buffer is filled with characters with the help of the
  /// output stream. The characters are consumed from the buffer with the 
  /// help of the input stream (read method). If the buffer becomes empty
  /// the iseof return true and the valid method false. The read method will
  /// return the eof character
  /// @author amaury darsch

  class InputOutput : public Input, public Output {
  public:
    /// create a new input-output stream without data
    InputOutput (void);

    /// create a new input-output stream with a string
    /// @param data the data for the buffer
    InputOutput (const String& data);

    /// create a new input-output stream with a buffer and a size
    /// @param data the data for the buffer
    /// @param size the size of the data
    InputOutput (const char* data, const long size);

    /// @return the class name
    String repr (void) const;

    /// rest this input-output stream
    void reset (void);

    /// @return a new character from the input stream
    char read (void);

    /// @return true if we are at the end of the input stream
    bool iseof (void) const;

    /// check if we can read a character
    /// @param tout the timeout value
    bool valid (const long tout) const;
      
    /// write one character on the output stream.
    /// @param value the character to write
    void write (const char value);

    /// write a character string to the output stream
    /// @param value the character string to write
    void write (const char* value);

  private:
    // make the copy constructor private
    InputOutput (const InputOutput&);
    // make the assignment operator private
    InputOutput& operator = (const InputOutput&);

  public:
    /// create a new object in a generic way
    /// @param argv the argument vector
    static Object* mknew (Vector* argv);

    /// @return true if the given quark is defined
    bool isquark (const long quark, const bool hflg) const;

    /// apply this object with a set of arguments and a quark
    /// @param robj  the current runnable
    /// @param nset  the current nameset    
    /// @param quark the quark to apply these arguments
    /// @param argv  the arguments to apply
    Object* apply (Runnable* robj, Nameset* nset, const long quark,
		   Vector* argv);
  };
}

#endif
