/*******************************************************************************
 *  PROJECT: GNOME Colorscheme
 *
 *  AUTHOR: Jonathon Jongsma
 *
 *  Copyright (c) 2005 Jonathon Jongsma
 *
 *  License:
 *    This program is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the 
 *    Free Software Foundation, Inc., 59 Temple Place, Suite 330, 
 *    Boston, MA  02111-1307  USA
 *
 *******************************************************************************/

#include <gtkmm/widget.h>

#include "core/log-stream.h"
#include "gcs-schemebox.h"
#include "gcs-conf.h"

namespace gcs
{
    namespace Widgets
    {

        SchemeBox::SchemeBox(void)
            : m_pScheme(new Scheme(Color::create("#FF0000"), SCHEME_COMPLEMENTS))
            {
                set_border_width(Conf::UI_SPACING_SMALL);
                set_spacing(Conf::UI_SPACING_SMALL);
                set_size_request(200, 150);
                init();
            }

        SchemeBox::SchemeBox(Scheme schm)
        {
            SchemeBox::SchemeBox();
            set_scheme(schm);
        }

        SchemeBox::~SchemeBox(void)
        {
            children().clear();
        }


        void SchemeBox::set_scheme(Scheme schm)
        {
            hide_all();
            m_pScheme.reset(new Scheme(schm));
            redraw();
        }


        void SchemeBox::set_color(ColorPtr clr)
        {
            hide_all();
            m_pScheme->set_color(clr);
            redraw();
        }


        void SchemeBox::set_scheme_type(tSchemeType t)
        {
            hide_all();
            m_pScheme->set_scheme_type(t);
            redraw();
        }


        void SchemeBox::init(void)
        {
            TextSwatch *pSwatch = 0;
            for (gint i = 0; i < NUM_SWATCHES; i++)
            {
                ColorPtr pColor = Color::create("#FF0000");
                pSwatch = Gtk::manage(new TextSwatch(pColor));
                (*(m_pSwatches + i)) = pSwatch;
                pack_start(*pSwatch, true, true, 0);
                pSwatch->signal_selected().connect(
                        sigc::bind(sigc::mem_fun(*this,
                                &SchemeBox::on_color_selected), i));
            }
        }


        void SchemeBox::redraw(void)
        {
            show();
            if (m_pScheme.get() != NULL)
            {
                Scheme::iterator iColor;
                gint i;
                for (iColor = m_pScheme->begin(), i = 0;
                        iColor != m_pScheme->end(); iColor++, i++)
                {
                    LOG("Swatch Color: " << *iColor);
                    if (*iColor == m_pScheme->get_color())
                    {
                        // this is the color that the scheme is based on, so
                        // let's emphasize it with a thicker border
                        m_pSwatches[i]->set_border_width(3);
                    }
                    else
                    {
                        m_pSwatches[i]->set_border_width(1);
                    }
                    // FIXME: is this ok?
                    m_pSwatches[i]->set_color(*iColor);
                    m_pSwatches[i]->show();
                }
            }
        }


        bool SchemeBox::on_color_selected(gint swatch_id)
        {
            signal_color_selected().emit(m_pSwatches[swatch_id]->get_color());
            LOG("SchemeBox::on_color_selected");
            return true;
        }

    } // namespace Widgets
} // namespace gcs
