//==============================================
//  copyright            : (C) 2003-2005 by Will Stokes
//==============================================
//  This program is free software; you can redistribute it
//  and/or modify it under the terms of the GNU General
//  Public License as published by the Free Software
//  Foundation; either version 2 of the License, or
//  (at your option) any later version.
//==============================================

//Systemwide includes
#include <qwidget.h>
#include <qlayout.h>
#include <qlabel.h>
#include <qiconview.h>
#include <qtabwidget.h>

//Projectwide includes
#include "layoutWidget.h"
#include "titleWidget.h"
#include "window.h"
#include "subalbumsWidget.h"
#include "subalbumWidget.h"
#include "editing/editingInterface.h"
#include "../backend/subalbum.h"
#include "../backend/photo.h"
#include "../config.h"

//==============================================
LayoutWidget::LayoutWidget(QWidget *parent, const char* name ) : QWidget(parent,name)
{
  window = (Window*)parent;
  subalbums = new SubalbumsWidget( this, "subalbums" );
  connect( subalbums, SIGNAL( collectionSelected(Subalbum*)),
           this,      SLOT( showCollection( Subalbum* )) );
  
  subalbum = new SubalbumWidget( NULL, this, "subalbum" );
  editingInterface = new EditingInterface( this, "editingInterface" );

  tabbedArea = new QTabWidget( this, "tabbedArea" );  
  tabbedArea->addTab(subalbum, tr("Organize") );
  tabbedArea->addTab(editingInterface, tr("Edit") );
  
  //by default no photo has been edited
  tabbedArea->setTabEnabled(editingInterface, false);   
  
  //catch changing tab selection in order to
  //initialize editing interface if it was selected
  connect( tabbedArea, SIGNAL( currentChanged(QWidget*) ),
           this, SLOT( tabChanged(QWidget*) ) );
  
  //catch selectedPhotoStateChanged signal in order to
  //enable/disable Photos and Tools menu items
  connect( subalbum, SIGNAL( selectedPhotoStateChanged() ),
           this, SLOT( photoStateChangedEvent() ) );
  
  //catch photo modified signal in order to 
  //enable/disable Photos and Tools menu items
  connect( editingInterface, SIGNAL( photoModified() ),
           this, SLOT( photoStateChangedEvent() ) );
    
  //place the subalbums list and tabbed area in grid
  grid = new QGridLayout( this, 1, 2, 0 );
  
  grid->addWidget( subalbums, 0, 0 );
  grid->setColSpacing( 0, subalbums->sizeHint().width() );

  grid->addWidget( tabbedArea, 0, 1 );
  grid->setColStretch( 1, 1 );
}
//==============================================
void LayoutWidget::showCollection(Subalbum* collection)
{
  //ensure currently in organize mode
  organize();
  
  //load collection
  subalbum->setSubalbum(collection);
  
  //pass signal on so title area can update as well
  emit collectionSelected( collection );
}
//==============================================
void LayoutWidget::refreshSelectedCollectionIconName()
{
  subalbums->refreshSelectedCollectionName();
}
//==============================================
void LayoutWidget::updateSubalbumImage( QPixmap* val)
{
  subalbums->updatedSelectedCollectionImage(val);
}
//==============================================
SubalbumWidget* LayoutWidget::getSubalbum()
{
  return subalbum;
}
//==============================================
SubalbumsWidget* LayoutWidget::getSubalbums()
{
  return subalbums;
}
//==============================================
Window* LayoutWidget::getWindow()
{
  return window;
}
//==============================================
void LayoutWidget::refresh()
{
  subalbums->refreshCollectionsList();
}
//==============================================
void LayoutWidget::tabChanged( QWidget* widget)
{
  //orignize tab seleced
  if(widget != editingInterface)
  {
    //refresh all thumbnails since any could have changed
    subalbum->refreshAllPhotos();

    //handle the selected/shown photo state having been changed
    photoStateChangedEvent();

    //find and select the last shown photo in the
    //editing interface, unselect all other items
    subalbum->setSelectedPhoto( editingInterface->getPhoto() );
    
    return;
  }
  //edit tab selected - init editor
  else
  {
    ///get current collection and photo pointers
    Subalbum* collection = subalbum->getSubalbum();
    Photo* photo = subalbum->getFirstSelectedPhoto();    

    //bail if either pointer is null (sanity check)
    if(collection == NULL || photo == NULL)
      return;
  
    //init editing interface for current collection:photo
    editingInterface->setPhoto( collection, photo);
    editingInterface->setFocus();

    //handle the selected/shown photo state having been changed
    photoStateChangedEvent();
  }
}
//==============================================
void LayoutWidget::editSelectedPhoto()
{
  tabbedArea->showPage( editingInterface );
}
//==============================================
void LayoutWidget::organize()
{
  tabbedArea->setCurrentPage( 0 );
}
//==============================================
void LayoutWidget::setEditTabEnabled(bool val)
{
  tabbedArea->setTabEnabled(editingInterface, val);   
}
//==============================================
void LayoutWidget::revertPhotos()
{
  if( tabbedArea->currentPage() == subalbum )
    subalbum->revertSelectedPhotos();
  else if( tabbedArea->currentPage() == editingInterface )
    editingInterface->revertCurrentPhoto();
}
//==============================================
void LayoutWidget::photoStateChangedEvent()
{
  //determine if:
  //1.) any photos are selected - false if in editing mode
  //2.) if any revertable photos are selected/shown
  bool anySelected = false;
  bool anyRevertable = false;
  
  if( tabbedArea->currentPage() == subalbum )
  {
    anySelected = subalbum->anyPhotosSelected();
    anyRevertable = anySelected && subalbum->anySelectedPhotosRevertable();    
  }
  else
  {
    //none selected in editing mode
    anySelected = false;
    anyRevertable = editingInterface->currentPhotoRevertable();
  }
  
  //update menus
  window->getTitle()->updateMenus(anySelected, anyRevertable);    
}
//==============================================
