// -*- mode: cpp; mode: fold -*-
// Description								/*{{{*/
// $Id: http.cc,v 1.59 2004/05/08 19:42:35 mdz Exp $
/* ######################################################################

   DEBTORRENT Acquire Method - This is the DEBTORRENT aquire method for APT.

   It is originally based on APT's HTTP Acquire Method (version 0.7.6).

   The DEBTORRENT protocol is almost identical to HTTP/1.1, with
   modifications to allow for extra pipelining of requests and the
   returning of files in a different order than requested.

   The pipelining change removes the limit on the number of outstanding
   requests that are allowed so that all requests received from APT are
   immediately requested from the DebTorrent client.

   To return files in a different order than requested, the request name
   must be used in the header line of the response. For example, if a
   request line is "GET /what/temp.html DEBTORRENT/1.1", and the HTTP/1.1
   response would be "HTTP/1.1 200 OK", then the DebTorrent response is
   "DEBTORRENT/1.1 /what/temp/html 200 OK".
   
   It uses HTTP/1.1 and many of the fancy options there-in, such as
   pipelining, range, if-range and so on. 

   It is based on a doubly buffered select loop. A groupe of requests are 
   fed into a single output buffer that is constantly fed out the 
   socket. This provides ideal pipelining as in many cases all of the
   requests will fit into a single packet. The input socket is buffered 
   the same way and fed into the fd for the file (may be a pipe in future).
   
   This double buffering provides fairly substantial transfer rates,
   compared to wget the http method is about 4% faster. Most importantly,
   when HTTP is compared with FTP as a protocol the speed difference is
   huge. In tests over the internet from two sites to llug (via ATM) this
   program got 230k/s sustained http transfer rates. FTP on the other 
   hand topped out at 170k/s. That combined with the time to setup the
   FTP connection makes HTTP a vastly superior protocol.
      
   ##################################################################### */
									/*}}}*/
// Include Files							/*{{{*/
#include <apt-pkg/fileutl.h>
#include <apt-pkg/acquire-method.h>
#include <apt-pkg/error.h>
#include <apt-pkg/hashes.h>

#include <sys/stat.h>
#include <sys/time.h>
#include <utime.h>
#include <unistd.h>
#include <signal.h>
#include <stdio.h>
#include <errno.h>
#include <string.h>
#include <iostream>

// Internet stuff
#include <netdb.h>

#include "connect.h"
#include "rfc2553emu.h"
#include "debtorrent.h"

									/*}}}*/
using namespace std;

string HttpMethod::FailFile;
int HttpMethod::FailFd = -1;
time_t HttpMethod::FailTime = 0;
unsigned long ConnectTimeOut = 60;
unsigned long PollTimeOut = 300;
bool Debug = false;

unsigned long CircleBuf::BwReadLimit=0;
unsigned long CircleBuf::BwTickReadData=0;
struct timeval CircleBuf::BwReadTick={0,0};
const unsigned int CircleBuf::BW_HZ=10;
  
// CircleBuf::CircleBuf - Circular input buffer				/*{{{*/
// ---------------------------------------------------------------------
/* */
CircleBuf::CircleBuf(unsigned long Size) : Size(Size), Hash(0)
{
   Buf = new unsigned char[Size];
   Reset();

   CircleBuf::BwReadLimit = 0;
}
									/*}}}*/
// CircleBuf::Reset - Reset to the default state			/*{{{*/
// ---------------------------------------------------------------------
/* */
void CircleBuf::Reset()
{
   InP = 0;
   OutP = 0;
   StrPos = 0;
   MaxGet = (unsigned int)-1;
   OutQueue = string();
   if (Hash != 0)
   {
      delete Hash;
      Hash = new Hashes;
   }   
};
									/*}}}*/
// CircleBuf::Read - Read from a FD into the circular buffer		/*{{{*/
// ---------------------------------------------------------------------
/* This fills up the buffer with as much data as is in the FD, assuming it
   is non-blocking.. */
bool CircleBuf::Read(int Fd)
{
   unsigned long BwReadMax;

   while (1)
   {
      // Woops, buffer is full
      if (InP - OutP == Size)
	 return true;

      // what's left to read in this tick
      BwReadMax = CircleBuf::BwReadLimit/BW_HZ;

      if(CircleBuf::BwReadLimit) {
	 struct timeval now;
	 gettimeofday(&now,0);

	 unsigned long d = (now.tv_sec-CircleBuf::BwReadTick.tv_sec)*1000000 +
	    now.tv_usec-CircleBuf::BwReadTick.tv_usec;
	 if(d > 1000000/BW_HZ) {
	    CircleBuf::BwReadTick = now;
	    CircleBuf::BwTickReadData = 0;
	 } 
	 
	 if(CircleBuf::BwTickReadData >= BwReadMax) {
	    usleep(1000000/BW_HZ);
	    return true;
	 }
      }

      // Write the buffer segment
      int Res;
      if(CircleBuf::BwReadLimit) {
	 Res = read(Fd,Buf + (InP%Size), 
		    BwReadMax > LeftRead() ? LeftRead() : BwReadMax);
      } else
	 Res = read(Fd,Buf + (InP%Size),LeftRead());
      
      if(Res > 0 && BwReadLimit > 0) 
	 CircleBuf::BwTickReadData += Res;
    
      if (Res == 0)
	 return false;
      if (Res < 0)
      {
	 if (errno == EAGAIN)
	    return true;
	 return false;
      }

      if (InP == 0)
	 gettimeofday(&Start,0);
      InP += Res;
   }
}
									/*}}}*/
// CircleBuf::Read - Put the string into the buffer			/*{{{*/
// ---------------------------------------------------------------------
/* This will hold the string in and fill the buffer with it as it empties */
bool CircleBuf::Read(string Data)
{
   OutQueue += Data;
   FillOut();
   return true;
}
									/*}}}*/
// CircleBuf::FillOut - Fill the buffer from the output queue		/*{{{*/
// ---------------------------------------------------------------------
/* */
void CircleBuf::FillOut()
{
   if (OutQueue.empty() == true)
      return;
   while (1)
   {
      // Woops, buffer is full
      if (InP - OutP == Size)
	 return;
      
      // Write the buffer segment
      unsigned long Sz = LeftRead();
      if (OutQueue.length() - StrPos < Sz)
	 Sz = OutQueue.length() - StrPos;
      memcpy(Buf + (InP%Size),OutQueue.c_str() + StrPos,Sz);
      
      // Advance
      StrPos += Sz;
      InP += Sz;
      if (OutQueue.length() == StrPos)
      {
	 StrPos = 0;
	 OutQueue = "";
	 return;
      }
   }
}
									/*}}}*/
// CircleBuf::Write - Write from the buffer into a FD			/*{{{*/
// ---------------------------------------------------------------------
/* This empties the buffer into the FD. */
bool CircleBuf::Write(int Fd)
{
   while (1)
   {
      FillOut();
      
      // Woops, buffer is empty
      if (OutP == InP)
	 return true;
      
      if (OutP == MaxGet)
	 return true;
      
      // Write the buffer segment
      int Res;
      Res = write(Fd,Buf + (OutP%Size),LeftWrite());

      if (Res == 0)
	 return false;
      if (Res < 0)
      {
	 if (errno == EAGAIN)
	    return true;
	 
	 return false;
      }
      
      if (Hash != 0)
	 Hash->Add(Buf + (OutP%Size),Res);
      
      OutP += Res;
   }
}
									/*}}}*/
// CircleBuf::WriteTillEl - Write from the buffer to a string		/*{{{*/
// ---------------------------------------------------------------------
/* This copies till the first empty line */
bool CircleBuf::WriteTillEl(string &Data,bool Single)
{
   // We cheat and assume it is unneeded to have more than one buffer load
   for (unsigned long I = OutP; I < InP; I++)
   {      
      if (Buf[I%Size] != '\n')
	 continue;
      ++I;
      
      if (Single == false)
      {
         if (I < InP  && Buf[I%Size] == '\r')
            ++I;
         if (I >= InP || Buf[I%Size] != '\n')
            continue;
         ++I;
      }
      
      Data = "";
      while (OutP < I)
      {
	 unsigned long Sz = LeftWrite();
	 if (Sz == 0)
	    return false;
	 if (I - OutP < Sz)
	    Sz = I - OutP;
	 Data += string((char *)(Buf + (OutP%Size)),Sz);
	 OutP += Sz;
      }
      return true;
   }      
   return false;
}
									/*}}}*/
// CircleBuf::Stats - Print out stats information			/*{{{*/
// ---------------------------------------------------------------------
/* */
void CircleBuf::Stats()
{
   if (InP == 0)
      return;
   
   struct timeval Stop;
   gettimeofday(&Stop,0);
/*   float Diff = Stop.tv_sec - Start.tv_sec + 
             (float)(Stop.tv_usec - Start.tv_usec)/1000000;
   clog << "Got " << InP << " in " << Diff << " at " << InP/Diff << endl;*/
}
									/*}}}*/

// ServerState::ServerState - Constructor				/*{{{*/
// ---------------------------------------------------------------------
/* */
ServerState::ServerState(URI Srv,HttpMethod *Owner) : Owner(Owner),
                        In(64*1024), Out(4*1024),
                        ServerName(Srv)
{
   Reset();
}
									/*}}}*/
// ServerState::Open - Open a connection to the server			/*{{{*/
// ---------------------------------------------------------------------
/* This opens a connection to the server. */
bool ServerState::Open()
{
   // Use the already open connection if possible.
   if (ServerFd != -1)
      return true;
   
   Close();
   In.Reset();
   Out.Reset();
   Persistent = true;
   
   // Determine what host and port to use based on the proxy settings
   int Port = 0;
   string Host;   
   if (ServerName.Port != 0)
      Port = ServerName.Port;
   Host = ServerName.Host;
   
   // Connect to the remote server
   if (Connect(Host,Port,"debtorrent",9988,ServerFd,ConnectTimeOut,Owner) == false)
      return false;
   
   return true;
}
									/*}}}*/
// ServerState::Close - Close a connection to the server		/*{{{*/
// ---------------------------------------------------------------------
/* */
bool ServerState::Close()
{
   close(ServerFd);
   ServerFd = -1;
   return true;
}
									/*}}}*/
// ServerState::RunHeaders - Get the headers before the data		/*{{{*/
// ---------------------------------------------------------------------
/* Returns 0 if things are OK, 1 if an IO error occursed and 2 if a header
   parse error occured */
int ServerState::RunHeaders()
{
   State = Header;
   
   // Owner->Status("Waiting for headers");

   Major = 0; 
   Minor = 0; 
   Result = 0; 
   Size = 0; 
   StartPos = 0;
   Encoding = Closes;
   HaveContent = false;
   time(&Date);
   Message[0] = '\0';

   do
   {
      string Data;
      if (In.WriteTillEl(Data) == false)
	 continue;

      if (Debug == true)
	 clog << Data;
      
      for (string::const_iterator I = Data.begin(); I < Data.end(); I++)
      {
	 string::const_iterator J = I;
	 for (; J != Data.end() && *J != '\n' && *J != '\r';J++);
	 if (HeaderLine(string(I,J)) == false)
	    return 2;
	 I = J;
      }

      // 100 Continue is a Nop...
      if (Result == 100)
	 continue;
      
      // 103 Status Update for apt
      if (Result == 103)
      {
	 Owner->Status(Message);
	 if (Debug == true)
	 {
	    char Buf[1000];
	    sprintf(Buf,"StatusMessage: %s\n", Message);
	    string Req = Buf;
	    cerr << Req << endl;
	 }
	 continue;
      }
      
      // Tidy up the connection persistance state.
      if (Encoding == Closes && HaveContent == true)
	 Persistent = false;
      
      return 0;
   }
   while (Owner->Go(false,this) == true);
   
   return 1;
}
									/*}}}*/
// ServerState::RunData - Transfer the data from the socket		/*{{{*/
// ---------------------------------------------------------------------
/* */
bool ServerState::RunData()
{
   State = Data;
   
   // Chunked transfer encoding is fun..
   if (Encoding == Chunked)
   {
      while (1)
      {
	 // Grab the block size
	 bool Last = true;
	 string Data;
	 In.Limit(-1);
	 do
	 {
	    if (In.WriteTillEl(Data,true) == true)
	       break;
	 }
	 while ((Last = Owner->Go(false,this)) == true);

	 if (Last == false)
	    return false;
	 	 
	 // See if we are done
	 unsigned long Len = strtol(Data.c_str(),0,16);
	 if (Len == 0)
	 {
	    In.Limit(-1);
	    
	    // We have to remove the entity trailer
	    Last = true;
	    do
	    {
	       if (In.WriteTillEl(Data,true) == true && Data.length() <= 2)
		  break;
	    }
	    while ((Last = Owner->Go(false,this)) == true);
	    if (Last == false)
	       return false;
	    return !_error->PendingError();
	 }
	 
	 // Transfer the block
	 In.Limit(Len);
	 while (Owner->Go(true,this) == true)
	    if (In.IsLimit() == true)
	       break;
	 
	 // Error
	 if (In.IsLimit() == false)
	    return false;
	 
	 // The server sends an extra new line before the next block specifier..
	 In.Limit(-1);
	 Last = true;
	 do
	 {
	    if (In.WriteTillEl(Data,true) == true)
	       break;
	 }
	 while ((Last = Owner->Go(false,this)) == true);
	 if (Last == false)
	    return false;
      }
   }
   else
   {
      /* Closes encoding is used when the server did not specify a size, the
         loss of the connection means we are done */
      if (Encoding == Closes)
	 In.Limit(-1);
      else
	 In.Limit(Size - StartPos);
      
      // Just transfer the whole block.
      do
      {
	 if (In.IsLimit() == false)
	    continue;
	 
	 In.Limit(-1);
	 return !_error->PendingError();
      }
      while (Owner->Go(true,this) == true);
   }

   return Owner->Flush(this) && !_error->PendingError();
}
									/*}}}*/
// ServerState::HeaderLine - Process a header line			/*{{{*/
// ---------------------------------------------------------------------
/* */
bool ServerState::HeaderLine(string Line)
{
   if (Line.empty() == true)
      return true;

   // The http server might be trying to do something evil.
   if (Line.length() >= MAXLEN)
      return _error->Error("Got a single header line over %u chars",MAXLEN);

   string::size_type Pos = Line.find(' ');
   if (Pos == string::npos || Pos+1 > Line.length())
   {
      // Blah, some servers use "connection:closes", evil.
      Pos = Line.find(':');
      if (Pos == string::npos || Pos + 2 > Line.length())
	 return _error->Error("Bad header line");
      Pos++;
   }

   // Parse off any trailing spaces between the : and the next word.
   string::size_type Pos2 = Pos;
   while (Pos2 < Line.length() && isspace(Line[Pos2]) != 0)
      Pos2++;
      
   string Tag = string(Line,0,Pos);
   string Val = string(Line,Pos2);
   
   if (stringcasecmp(Tag.c_str(),Tag.c_str()+10,"DEBTORRENT") == 0)
   {
      if (sscanf(Line.c_str(),"DEBTORRENT/%u.%u %s %u %[^\n]",&Major,&Minor,
		 ResultPath,&Result,Code) != 5)
	 return _error->Error("The DebTorrent client sent an invalid reply header");

      // DEBTORRENT connections are always persistent
      Persistent = true;

      return true;
   }      

   // The new Pieces-Downloaded header indicating how much of the file is available      
   if (stringcasecmp(Tag,"Pieces-Downloaded:") == 0)
   {
      if (sscanf(Val.c_str(),"bytes %lu/%lu",&StartPos,&Size) != 2)
	 return _error->Error("The DebTorrent client sent an invalid Pieces-Downloaded header");
      if ((unsigned)StartPos > Size)
	 return _error->Error("This DebTorrent client has broken Pieces-Downloaded support");
      return true;
   }
   
   // The new Message header indicating the DebTorrent client's current status      
   if (stringcasecmp(Tag,"Message:") == 0)
   {
      if (sscanf(Val.c_str(),"%[^\n]",Message) != 1)
	 return _error->Error("The DebTorrent client sent an invalid Message header");
      return true;
   }
   
   if (stringcasecmp(Tag,"Content-Length:") == 0)
   {
      if (Encoding == Closes)
	 Encoding = Stream;
      HaveContent = true;
      
      // The length is already set from the Content-Range header
      if (StartPos != 0)
	 return true;
      
      if (sscanf(Val.c_str(),"%lu",&Size) != 1)
	 return _error->Error("The DebTorrent client sent an invalid Content-Length header");
      return true;
   }

   if (stringcasecmp(Tag,"Content-Type:") == 0)
   {
      HaveContent = true;
      return true;
   }
   
   if (stringcasecmp(Tag,"Content-Range:") == 0)
   {
      HaveContent = true;
      
      /* Remove Range headers as DebTorrent doesn't support them and it could cause
       * problems with sparse file writing.
      if (sscanf(Val.c_str(),"bytes %lu-%*u/%lu",&StartPos,&Size) != 2)
	 return _error->Error("The DebTorrent client sent an invalid Content-Range header");
      if ((unsigned)StartPos > Size)
	 return _error->Error("This DebTorrent client has broken range support"); */
      return true;
   }
   
   if (stringcasecmp(Tag,"Transfer-Encoding:") == 0)
   {
      HaveContent = true;
      if (stringcasecmp(Val,"chunked") == 0)
	 Encoding = Chunked;      
      return true;
   }

   if (stringcasecmp(Tag,"Connection:") == 0)
   {
      if (stringcasecmp(Val,"close") == 0)
	 Persistent = false;
      if (stringcasecmp(Val,"keep-alive") == 0)
	 Persistent = true;
      return true;
   }
   
   if (stringcasecmp(Tag,"Last-Modified:") == 0)
   {
      if (StrToTime(Val,Date) == false)
	 return _error->Error("Unknown date format");
      return true;
   }

   return true;
}
									/*}}}*/

// HttpMethod::SendReq - Send the HTTP request				/*{{{*/
// ---------------------------------------------------------------------
/* This places the http request in the outbound buffer */
void HttpMethod::SendReq(FetchItem *Itm,CircleBuf &Out)
{
   URI Uri = Itm->Uri;

   // The HTTP server expects a hostname with a trailing :port
   char Buf[1000];
   string ProperHost = Uri.Host;
   if (Uri.Port != 0)
   {
      sprintf(Buf,":%u",Uri.Port);
      ProperHost += Buf;
   }   
      
   // Just in case.
   if (Itm->Uri.length() >= sizeof(Buf))
       abort();
       
   /* Build the request. We include a keep-alive header to tweak old
      http/1.0 servers that do support keep-alive but not HTTP/1.1
      automatic keep-alive.  */
   sprintf(Buf,"GET %s DEBTORRENT/0.2\r\nHost: %s\r\nConnection: keep-alive\r\n",
	   QuoteString(Uri.Path,"~").c_str(),ProperHost.c_str());
   
   string Req = Buf;

   // Check for a partial file
   /* Remove Range headers as DebTorrent doesn't support them and it could cause
    * problems with sparse file writing.
   struct stat SBuf;
   if (stat(Itm->DestFile.c_str(),&SBuf) >= 0 && SBuf.st_size > 0)
   {
      // In this case we send an if-range query with a range header
      sprintf(Buf,"Range: bytes=%li-\r\nIf-Range: %s\r\n",(long)SBuf.st_size - 1,
	      TimeRFC1123(SBuf.st_mtime).c_str());
      Req += Buf;
   }
   else
   { */
      if (Itm->LastModified != 0)
      {
	 sprintf(Buf,"If-Modified-Since: %s\r\n",TimeRFC1123(Itm->LastModified).c_str());
	 Req += Buf;
      }
   /*   } */

   if (Uri.User.empty() == false || Uri.Password.empty() == false)
      Req += string("Authorization: Basic ") + 
          Base64Encode(Uri.User + ":" + Uri.Password) + "\r\n";
   
   Req += "User-Agent: Debian APT-DEBTORRENT/0.2 ("VERSION")\r\n\r\n";
   
   if (Debug == true)
      cerr << Req << endl;

   Out.Read(Req);
}
									/*}}}*/
// HttpMethod::Go - Run a single loop					/*{{{*/
// ---------------------------------------------------------------------
/* This runs the select loop over the server FDs, Output file FDs and
   stdin. */
bool HttpMethod::Go(bool ToFile,ServerState *Srv)
{
   // Server has closed the connection
   if (Srv->ServerFd == -1 && (Srv->In.WriteSpace() == false || 
			       ToFile == false))
      return false;
   
   fd_set rfds,wfds;
   FD_ZERO(&rfds);
   FD_ZERO(&wfds);
   
   /* Add the server. We only send more requests if the connection will 
      be persisting */
   if (Srv->Out.WriteSpace() == true && Srv->ServerFd != -1 
       && Srv->Persistent == true)
      FD_SET(Srv->ServerFd,&wfds);
   if (Srv->In.ReadSpace() == true && Srv->ServerFd != -1)
      FD_SET(Srv->ServerFd,&rfds);
   
   // Add the file
   int FileFD = -1;
   if (File != 0)
      FileFD = File->Fd();
   
   if (Srv->In.WriteSpace() == true && ToFile == true && FileFD != -1)
      FD_SET(FileFD,&wfds);
   
   // Add stdin
   FD_SET(STDIN_FILENO,&rfds);
	  
   // Figure out the max fd
   int MaxFd = FileFD;
   if (MaxFd < Srv->ServerFd)
      MaxFd = Srv->ServerFd;

   // Select
   struct timeval tv;
   tv.tv_sec = PollTimeOut;
   tv.tv_usec = 0;
   int Res = 0;
   if ((Res = select(MaxFd+1,&rfds,&wfds,0,&tv)) < 0)
   {
      if (errno == EINTR)
	 return true;
      return _error->Errno("select","Select failed");
   }
   
   if (Res == 0)
   {
      _error->Error("Connection timed out");
      return ServerDie(Srv);
   }
   
   // Handle server IO
   if (Srv->ServerFd != -1 && FD_ISSET(Srv->ServerFd,&rfds))
   {
      errno = 0;
      if (Srv->In.Read(Srv->ServerFd) == false)
	 return ServerDie(Srv);
   }
	 
   if (Srv->ServerFd != -1 && FD_ISSET(Srv->ServerFd,&wfds))
   {
      errno = 0;
      if (Srv->Out.Write(Srv->ServerFd) == false)
	 return ServerDie(Srv);
   }

   // Send data to the file
   if (FileFD != -1 && FD_ISSET(FileFD,&wfds))
   {
      if (Srv->In.Write(FileFD) == false)
	 return _error->Errno("write","Error writing to output file");
   }

   // Handle commands from APT
   if (FD_ISSET(STDIN_FILENO,&rfds))
   {
      if (Run(true) != -1)
	 exit(100);
   }   
       
   return true;
}
									/*}}}*/
// HttpMethod::Flush - Dump the buffer into the file			/*{{{*/
// ---------------------------------------------------------------------
/* This takes the current input buffer from the Server FD and writes it
   into the file */
bool HttpMethod::Flush(ServerState *Srv)
{
   if (File != 0)
   {
      // on GNU/kFreeBSD, apt dies on /dev/null because non-blocking
      // can't be set
      if (File->Name() != "/dev/null")
	 SetNonBlock(File->Fd(),false);
      if (Srv->In.WriteSpace() == false)
	 return true;
      
      while (Srv->In.WriteSpace() == true)
      {
	 if (Srv->In.Write(File->Fd()) == false)
	    return _error->Errno("write","Error writing to file");
	 if (Srv->In.IsLimit() == true)
	    return true;
      }

      if (Srv->In.IsLimit() == true || Srv->Encoding == ServerState::Closes)
	 return true;
   }
   return false;
}
									/*}}}*/
// HttpMethod::ServerDie - The server has closed the connection.	/*{{{*/
// ---------------------------------------------------------------------
/* */
bool HttpMethod::ServerDie(ServerState *Srv)
{
   unsigned int LErrno = errno;
   
   // Dump the buffer to the file
   if (Srv->State == ServerState::Data)
   {
      // on GNU/kFreeBSD, apt dies on /dev/null because non-blocking
      // can't be set
      if (File->Name() != "/dev/null")
	 SetNonBlock(File->Fd(),false);
      while (Srv->In.WriteSpace() == true)
      {
	 if (Srv->In.Write(File->Fd()) == false)
	    return _error->Errno("write","Error writing to the file");

	 // Done
	 if (Srv->In.IsLimit() == true)
	    return true;
      }
   }
   
   // See if this is because the server finished the data stream
   if (Srv->In.IsLimit() == false && Srv->State != ServerState::Header && 
       Srv->Encoding != ServerState::Closes)
   {
      Srv->Close();
      if (LErrno == 0)
	 return _error->Error("Error reading from DebTorrent client. Remote end closed connection");
      errno = LErrno;
      return _error->Errno("read","Error reading from DebTorrent client");
   }
   else
   {
      Srv->In.Limit(-1);

      // Nothing left in the buffer
      if (Srv->In.WriteSpace() == false)
	 return false;
      
      // We may have got multiple responses back in one packet..
      Srv->Close();
      return true;
   }
   
   return false;
}
									/*}}}*/
// HttpMethod::RearrangeQueue - Move the item returned to the top	/*{{{*/
// ---------------------------------------------------------------------
/* Look at the ResultPath we got back from the server and find the
   item to move to the top. Returns false if the item is not in the
   queue.  */
bool HttpMethod::RearrangeQueue(ServerState *Srv)
{
   URI Uri;
   FetchItem *PrevI;
   PrevI = 0;

   // Search the queue for the returned item
   for (FetchItem *I = Queue; I != 0; I = I->Next)
   {
      Uri = I->Uri;

      // If this is the end of the queue (QueueBack always points to the
      // next item to request), then the item is not found.
      if (I == QueueBack || I == 0)
      {
	 break;
      }

      // Compare this item's path with the returned path
      if (stringcmp(QuoteString(Uri.Path,"~"),Srv->ResultPath) == 0)
      {
      	 if (PrevI == 0)
      	 {
	    // The item is already at the head of the queue
	    return true;
      	 }

	 // Pop the item out of the queue
	 PrevI->Next = I->Next;
	 
	 // Add it to the front of the queue
	 I->Next = Queue;
	 Queue = I;
	 
	 return true;
      }
      
      PrevI = I;
   }
   
   // The item was not found
   return false;
}
									/*}}}*/
// HttpMethod::DealWithHeaders - Handle the retrieved header data	/*{{{*/
// ---------------------------------------------------------------------
/* We look at the header data we got back from the server and decide what
   to do. Returns 
    -1 - Piece downloaded update
     0 - File is open,
     1 - IMS hit
     3 - Unrecoverable error 
     4 - Error with error content page
     5 - Unrecoverable non-server error (close the connection) */
int HttpMethod::DealWithHeaders(FetchResult &Res,ServerState *Srv)
{
   // Piece Downloaded update message
   if (Srv->Result == 102)
   {
      // Hack: use ResumePoint to temporarily store the current size 
      Res.Size = Srv->Size;
      Res.ResumePoint = Srv->StartPos;
      return -1;
   }

   // Not Modified
   if (Srv->Result == 304)
   {
      unlink(Queue->DestFile.c_str());
      Res.IMSHit = true;
      Res.LastModified = Queue->LastModified;
      return 1;
   }
   
   /* We have a reply we dont handle. This should indicate a perm server
      failure */
   if (Srv->Result < 200 || Srv->Result >= 300)
   {
      _error->Error("%u %s",Srv->Result,Srv->Code);
      if (Srv->HaveContent == true)
	 return 4;
      return 3;
   }

   // This is some sort of 2xx 'data follows' reply
   Res.LastModified = Srv->Date;
   Res.Size = Srv->Size;
   
   // Open the file
   delete File;
   File = new FileFd(Queue->DestFile,FileFd::WriteEmpty);
   if (_error->PendingError() == true)
      return 5;

   FailFile = Queue->DestFile;
   FailFile.c_str();   // Make sure we dont do a malloc in the signal handler
   FailFd = File->Fd();
   FailTime = Srv->Date;
      
   // Set the expected size
   if (Srv->StartPos >= 0)
   {
      Res.ResumePoint = Srv->StartPos;
      ftruncate(File->Fd(),Srv->StartPos);
   }
      
   // Set the start point
   lseek(File->Fd(),0,SEEK_END);

   delete Srv->In.Hash;
   Srv->In.Hash = new Hashes;
   
   // Fill the Hash if the file is non-empty (resume)
   if (Srv->StartPos > 0)
   {
      lseek(File->Fd(),0,SEEK_SET);
      if (Srv->In.Hash->AddFD(File->Fd(),Srv->StartPos) == false)
      {
	 _error->Errno("read","Problem hashing file");
	 return 5;
      }
      lseek(File->Fd(),0,SEEK_END);
   }
   
   SetNonBlock(File->Fd(),true);
   return 0;
}
                                                                        /*}}}*/
// HttpMethod::UpdateFileSize - Update the output file's size           /*{{{*/
// ---------------------------------------------------------------------
/* Increase the output file's size (sparsely) to indicate the current 
   amount of the file that has been downloaded by DebTorrent. Returns
   false if an error occurred. */
bool HttpMethod::UpdateFileSize(FetchResult &Res, bool Started)
{
   // Open the file (reset to empty if not started)
   if (Started)
      File = new FileFd(Res.Filename, FileFd::WriteAny);
   else
      File = new FileFd(Res.Filename, FileFd::WriteEmpty);
   if (_error->PendingError() == true)
   {
      delete File;
      File = 0;
      return false;
   }

   // Get the old size of the file
   unsigned long OldSize = File->Size();
   if (_error->PendingError() == true)
   {
      delete File;
      File = 0;
      return false;
   }

   // Don't make files smaller (it will confuse apt)
   if (OldSize >= Res.ResumePoint)
   {
      delete File;
      File = 0;
      return true;
   }
   
   // Set the file to the new length
   File->Truncate(Res.ResumePoint);
   if (_error->PendingError() == true)
   {
      delete File;
      File = 0;
      return false;
   }

   // Sync the file to disk and close it
   File->Sync();
   delete File;
   File = 0;
   if (_error->PendingError() == true)
   {
      return false;
   }

   if (Debug == true)
   {
      char Buf[1000];
      sprintf(Buf,"UpdateFileSize: %s was %ld, set to %ld/%ld\n",
	      Res.Filename.c_str(), OldSize, Res.ResumePoint, Res.Size);
      string Req = Buf;
      cerr << Req << endl;
   }
   
   return true;
}
									/*}}}*/
// HttpMethod::SigTerm - Handle a fatal signal				/*{{{*/
// ---------------------------------------------------------------------
/* This closes and timestamps the open file. This is neccessary to get 
   resume behavoir on user abort */
void HttpMethod::SigTerm(int)
{
   if (FailFd == -1)
      _exit(100);
   close(FailFd);
   
   // Timestamp
   struct utimbuf UBuf;
   UBuf.actime = FailTime;
   UBuf.modtime = FailTime;
   utime(FailFile.c_str(),&UBuf);
   
   _exit(100);
}
									/*}}}*/
// HttpMethod::Fetch - Fetch an item					/*{{{*/
// ---------------------------------------------------------------------
/* This adds an item to the pipeline. We keep the pipeline at a fixed
   depth. */
bool HttpMethod::Fetch(FetchItem *)
{
   if (Server == 0)
      return true;

   // Queue the requests
   int Depth = -1;
   for (FetchItem *I = Queue; I != 0; I = I->Next, Depth++)
   {
      // Make sure we stick with the same server
      if (Server->Comp(I->Uri) == false)
	 break;
      if (QueueBack == I)
      {
	 QueueBack = I->Next;
	 
	 // Initialize the unstarted flag
	 I->IndexFile = false;
	 
	 SendReq(I,Server->Out);
	 continue;
      }
   }
   
   return true;
};
									/*}}}*/
// HttpMethod::Configuration - Handle a configuration message		/*{{{*/
// ---------------------------------------------------------------------
/* We stash the desired pipeline depth */
bool HttpMethod::Configuration(string Message)
{
   if (pkgAcqMethod::Configuration(Message) == false)
      return false;
   
   PollTimeOut = _config->FindI("Acquire::debtorrent::Timeout",PollTimeOut);
   Debug = _config->FindB("Debug::Acquire::debtorrent",false);
   
   return true;
}
									/*}}}*/
// HttpMethod::Loop - Main loop						/*{{{*/
// ---------------------------------------------------------------------
/* */
int HttpMethod::Loop()
{
   signal(SIGTERM,SigTerm);
   signal(SIGINT,SigTerm);
   
   Server = 0;
   
   int FailCounter = 0;
   while (1)
   {      
      // We have no commands, wait for some to arrive
      if (Queue == 0)
      {
	 if (WaitFd(STDIN_FILENO) == false)
	    return 0;
      }
      
      /* Run messages, we can accept 0 (no message) if we didn't
         do a WaitFd above.. Otherwise the FD is closed. */
      int Result = Run(true);
      if (Result != -1 && (Result != 0 || Queue == 0))
	 return 100;

      if (Queue == 0)
	 continue;
      
      // Connect to the server
      if (Server == 0 || Server->Comp(Queue->Uri) == false)
      {
	 delete Server;
	 Server = new ServerState(Queue->Uri,this);
      }
      /* If the server has explicitly said this is the last connection
         then we pre-emptively shut down the pipeline and tear down 
	 the connection. This will speed up HTTP/1.0 servers a tad
	 since we don't have to wait for the close sequence to
         complete */
      if (Server->Persistent == false)
	 Server->Close();
      
      // Reset the pipeline
      if (Server->ServerFd == -1)
	 QueueBack = Queue;	 
	 
      // Connnect to the host
      if (Server->Open() == false)
      {
	 Fail(true);
	 delete Server;
	 Server = 0;
	 continue;
      }

      // Fill the pipeline.
      Fetch(0);
      
      // Fetch the next URL header data from the server.
      switch (Server->RunHeaders())
      {
	 case 0:
	 break;
	 
	 // The header data is bad
	 case 2:
	 {
	    _error->Error("Bad header data");
	    Fail(true);
	    RotateDNS();
	    continue;
	 }
	 
	 // The server closed a connection during the header get..
	 default:
	 case 1:
	 {
	    FailCounter++;
	    _error->Discard();
	    Server->Close();
	    
	    if (FailCounter >= 3)
	    {
	       Fail("Connection failed",true);
	       FailCounter = 0;
	    }
	    
	    RotateDNS();
	    continue;
	 }
      };

      // Decide what to do.
      FetchResult Res;
      if (RearrangeQueue(Server))
      {
	 Res.Filename = Queue->DestFile;
	 switch (DealWithHeaders(Res,Server))
	 {
	    // Got an informational file size message
	    case -1:
	    {
	       // Set the file to the new length
	       if (UpdateFileSize(Res, Queue->IndexFile) == false)
	       {
		  Fail();
		  RotateDNS();
		  Server->Close();
	       }
	       // Hack: use the IndexFile flag to indicate if it's been started
	       else if (Queue->IndexFile == false)
	       {
		  // Make sure to unset the ResumePoint for the call to URIStart
		  Res.ResumePoint = 0;
		  
		  URIStart(Res);
		  
		  Queue->IndexFile = true;
	       }

	       break;
	    }

	    // Ok, the file is Open
	    case 0:
	    {
	       // Make sure to send the URIStart first
	       if (Queue->IndexFile == false)
	       {
		  URIStart(Res);
		  Queue->IndexFile = true;
	       }

	       // Run the data
	       bool Result =  Server->RunData();

	       /* If the server is sending back sizeless responses then fill in
		  the size now */
	       if (Res.Size == 0)
		  Res.Size = File->Size();
	    
	       // Close the file, destroy the FD object and timestamp it
	       FailFd = -1;
	       delete File;
	       File = 0;
	    
	       // Timestamp
	       struct utimbuf UBuf;
	       time(&UBuf.actime);
	       UBuf.actime = Server->Date;
	       UBuf.modtime = Server->Date;
	       utime(Queue->DestFile.c_str(),&UBuf);

	       // Send status to APT
	       if (Result == true)
	       {
		  Res.TakeHashes(*Server->In.Hash);
		  URIDone(Res);
	       }
	       else
	       {
		  if (Server->ServerFd == -1)
		  {
		     FailCounter++;
		     _error->Discard();
		     Server->Close();
		  
		     if (FailCounter >= 3)
		     {
		        Fail("Connection failed",true);
		        FailCounter = 0;
		     }
		  
		     QueueBack = Queue;
	          }
	          else
		     Fail(true);
	       }
	       break;
	    }
	 
	    // IMS hit
	    case 1:
	    {
	       URIDone(Res);
	       break;
	    }
	 
	    // Hard server error, not found or something
	    case 3:
	    {
	       Fail();
	       break;
	    }
	  
	    // Hard internal error, kill the connection and fail
	    case 5:
	    {
	       delete File;
	       File = 0;

	       Fail();
	       RotateDNS();
	       Server->Close();
	       break;
	    }

	    // We need to flush the data, the header is like a 404 w/ error text
	    case 4:
	    {
	       Fail();
	    
	       // Send to content to dev/null
	       File = new FileFd("/dev/null",FileFd::WriteExists);
	       Server->RunData();
	       delete File;
	       File = 0;
	       break;
	    }
	 
	    default:
	    Fail("Internal error");
	    break;
	 }
      }
      else
      {
	 // The result could not be found in the requests
	 Fail("Data returned for an unrequested file");
	    
	 // Send to content to dev/null
	 File = new FileFd("/dev/null",FileFd::WriteExists);
	 Server->RunData();
	 delete File;
	 File = 0;

	 RotateDNS();
	 Server->Close();
      }
      FailCounter = 0;
   }
   
   return 0;
}
									/*}}}*/

int main()
{
   setlocale(LC_ALL, "");

   HttpMethod Mth;
   
   return Mth.Loop();
}


