/* $Id: ArkMaterial.h,v 1.21 2003/03/19 09:25:05 zongo Exp $
**
** Ark - Libraries, Tools & Programs for MMORPG developpements.
** Copyright (C) 1999-2002 The Contributors of the Ark Project
** Please see the file "AUTHORS" for a list of contributors
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#ifndef ARK_MATERIAL_H
#define ARK_MATERIAL_H

#include <Ark/Ark.h>
#include <Ark/ArkObject.h>
#include <Ark/ArkTexture.h>
#include <Ark/ArkMath.h>

#include <vector>

namespace Ark
{
    // Forward declaration and typedefs
    class Material;
    typedef Ptr<Material> MaterialPtr;
    typedef std::vector<MaterialPtr> MaterialList;

   
   const int MAT_MAXPASSES = 4;

   class Lexer;
   class Renderer;

   enum PassFlags
   {
      PASS_HAS_TEXTURE = (1<<0),
      PASS_HAS_TEXTUREANIM = (1<<1),
      PASS_HAS_TEXTUREGEN = (1<<2),
      PASS_HAS_TEXTUREENV = (1<<3),
      PASS_HAS_BLENDING = (1<<4),
      PASS_HAS_DEPTHFUNC = (1<<5),
      PASS_HAS_DEPTHWRITE = (1<<6),
      PASS_HAS_ALPHATEST = (1<<7)
   };

   // Blend functions. Warning : if you change the values of the
   // enum, you'll also have to change them in renderer specific
   // sources...
   enum BlendFunc
   {
      BLEND_UNDEFINED,
      BLEND_ONE,
      BLEND_ZERO,
      BLEND_DST_COLOR,
      BLEND_ONE_MINUS_DST_COLOR,
      BLEND_SRC_ALPHA,
      BLEND_ONE_MINUS_SRC_ALPHA,
      BLEND_DST_ALPHA,
      BLEND_ONE_MINUS_DST_ALPHA,
      BLEND_SRC_COLOR,
      BLEND_ONE_MINUS_SRC_COLOR,
      BLEND_CONSTANT_ALPHA,
      BLEND_ONE_MINUS_CONSTANT_ALPHA,
   };

   enum DepthFunc
   {
      DEPTHFUNC_UNDEFINED,
      DEPTHFUNC_ALWAYS,
      DEPTHFUNC_LEQUAL,
      DEPTHFUNC_EQUAL,
      DEPTHFUNC_GEQUAL,
   };

   enum MapRepeatMode
   {
      TEXTURE_REPEAT_REPEAT,
      TEXTURE_REPEAT_CLAMP,
      TEXTURE_REPEAT_UNKNOWN,
   };

   enum TextureEnv
   {
      TEXTUREENV_UNDEFINED,
      TEXTUREENV_MODULATE,
      TEXTUREENV_REPLACE,
      TEXTUREENV_DECAL
   };

   enum TextureGen
   {
      TEXTUREGEN_UNDEFINED,
      TEXTUREGEN_ENVIRONMENT,
      TEXTUREGEN_LINEAR
   };

   union TexgenData
   {
	 struct
	 {
	       scalar m_PlaneS[4];
	       scalar m_PlaneT[4];
	 } m_Linear;
   };

   /**
    * A shader pass contains all data needed to do a single-pass object
    * rendering. As for now it's quite limited since a pass rendering is
    * only rendering the object with a texture : a multipass-rendered object
    * is simply a mulitexture object.
    */
   class ARK_DLL_API ShaderPass
   {
	 // Disable copy
	 ShaderPass (const ShaderPass &src);
	 ShaderPass &operator = (const ShaderPass &src);

      public:
	 /// An or'ed bitfield of PASS_HAS_* values.
	 int m_Flags;


	 /// Number of frame per seconds in case of an animated texture.
	 int m_TexPerSec;
	 
	 /// Array of animation frames.
	 TextureList m_AnimTex;
	 
	 /// Pointer to the pass texture.
	 TexturePtr m_Texture;


	 /// Environment value for the texture.
	 TextureEnv m_TextureEnv;
	 /// Texture coordinate generation (if PASS_HAS_TEXTUREGEN is set).
	 TextureGen m_TextureGen;
	 /**
	  * Texture coordinate generation data (ie plane to which projection
	  * in case of TEXTURE_GEN_LINEAR, etc).
	  */
	 TexgenData m_TextureGenData;

	 /// Source and destination blend functions.
	 BlendFunc m_BlendSrc, m_BlendDst;

	 /// Depth function.
	 DepthFunc m_DepthFunc;

	 /// Texture repeat mode.
	 MapRepeatMode m_RepeatMode;

	 /**
	  * Tells wether the the depth buffer should be tested when rendering
	  * using this pass. This field is taken into account only if
	  * PASS_HAS_DEPTHFUNC is set.
	  */
	 bool m_DepthTest;
	 
	 /**
	  * Tells wether the the depth should buffer be filled when rendering
	  * using this pass. This field is taken into account only if
	  * PASS_HAS_DEPTHWRITE is set.
	  */
	 bool m_DepthWrite;
	 
	 /**
	  * If blending is done with this material, set the object color
	  * to m_BlendColor (if there's no per-vertex color in the vertex
	  * buffer, of course).
	  */
	 Color m_BlendColor;

	 /**
	  * Should a pixel be drawn if the alpha value is above/below a given
	  * level. (this is important if you want to be able to draw alpha
	  * blended objects without having to sort their polygons). This
	  * field is taken into account only if PASS_HAS_ALPHAFUNC is set.
	  */
	 DepthFunc m_AlphaFunc;

	 /// Alpha value under/above which a pixel isn't drawn.
	 scalar m_AlphaValue;

	 /**
	  * Fill this pass with values from a given lexer, and getting
	  * textures from the given cache.
	  */
	 bool Parse (Lexer &from, Cache &cache);
	 
      public:
	 /// Create an empty ShaderPass (ie its flag is set to 0).
	 ShaderPass ();

	 /**
	  * Create a ShaderPass with a simple texture mapping with the given
	  * texture.
	  */
	 ShaderPass (const TexturePtr& texture);

	 /// Destroy the pass.
	 ~ShaderPass ();

	 /// Return the texture of the pass.
	 Texture* GetTexture()
	    {return &*m_Texture;}
   };

   enum
   {
      /// Set if material has ambient, diffuse and specular colors.
      MATERIAL_HAS_LIGHTING = (1<<0),

      MATERIAL_HAS_PASS1 = (1<<1),
      MATERIAL_HAS_PASS2 = (1<<2),
      MATERIAL_HAS_PASS3 = (1<<3),
      MATERIAL_HAS_PASS4 = (1<<4),

      /// Set if one can walk on this material
      MATERIAL_IS_WALKABLE = (1<<5),

      /// Set if no backface culling should be done.
      MATERIAL_IS_DOUBLESIDED = (1<<6)
   };

   /**
    * A material is a list of pass that will be used to render an object. It
    * also have some informations on objects characteristics.
    */
   class ARK_DLL_API Material : public Object
   {
	 // Disable cop
	 Material &operator = (const Material &src);
	 Material (const Material &src);

      public:
	 /**
	  * Decorated name of the material, to be displayed in the world
	  * editor.
	  */
	 String m_DecorName;

	 /**
	  * This a bitfield of MATERIAL_* flags to specify which informations
	  * are available in this material.
	  */
	 int m_Flags;

	 /// Color of the ambient reflection for this material.
	 Color m_Ambient;
	 /// Color of the diffuse reflection for this material.
	 Color m_Diffuse;
	 /// Color of the specular reflection for this material.
	 Color m_Specular;

	 /// Rendering passes for the material.
	 ShaderPass m_Passes[MAT_MAXPASSES];

	 /**
	  * Fill this material with values read from the given stream, and
	  * getting textures from the given cache.
	  */
         bool Parse (const String& name, Lexer &lexer, Cache &cache);
         bool Parse (const String& name, Stream &stream, Cache &cache);	 
      public:
	 /// Create an empty material with the given name.
	 Material(const String &name);

	 /// Destroy the material, unreference all textures.
	 ~Material();

	 /// Returns a string describing the material;
	 String Dump (bool long_version);

	 /// Return the number of passes.
	 int NumPasses() const;
   };

}

#endif
