/* $Id: ArkMath.cpp,v 1.29 2003/03/20 17:23:25 zongo Exp $
**
** Ark - Libraries, Tools & Programs for MMORPG developpements.
** Copyright (C) 1999-2002 The Contributors of the Ark Project
** Please see the file "AUTHORS" for a list of contributors
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/


#include <Ark/ArkMath.h>
#include <Ark/ArkSystem.h>

#include <math.h>

namespace Ark
{

/*====================================================================*/
/*======================================================== Math Stuff */
/*====================================================================*/
void
Math::Init ()
{
	// Body is in Ark/ArkInvSqrt.cpp
	MakeInvSqrtTable ();
}

/*====================================================================*/
/*============================================== 3D-Vector operations */
/*====================================================================*/

Vector3 &Vector3::Normalize  ()
{
  const scalar length2 = X*X + Y*Y + Z*Z;

  if (length2 != 0.0f)
    Scale (Math::InvSqrt (length2));
  else
  {
    /* FIXME: should be at infinite */
    X = 0;
    Y = 0;
    Z = 0;
  }

  return *this;
}

Vector3
Vector3::GetUnitVector () const
{
  Vector3 R = *this;

  return R.Normalize();
}


Vector3
Vector3::ComputeNormal (const Vector3 &A,
                        const Vector3 &B,
                        const Vector3 &C)
{
  Vector3 R;

  R = (B - A) ^ (C - A);
  R.Normalize (); 

  return R;
}

scalar Vector3::GetMagnitude() const
{
  return sqrtf(GetMagnitude2());
}

scalar Vector3::GetMagnitude2() const
{
  return (*this) * (*this);
}

Vector3 &Vector3::operator += (const Vector3 &B)
{
  X += B.X;
  Y += B.Y;
  Z += B.Z;
  return *this;
}

Vector3 &Vector3::operator -= (const Vector3 &B)
{
  X -= B.X;
  Y -= B.Y;
  Z -= B.Z;
  return *this;
}

Vector3 &Vector3::operator ^= (const Vector3 &B)
{
  Vector3 A = *this;
  X = (A.Y * B.Z) - (A.Z * B.Y);
  Y = (A.Z * B.X) - (A.X * B.Z);
  Z = (A.X * B.Y) - (A.Y * B.X);
  return *this;
}

Vector3 &Vector3::Scale (scalar B)
{
  X *= B;
  Y *= B;
  Z *= B;

  return *this;
}

Vector3 &Vector3::Scale (const Vector3 &B)
{
  X *= B.X;
  Y *= B.Y;
  Z *= B.Z;

  return *this;
}

Vector3 Vector3::operator + (const Vector3 &B) const
{
  return Vector3 (X + B.X, Y + B.Y, Z + B.Z);
}

Vector3 Vector3::operator - (const Vector3 &B) const
{
  return Vector3 (X - B.X, Y - B.Y, Z - B.Z);
}

/* Dot product */
scalar Vector3::operator * (const Vector3 &B) const
{
  return X * B.X + Y * B.Y + Z * B.Z;
}

/* Cross product */
Vector3 Vector3::operator ^ (const Vector3 &B) const
{
  Vector3 R = *this;
  R ^= B;
  return R;
}

/*====================================================================*/
/*============================================== 2D-Vector operations */
/*====================================================================*/
void Vector2::Normalize  ()
{
  scalar length2 = (*this) * (*this);

  if (length2 != 0.0f)
    Scale (Math::InvSqrt (length2));
  else
  {
    /* FIXME: should be at infinite */
    X = 0;
    Y = 0;
  }
}

scalar Vector2::GetMagnitude() const
{
  return sqrtf(GetMagnitude2());
}

scalar Vector2::GetMagnitude2() const
{
  return (*this) * (*this);
}

Vector2 &Vector2::operator += (const Vector2 &B)
{
  X += B.X;
  Y += B.Y;
  return *this;
}

Vector2 &Vector2::operator -= (const Vector2 &B)
{
  X -= B.X;
  Y -= B.Y;
  return *this;
}

Vector2 &Vector2::Scale (scalar B)
{
  X *= B;
  Y *= B;

  return *this;
}

Vector2 &Vector2::Scale (const Vector2 &B)
{
  X *= B.X;
  Y *= B.Y;

  return *this;
}

Vector2 Vector2::operator + (const Vector2 &B) const
{
  return Vector2 (X + B.X, Y + B.Y);
}

Vector2 Vector2::operator - (const Vector2 &B) const
{
  return Vector2 (X - B.X, Y - B.Y);
}

/* Dot product */
scalar Vector2::operator * (const Vector2 &B) const
{
  return X * B.X + Y * B.Y;
}

/*====================================================================*/
/*================================================= Matrix operations */
/*====================================================================*/
static const scalar IdentityMatrixElements[16] = {
	1.f, 0.f, 0.f, 0.f,
	0.f, 1.f, 0.f, 0.f,
	0.f, 0.f, 1.f, 0.f,
	0.f, 0.f, 0.f, 1.f,
};

void Matrix44::MakeIdentity()
{
	memcpy(&m_Elems[0], &IdentityMatrixElements[0], sizeof(IdentityMatrixElements));
}

void Matrix44::Multiply (const Matrix44 &B)
{
  Matrix44 A = *this;
  int x, y;

  for (x = 0; x < 4; x ++)
  {
    for(y = 0; y < 4; y ++)
    {
      M(x,y) = (A.M(x,0) * B.M(0,y)) +
               (A.M(x,1) * B.M(1,y)) +
               (A.M(x,2) * B.M(2,y)) +
               (A.M(x,3) * B.M(3,y));
    }
  }
}

/*
  1     0        0         0
  0     cos(rx)  sin(rx)   0
  0     -sin(rx) cos(rx)   0
  0     0        0         1
*/
void Matrix44::MakeRotationX(scalar angle)
{
	const float rad = Math::DegToRad(angle);
	const float s = sinf(rad);
	const float c = cosf(rad);

  MakeIdentity();

  M(1,1) =  c;
  M(1,2) =  s;
  M(2,1) = -s;
  M(2,2) =  c;
}

/*
  cos(ry)  0     -sin(ry)   0
  0        1     0          0
  sin(ry)  0     cos(ry)    0
  0        0     0          1
*/
void Matrix44::MakeRotationY(scalar angle)
{
	const float rad = Math::DegToRad(angle);
	const float s = sinf(rad);
	const float c = cosf(rad);

  MakeIdentity();

  M(0,0) =  c;
  M(0,2) = -s;
  M(2,0) =  s;
  M(2,2) =  c;
}

/*
  cos(rz)  sin(rz)   0      0
  -sin(rz) cos(rz)   0      0
  0        0         1      0
  0        0         0      1
*/
void Matrix44::MakeRotationZ(scalar angle)
{
	const float rad = Math::DegToRad(angle);
	const float s = sinf(rad);
	const float c = cosf(rad);

  MakeIdentity();

  M(0,0) =  c;
  M(0,1) =  s;
  M(1,0) = -s;
  M(1,1) =  c;
}

void Matrix44::MakeTranslation(const Vector3 &pos)
{
  MakeIdentity();

  M(3,0) = pos.X;
  M(3,1) = pos.Y;
  M(3,2) = pos.Z;
}

void Matrix44::MakeScale(const Vector3 &factors)
{
  MakeIdentity();

  M(0,0) = factors.X;
  M(1,1) = factors.Y;
  M(2,2) = factors.Z;
}

void 
Matrix44::PostTranslate(const Vector3& vector)
{
	const scalar projX = M(0,3);
	M(0,0) += projX * vector.X;
	M(0,1) += projX * vector.Y;
	M(0,2) += projX * vector.Z;

	const scalar projY = M(1,3);
	M(1,0) += projY * vector.X;
	M(1,1) += projY * vector.Y;
	M(1,2) += projY * vector.Z;

	const scalar projZ = M(2,3);
	M(2,0) += projZ * vector.X;
	M(2,1) += projZ * vector.Y;
	M(2,2) += projZ * vector.Z;

	const scalar projW = M(3,3);
	M(3,0) += projW * vector.X;
	M(3,1) += projW * vector.Y;
	M(3,2) += projW * vector.Z;
#if 0
  M(3,0) += translation.X;
  M(3,1) += translation.Y;
  M(3,2) += translation.Z;
#endif
}

Vector3 Matrix44::Transform (const Vector3 &vector) const
{
  return Vector3
  (
    vector.X * M(0,0) + vector.Y * M(1,0) + vector.Z * M(2,0) + M(3,0),
    vector.X * M(0,1) + vector.Y * M(1,1) + vector.Z * M(2,1) + M(3,1),
    vector.X * M(0,2) + vector.Y * M(1,2) + vector.Z * M(2,2) + M(3,2)
  );
}

/* Rotate only */
Vector3 Matrix44::Rotate (const Vector3 &vector) const
{
  return Vector3
  (
    vector.X * M(0,0) + vector.Y * M(1,0) + vector.Z * M(2,0),
    vector.X * M(0,1) + vector.Y * M(1,1) + vector.Z * M(2,1),
    vector.X * M(0,2) + vector.Y * M(1,2) + vector.Z * M(2,2)
  );
}

Vector3 Matrix44::InvRotate (const Vector3 &vector) const
{
  return Vector3
  (
    vector.X * M(0,0) + vector.Y * M(0,1) + vector.Z * M(0,2),
    vector.X * M(1,0) + vector.Y * M(1,1) + vector.Z * M(1,2),
    vector.X * M(2,0) + vector.Y * M(2,1) + vector.Z * M(2,2)
  );
}

bool Matrix44::Invert ()
{
  Matrix44 m = *this;

  scalar wtmp[4][8];
  scalar m0, m1, m2, m3, s;
  scalar *r0, *r1, *r2, *r3;

  /* NB. OpenGL Matrices are COLUMN major. */
  #define SWAP_ROWS(a, b) { scalar *_tmp = a; (a)=(b); (b)=_tmp; }

  r0 = wtmp[0], r1 = wtmp[1], r2 = wtmp[2], r3 = wtmp[3];

  r0[0] = m.M(0,0), r0[1] = m.M(0,1),
  r0[2] = m.M(0,2), r0[3] = m.M(0,3),
  r0[4] = 1.0f, r0[5] = r0[6] = r0[7] = 0.0f,

  r1[0] = m.M(1,0), r1[1] = m.M(1,1),
  r1[2] = m.M(1,2), r1[3] = m.M(1,3),
  r1[5] = 1.0f, r1[4] = r1[6] = r1[7] = 0.0f,

  r2[0] = m.M(2,0), r2[1] = m.M(2,1),
  r2[2] = m.M(2,2), r2[3] = m.M(2,3),
  r2[6] = 1.0f, r2[4] = r2[5] = r2[7] = 0.0f,

  r3[0] = m.M(3,0), r3[1] = m.M(3,1),
  r3[2] = m.M(3,2), r3[3] = m.M(3,3),
  r3[7] = 1.0f, r3[4] = r3[5] = r3[6] = 0.0f;

  /* choose pivot - or die */
  if (fabs (r3 [0]) > fabs (r2 [0])) SWAP_ROWS(r3, r2);
  if (fabs (r2 [0]) > fabs (r1 [0])) SWAP_ROWS(r2, r1);
  if (fabs (r1 [0]) > fabs (r0 [0])) SWAP_ROWS(r1, r0);

  if (r0[0] == 0.0f)  return false;

  /* Eliminate first variable     */
  m1 = r1[0]/r0[0]; m2 = r2[0]/r0[0]; m3 = r3[0]/r0[0];
  s = r0[1]; r1[1] -= m1 * s; r2[1] -= m2 * s; r3[1] -= m3 * s;
  s = r0[2]; r1[2] -= m1 * s; r2[2] -= m2 * s; r3[2] -= m3 * s;
  s = r0[3]; r1[3] -= m1 * s; r2[3] -= m2 * s; r3[3] -= m3 * s;

  s = r0[4];
  if (s != 0.0f)
  {
    r1[4] -= m1 * s;
    r2[4] -= m2 * s;
    r3[4] -= m3 * s;
  }

  s = r0[5];
  if (s != 0.0f)
  {
    r1[5] -= m1 * s;
    r2[5] -= m2 * s;
    r3[5] -= m3 * s;
  }

  s = r0[6];
  if (s != 0.0f)
  {
    r1[6] -= m1 * s;
    r2[6] -= m2 * s;
    r3[6] -= m3 * s;
  }

  s = r0[7];
  if (s != 0.0f)
  {
    r1[7] -= m1 * s;
    r2[7] -= m2 * s;
    r3[7] -= m3 * s;
  }

  /* Choose pivot - or die */
  if (fabs (r3 [1]) > fabs(r2 [1])) SWAP_ROWS(r3, r2);
  if (fabs (r2 [1]) > fabs(r1 [1])) SWAP_ROWS(r2, r1);
  if (r1[1] == 0.0f) return false;

  /* Eliminate second variable */
  m2 = r2[1]/r1[1]; m3 = r3[1]/r1[1];
  r2[2] -= m2 * r1[2]; r3[2] -= m3 * r1[2];
  r2[3] -= m2 * r1[3]; r3[3] -= m3 * r1[3];
  s = r1[4]; if (0.0f != s) { r2[4] -= m2 * s; r3[4] -= m3 * s; }
  s = r1[5]; if (0.0f != s) { r2[5] -= m2 * s; r3[5] -= m3 * s; }
  s = r1[6]; if (0.0f != s) { r2[6] -= m2 * s; r3[6] -= m3 * s; }
  s = r1[7]; if (0.0f != s) { r2[7] -= m2 * s; r3[7] -= m3 * s; }

  /* Choose pivot - or die */
  if (fabs(r3 [2]) > fabs(r2 [2])) SWAP_ROWS(r3, r2);
  if (0.0f == r2[2])  return false;

  /* Eliminate third variable */
  m3 = r3[2]/r2[2];
  r3[3] -= m3 * r2[3], r3[4] -= m3 * r2[4],
  r3[5] -= m3 * r2[5], r3[6] -= m3 * r2[6],
  r3[7] -= m3 * r2[7];

  /* Last check */
  if (r3[3] == 0.0f)
    return false;

  /* Now back substitute row 3 */
  s = 1.0f/r3[3];
  r3[4] *= s; r3[5] *= s; r3[6] *= s; r3[7] *= s;

  /* Now back substitute row 2 */
  m2 = r2[3];
  s  = 1.0f/r2[2];
  r2[4] = s * (r2[4] - r3[4] * m2), r2[5] = s * (r2[5] - r3[5] * m2),
  r2[6] = s * (r2[6] - r3[6] * m2), r2[7] = s * (r2[7] - r3[7] * m2);
  m1 = r1[3];
  r1[4] -= r3[4] * m1, r1[5] -= r3[5] * m1,
  r1[6] -= r3[6] * m1, r1[7] -= r3[7] * m1;
  m0 = r0[3];
  r0[4] -= r3[4] * m0, r0[5] -= r3[5] * m0,
  r0[6] -= r3[6] * m0, r0[7] -= r3[7] * m0;

  /* Now back substitute row 1 */
  m1 = r1[2];
  s  = 1.0f/r1[1];
  r1[4] = s * (r1[4] - r2[4] * m1), r1[5] = s * (r1[5] - r2[5] * m1),
  r1[6] = s * (r1[6] - r2[6] * m1), r1[7] = s * (r1[7] - r2[7] * m1);
  m0 = r0[2];
  r0[4] -= r2[4] * m0, r0[5] -= r2[5] * m0,
  r0[6] -= r2[6] * m0, r0[7] -= r2[7] * m0;


  /* Now back substitute row 0 */
  m0 = r0[1];
  s  = 1.0f/r0[0];
  r0[4] = s * (r0[4] - r1[4] * m0), r0[5] = s * (r0[5] - r1[5] * m0),
  r0[6] = s * (r0[6] - r1[6] * m0), r0[7] = s * (r0[7] - r1[7] * m0);

  M(0,0) = r0[4]; M(0,1) = r0[5],
  M(0,2) = r0[6]; M(0,3) = r0[7],
  M(1,0) = r1[4]; M(1,1) = r1[5],
  M(1,2) = r1[6]; M(1,3) = r1[7],
  M(2,0) = r2[4]; M(2,1) = r2[5],
  M(2,2) = r2[6]; M(2,3) = r2[7],
  M(3,0) = r3[4]; M(3,1) = r3[5],
  M(3,2) = r3[6]; M(3,3) = r3[7];

  return true;
}


/*====================================================================*/
/*================================================== Plane operations */
/*====================================================================*/

/* Rotate the plane from view coords into world coords. I'm pretty sure
** this is not a slick way to do this :)
** Taken from a Gamasutra article by Thatcher Ulrich.
*/

void Plane::TransformInv (const Matrix44 &mat)
{
  Vector3 dir = Normal; dir.Scale (-D);
  Vector3 pt = mat.Transform (dir);

  Normal = mat.Rotate (Normal);
  D = - (Normal * pt);
}

void Plane::Transform (const Matrix44 &mat)
{
  Matrix44 m = mat;
  m.Invert();

  TransformInv (m);
}

Plane
Plane::GetTriPlane (const Vector3& A, const Vector3& B, const Vector3& C)
{
  Vector3 normal = Vector3::ComputeNormal (A, B, C);

  /* Ax + By + Cz + D = 0
  ** => D = - (Ax + By + Cz)
  */
  scalar D = - (normal.X * A.X +
                normal.Y * B.Y +
                normal.Z * C.Z);

  return Plane (normal, D);
}

/*====================================================================*/
/*================================================ Frustum operations */
/*====================================================================*/

Frustum::Frustum (scalar near_, scalar far_, scalar hfov, scalar vfov)
{
  /* Near & Far */
  m_Planes [0] = Plane (0, 0, 1, near_);
  m_Planes [1] = Plane (0, 0, -1, -far_);

  /* Left & Right*/
  const float hrad = Math::DegToRad(hfov / 2.f);
  const float hs = sinf(hrad);
  const float hc = cosf(hrad);
  m_Planes [2] = Plane (-hc, 0, hs, 0);
  m_Planes [3] = Plane ( hc, 0, hs, 0);

  /* Top & Bottom*/
  const float vrad = Math::DegToRad(vfov / 2.f);
  const float vs = sinf(vrad);
  const float vc = cosf(vrad);
  m_Planes [4] = Plane (0,  vc, vs, 0);
  m_Planes [5] = Plane (0, -vc, vs, 0);

  ComputeVCode();
}

Frustum::Frustum (Plane *planes)
{
  if (!planes)
    return;

  for (int i = 0; i < 6; i++)
    m_Planes[i] = planes[i];

  ComputeVCode();
}

void Frustum::Transform  (const Matrix44 &m)
{
  Matrix44 inv = m;
  int i = 0;

  inv.Invert ();
  m_Planes[i].TransformInv (inv);
  ComputeVCode();
}

void Frustum::ComputeVCode ()
{
   for (int i = 0; i < 6; i++)
   {
      m_VertexCode[i] = 0;
      
      if (m_Planes[i].Normal.X < 0)
	 m_VertexCode[i] |= 4;
      if (m_Planes[i].Normal.Y < 0)
	 m_VertexCode[i] |= 2;
      if (m_Planes[i].Normal.Z < 0)
	 m_VertexCode[i] |= 1;
    }
}

void Frustum::Dump ()
{
  for (int i = 0; i < 6; i++)
  {
    Sys()->Log ("Plane %d: (%f, %f, %f), %f\n",
                      i,
                      m_Planes[i].Normal.X,
                      m_Planes[i].Normal.Y,
                      m_Planes[i].Normal.Z,
                      m_Planes[i].D);
  }
}

int Frustum::Outcode (const Vector3 &point) const
{
  int outcode = 0, i, bit = 1;

  for (i = 0; i < 6; i++, bit <<= 1)
  {
    if (point * m_Planes[i].Normal + m_Planes[i].D < 0.0f)
      outcode |= bit;
  }

  return outcode;
}


Visibility
Frustum::GetVisibility (const BBox &box) const
{
#if 0
    bool intersect = false;
    Vector3 p,n;
 
    for (int i = 0; i < 6; i++)
    {
        p = box.m_Min;
        n = box.m_Max;
 
        if (m_VertexCode[i] & 4)
	{
            p.X = box.m_Max.X;
            n.X = box.m_Min.X;
        }

        if (m_VertexCode[i] & 2)
	{
            p.Y = box.m_Max.Y;
            n.Y = box.m_Min.Y;
        }

        if (m_VertexCode[i] & 1)
	{
            p.Z = box.m_Max.Z;
            n.Z = box.m_Min.Z;
        }
 
        if (n * m_Planes[i].Normal + m_Planes[i].D < 0)
        {
            return OUTSIDE;
        }

        if (p * m_Planes[i].Normal + m_Planes[i].D < 0)
        {
            intersect = true;
        }
    }

    if (intersect)
       return SOME_CLIP;
   
    return INSIDE;

#else
   int i, j;
   static Vector3 boxv[8];
   
   for (i = 0; i < 8; i++)
   {
      Vector3 &v = boxv[i];
      
      if (i & 1) v.X = box.m_Max.X; else v.X = box.m_Min.X;
      if (i & 2) v.Y = box.m_Max.Y; else v.Y = box.m_Min.Y;
      if (i & 4) v.Z = box.m_Max.Z; else v.Z = box.m_Min.Z;
   }
   
   Visibility returncode = INSIDE;
   for (i = 0; i < 6; i++)
   {
      Visibility curcode = OUTSIDE;

      /*
       * Classify first vertice. If it's INSIDE, one OUTSIDE vertex will stop
       * the loop. If it's OUTSIDE, one INSIDE vert will stop the loop..
       * -> 2 different classifications are enough to say that a plane clips a
       * part of the box.
       */
      if (boxv[0] * m_Planes[i].Normal + m_Planes[i].D > 0.0)
	 curcode = INSIDE;
	 
      for (j = 1; j < 8; j++)
      {
	 bool inside = boxv[j] * m_Planes[i].Normal + m_Planes[i].D > 0.0;

	 if (inside == true && curcode == OUTSIDE
	     || inside == false && curcode == INSIDE)
	 {
	    curcode = SOME_CLIP;
	    break;
	 }
      }

      // If all the points are on the "bad" side of a plane, then the box is
      // outside the frustum.
      if (curcode == OUTSIDE)
	 return OUTSIDE;

      // If only one vertex is out, the box cant be completely in the frustum
      // We should continue looping because the box can still be outside;
      if (curcode == SOME_CLIP)
	 returncode = SOME_CLIP;
   }

   return returncode;
#endif
}

/*====================================================================*/
/*================================================== Quaternion stuff */
/*====================================================================*/

Quaternion::Quaternion (scalar angle, const Vector3 &axis)
{
  Vector3 v = axis.GetUnitVector();

  const float rad = Math::DegToRad(angle / 2.f);
  const float sinHalfAngle = sinf(rad);

  X = v.X * sinHalfAngle;
  Y = v.Y * sinHalfAngle;
  Z = v.Z * sinHalfAngle;
  W = -cosf(rad);
}

Quaternion::Quaternion (scalar y, scalar p, scalar r)
{
	const float ry = Math::DegToRad(y/2.f);
	const float sy = sinf(ry);
	const float cy = cosf(ry);

	const float rp = Math::DegToRad(p/2.f);
	const float sp = sinf(rp);
	const float cp = cosf(rp);

	const float rr = Math::DegToRad(r/2.f);
	const float sr = sinf(rr);
	const float cr = cosf(rr);

  X = sr * cp * cy - cr * sp * sy;
  Y = cr * sp * cy + sr * cp * sy;
  Z = cr * cp * sy - sr * sp * cy;
  W = -(cr * cp * cy + sr * sp * sy);
}

void Quaternion::Zero ()
{
  W = 1.0f;
  X = 0.0f;
  Y = 0.0f;
  Z = 0.0f;
}

void  Quaternion::Normalize ()
{
  scalar factor = X * X + Y * Y + Z * Z + W * W;

  if (factor != 0.0f)
  {
    scalar scaleBy = Math::InvSqrt (factor);

    X *= scaleBy;
    Y *= scaleBy;
    Z *= scaleBy;
    W *= scaleBy;
  }
}

const Quaternion& 
Quaternion::operator*= (const Quaternion &q)
{
	const scalar a = (W + X) * (q.W + q.X);
	const scalar b = (Z - Y) * (q.Y - q.Z);
	const scalar c = (W - X) * (q.Y + q.Z); 
	const scalar d = (Y + Z) * (q.W - q.X);
	const scalar e = (X + Z) * (q.X + q.Y);
	const scalar f = (X - Z) * (q.X - q.Y);
	const scalar g = (W + Y) * (q.W - q.Z);
	const scalar h = (W - Y) * (q.W + q.Z);

	W = b - (e + f - g - h) * 0.5f;
	X = a - (e + f + g + h) * 0.5f; 
	Y = c + (e - f + g - h) * 0.5f; 
	Z = d + (e - f - g + h) * 0.5f;
	return *this;
}

Vector3 
Quaternion::Transform(const Vector3 &vector) const
{
	const Quaternion& q = *this;
	Quaternion p(0.f, vector.X, vector.Y, vector.Z);
	Quaternion result = Conjugate(q) * p * q;
	
	return Vector3(result.X, result.Y, result.Z);
}

Matrix44 Quaternion::GetInvMatrix () const
{
  Matrix44 R;
  Quaternion quat = *this;

  quat.Normalize ();
  scalar w = quat.X;
  scalar x = quat.Y;
  scalar y = quat.Z;
  scalar z = quat.W;

  scalar xx = x * x;
  scalar yy = y * y;
  scalar zz = z * z;

  R.M (0, 0) = -(1.0f - 2.0f *  (yy + zz));
  R.M (0, 1) = -(2.0f *  (x * y + w * z));
  R.M (0, 2) = -(2.0f *  (x * z - w * y));
  R.M (0, 3) = 0.0f;

  R.M (1, 0) = 2.0f *  (x * y - w * z);
  R.M (1, 1) = 1.0f - 2.0f *  (xx + zz);
  R.M (1, 2) = 2.0f *  (y * z + w * x);
  R.M (1, 3) = 0.0f;

  R.M (2, 0) = 2.0f *  (x * z + w * y);
  R.M (2, 1) = 2.0f *  (y * z - w * x);
  R.M (2, 2) = 1.0f - 2.0f *  (xx + yy);
  R.M (2, 3) = 0.0f;

  R.M (3, 0) = 0.0f;
  R.M (3, 1) = 0.0f;
  R.M (3, 2) = 0.0f;
  R.M (3, 3) = 1.0f;

  return R;
}

Matrix44  Quaternion::GetMatrix () const
{
  Matrix44 R;
  Quaternion q = *this;
  q.Normalize ();

  scalar two_xx = q.X * (q.X + q.X);
  scalar two_xy = q.X * (q.Y + q.Y);
  scalar two_xz = q.X * (q.Z + q.Z);

  scalar two_wx = q.W * (q.X + q.X);
  scalar two_wy = q.W * (q.Y + q.Y);
  scalar two_wz = q.W * (q.Z + q.Z);

  scalar two_yy = q.Y * (q.Y + q.Y);
  scalar two_yz = q.Y * (q.Z + q.Z);

  scalar two_zz = q.Z * (q.Z + q.Z) ;

  R.M (0, 0) = 1.0f-(two_yy+two_zz);
  R.M (0, 1) = two_xy-two_wz;
  R.M (0, 2) = two_xz+two_wy;
  R.M (0, 3) = 0.0f;

  R.M (1, 0) = two_xy+two_wz;
  R.M (1, 1) = 1.0f-(two_xx+two_zz);
  R.M (1, 2) = two_yz-two_wx;
  R.M (1, 3) = 0.0f;

  R.M (2, 0) = two_xz-two_wy;
  R.M (2, 1) = two_yz+two_wx;
  R.M (2, 2) = 1.0f-(two_xx+two_yy);
  R.M (2, 3) = 0.0f;

  R.M (3, 0) = 0.0f;
  R.M (3, 1) = 0.0f;
  R.M (3, 2) = 0.0f;
  R.M (3, 3) = 1.0f;

  return R;
}

Vector3 Quaternion::GetDirVector () const
{
  Vector3 R;
  Quaternion quat = *this;

  quat.Normalize ();
  scalar w = quat.X;
  scalar x = quat.Y;
  scalar y = quat.Z;
  scalar z = quat.W;

  R.X = 2.0f *  (x * z - w * y);
  R.Y = 2.0f *  (y * z + w * x);
  R.Z = 1.0f - 2.0f *  (x * x + y * y);

  return R;
}

void Quaternion::GetAxisAngle (Vector3 *axis, scalar *angle) const
{
#define DEF_EPSILON 0.00000001f

  scalar lenOfVector2 = Y * Y + Z * Z + W * W;

  if (axis == NULL || angle == NULL)
    return;

  if (lenOfVector2 < DEF_EPSILON)
  {
    axis->X = 1.0f;
    axis->Y = 0.0f;
    axis->Z = 0.0f;
    *angle = 0.0f;
  }
  else
  {
    const float invLen = Math::InvSqrt (lenOfVector2);
    axis->X = Y * invLen;
    axis->Y = Z * invLen;
    axis->Z = W * invLen;
    *angle = 2.0f * Math::RadToDeg(acosf(X));
  }
#undef DEF_EPSILON
}

void
Quaternion::GetEuler (scalar *y, scalar *p, scalar *r)
{
   assert (y != NULL);
   assert (p != NULL);
   assert (r != NULL);

   scalar matrix[3][3];
   scalar cx,sx;
   scalar cz,sz;
   scalar euler[3];
   
   // CONVERT QUATERNION TO MATRIX - I DON'T REALLY NEED ALL OF IT
   
   matrix[0][0] = (1.f - (2.f * Y * Y) - (2.f * Z * Z));
   
   matrix[1][0] = (2.f * X * Y) + (2.f * W * Z);
   
   matrix[2][0] = (2.f * X * Z) - (2.f * W * Y);
   matrix[2][1] = (2.f * Y * Z) + (2.f * W * X);
   matrix[2][2] = 1.f - (2.f * X * X) - (2.f * Y * Y);
   
   const float sy = matrix[2][0];
   const float cy = sqrtf(1.f - (sy * sy));
   const float yr = atan2f(sy,cy);
   euler[1] = Math::RadToDeg (yr);
 
   // AVOID DIVIDE BY ZERO ERROR ONLY WHERE Y= +-90 or +-270
   // NOT CHECKING cy BECAUSE OF PRECISION ERRORS
   if (sy != 1.f && sy != -1.f)
   {
      cx = matrix[2][2] / cy;
      sx = matrix[2][1] / cy;
      euler[0] = Math::RadToDeg(atan2f(sx,cx));
      
      cz = matrix[0][0] / cy;
      sz = matrix[1][0] / cy;
      euler[2] = Math::RadToDeg(atan2f(sz,cz));
   }
   else
   {
      // SINCE Cos(Y) IS 0, I AM SCREWED.  ADOPT THE STANDARD Z = 0
      // I THINK THERE IS A WAY TO FIX THIS BUT I AM NOT SURE.  EULERS SUCK
      // NEED SOME MORE OF THE MATRIX TERMS NOW 
      
      matrix[1][1] = 1.f - (2.f * X * X) - (2.f * Z * Z);
      matrix[1][2] = (2.f * Y * Z) - (2.f * W * X);
      
      cx =  matrix[1][1];
      sx = -matrix[1][2];
      euler[0] = Math::RadToDeg(atan2f(sx,cx));
      
      cz = 1.0;
      sz = 0.0;
      euler[2] = Math::RadToDeg(atan2f(sz,cz));
   }     
   
   *y = euler[0];
   *p = euler[1];
   *r = euler[2];
}

#define DELTA 0.00001f
Quaternion Quaternion::Slerp (const Quaternion &from, const Quaternion &to, scalar t)
{
  Quaternion R;

  scalar to1[4];
  scalar omega, sinom, scale0, scale1;

  // calc cosine
  scalar cosom =  from.X * to.X 
                + from.Y * to.Y 
        	    + from.Z * to.Z 
			    + from.W * to.W;

  // adjust signs (if necessary)
  if (cosom < 0.0)
  {
    cosom = -cosom;
    to1[0] = - to.X;
    to1[1] = - to.Y;
    to1[2] = - to.Z;
    to1[3] = - to.W;
  }
  else
  {
    to1[0] = to.X;
    to1[1] = to.Y;
    to1[2] = to.Z;
    to1[3] = to.W;
  }

  // calculate coefficients
  if ((1.f-cosom) > DELTA)
  {
    // standard case (slerp)
    omega = acosf(cosom);
    sinom = sinf(omega);
    scale0 = sinf((1.0f - t) * omega) / sinom;
    scale1 = sinf(t * omega) / sinom;
  }
  else
  {
    // "from" and "to" quaternions are very close
    //  ... so we can do a linear interpolation
    scale0 = 1.0f - t;
    scale1 = t;
  }

  // calculate final values
  R.X = scale0 * from.X + scale1 * to1[0];
  R.Y = scale0 * from.Y + scale1 * to1[1];
  R.Z = scale0 * from.Z + scale1 * to1[2];
  R.W = scale0 * from.W + scale1 * to1[3];

  return R;


}
#undef DELTA

/*====================================================================*/
/*==================================================== Bounding Boxes */
/*====================================================================*/

BBox::BBox ()
{MakeEmpty();}

/// Make this bounding box empty.
void
BBox::MakeEmpty()
{
  m_Min = Vector3 ( 9999999.f,  9999999.f,  9999999.f);
  m_Max = Vector3 (-9999999.f, -9999999.f, -9999999.f);
}

/// Test wether this bounding box is empty.
bool
BBox::Empty() const
{
   Vector3 v = m_Max-m_Min;
   if (v.X <= .0 || v.Y <= .0 || v.Z <= .0) return true;
   return false;
}


bool BBox::IsInside (const Vector3 &point) const
{
  return (point.X >= m_Min.X && point.X <= m_Max.X &&
          point.Y >= m_Min.Y && point.Y <= m_Max.Y &&
          point.Z >= m_Min.Z && point.Z <= m_Max.Z);
}

void BBox::AddPoint (const Vector3 &point)
{
  if (point.X > m_Max.X) m_Max.X = point.X;
  if (point.X < m_Min.X) m_Min.X = point.X;

  if (point.Y > m_Max.Y) m_Max.Y = point.Y;
  if (point.Y < m_Min.Y) m_Min.Y = point.Y;

  if (point.Z > m_Max.Z) m_Max.Z = point.Z;
  if (point.Z < m_Min.Z) m_Min.Z = point.Z;
}

/// Find the smaller box containing all the transformed points (8) of this
/// bounding box.
void
BBox::Transform (const Ark::Matrix44 &m)
{
  Vector3 v[8];
  GetCorners (v);

  MakeEmpty();
  for (int i = 0; i < 8; i++)
    AddPoint(m.Transform(v[i]));
}

void
BBox::GetCorners (Ark::Vector3 *v) const
{
  assert (v != NULL);

  v[0] = Vector3 (m_Min.X, m_Max.Y, m_Min.Z);
  v[1] = Vector3 (m_Min.X, m_Min.Y, m_Min.Z);
  v[2] = Vector3 (m_Max.X, m_Max.Y, m_Min.Z);
  v[3] = Vector3 (m_Max.X, m_Min.Y, m_Min.Z);
  v[4] = Vector3 (m_Max.X, m_Max.Y, m_Max.Z);
  v[5] = Vector3 (m_Max.X, m_Min.Y, m_Max.Z);
  v[6] = Vector3 (m_Min.X, m_Max.Y, m_Max.Z);
  v[7] = Vector3 (m_Min.X, m_Min.Y, m_Max.Z);
}

bool
BBox::Overlap (const BBox &b) const
{
   for (int i = 0; i < 3; i++)
   {
      if ((&m_Min.X)[i] > (&b.m_Max.X)[i] ||
	  (&m_Max.X)[i] < (&b.m_Min.X)[i])
	 return false;
   }

   return true;
}

/* namespace Ark */
}


