/* $Id: ArkModel.h,v 1.42 2003/03/19 09:25:05 zongo Exp $
**
** Ark - Libraries, Tools & Programs for MMORPG developpements.
** Copyright (C) 1999-2002 The Contributors of the Ark Project
** Please see the file "AUTHORS" for a list of contributors
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#ifndef ARK_MODEL_H
#define ARK_MODEL_H

#include <Ark/ArkSkeleton.h>
#include <Ark/ArkMaterial.h>
#include <Ark/ArkPrimBlock.h>
#include <Ark/ArkVBuffer.h>
#include <Ark/ArkSkin.h>


namespace Ark
{
    class Model;
    typedef Ptr<Model> ModelPtr;

   /**
    * An attachment point allows to bind an model to another model.
    * For example you can bind a sword to a player. A model may have
    * several attachment points.
    */
   struct ARK_DLL_API Attachment
   {	 
	 /**
	  * Bone to which this attachment bounds (ie we'll transform
	  * m_Origin using this bone matrix)
	  */
	 int m_Bone;
	 
	 /// Attachment point
	 Vector3 m_Origin;
	 
	 /// Up, left and far vectors.
	 Vector3 m_Vectors[3];
   };

   class ARK_DLL_API Mesh
   {
      public:
	 int m_Material;
	 BlockList m_Blocks;
	 
      public:
	 /// Optimize this mesh by creating triangle strips / fans.
	 void Optimize ();
   };

   class ARK_DLL_API SubModel
   {
	 friend class ModelBuilder;
	 ModelBuilder *m_Builder;

      public:
	 /**
	  * Name of the submodel, given to Hit event handlers so they can
	  * react differently depending on the part of the body which has been
	  * hit.
	  */
	 String m_Name;

	 /// Vertex buffer
	 VertexBuffer m_VB;
	 VertexBuffer* m_SkinnedVB;

	 // Typedef for bone bindings
	 typedef std::vector< uchar > BoneBindingList;
	 typedef BoneBindingList::iterator BoneBindingLI;

	 /// Bone bindings
	 BoneBindingList m_BoneBindings;

	 // Typedefs for a list of meshes
	 typedef std::vector<Mesh> MeshList;
	 typedef MeshList::iterator MeshLI;
	 
	 /// List of meshes for the model
	 MeshList  m_Meshes;

	 /// Axis-aligned bounding box
	 BBox         m_BBox;

      public:
	 SubModel ();
	 ~SubModel ();

      public:
	 /// Return the current model builder (a new one might be allocated).
	 ModelBuilder *Builder ();

	 /**
	  * Call the Build() method of the model builder, then destroy the
	  * builder. This creates the vertex buffers, the meshes, and so on..
	  */
	 bool Build ();

	 /**
	  * Optimize the model, by stripifying the meshes, and reorganizing
	  * the vertex data in order to reduce cache loss.
	  */
	 void Optimize();

	 /**
	  * Transform all points in this submodel by the \c transform
	  * matrix. Rotate normals using the same matrix.
	  */
	 void Transform (const Matrix44 &transform, bool scale = false);

	 void SkinTransform (const Matrix44* boneMatrices);
   };
   
   class ColModel;

   /**
    * A complex 3D model made of several materials, meshes, eventually a 
    * skeleton and some attachment points. Models are "resources" and as
    * such they can be shared by some entities : to keep track of a model
    * state (position, animation sequence, etc) you will likely use the
    * ModelState class.
    */
   class ARK_DLL_API Model : public Object
   {
     private:
	 friend class ModelState;
	 scalar m_Scale;
	 Model *m_SharedCDModel;
	 String m_SharedCDModelName;

      public:
	 /// Submodels typedef
	 typedef std::vector<SubModel> SubModelList;
	 typedef SubModelList::iterator SubModelLI;
	 
	 /// A list of submodels.
	 SubModelList m_SubModels;

	 /// Attachment typedef
	 typedef std::map<String,Attachment> AttachmentMap;
	 typedef AttachmentMap::iterator AttachmentMI;

	 /// Attachment points.
	 AttachmentMap m_Attachments;

	 /**
	  * Skeleton bound to this model (it can be nil, meaning
	  * the model isn't animated).
	  */
	 Skeleton *m_Skeleton;

	 /**
	  * Default skin for this model. It is used by the Render() function
	  * if the model state has no skin information.
	  */
	 Skin m_Skin;

	 /**
	  * Bounding box of the model. With animated models it depends on
	  * the current sequence, that's why you shouldn't access this member
	  * directly, but instead use the ModelState::ExtractBbox function.
	  */
	 BBox m_BBox; 

	 /// Collision model. It will be nil, if there's no collision system.
	 ColModel *m_CDModel;

	 /// This is set to true if the model has been optimised.
	 bool m_Optimized;
	 
     public:
	 /// Initialises an empty model, with the given name.
	 Model (const String &name);

	 /**
	  * Destroy the model data (meshes) and unreference all ressources
	  * previously used by the model : skeleton, materials...
	  */
	 ~Model ();

	 /// Render the model using the given renderer.
	 void Render (Ark::Renderer &renderer, const ModelState &state);

	 /**
	  * This transforms the model by the given matrix. Warning : all
	  * instantiations of the model will be modified : if you only want
	  * to change one of them, please have a look at the ModelState class.
	  */
	 void Transform (const Matrix44 &mat,
			 bool scale = false);

	 /**
	  * Scale the whole model in an uniform way : submodels, skeleton,
	  * attachment points, controllers, bbox and so on are transformed..
	  */
	 void Scale (float scale_factor);

	 /**
	  * Scale the model so that the bounding box has the good \c height ; 
	  * this is especially useful when there are several models, each
	  * of them having a different scale factor (ie different units, for
	  * example).
	  */
	 void SetRealHeight (scalar height)
	 {
	    Scale (height / (m_BBox.m_Max.Y - m_BBox.m_Min.Y));
	 }

	 /** Write all the data contained in this model in an file. This
	  * file format is used as the preferred format for game data since it
	  * supports all operations the engine supports...
	  */
	 bool Write (WriteStream &stream);

	 /** Fill this model with data read from a native file. */
	 bool Read (const String& name, Stream &stream, Cache *cache);

	 /**
	  * This function creates the collision model after the model has 
	  * been completely loaded. 
	  */
	 virtual bool PostLoad (Cache* cache);

     public:
	 /**
	  * Compute the bounding box for this model from the bounding boxes
	  * of all submodels.
	  * \arg center : if set to true then the model is centered.
	  */
	 void SetupBBox (bool center);

	 /** Find the controller whose identifier is \c id */
	 BoneController *GetController (int id);

	 /// Returns a string describing the object
	 String Dump (bool long_version);

	 /**
	  * Optimize this model. This is a quite compute-intensive operation,
	  * so models would better be optimized statically.
	  */
	 void Optimize();
   };


} // namespace Ark

#endif
