/* $Id: ArkModelBuild.cpp,v 1.10 2002/01/29 15:51:49 nekeme Exp $
** 
** Ark - Libraries, Tools & Programs for MMORPG developpements.
** Copyright (C) 1999-2000 The Contributors of the Ark Project
** Please see the file "AUTHORS" for a list of contributors
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <Ark/ArkModelBuild.h>
#include <algorithm>
#include <set>

namespace Ark
{
   static int g_CurrentFormat = 0;

/////////////////////////////////////////////////////////////////////////////

   Vertex::Vertex ()
   {
      m_Bone = 0;
      m_Color.R = m_Color.G = m_Color.B = 255;
      m_TexCoord = Vector2();
      m_Coord = Vector3();
   }

   bool operator < (const Vertex &x, const Vertex &y)
   {
      if (g_CurrentFormat & VertexBuffer::VB_HAS_COORD 
	 && x.m_Coord != y.m_Coord)
      {  
	 return x.m_Coord < y.m_Coord;
      }

      if (g_CurrentFormat & VertexBuffer::VB_HAS_UV0
	 && x.m_TexCoord != y.m_TexCoord)
      {  
	 return x.m_TexCoord < y.m_TexCoord;
      }

      if (g_CurrentFormat & VertexBuffer::VB_HAS_NORMAL
	 && x.m_Normal != y.m_Normal)
      {  
	 return x.m_Normal < y.m_Normal;
      }

      if (g_CurrentFormat & VertexBuffer::VB_HAS_COLOR)
      {
	 if (x.m_Color.R != y.m_Color.R)
	    return x.m_Color.R < x.m_Color.R;

	 if (x.m_Color.G != y.m_Color.G)
	    return x.m_Color.G < x.m_Color.G;

	 if (x.m_Color.B != y.m_Color.B)
	    return x.m_Color.B < x.m_Color.B;
      }

      return x.m_Bone < y.m_Bone;
   }

////////////////////////////////////////////////////////////////////////////

   typedef std::less< Vertex > VertexSort;
   typedef std::set<Vertex, VertexSort> VertexBMap;
   class VBMap : public VertexBMap {};

////////////////////////////////////////////////////////////////////////////

   /// initialization
   ModelBuilder::ModelBuilder (SubModel *m) :
      m_Model (m), m_Map (new VBMap)
   {}

   ModelBuilder::~ModelBuilder ()
   {
      delete m_Map;
   }

   int 
   ModelBuilder::FindVertex (const Vertex &v)
   {
      // Reset global VB format, because of the comparison function
      // (see above)
      g_CurrentFormat = m_Model->m_VB.Format();

      if (m_ComputeNormal == true)
	 g_CurrentFormat &= ~VertexBuffer::VB_HAS_NORMAL;
     
      VBMap::iterator it = m_Map->find (v);

      if (it != m_Map->end())
	 return it->m_VBId;

      int vbid = m_Model->m_VB.Size();

      /// Insert in VB
      VertexBuffer &vb = m_Model->m_VB;
      vb.Resize(vbid + 1);
      vb.Coord (vbid) = v.m_Coord;

      if (m_ComputeNormal)
	 vb.Normal (vbid) = Vector3(); // Set later
      else
	 vb.Normal (vbid) = v.m_Normal;

      if (vb.Format() & VertexBuffer::VB_HAS_UV0)
	 vb.UV0 (vbid) = v.m_TexCoord;
      if (vb.Format() & VertexBuffer::VB_HAS_COLOR)
	 vb.Color4 (vbid) = v.m_Color;

      /// Bone bindings..
      m_Model->m_BoneBindings.push_back( v.m_Bone );

      /// Insert in map
      Vertex vtx = v;
      vtx.m_VBId = vbid;
      m_Map->insert (vtx);

      return vbid;
   }
   
   /*
    * Sets the format of the vertex buffer. This should be called
    * before any other function is called.
    */
   bool
   ModelBuilder::SetFormat (int format)
   {
      if (format & VertexBuffer::VB_HAS_NORMAL)
	 m_ComputeNormal = false;
      else
	 m_ComputeNormal = true;

      m_Model->m_VB.SetFormat(VertexBuffer::VertexFormat(format | VertexBuffer::VB_HAS_NORMAL));
      return true;
   }
   
   bool
   ModelBuilder::SetVertices (Vertex *vertices, size_t n)
   {
      m_CMap.resize (n);

      /** Map current vertex indice to vertex buffer indice */
      for (size_t i = 0; i < n; i++)
	 m_CMap[i] = FindVertex (vertices[i]);
  
      return true;
   }


   /// Add a triangle to the current model
   bool
   ModelBuilder::AddTriangle (int material,
			      const Vertex &a,
			      const Vertex &b,
			      const Vertex &c)
   {
       Vertex v[3] = {a, b, c};
       PrimitiveBlock pb;

       pb.SetType (PRIM_TRIANGLES);
       pb.Add( 0 );
       pb.Add( 1 );
       pb.Add( 2 );
       
       return SetVertices (v, 3)
	  && AddBlock (material, pb);
   }


   /**
    * Add a primitive block to the current model ; put it in the
    * mesh corresponding to material \c mat (create one if it
    * does not currently exist).
    */
   bool
   ModelBuilder::AddBlock (int mat, const PrimitiveBlock &block)
   {
      Mesh* mesh = 0;
      size_t i;

      // Find mesh in current meshes.
      SubModel::MeshList& meshes = m_Model->m_Meshes;
      for (SubModel::MeshLI m=meshes.begin(); m!=meshes.end(); ++m)
      {
	 if (m->m_Material == mat)
	 {
	    mesh = &*m;
	    break;
	 }
      }

      // Not found. insert a new mesh
      if (! mesh)
      {
	 meshes.push_back(Mesh());
	 mesh = &meshes.back();
	 mesh->m_Material = mat;
      }
     
      // Create a new primitive block & convert indices
      PrimitiveBlock pb;
      pb.SetType (block.Type());
      pb.Resize (block.Size());

      for (i = 0; i < block.Size(); i++)
		  pb[i] = m_CMap [block[i]];

      mesh->m_Blocks.push_back(pb);

      if (!m_ComputeNormal)
	 return true;

      VertexBuffer &vb = m_Model->m_VB;

      // Normal computation.
      switch (pb.Type())
      {
	 case PRIM_TRIANGLES:
	    for (i = 2; i < pb.Size(); i += 3)
	    {
	       Vector3 normal = Vector3::ComputeNormal
		  (vb.Coord(pb[i-2]), 
		   vb.Coord(pb[i-1]),
		   vb.Coord(pb[i  ]));

	       vb.Normal(pb[i-2]) += normal;
	       vb.Normal(pb[i-1]) += normal;
	       vb.Normal(pb[i  ]) += normal;
	    }
	    break;

	 case PRIM_TRIANGLE_FAN:
	 {
	    Vector3 &fst = vb.Coord (pb[0]);
	    for (i = 2; i < pb.Size(); i += 2)
	    {
	       Vector3 normal = Vector3::ComputeNormal
		  (fst,
		   vb.Coord(pb[i-1]), 
		   vb.Coord(pb[i]));

	       vb.Normal(pb[i-1]) += normal;
	       vb.Normal(pb[i]) += normal;
	       vb.Normal(pb[0]) += normal;
	    }
	 }
	 break;

	 case PRIM_TRIANGLE_STRIP:
	    for (i = 2; i < pb.Size(); i++)
	    {
	       // the order of vertices in triangles is not the same
	       // when i is a multiple of two and when it's not..

	       Vector3 normal;
	       if (i & 1) 
	       {
		  normal = Vector3::ComputeNormal
		     (vb.Coord(pb[i-2]),
		      vb.Coord(pb[i-1]),
		      vb.Coord(pb[i]));
	       }
	       else
	       {
		  normal = Vector3::ComputeNormal
		     (vb.Coord(pb[i-2]),
		      vb.Coord(pb[i]),
		      vb.Coord(pb[i-1]));
	       }

	       vb.Normal(pb[i-2]) += normal;
	       vb.Normal(pb[i-1]) += normal;
	       vb.Normal(pb[i  ]) += normal;
	    }
	    break;
	    
	 default:
	    break;
      }

      return true;
   }


   /// Finish to build the model.
   bool
   ModelBuilder::Build ()
   {
      const size_t sz = m_Model->m_VB.Size();

      if (m_ComputeNormal)
      {
	 // Normalize normals...
	 for (size_t i = 0; i < sz; i++)
	    m_Model->m_VB.Normal(i).Normalize();
      }

      for (size_t i = 0; i < sz; i++)
      {
	 m_Model->m_BBox.AddPoint (m_Model->m_VB.Coord(i));
      }
	 
      return true;
   }
}
