/* $Id: ArkSequence.cpp,v 1.10 2003/03/20 17:23:25 zongo Exp $
**
** Ark - Libraries, Tools & Programs for MMORPG developpements.
** Copyright (C) 1999-2000 The Contributors of the Ark Project
** Please see the file "AUTHORS" for a list of contributors
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/


#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <Ark/ArkSkeleton.h>
#include <Ark/ArkLexer.h>

namespace Ark
{
   /**
    * Create an empty sequence with the given name.
    */
   Sequence::Sequence (const String &name) :
      Object (name, V_SEQUENCE)
   {
   }
   

   /**
    * Delete the given sequence.
    */
   Sequence::~Sequence()
   {
   }

   /** Read that sequence from a file. */
   bool
   Sequence::Read (const String &filename, Stream &file)
   {
      Lexer lexer(filename, file);

      if (!lexer.CheckToken("ArkSequence")
	 || !lexer.CheckToken("{")) return false;
      if (!lexer.CheckToken("NumFrames")) return false;
      int nframes; lexer.ReadInteger(&nframes);

      if (!lexer.CheckToken("FPS")) return false;
      lexer.ReadScalar(&m_FPS);

      Skeleton skel(filename); skel.Read(lexer);
      int nbones = skel.m_Bones.size();

      m_EndTime = nframes / m_FPS;

      std::vector<int> invOrder;
      invOrder.resize(nbones);
      for (size_t i = 0; i < (size_t)nbones; ++i)
	 invOrder[skel.m_BoneOrder[i]] = i;


      m_Keyframes.resize(nframes);
      for (KeyframeList::iterator it = m_Keyframes.begin();
	   it != m_Keyframes.end(); ++it)
      {
	 it->resize(nbones);
	 size_t j;
	 for (j = 0; j < (size_t)nbones; ++j)
	 {
	    BoneFrame &kf = (*it)[invOrder[j]];
	    for (size_t i = 0; i < 6; i++)
	    {
	       if (! lexer.ReadScalar(& (&kf.m_Position.X)[i]) )
	       {
		  m_Keyframes.resize(0);
		  return false;
	       }
	    }
	 }
      }

      return lexer.CheckToken("}");
   }
   
   /**
    * Find the two frame corresponding to the given time (the one
    * before, and the one after), and give factors between 0 and
    * 1 for each of those frames.
    *
    * \return false if time is out of range
    */
   bool
   Sequence::ComputeFrameFactors (scalar time,
				  int *frame0, scalar *fact0,
				  int *frame1, scalar *fact1) const
   {
      assert (frame0 != NULL);
      assert (frame1 != NULL);
      assert (fact0 != NULL);
      assert (fact1 != NULL);

      if (time >= m_EndTime)
	 return false;

      // FPS = frame / time <=> frame = FPS * time
      scalar frame = time * m_FPS;

      *frame0 = int(frame);
      *frame1 = int(frame) + 1;

      // Compute scale factors.
      // fact0 + fact1 = 1.0

      *fact0 = (frame - scalar(*frame0));
      *fact1 = 1.0f - *fact0;

      if (*frame1 >= int(m_Keyframes.size()))
	 *frame1 = 0;

      return true;
   }

   /**
    * Return the transformation and rotation at a given time. If
    * we are between two frame, linear interpolation is done to 
    * "smooth" the movement.
    */
   void
   Sequence::GetTransformation (Bone &bone,
				const BoneFrame &f0, scalar fact0,
				const BoneFrame &f1, scalar fact1,
				const scalar *adjs,
				Vector3 *translation,
				Quaternion *rotation) const
   {
      assert (translation != NULL);
      assert (rotation != NULL);
      assert (adjs!= NULL);

//#define FIXME_SUPPORT_LEGACY_MDL
#ifdef FIXME_SUPPORT_LEGACY_MDL
      *translation = bone.m_DefPosition;
      *translation += fact0 * f0.m_Position + fact1 * f1.m_Position;
#else
      *translation = fact0 * f0.m_Position + fact1 * f1.m_Position;
#endif

      Vector3 r0 = f0.m_Rotation;
      Vector3 r1 = f1.m_Rotation;

      for (int i = 0; i < 3; i++)
      {
	 if (bone.m_BoneControllers[i+3] != -1)
	 {
	    scalar adj = adjs[bone.m_BoneControllers[i+3]];
	    (&r0.X)[2] += adj;
	    (&r1.X)[2] += adj;
	 }
      }

      // Use quaternions to interpolate angles..
      Quaternion q0 (r0.X, r0.Y, r0.Z);
      Quaternion q1 (r1.X, r1.Y, r1.Z);
      *rotation = Quat::Slerp (q0, q1, fact0);
   }

}
