/* $Id: LuaTable.h,v 1.3 2003/01/10 20:17:20 tharibo Exp $
**
** Ark - Libraries, Tools & Programs for MMORPG developpements.
** Copyright (C) 1999-2000 The Contributors of the Ark Project
** Please see the file "AUTHORS" for a list of contributors
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#ifndef LUA_TABLE_H
#define LUA_TABLE_H

extern "C"
{
#include <lua.h>
}

#include <Modules/Lua/ArkLua.h>

namespace Ark
{

  /**
   * \brief Wraps a Lua table.
   * \author tharibo tharibo@nekeme.net
   *
   * This class is used to handle tables when writing functions called from Lua
   * code. It should be destroyed at the end of the function.
   */
  class ARKLUA_DLL_API LuaTable
  {
    public:
      
      /**
       * Construct a LuaTable, the table being at the given index.
       */
      LuaTable(lua_State* L, int index = -1);

      /**
       * Destructor
       */
      virtual ~LuaTable();

      /**
       * Return the table's index on the stack
       */
      int index() const;

      /**
       * The following functions get a value of the table on the stack, given
       * the key of the value in the table.
       *
       * TODO: add other types as keys...
       * TODO: add the CFunction type.
       */

      /**
       * Read a number in the table.
       * 
       * \param key The key of the element to get in the table.
       * \return The value at the given key or 0 if the value is not a number
       * or is not a string convertible to a number.
       */
      double getNumber(double key);

      /**
       * Read a string in the table.
       *
       * \param key The key of the element to get in the table.
       * \return The value (a string or a number) at the given key or 0 if the
       * value isn't a string or a number.
       *
       * \bug In fact, it is not a bug but... Well, here is it ;-) Lua strings
       * may have embeddend \0 insided them, whereas the real string continues.
       * This function doesn't handle the problem. There should be a subclass
       * LuaStringTable that handles a table of strings, or a LuaString class
       * that return a correct string given the index of the string in the
       * stack. Feel free to write these functions if they don't exist and you
       * need them... ;-)
       */
      const char* getString(double key);

      /**
       * Read a user data in the table.
       *
       * \param key The key of the element to get in the table.
       * \return A pointer to some userdata, or 0 if the value in the table is
       * not of luatype \e userdata.
       */
      void* getUserData(double key);

      /**
       * Read a table in the table.
       *
       * \param key The key of the element to get in the table.
       * \return A pointer on a LuaTable or 0 if there's no table at the given
       * key.
       */
      LuaTable* getTable(double key);

      /**
       * The following functions add element to the table at a given key.
       * TODO: add more types...
       */
      void addElement(const char* key, double value);

      /**
       * Use this function to test if the value at the given key is nil.
       *
       * \param key The value's key in the table.
       * \return True if the value at the given key is nil.
       */
      bool isNil(double key);

      /**
       * Use this function if you think that the get*() functions of this class
       * are not efficient for your purpose. It pushes the value at the given
       * table's key on the stack. You may then use the classical lua
       * functions...
       *
       * \param key The key of the element to get in the table.
       */
      void pushElementAt(double key);

      /**
       * Constructs a LuaTable from scratch. This creates a new table on the
       * stack.
       *
       * \param L The lua state where we are going to create the table.
       */
      static LuaTable* createTable(lua_State* L);

    protected:

      /**
       * The lua state
       */
      lua_State* m_L;
      
      /**
       * The index of the table in the stack
       */
      int m_Index;
  };
}

#endif
